function [z,p,k] = cheb2np(n, rs, f3)
%CHEB2NP Chebyshev type II analog lowpass filter prototype.
%	[Z,P,K] = CHEB2NP(N,Rs) returns the zeros, poles, and gain
%	of an N-th order normalized prototype type II Chebyshev analog
%	lowpass filter with Rs decibels of ripple in the stopband.
%	Type II Chebyshev filters are maximally flat in the passband and
%	its frequency response is Rs dB at 1Hz.
%	Optionally, [Z,P,K]=CHEB2NP(N,Rs,F3), where F3 is any non-zero input
%	argument, normalizes the response to -3 dB at 1Hz.
%	
%	See also CHEB2AP, CHEB1NP, CHEB1AP, CHEBY2, and CHEB2ORD.

%	L. Shure 1-13-88
%	(c) Copyright 1988, by The MathWorks, Inc.
%	Normalization to -3dB at 1Hz by Erik Margan, 9-14-88.

if n == 1
	z=[];
	p=-1;
	k=1;
	return
end
% j = sqrt(-1);
delta = 1/sqrt(10^(.1*rs)-1);
mu = asinh(1/delta)/n;
if (rem(n,2))
	m = n - 1;
	z = j ./ cos([1:2:n-2 n+2:2:2*n-1]*pi/(2*n))';
else
	m = n;
	z = j ./ cos((1:2:2*n-1)*pi/(2*n))';
end
% Organize zeros in complex pairs:
i = [1:m/2; m:-1:m/2+1];
z = z(i(:));

p = exp(j*(pi*(1:2:2*n-1)/(2*n) + pi/2)).';
p = sinh(mu)*real(p) + j*cosh(mu)*imag(p);
p = 1 ./ p;
k = real(prod(-p)/prod(-z));

if nargin == 3 % --- Normalize H(s) to -3dB at f0=1Hz --- :
	F=abs(freqw(z,p,1));
	wr=2;
	while wr > 1.0001
		wr=1+(.707-F)/3;
		z=wr*z;
		p=wr*p;
		k=real(prod(-p)/prod(-z));
		F=abs(freqw(z,p,1));
	end
end
