function [z,p,k] = ellipnp(n, rp, rs)
%ELLIPNP Elliptic analog lowpass filter prototype.
%	[Z,P,K] = ELLIPNP(N,Rp,Rs) returns the zeros, poles, and gain
%	of an N-th order normalized prototype elliptic analog lowpass
%	filter with Rp decibels of ripple in the passband and a
%	stopband Rs decibels down.
%	
%	ELLIPAP is normalized to Rs dB at 1Hz.
%	ELLIPNP is normalized to -3 dB at 1Hz.

%	L. Shure 3-8-88
%	(c) Copyright 1988, by The MathWorks, Inc.
%	Normalized to -3 dB at 1Hz by Erik Margan, 9-14-88.

%	References:
%	  [1] T. W. Parks and C. S. Burrus, Digital Filter Design,
%	      John Wiley & Sons, 1987, chapter 7, section 7.3.7-8.

if n == 1
	z=[];
	p=-1;
	k=1;
	return
end
epsilon = sqrt(10^(0.1*rp)-1);	%rp, dB of passband ripple
k1 = epsilon/sqrt(10^(0.1*rs)-1);	%rs, dB of stopband ripple
k1p = sqrt(1-k1^2);
wp = 1;	% passband edge - normalized
if abs(1-k1p^2) < eps(1)
	krat = 0;
else
	capk1 = ellipk(1,[k1^2,k1p^2]);
	krat = n*capk1(1)/capk1(2);	% krat = K(k)/K'(k) -- need to find relevant k
end

% put krat into eps(2), a global variable, so kratio.m can access it
% try to find m, elliptic parameter, so that K(m)/K'(m) = krat -- K, complete
% elliptic integral of first kind
eps(2) = krat;
m = fmins('kratio',.5);
% get rid of fabricated global eps(2) now that it is no longer needed
eps(2) = [];

capk = ellipk(1,m);
ws = wp/sqrt(m);	% stopband edge (=> transition band is ws-wp in width)
m1 = 1 - m;
i = sqrt(-1);

% find zeros; they are purely imaginary and paired in complex conjugates
j = (1-rem(n,2)):2:n-1;
[ij,jj] = size(j);
% s is Jacobi elliptic function sn(u)
[s,c,d] = ellipj(j*capk/n,m*ones(ij,jj));
is = find(abs(s) > eps(1));
z = 1 ./(sqrt(m)*s(is));
z = i*z(:);	% make column vector
z = [z ; conj(z)];
% order the zeros for convenience later on
z = cplxpair(z);

% poles; one purely real if n is odd - the remainder are complex conjugate pairs
% put 1/epsilon, mp into global variables for calculating v0
eps(2) = 1 / epsilon;
mp = k1p^2;
eps(3) = mp;
% calculate v0, a 'fundamental' parameter for the poles related to inverse sc
% function . I.e. find r so sn(r)/cn(r) = 1/epsilon for the given parameter mp
r = fmins('vratio', ellipk(1,1-m));
% get rid of extra global values stuffed into eps(2:3)
eps([2 3]) = [];
v0 = capk*r/(n*capk1(1));
[sv,cv,dv] = ellipj(v0,1-m);
p = -(c.*d*sv*cv + i*s*dv)./(1-(d*sv).^2);
p = p(:);   % make column vector
% check to see if there is a real pole
if rem(n,2)
	ip = find(abs(imag(p)) < eps(1)*norm(p));
	[pm,pn] = size(p);
	pp = 1:pm;
	pp(ip) = [];
	p = [p ; conj(p(pp))];
else
	p = [p; conj(p)];
end
p = cplxpair(p);	% order poles for later use

% --- Normalize H(s) to -3dB for f0=1Hz --- :
F=abs(freqw(z,p,1));
wr=0;
k=1;
while abs(1-wr) > .001
	wr=(.707/F)^(1/32);
	z=wr*z;
	p=wr*p;
	F=abs(freqw(z,p,1));
	k=k+1;
	if k > 100
		disp('WARNING : Normalization won''t converge! Exiting loop.')
		wr=1;
	end
end

% gain
k = real(prod(-p)/prod(-z));
if (~rem(n,2))	% n is even order so patch gain
	k = k/sqrt((1 + epsilon^2));
end
