function h=freqw(Z,P,w,f0)
%FREQW  H=freqw(Z,P,w) returns the complex frequency response H(jw) 
%       of the system described by zeros (vector Z=[z1,z2,...,zm])
%       and poles (vector P=[p1,p2,...,pn]) in the complex plane.
%
%       H=freqw(P,s) assumes a system with poles only.
%
%	The frequency vector w is usually normalized ( w --> w/w0 ).
%	If Z and P have been denormalized ( with the expected -3dB
%	frequency at f0 ), then :
%
%	H=freqw(Z,P,w,f0)	will denormalize the response to
%				the actual frequency.
%	
%	Example :	[z,p]=cheb1np(5,0.1) ;
%			w=logspace(-1,1,401) ;
%			H=freqw(z,p,w) ;
%			M=20*log10(abs(H)) ;        % magnitude in dB
%			phi=180*ephd(angle(H))/pi ; % phase angle in degrees.
%			semilogx(w,M,w,phi/4.5) ; % rescale phi for 90 deg/div.
%
%	See PATS. See FREQS and FREQZ. See POLY and ROOTS.
%	See also RESIDUE and SYSVIEW.

%   Erik Margan, November 10th, 1988 
%   Jozef Stefan Institute, Ljubljana, Slovenia
%	Last revision :  980923

if nargin == 2
	w = P ;
	P = Z ;
	Z = [];		% A normalized system with poles only!
end

if any( real( P ) > 0 )
	disp('WARNING : This is a non-Hurwitz system!')
end

% if w is entered in real form, assume it to be purely imaginary!
% i = sqrt(-1) ;
if all( w == real( w ) )
	w = i * w ;
end

if nargin == 4
	w = 2 * pi * f0 * w ;		% denormalize to f0.
end

if isempty(Z)
	h=1 ./ pats(P,w);
else
	h=pats(Z,w)./pats(P,w);
end
