function [T,t]=maxthetm(Tz,Tm,plt,Po,v)
% MAXTHERM  THERMal time-domain analysis for Max B. - with real-time plotting.
%               A linear multi-step numerical integration method is used.
%       ------- Sytem equations : -----------------------------------
%       Pd=Po*exp(-v/T) ... Power dissipation vs. temperature
%       Pp=Cp*dT/dt     ... Power absorbed by the mass Cp
%       Ppx=(T-Tx)/Rpx  ... Power flow through thermal resistance Rpx
%       Pd=Pp+Ppx       ... Node equation of power flow at T
%       Px=Cx*dTx/dt    ... Power absorbed by the mass Cx
%       Pxz=(Tx-Tz)/Rxz ... Power flow through thermal resistance Rxz
%       Ppx=Px+Pxz      ... Node equation of power flow at Tx
%       -------------------------------------------------------------
%       Call :  [T,t]=mxt2(Tz,Tm,plt,Po,v);
%       where : Tz --> initial air temperature (273+20 [K]).
%               Tm --> inner temperature raise, after power transient 
%                      at t=0, T=T0+Tm  ( critical 143<Tm<144 [K] )
%               T  --> resulting temperatures ( T = [ Ti(:), Tx(:) ],
%                      Ti - inner temp., Tx - external case temp.)          
%               t  --> time vector, normalized to the system time-constant.
%               plt--> optional : if non-zero, plot heathing/cooling
%                      comparison is also ploted.
%               Po --> Power dissipation system amplitude constant
%               v  --> Power dissipation system exponent constant
 
%       Erik Margan, *930719*
%       Last editing *991225*

% --- define default parameters : ---
if nargin < 5
	v=1990;         % 2000, see file "XTERM.M" ;
end
if nargin < 4
	Po=68;          % 65, see file "XTERM.M" ;
end
if nargin < 3
	plt = 0 ;
end
if nargin < 2
	Tm = 144.3 ;    % [K]
end
if nargin < 1
	Tz = 273+20 ;   % [K]
end

% --- define time & init T-matrix : ---
% --- new requirement : t in hours : ---
tstart = -20 * 3600 ;           % before : -5e+4 [s]
deltat = 3600 ;                 % before :  5e+3 [s]
tend   = 140 * 3600 ;           % before :  1e+6 [s]
t = ( tstart : deltat : tend ) ;
% t=round(10*t)/10;
t = t( : ) ;
T = zeros( length(t), 1 ) ;
T = [ T, T ] ;

% --- define system constants : ---
Cp  = 350.00 ;  % [Ws/K]
% Rpx = 159.90 ;  % [K/W]
Rpx = 120.90 ;  % [K/W]
Cx  = 512.80 ;  % [Ws/K]
% Rxz =  28.37 ;  % [K/W]
Rxz =  68.37 ;  % [K/W]

% --- Normalize time vector to the system time constant : ---
% t = t / sqrt( Rpx * Rxz * Cp * Cx ) ;
% New requirement :  time vector in hours :
t = t / 3600 ;

% --- Compare dissipation and cooling : ---
if plt
	TT=273+(20:5:200);
	Pr=(TT-Tz)/(Rpx+Rxz);
	Pd=Po*exp(-v ./TT);
	plot(TT-273, Pr, '--r', TT-273, Pd, '-g')
	title('Comparison of dissipation and cooling characteristics')
	xlabel('Temperature [K-273]')
	pause
end

% --- calculate initial conditions : ---
deltaP=1;
T0=Tz;
Pq=0;
while deltaP > 0.001
	Pd=Po*exp(-v/T0);
	Tx=Tz+Pd*Rxz;
	T0=Tz+Pd*(Rpx+Rxz);
	deltaP=Pd-Pq;
	Pq=Pd;  
end
xt0=find(t==0);
T(1:xt0-1, 1)=T0*(ones(xt0-1,1));
T(1:xt0-1, 2)=Tx*(ones(xt0-1,1));

% --- plot initial conditions : ---
plot([t(1), t(max(size(t)))], [Tz-273, Tz-273],'b', t(1), T0+Tm-273+50, 0, 0)

% --- a power transient at t=0 heaths the system to T0+Tm : ---
Ti=T0+Tm;
T(xt0,:)=[Ti, Tx];

for k=1:xt0-1
	line(t(k:k+1), T(k:k+1,1)-273);
	line(t(k:k+1), T(k:k+1,2)-273);
end 

% --- calculate initial conditions after the transient and
%       loop for each new time step : ---
for k=1:max(size(t))-xt0
	Pd=Po*exp(-v/Ti);
	Ppx=(Ti-Tx)/Rpx;

	% --- calculate new T and Tx at the next time step : ---
	Ti=Ti+deltat*(Pd-Ppx)/Cp;               % integrate Pp ;
	Pxz=(Tx-Tz)/Rxz;
	Ppx=(Ti-Tx)/Rpx;
	Tx=Tx+deltat*(Ppx-Pxz)/Cx;              % integrate Px ;
	T(xt0+k,:)=[Ti,Tx];                     % new value of T ;
	line(t(xt0+k-1:xt0+k), T(xt0+k-1:xt0+k,1)-273);
	line(t(xt0+k-1:xt0+k), T(xt0+k-1:xt0+k,2)-273);
end
% plot(t, T-273, [t(1), t(max(size(t)))], [Tz-273, Tz-273], 0, 0)
xlabel('t [h]')
ylabel('Temperature [K-273]')
title(['Thermal transient for delta-T = ', num2str(Tm), ' [K]'])
