function xtherm(Tmin)
% XTHERM        ..returns system thermal constants for MAXTHERM.

%       Erik Margan, *930706*

kB = 1.38e-23 ; % [VAs/K] Boltzmann constant
qe = -1.602e-19 ; % [As] electron charge
Is = 1e-14 ; % [A] saturation current in silicon
Rthjc = 25 ; % [K/W] thermal resistance, junction to case
Rthca = 55 ; % [K/W] thermal resistance, case to ambient (TO-5, no heatsink)
Cthj = 0.05 ; % [Ws/K] junction+die thermal capacitance
Cthc = 0.65 ; % [Ws/K] case thermal capacitance


if nargin == 0
	Tmin = 273 + 40 ;
    % on a hot summer day, the room temperature can easily reach 40 deg.C;
    Tmax = 273 + 180 ;
    % 180 deg.C is the maximum temperature for silicon devices, but, at this
    % temperature, the allowable power dissipation is zero!
end
% T = Tmin + ( 0 : 5 : Tmax ) ;

Rc = 50 ; % [Ohm] collector load
Ic = 0.03 ; % [A] collector current
Vcc = 15 ; % [V] supply voltage
Pd = Vcc * Ic - (Ic^2) * Rc ; % [W] initial power dissipation
Vo = Vcc - Ic * Rc ; % [V] initial conllector voltage

% base voltage calculated from the required collector current at power-up
Vbe = ( kB * Tmin /qe ) * log( ( Ic / Is ) + 1 ) ;

% time vector
deltat = 0.2 * Rthjc * Cthj ; % increment must be smaller than the shortest time constant
tmax = 5 * Rthca * Cthc ; % max. time must be larger than the longest time constant
t = 0 : deltat : tmax ;

Tj = Tmin ; % initial junction temperature

Tc = Tmin ; % initial case temperature

% calculate working conditions iteratively
for k = 2:length(t)
    % Pd increases juction temperature
    Tj(k,:) = Tj(k-1) + deltat * ( Pd(k-1) / Cthj  - ( Tj(k-1) - Tc(k-1) ) / Rthjc ) ;
    % higher junction temperature increases case temperature
    Tc(k,:) = Tc(k-1) + deltat * ( ( ( Tj(k) - Tc(k-1) ) / Rthjc ) - ( ( Tc(k-1) - Tmin ) / Rthca ) ) / Cthc ;
    % Tj also increases collector current
    Ic(k,:) = Is * ( exp( Vbe * qe / ( kB * Tj(k) ) ) - 1 ) ;
    % higher Ic increases Pd
    Pd(:,k) = Vcc * Ic(k) - (Ic(k)^2) * Rc ;
end

plot( t, Tj-273, '-r', t, Tc-273, '-m', t, Ic*1000, '-b', t, Pd*100, '-k' )
