function [y,t]=ttresp(F,w,r,g,h)
%TRESP   Transient RESPonse, using Fast-Fourier-Transform algorhythm.
%
%	Call : [y,t]=tresp(F,w,r,g,h);
%
%	input arguments :
%		F is a 2^N length vector of complex-frequency response
%		w can be the related frequency vector of F, or it
%		  can be the normalized frequency unit index, or it
%		  can be zero and the n.f.u. index is found from F ;
%		r is a character, selects the response type returned in y:
%		- 'u' is the unity-area impulse response (the default)
%		- 'i' is the ideal impulse response
%		- 's' is the step response
%		g is an optional flag; if present, plot the response.
%		h is an optional flag; if nonzero, force windowing for all systems.
%	output results :
%		y is the selected system response
%		t is the normalized time-scale vector.

%	Erik Margan, 880414, Last revision 000315

% ----------- Preparation and check of input data ; ----------------
if nargin < 5
	h=-1;
end
if nargin < 4
	g=0;
end
if nargin < 3
	r='u';			% default response selector ;
end
if nargin < 2
	w=0;
end
G=abs(F(1));			% find system d.c. gain ;
N=max(size(F));			% find number of input points ;
v=max(size(w));
if v == 1
	m = w ;			% w is the normalized frequency unit or zero ;
elseif v == N
	m=find(abs(w-1)==min(abs(w-1)))-1;	% find the normalized frequency unit ;
	if isempty(m)
		m = 0 ;		% could not find it, try from half-power bandwidth ;
	end
else
	error('F and w must be of equal length !') ;
end
if m == 0
	m=max(find(abs(F)>=G/sqrt(2)))-1 ;
end

% M=abs(F(1*10*m));		% check the magnitude a decade above cut-off ;
M=abs(diff(20*log10(abs(F(1+[4*m,2*m]))))); % check magn.slope 2 octaves above cut-off ;
x=3;				% system is third-order or higher ;
% if M > 0.0005*G
if M < 15
	x=2;			% probably a second-order system ;
end
% if M > 0.005*G
if M < 9
	x=1;			% probably a first-order system ;
end

% ----------- Normalize the time-scale ; ---------------------------
A=2*pi*m;			% amplitude normaliza tion factor ;
dt=A/N;				% calculate delta_t ;
if v == N
	dt=dt/w(m+1);		% correct for the actual frequency unit ;
end
t=dt*(0:1:N-1);			% form normalized time scale ;

% ----------- Form window function ; -------------------------------
if (h > 0) | ( ( x < 3 ) & h )
	W=(1+cos(pi*(0:1:N-1)/(N-1)))/2; % right-half Hanning window ;
	F=W.*F;			% frequency response windowed ;
end

% ----------- Calculate impulse response ; -------------------------
y=2*real(fft(conj(F)))-G;	% calculate iFFT and null d.c. offset ;
if x == 1
	er1=A*G-y(1);
	y(1)=A*G;
elseif x == 2
	er1=-y(1);
	y(1)=0;
end
if r == 'u' | r == 'U' | r == 's' | r == 'S'
	y=y/N;			% normalize area to M ;
end

% ----------- Calculate step response ; ----------------------------
if r == 's' | r == 'S'
	if x < 3
		y=y*(1/(1+er1/N));	% correct first-point error ;
		if x == 1
			y=[0, y(1:N-1)];
		end
	end
	y=cumsum(y);		% integrate to get the step resp. ;
end

% ----------- Normalize Amplitude to Ideal ; -----------------------
if r == 'i' | r == 'I'
	y=y/(2*pi*m);		% normalize impulse amplitude ;
end

% ----------- Display results ; ------------------------------------
if g
	plot( t, y, '-r' )
	title('Transient responses')
	xlabel('Time')
end
