function aamxfd(select)
% Anti-Alias mixed-mode filter design example ;
% Script.
% Call :        aamxfd(select) ;
%               select : 0 --> default pole selection
%                        1 --> top + real pole selection

% Author : E.Margan, 1989.10.17.
% No copyright protection applied !

 if ~nargin
     select = 0 ;
 end
 
% Sysrem definitions and requirements determination

 Nf=601;                % number of frequency samples
 Nt=500;                % number of time samples
 M=1e+6;                % 1 MHz scale factor
 A=2^12;				% ADC resolution limit sets the attenuation
 a=20*log10(1/A);       % same in dB
 fs=2e+7;				% ADC sampling frequency, 2MHz
 fN=fs/2;				% Nyquist frequency, 1MHz
 f=fN*logspace(-2,1,Nf);	% frequency vector (0.1 to 10 MHz, log-scale)
 % cut-off frequencies of the 5th-, 7th- and 9th-order system asymptotes,
 % all crossing the ADC resolution limit, 1/A, at fN, after Eq.6.3.10:
 fa5=fN/10^(log10(A^2-1)/(2*5));
 fa7=fN/10^(log10(A^2-1)/(2*7));
 fa9=fN/10^(log10(A^2-1)/(2*9));
 disp(['fa5 = ', num2str(fa5/M), ' MHz'])
 disp(['fa7 = ', num2str(fa7/M), ' MHz'])
 disp(['fa9 = ', num2str(fa9/M), ' MHz'])
 	% the disp commands return the following values :
%  fa5 = 1.8946 MHz
%  fa7 = 3.0475 MHz
%  fa9 = 3.9685 MHz



 [z5,p5]=bestap(5,'a');	% Bessel-Thomson 'asymptote' normalized systems
 [z7,p7]=bestap(7,'a');	% of order 5, 7
 [z9,p9]=bestap(9,'a'); % and 9 ;

 p5=p5*2*pi*fa5;		% The poles are scaled-up by the previously
 p7=p7*2*pi*fa7;		% calculated cut-off frequencies, so that all
 p9=p9*2*pi*fa9;		% three responses have 1/A attenuation at fN ;

 F5=20*log10(abs(freqw(p5,2*pi*f))); % Calculate magnitudes in dB ;
 F7=20*log10(abs(freqw(p7,2*pi*f)));
 F9=20*log10(abs(freqw(p9,2*pi*f)));
 				% plot magnitudes in dB vs. log-scaled frequency
 db3=-3.0103;		% and the -3dB reference level (see Fig.7.3.2) :
 % -------------------------------------------------------------------------
 figure(1)
 semilogx( f/M, F5, '-r',...
           f/M, F7, '-g',...
           f/M, F9, '-b',...
           fN*[0.02, 0.9]/M, [db3, db3], '-k',...
           [f(1), f(Nf)]/M, [a, a], '--k' )
 axis([0.1, 100, -80, 0]);
 xlabel('f [MHz]')
 % -------------------------------------------------------------------------


 % compare the magnitudes with the -3dB level and
 % find the index of each frequency limit:

 m=abs(F5-(-3.0103));
 x5=find(m==min(m));

 m=abs(F7-(-3.0103));
 x7=find(m==min(m));

 m=abs(F9-(-3.0103));
 x9=find(m==min(m));

 f5=f(x5);
 f7=f(x7);
 f9=f(x9);
 % display the Bessel-Thomson systems cut-off frequencies :
 disp(['f5 = ', num2str(f5/M), ' MHz'])
 disp(['f7 = ', num2str(f7/M), ' MHz'])
 disp(['f9 = ', num2str(f9/M), ' MHz'])

 %  f5 = 1.166 MHz
 %  f7 = 1.660 MHz
 %  f9 = 1.965 MHz

 Tx=2e-9;                   % 2 ns samplig
 t=Tx * (0:1:Nt-1);         % time vector,

 S5=atdr(z5,p5,t,'s');		% Step responses
 S7=atdr(z7,p7,t,'s');
 S9=atdr(z9,p9,t,'s');

 % plot the step responses with the 0.1 and 0.9 reference levels to
 % compare the rise-times :
 x10=round(Nt*[0.10, 0.25]);
 x90=round(Nt*[0.25, 0.65]);
 y10=[0.1, 0.1];
 y90=[0.9, 0.9];
 % -------------------------------------------------------------------------
 figure(2)
 plot( t*M, S5, '-r',...
       t*M, S7, '-g',...
       t*M, S9, '-b',...
       t(x10)*M, y10, '-k',...
       t(x90)*M, y90, '-k' )
 xlabel('t [us]')
 axis([t(1)*M, t(Nt)*M, 0, 1.2]);
 % -------------------------------------------------------------------------

% anti-alias filtering

 B=1.73;	% bandwidth improvement factor ( 1.5 <= Bi <= 1.8 )
 fA=fN*B;				    % fA is the shifted fN
 xn=min(find(f>=fN));		% index of fN in f
 xa=min(find(f>=fA));		% index of fA in f
                            % original 7-pole system (same as F7) :
 F7o=20*log10(abs(freqw(p7,2*pi*f)));
                            % same as F7o, but shifted up by B :
 F7B=20*log10(abs(freqw(B*p7,2*pi*f)));
 S7a=F7B(xn:xa);			% source of alias spectrum
 fa=f(xn:xa);               % f.band of S7a 
 S7b=F7B(xa:-1:xn);         % alias spectrum, from fs-fA to fN
 fb=fs-f(xa:-1:xn);         % f.band of S7b
 Frq=a-S7b;                 % required dig.filt. magnitude in dB-scale
 fr=fs-f(xa:-1:xn);		    % in the same freq. range: fs-fa to fN
 
 % -------------------------------------------------------------------------
 figure(3)
 semilogx(  f/M, F7o, '-k', ...
            f/M, F7B, '-b', ...
           fa/M, S7a, '-y', ...
           fb/M, S7b, '-c', ...
           fr/M, Frq, '-m', ...
           [f(1),f(Nf)]/M, [a,a], '--k',...
           [fN, fN]/M, [-80, 0], '-k',...
           [fs, fs]/M, [-80, 0], '-.k',...
           [fA, fA]/M, [-80, 0], ':k' )
 axis([0.1, 100, -80, 0]);
 % -------------------------------------------------------------------------


 % mixed-mode filter

 [z13,p13]=bestap(13,'a');	        % the 13th-order system
 p13=B*2*pi*fa7*p13;                % denormalized poles of the compound system
 p13=sort(p13);			            % sort the poles in ascending order
 
 if ~select
     p7a=p13([1,4,5,8,9,12,13]);    % analogue filter pole selection
     p6d=p13([2,3,6,7,10,11]);	    % digital-equivalent filter poles
 else
     p7a=p13([1,8,9,10,11,12,13]);	% analogue filter pole selection
     p6d=p13([2,3,4,5,6,7]);	    % digital-equivalent filter poles
 end
 
 rp=find(p7a==real(p7a));
 ux=log10(abs(p7a(rp)));
 qx=floor(ux);
 gx=ceil(10*(ux-qx))/10;
 gx=ceil(10^gx);
 ex=gx*10^qx;
 
 % -------------------------------------------------------------------------
 figure(4)
 plot( real(p7a), imag(p7a), 'xb',...
       real(p6d), imag(p6d), '+r',...
       real(p7),  imag(p7),  '*g',...
       [-ex, ex], [0, 0],    '-k',...
       [0, 0], [-ex, ex],    '-k' )
 axis('square');
 axis('equal');
 axis([-ex, ex, -ex, ex]);
 title( 'poles')
 % -------------------------------------------------------------------------
 
 X7a=20*log10(abs(freqw(p7a,2*pi*f))); % analogue 7(of 13)-pole magnitude
 X6d=20*log10(abs(freqw(p6d,2*pi*f))); % digital-equivalent 6(of 13)-pole magnitude
 X13=20*log10(abs(freqw(p13,2*pi*f))); % total a+d 13-pole magnitude
 xa=max(find(X7a>=a));		% response index at crossing of a
 xn=min(find(f>=fN));		% index of the Nyquist frequency
 fa=f(xn:xa);			% frequency range above Nyquist
 Sxa=X7a(xn:xa);		% source of alias
 fb=fs-f(xa:-1:xn);		% alias of fa
 Sxb=X7a(xa:-1:xn);		% alias band (plotted as Sxa in reverse order, as fb)
 Frq=a-Sxb;				% alias suppression requirement
 fr=fb;
 % -------------------------------------------------------------------------
 figure(5)
 semilogx(  f/M, F7o, ':k', ...
            f/M, F7B, ':b', ...
            f/M, X7a, '-b', ...
            f/M, X6d, '-g', ...
            f/M, X13, '-r', ...
           fa/M, Sxa, '-y', ...
           fb/M, Sxb, '-c', ...
           fr/M, Frq, '-m', ...
           [f(1),f(Nf)]/M, [a,a],'--k' )
 axis([0.1, 100, -80, 10]);
 % -------------------------------------------------------------------------
 cx=min(find(X13<=-3.0103));
 f13=f(cx);
 disp(['f13 = ', num2str(f13/M), ' MHz'])


 sr7o=atdr(z7,p7,t,'s'); 	% reference 7th-order system step-response
 sr13=atdr(z13,p13,t,'s');	% 13th-order step-response
 % plot the step responses with the 0.1 and 0.9 reference levels :
 % -------------------------------------------------------------------------
 figure(6)
 plot( t*M, sr7o, '-r',...
       t*M, sr13, '-b',...
       t([25, 150])*M, [0.1, 0.1], '-k',...
       t([95, 250])*M, [0.9, 0.9], '-k' )
 axis([t(1)*M, t(Nt)*M, 0, 1.2]);
 xlabel('t [us]')
 % -------------------------------------------------------------------------


 gd7o=gdly(z7,p7,2*pi*f);	% group-delay for the original 7th-order and
 gd13=gdly(z13,p13,2*pi*f);	% the improved 13th-order system
 
 % -------------------------------------------------------------------------
 figure(7)
 semilogx( f/M, gd7o*M, '-r', f/M, gd13*M, '-b')
 title('Normalized group delay')
 xlabel('f [MHz]')
 % -------------------------------------------------------------------------

% --------------------------------------------
%  p7a :  1e+7 *  -3.0181
%                  -2.8572 - 1.1751i
%                  -2.8572 + 1.1751i
%                  -2.3275 - 2.385i
%                  -2.3275 + 2.385i
%                  -1.1637 - 3.7353i
%                  -1.1637 + 3.7353i
%
%  p6d :  1e+7 *  -2.9785 - 0.58579i
%                  -2.9785 + 0.58579i
%                  -2.6460 - 1.7725i
%                  -2.6460 + 1.7725i
%                  -1.8655 - 3.0264i
%                  -1.8655 + 3.0264i
% ---------------------------------------------

 Ts=1/fs;                   % actual ADC sampling interval (50 ns)
 kx=round(Ts/t(2));         % fine time-scale resolution
 ky=floor(Nt/kx);           % number of actual samples in the time window
 jx=[1, (kx:kx:kx*ky)];     % coarse time-scale index (actual sampling)
 ta=t(jx);	                % actual time vector
 S7a=atdr(z7,p7a,t,'s');	% step-response of the analogue part 
 h6d=atdr(z7,p6d,t,'n');	% impulse-response, digital part
 S13=conv(S7a,h6d);         % digital filtering = convolution
 S13=S13(1:max(size(t)));	% take only length(t) samples
 h6d=0.43*(h6d/max(h6d));   % amplify h6d for display only
 % -------------------------------------------------------------------------
 figure(8)
 plot(  t*M, S7a,     '-r',...
       ta*M, S7a(jx), 'or',...
        t*M, h6d,     '-g',...
       ta*M, h6d(jx), 'og',...
        t*M, S13,     '-b',...
       ta*M, S13(jx), 'ob' )
 xlabel('t [us]')
 % -------------------------------------------------------------------------

