function fml(M,N,k)
% FAMILY FAMILY without input arguments is a menu-driven utility.
%	 
%	 FAMILY(M) plots a family of filter responses in decreasing order
%	 from M to 2.
%	 
%	 FAMILY(M,N) will plot the responses from order M to N.
%	 By making  N = 1  the first order system response is also ploted.
%	 
%	 FAMILY(M,N,k) will plot the responses from order M to N of the filter
%	 type selected by k :
%	 
%	 1) Bessel-Thomson ..... maximaly flat envelope delay, no overshoot ;
%	 2) Butterworth ........ maximally flat amplitude response ;
%	 3) Chebyschew type-1 .. max. stopband attenuation, passband ripple ;
%	 4) Chebyshew type-2 ... max. flat passband, ripple in stopband ;
%	 5) Elliptic (Cauer) ... max. steep pass-to-stopbadn transition.
%	 
%	 The frequency interval is +/- one decade about the filter cuttoff
%	 frequency (-3 dB).
%
%	 See also SYSVIEW and COMPFILT.

%	 Erik Margan, October 12th, 1988
%	 Ljubljana, Slovenia
%	 Last revision : May 8th, 1992

nw=200;
w=logspace(-1,1,nw);

if nargin < 3
	k=0;
	k=menu('Choose Filter Family :',...
	       'Bessel-Thomson',...
	       'Butterworth',...
	       'Chebyshew type-1',...
	       'Chebyshew type-2',...
	       'Elliptic (Cauer)',...
	       'Return to MATLAB');
end

if isempty( k )
	return
end

if k==6
	return
end

if k==3 | k==5
   	rp=input('Enter allowable passband ripple in dB [ CR=.5dB ] : ');
	if isempty(rp)==1
		rp=.5;
	end
end

if k==4 | k==5
   	rs=input('Enter allowable stopband ripple in dB [ CR=50dB ] : ');
	if isempty(rs)==1
		rs=50;
	end
end

if nargin < 2
	N=2;
end

if nargin == 0
	disp(' ')
	disp(' Minimum filter order is 2.')
   	M=input('Choose maximum filter order [ CR=5 ] : ');
	if isempty(M)==1
		M=5;
	end
	M=fix(abs(M));
end

N=fix(abs(N));
m=max(M,N);
n=min(M,N);
M=m;
N=n;
m=M-N+1;
n=1;

zp=input('Show numeric zeros and poles ?   [Y/N; CR=Y] : ','s');
fr=input('Show frequency responses ?       [Y/N; CR=Y] : ','s');
ph=input('Show phase shifts ?              [Y/N; CR=Y] : ','s');
ed=input('Show envelope (group) delays ?   [Y/N; CR=Y] : ','s');
ir=input('Show impulse responses ?         [Y/N; CR=Y] : ','s');
sr=input('Show step responses ?            [Y/N; CR=Y] : ','s');

if isempty(zp)==1
	zp='y';
end
if isempty(fr)==1
	fr='y';
end
if isempty(ph)==1
	ph='y';
end
if isempty(ed)==1
	ed='y';
end
if isempty(ir)==1
	ir='y';
end
if isempty(sr)==1
	sr='y';
end

clc;
disp('Calculating system poles and zeros ...')
Z=[];
P=[];

FT=[' bestap';' buttap';'cheb1np';'cheb2np';'ellipnp'];
fname = FT( k, : ) ;
if fname(1) == ' '
	fname(1) = [] ;
end

for gk=M:-1:N
	if k==3 & abs(gk)~=1
	   	[z,p]=feval(fname,abs(gk),rp,-3);
   	elseif k==4 & abs(gk)~=1
	   	[z,p]=feval(fname,abs(gk),rs,-3);
   	elseif k==5 & abs(gk)~=1
	   	[z,p]=feval(fname,abs(gk),rp,rs);
	elseif k<3 & abs(gk)~=1
		[z,p]=feval(fname,abs(gk));
	elseif abs(gk)==1
		[z,p]=feval(fname,abs(gk));
	end
	
	if zp == 'y' | zp == 'Y'
		clc
		disp(['Zeros and poles of the system of order ',num2str(gk)])
		z
		p
		disp('Press any key to continue ...'), pause
	end
	
	Z=[Z, ([z; NaN*ones(M-length(z),1)])];
	P=[P, ([p; NaN*ones(M-gk,1)])];
end
clc

if fr=='y' | fr=='Y'
	disp('Calculating frequency responses...');
	for ix=n:m
		z=Z(:,ix);
		p=P(:,ix);
		% ixx=find(z==NaN);
		ixx=find(isnan(z));
		z(ixx)=[];
		% ixx=find(p==NaN);
		ixx=find(isnan(p));
		p(ixx)=[];
		H=freqw(z,p,w);
		h=20*log10(abs(H));
		semilogx(w,h)
		hold on
	end
	title('Family of filter responses')
	xlabel('Normalized frequency [f/fo]')
	ylabel('Gain [dB]')
	pause
	hold off;

	clc;
	zz=input('Zoom passband details? [Y/N; CR=Y] : ','s');
	if isempty(zz) == 1
		zz='y';
	end
	if zz=='y' | zz=='Y'
		at=input('Enter maximum attenuation in dB [ CR=3dB ] : ');
		if isempty(at)==1
			at=3;
		end
		axis([-1 1 -at 1]);
		% for ix=n:m
		%	z=Z(:,ix);
		%	p=P(:,ix);
		%	% ixx=find(z==NaN);
		%	ixx=find(isnan(z));
		%	z(ixx)=[];
		%	% ixx=find(p==NaN);
		%	ixx=find(isnan(p));
		%	p(ixx)=[];
		%	H=freqw(z,p,w);
		%	h=20*log10(abs(H));
		%	semilogx(w,h)
		%	hold on
		% end
		% title('Family of filter responses')
		% xlabel('Normalized frequency [f/fo]')
		% ylabel('Gain [dB]')
		pause
		hold off;
		% axis;
	end
end
clc;

if ph=='y' | ph=='Y'
	clc;
	disp('Calculating phase shifts...');
	for ix=n:m
		z=Z(:,ix);
		p=P(:,ix);
		% ixx=find(z==NaN);
		ixx=find(isnan(z));
		z(ixx)=[];
		% ixx=find(p==NaN);
		ixx=find(isnan(p));
		p(ixx)=[];
		H=freqw(z,p,w);
		phase=180*ephd(angle(H))/pi;	
		semilogx(w,phase)
		hold on
	end
	title('Family of phase shifts')
	xlabel('Frequency')
	ylabel('Phase [Degrees]')
	pause
	hold off;
end

if ed=='y' | ed=='Y'
	clc;
	disp('Calculating envelope delays...');
	deltaw=w(2:1:nw)-w(1:1:nw-1);
	for ix=n:m
		z=Z(:,ix);
		p=P(:,ix);
		% ixx=find(z==NaN);
		ixx=find(isnan(z));
		z(ixx)=[];
		% ixx=find(p==NaN);
		ixx=find(isnan(p));
		p(ixx)=[];
		H=freqw(z,p,w);
		phase=ephd(angle(H));	
		td=(phase(2:1:nw)-phase(1:1:nw-1))./deltaw;
		semilogx(w(1:nw-1),td,w(nw),0)
		hold on
	end
	title('Family of envelope delays'),xlabel('Frequency'),ylabel('Delay');   
	pause
	hold off;
end
clear H;
clear h;
clear phase;
clear deltaw;
clear td;
clear w;

if ir=='y' | ir=='Y'
	clc;
	disp('Calculating impulse responses...');
	t=(0:1:10*6)/6;
	w=(0:1:255);
	w0=8;
	nw=length(w);
	xw=(1+cos(pi*(0:1:nw-1)/(nw-1)))/2;
	w=w/w0;
	axis([0 max(t) -.2 1.2]);
	for ix=n:m
		z=Z(:,ix);
		p=P(:,ix);
		% ixx=find(z==NaN);
		ixx=find(isnan(z));
		z(ixx)=[];
		% ixx=find(p==NaN);
		ixx=find(isnan(p));
		p(ixx)=[];
		f=xw.*freqw(z,p,w);
		f=(2*real(fft(conj(f)))-1)/nw;
		aa=10.23*(nw/256)*(4/w0);
		f=f*aa;
		f(1)=0;
		f=f(1:length(t));
		plot(t,f)
		hold on
	end
	title('Family of impulse responses'), xlabel('Normalized time');
	pause
	hold off
	axis;
end
clc;

if sr=='y' | sr=='Y'
	clc;
	disp('Calculating step responses...');
	t=(0:1:10*6)/6;
	w=(0:1:255);
	w0=8;
	nw=length(w);
	xw=(1+cos(pi*(0:1:nw-1)/(nw-1)))/2;
	ax=[0 max(t) 0 1.2];
	w=w/w0;
	axis(ax);
	for ix=n:m
		z=Z(:,ix);
		p=P(:,ix);
		% ixx=find(z==NaN);
		ixx=find(isnan(z));
		z(ixx)=[];
		% ixx=find(p==NaN);
		ixx=find(isnan(p));
		p(ixx)=[];
		f=xw.*freqw(z,p,w);
		f=(2*real(fft(conj(f)))-1)/nw;
		f=f(1:length(t));
		f=cumsum(f);
		f=(f-f(1))/(1-f(1));
		plot(t,f)
		hold on
	end
	title('Family of step responses'), xlabel('Normalized time');
	axis;
	hold off
end
clc;
