function O=afs
% AFS   Active Filter Synthesis.
%	A menu-driven utility using standard filter types and circuit configurations.
%	Call :
%		 O=afs;
%
%	where:	 O ==> returned component values of the selected filter type.
%
%       Circuit layout display option is provided (via "LAYOUT*.CFG" files).
%       
%       See also VIEW, COMPFILT and FAMILY.
%       
%       Note :
%	       At present only 3 filter types and 2 configurations are available.

%       Erik Margan
%				First editing : 910520
%		  Matlab-IV version : 980109
%				Last revision : 000608
%				Last editing  : 000608
%		  Matlab-V version  : 000805
%       Not copyright protected.

figh=1;
cont='y';
while cont == 'y' | cont == 'Y'
	ntype=0;
	while ntype < 1 | ntype > 5   % ( Change the upper limit if expanding menu )
		clc
		disp(' ')
		disp('----- Select Design Criteria (Filter Type) -----')
		disp(' ')
		disp(' All-pole systems :')
		disp('  1) Butterworth ......... Flattest Pass-Band                    ')
		disp('  2) Chebyshew ........... Steepest Pass-to-Stop-Band Knee       ')
		disp('  3) Bessel-Thomson ...... Flattest Group-Delay, Best Transient  ')
		disp(' Systems with poles and zeros : - not available yet')
		disp('  4) Inverse Chebyshew ... Steepest Stop-Band Knee               ')
		disp('  5) Cauer/Elliptic ...... Steepest Pass-to-Stop-Band Transition ')
		disp(' ')
		ntype=input('     Select a menu number [CR=Exit] : ');
		if isempty(ntype) == 1
			return
		else
			ntype=fix(abs(ntype));
			if ntype > 5 | ntype < 1
				disp(' ')
				disp('No such menu number !')
				disp(' ')
				return
			end
		end
	
		% ----- ???? ---------- ;
		if ntype == 4 | ntype == 5
			disp(' ')
			disp('Sorry, menu No. 4 & 5 have''nt been programmed yet.')
			disp(' ')
			return
		end
		% ----- ???? ---------- ;
	
	end
   
	% Select the system order number (number of poles) :
	nord=0;
	while nord < 2 | nord > 13
		disp(' ')
		nord=input('     Select system order [2 to 13] :  ');
		nord=fix(abs(nord));
	end
	
	% Select the first-stage configuration if nord is odd :
	stage1 = 2 ;	% initially assume stage1 of 2nd-order !
	if rem( nord, 2 )
		disp( ' ' )
		stage1=input( '     Select first-stage order [ 1 or 3 ? CR=3] : ' );
		if isempty( stage1 )
			stage1 = 3 ;
		end
		if stage1 <= 2
			stage1 = 1 ;
		else
			stage1 = 3 ;
		end
	end
	
	% Calculate the required number of stages :
	nstages=1+(nord-stage1)/2;
	
	% Select the system layout sceme :
	layo=0;
	while layo < 1 | layo > 5    % ( Change the upper limit if expanding menu )
		disp(' ')
		disp(['----- ','Select System Layout',' -----'])
		disp(' ')
		disp('       1) Sallen-Key')
		disp('       2) Equal-Components Sallen-Key')
		disp('       3) Multyple-Feedback')
		disp('       4) TDCC')
		disp('       5) GIC')
		disp(' ')
		layo=input('     Select a menu number [CR = Show circuit] : ');
		
		% configuration display loop :
      if isempty(layo) == 1
         layn = -1 ;
         while layn ~= 0
				layn=input('Enter the circuit number to show [CR = Return] : ');
				if isempty( layn )
					layn = 0 ;
				end
            if ( exist( 'figh' ) == 1 )
               if figh > 0
                  close( figh ) ;
                  figh = figh - 1 ;
               end
            end
				if layn > 3
					disp('Sory, this circuit is not available yet!');
					layn=-1;
				end
            if ( layn > 0 ) & ( layn < 4 )
               figh=schema(layn);
            end
			end
			layo=0;
		else
			layo=fix(abs(layo));
		end
	end
	
	% ftype defines the system subroutine string,
	% which has to be called to return poles and zeros :
	ftype=[' buttap' ; ...
	       'cheb1np' ; ...
	       ' bestap' ; ...
	       'cheb2np' ; ...
	       'ellipap' ] ;
	
	if ntype == 1
		optionstr = [] ;
	elseif ntype == 2
		rr = input( '        Enter the passband ripple in dB [ 0.01 ... 1 ; CR=0.1 ] : ' ) ;
		if isempty( rr )
			rr = 0.1 ;
		end
		if rr < 0.01
			rr = 0.01 ;
		end
		if rr > 1
			rr = 1 ;
		end
		optionstr = [ ', ', num2str( rr ), ', 1' ]  ;
		% the '1' at the end of optionstr enables frequency cut-off normalization
		% to the half-power bandwidth. Else, the ripple level sets the bandwidth.
	elseif ntype == 3
		optionstr = [ ', ''n''' ] ;
		% 'n' enables frequency cut-off normalization to half-power bandwidth.
	elseif ntype == 4
		rr = input( '        Enter the stopband ripple in dB [ > 20 ; CR=50 ] : ' ) ;
		if isempty( rr )
			rr = 50 ;
		end
		if rr < 20
			rr = 20 ;
		end
		optionstr = [ ', ', num2str( rr ), ', 1' ] ;
		% the '1' at the end of optionstr enables frequency cut-off normalization
		% to the half-power bandwidth. Else, the ripple level sets the bandwidth.
	elseif ntype == 5
		rr = input( '        Enter the passband ripple in dB [0.01 ... 1 ; CR=0.1 ] : ' ) ;
		if isempty( rr )
			rr = 0.1 ;
		end
		if rr > 1
			rr = 1 ;
		end
		optionstr = [ ', ', num2str( rr ) ] ;
		rr = input( '        Enter the stopband ripple in db [ > 40 ; CR=60 ] : ' ) ;
		if isempty( rr )
			rr = 60 ;
		end
		if rr < 20
			rr = 40 ;
		end
		optionstr = [ optionstr, ', ', num2str( rr ) ] ;
		% frequency cut-off normalization flag should be added to optionstr.
	end
	
	% get the system poles and zeros :
	eval( [ '[zzz, ppp] = ', ftype(ntype,:), '(', num2str(nord), optionstr, ' ) ; '] )
	
	% Sort poles by ascending value of the imaginary part ;
	[ip,sx]=sort(abs(imag(ppp)));
	ppp=ppp(sx);
	clear sx
	clear ip

	% preserve poles and zeros for later use in pporg and zzorg :
	pporg = ppp ;
	zzorg = zzz ;

	% extract the real pole if system order is odd and the first-stage is single-pole :
	real_pole = 0 ;
	if stage1 == 1
		kx=find( abs( imag( ppp ) ) < 1e-5  ) ;
		real_pole = ppp( kx ) ;
		
		% Remove the real-pole and
		% decrement the system order number
		% to be able to use the even-order loop below.
		% Will be restored at the end of the loop !
		ppp( kx ) = [] ;
		nord = nord - 1 ;
	end
	
	% Find layout-specific normalized values of Ci :
	if layo==1
		% Sallen-Key layout scheme :
		Ao=1;	% system gain is 1 for this layout.
		if rem( nord, 2 )
			% the first-stage is 3rd-order :
			C=[0 0 0];
			a=ppp(1);
			b=ppp(2);
			c=ppp(3);
			
			p=(a*b+b*c+a*c)/(a*b*c);
			q=3*(a+b+c)/(2*a*b*c);
			r=6/(2*a*b*c);
		
			%	C(3) polynomial :
			%	c^3 + p*c^2 + q*c + r = 0
			
			% general solution :
			%	C(3)=-2*sqrt(p^2-3*q)*sin(atan(sqrt(3)*(p^2-3*q)^(3/2)*...
			%	     (3*q-p^2)^(-3/2)*(2*p^3-9*p*q+27*r)/(9*...
			%	     sqrt(4*p^3*r-p^2*q^2-18*p*q*r+4*q^3+27*r^2)))/3)/3-p/3;
			
			% general solution simplified :
			%	C(3) = 4*r*p^3 - p^2 * q^2 - 18*p*q*r + 4*q^3 + 27*r^2 ;
			%	C(3) = 3 * sqrt( 3 ) * sqrt( C(3) );
			%	C(3) = ( 2*p^3 - 9*p*q + 27*r ) / C(3) ;
			%	C(3) = sin( atan( i * C(3) ) / 3 ) ;
			%	C(3) = (-2/3) *sqrt( p^2 - 3*q ) * C(3) - p/3 ;
			
			% general solution, numerically :
			M=roots([1,p,q,r]);
			kx=find(abs(imag(M))<1e-4);
			C(3) = M(kx(1));

			C(1)=-(a*b*c*C(3)+a*(b+c)+b*c)/(3*a*b*c);
			
			C(2)=-2*C(3)/(C(3)*(a+b+c)+2);
			
			clear p
			clear q
			clear r
			clear a
			clear b
			clear c
			clear M
			clear kx
			
			% solutions must be pure real :
			C=real(C);

			% all other stages are 2nd-order :			
			if nord > 4
				a=ppp(4:2:nord-1);
				b=ppp(5:2:nord);
				C=[C;zeros((nord-3)/2,3)];
				C1=-real((a+b)./(2*a.*b));
				C2=-2*real((a+b).\1);
				C(:,1)=[C(1,1); C1];
				C(:,2)=[C(1,2); C2];
				C(:,3)=[C(1,3); NaN*C2];
			end
			clear C1
			clear C2
			clear b
			clear a
		else
			% system order number, nord, is even :
			C=zeros(nord/2,3);
			a=ppp(1:2:nord-1);
			b=ppp(2:2:nord);
			C1=-real((a+b)./(2*a.*b));
			C2=-2*real((a+b).\1);
			C(:,1)=C1;
			C(:,2)=C2;
			C(:,3)=NaN*C2;
			
			% if the first-stage is single-pole :
			if stage1 == 1 & real_pole < 0
				C(1,3) = 1/abs( real_pole ) ;
			end
			clear C1
			clear C2
			clear b
			clear a
		end
		
		% select low-pass or high-pass configuration :
		pass=input('Low-pass or High-pass configuration ? [L/H,  CR=L] : ','s');
		if isempty( pass )
			pass = 'L' ;
		end

		% *****************************************************
		% invert for high-pass :
		if pass == 'h' | pass == 'H'
			C=C.\1;
		end
		% *****************************************************
		
		disp(' ')
		disp('    Normalized results :');
		disp(' ')
		if pass == 'h' | pass == 'H'
			disp('    R1        R2        R3');
		else
			disp('    C1        C2        C3');
		end
		disp('---------------------------------------------');
		disp(C);
		disp(' ')

	elseif layo == 2
		% Sallen-Key, equal R-C layout scheme, gain sets Q for each stage :
		Ao=1;	% this line is only for result display, actual gain is gi !
		C=zeros(fix(nord/2),2);
		if rem( nord, 2 )
			% first-stage is 3rd-order :
			a=ppp(1);
			b=ppp(2);
			c=ppp(3);
			C(1,1)=real((-1/(a*b*c))^(1/3));        % C(1,123)
			C(1,2)=5+C(1,1)*real((a+b+c));          % g1
			
			if nord > 4
				% all other stages are 2nd-order :
				a=ppp(4:2:nord-1);
				b=ppp(5:2:nord);
				c=real(sqrt(a.*b).\1);
				C(:,1)=[C(1,1); c];                  % Ci
				C(:,2)=[C(1,2); 3+real((a+b).*c)];   % gi
			end
			clear a
			clear b
			clear c
		else
			% the system order number, nord, is even :
			a=ppp(1:2:nord-1);
			b=ppp(2:2:nord);
			C(:,1)=real(sqrt(a.*b).\1);             % Ci
			C(:,2)=3+real((a+b).*C(:,1));           % gi
			C(:,3)=NaN*C2;
			
			% if the first-stage is single-pole :
			if stage1 == 1 & real_pole < 0
				C(1,3)=1/abs( real_pole ) ;
			end
			clear a
			clear b
		end

		disp(' ')
		disp('    Normalized results :');
		disp(' ')
		if pass == 'h' | pass == 'H'
			disp('    Ri        gi');
		else
			disp('    Ci        gi');
		end
		disp('---------------------------');
		disp( C );
		disp(' ')

	elseif layo == 3
		% Multiple-Feedback layout scheme :
		% Select the total system d.c. gain ;
		% the gain for each stage is the nstages-root of the total gain :
		Ao=input('Enter the total system d.c. gain [CR=1] :  ') ;
		if isempty(Ao)
			Ao=1;
		else
			Ao=abs(Ao);			
			Ao=Ao^(1/nstages);
		end;

		if rem( nord, 2 )
			% first-stage is 3rd-order :
			C=[0 0 0];
			a=ppp(1);
			b=ppp(2);
			c=ppp(3);

			% x=-(a+b+c)
			% y=a*b+b*c+a*c
			% z=-a*b*c
			
			Ai=2*(3*Ao+1)/(4*Ao+1);
			 
			% p = - 2 * y / z ;
			% q = Ai * 2 * x / z ;
			% r = - Ai * 4 / z ;
			% C(3) polynomial :
			% c^3 + p*c^2 + q*c + r = 0 ;
			
			p=2*(a*b+b*c+a*c)/(a*b*c);
			q=Ai*2*(a+b+c)/(a*b*c);
			r=Ai*4/(a*b*c);
			
			% general 3rd-order solution :
			%	C(3)=-2*sqrt(p^2-3*q)*sin(atan(sqrt(3)*(p^2-3*q)^(3/2)*...
			%	     (3*q-p^2)^(-3/2)*(2*p^3-9*p*q+27*r)/(9*...
			%	     sqrt(4*p^3*r-p^2*q^2-18*p*q*r+4*q^3+27*r^2)))/3)/3-p/3;
			
			% general 3rd-order solution, simplified :
			%	C(3) = 4*r*p^3 - p^2 * q^2 - 18*p*q*r + 4*q^3 + 27*r^2 ;
			%	C(3) = 3 * sqrt( 3 ) * sqrt( C(3) );
			%	C(3) = ( 2*p^3 - 9*p*q + 27*r ) / C(3) ;
			%	C(3) = sin( atan( i * C(3) ) / 3 ) ;
			%	C(3) = (-2/3) *sqrt( p^2 - 3*q ) * C(3) - p/3 ;
			
			% general 3rd-order solution, numerically :
			M=roots([1,p,q,r]);
			kx=find(abs(imag(M))<1e-4);
			C(3) = M(kx(1));

			% C(2)=(3+1/Ao)/(C(3)*(y-C(3)*z/2));
			% C(1)=1/(Ao*z*C(2)*C(3));
			
			C(2) = (3 + 1/Ao) / ( C(3) * ( a*b + b*c + a*c + C(3)*a*b*c/2 ) ) ;
			C(1) = -1 / ( Ao*a*b*c*C(2)*C(3) ) ;
			
			clear p
			clear q
			clear r
			clear a
			clear b
			clear c
			clear M
			C=real(C);
			
			if nord > 4
				% all other stages are 2nd-order :
				a=ppp(4:2:nord-1);
				b=ppp(5:2:nord);
				C=[C;zeros((nord-3)/2,3)];
				C1=-real((a+b)./((1+2*Ao)*a.*b));
				C2=-real((a+b).\(2+1/Ao));
				C(:,1)=[C(1,1); C1];
				C(:,2)=[C(1,2); C2];
				C(:,3)=[C(1,3); NaN*C2];
			end
			clear C1
			clear C2
			clear b
			clear a
		else
			% if system order number, nord, is even :
			C=zeros(nord/2,3);
			a=ppp(1:2:nord-1);
			b=ppp(2:2:nord);
			C1=-real((a+b)./((1+2*Ao)*a.*b));
			C2=-real((a+b).\(2+1/Ao));
			C(:,1)=C1;
			C(:,2)=C2;
			C(:,3)=NaN*C2;
			
			% if the first-stage is single-pole :
			if ( stage1 == 1 ) & ( real_pole < 0 )
				C(1,3) = 1 / ( Ao * abs( real_pole ) ) ;
			end
			clear C1
			clear C2
			clear b
			clear a
		end
		
		disp('WARNING :  This configuration can be Low-Pass only !');
		% pass=input('Low-pass or High-pass configuration ? [L/H,  CR=L] : ','s');
		% if isempty( pass )
			pass = 'L' ;
		% end
		
		disp(' ')
		disp('    Normalized results :');
		disp(' ')
		disp('    C1        C2        C3');
		disp('---------------------------------------------');
		disp(C);
		disp(' ')
	
	elseif layo >= 4
		disp('Sorry, this configuration has not been programmed yet!')
		C=[];
	end
	
	% restore system order number  - see above !!!
	if stage1 == 1 & real_pole < 0
		nord = nord + 1 ;
		% ???????? restore the real-pole also ????????
		% ppp = [ real_pole ; ppp ] ;
	end
	
	% cont=input('Again ? [Y/N ; CR=Y] : ','s');
	% if isempty(cont)==1
	% 	cont='y';
	% end
	cont = 'n' ;	
end

% 2*pi compensation, since the poles and zeros are normalized :
C=C/(2*pi);

if isempty(C)==1
	return
end

% clear cont
clear layn
clear next

% -------- Approximate to standard E48 or E24 scale of values -------- ;
opt=input('Approximate to nearest standard values ? [Y/N ; CR=Y] : ','s');
if isempty(opt)==1
	opt='y';
end
if opt == 'y' | opt == 'Y'
	% this part of the routine works columnwise, so the rows and columns
	% of the component-value matrix C must be exchanged and exchanged back 
	% at the end of the process. Matrix O contains the final values :

	C=C.';

	% How to solve stage1==1 case ?
	% --> by running the case C(3,1) separately and make it NaN for normal run ???
	if stage1 == 1
		M = [NaN;NaN;C(3,1)];
		C(3,1) = NaN ;
		C = [ M, C ] ;
	end

	[rows,cols]=size(C);
	O=[zeros(1,cols); zeros(rows,cols)];

	rsc=input('Enter standard R scale [E12, E24, E48 or E96; CR=E24] : ','s');
	csc=input('Enter standard C scale [E12, E24, E48 or E96; CR=E12] : ','s');
	
	if isempty(rsc)==1
		rsc='e24';
	end;
	if isempty(csc)==1
		csc='e12';
	end;
	
	e12=[1.0;1.2;1.5;1.8;2.2;2.7;3.3;3.9;4.7;5.6;6.8;8.2];
	e24=[1.0;1.1;1.2;1.3;1.5;1.6;1.8;2;2.2;2.4;2.7;3;3.3];
	e24=[e24;3.6;3.9;4.3;4.7;5.1;5.6;6.2;6.8;7.5;8.2;9.1];
	
	e48=[1.0;1.05;1.10;1.15;1.21;1.27;1.33;1.40;1.47;1.54;1.62;1.69;1.78;1.87];
	e48=[e48;1.96;2.05;2.15;2.26;2.37;2.49;2.61;2.74;2.87;3.01;3.16;3.32;3.48];
	e48=[e48;3.65;3.83;4.02;4.22;4.42;4.64;4.87;5.11;5.36;5.62;5.90;6.19;6.49];
	e48=[e48;6.81;7.15;7.50;7.87;8.25;8.66;9.09;9.53];
	
	e96=[1.02;1.07;1.13;1.18;1.24;1.30;1.37;1.43;1.50;1.58;1.65;1.74;1.82;1.91];
	e96=[e96; 2.00;2.10;2.15;2.21;2.32;2.43;2.67;2.80;2.94;3.09;3.24;3.40;3.57];
	e96=[e96; 3.74;3.92;4.12;4.32;4.53;4.75;4.99;5.23;5.49;5.76;6.04;6.34;6.65];
	e96=[e96; 6.98;7.32;7.68;8.06;8.45;8.87;9.31;9.76];
	
	if rsc(2:3)=='96'
		scaler=[e48 e96];
		scaler=scaler(:).';
	elseif rsc(2:3)=='48'
		scaler=e48;
	elseif rsc(2:3)=='24'
		scaler=e24;
	else
		scaler=e12;
	end
	scaler=[scaler;10*scaler;100*scaler];
	
	if csc(2:3)=='96'
		scalec=[e48 e96];
		scalec=scalec(:).';
	elseif csc(2:3)=='48'
		scalec=e48;
	elseif csc(2:3)=='24'
		scalec=e24;
	else
		scalec=e12;
	end
	scalec=[scalec;10*scalec;100*scalec];

	if pass == 'h' | pass == 'H'
		ssss=scalec;
		scalec=scaler;
		scaler=ssss;
		clear ssss
	end
	
	f0=input('Enter the filter cut-off (-3dB) frequency in Hz [CR=1000] : ');
	if isempty(f0)
		f0=1000;
	end
	f0=abs(f0);

	R=1000000*ones(1,cols);              % R [k]  &  C [nF] ;
	% R=R/(2*pi*f0) ;	% poles are normalized, 2*pi is not needed !!!!
	R=R/f0;  	                     % at normalized freqency ;

	for k=1:cols
		colk=C(:,k);
		[rc,cc]=size(colk);
		orig=ones(rc,cc);
		xnan=find(isnan(colk)==1);        % ignore NaN's ;
		if isempty(xnan) ~= 1
			colk(xnan)=[];
			[mxn,nxn]=size(xnan);
			xzer=zeros(mxn,nxn);
			orig(xnan)=xzer;
		end

		cref = 1/min(colk);	% norm. to min. ;
		colk = colk * cref ;

		[colk,xsort]=sort(colk);
		for xcolk=2:max(size(colk))
			dd1 = zeros( max( size( scalec ) )/3, 1 ) ;
			% size(dd1), pause
			% dd1 = zeros( 1:max( size( scalec ) )/3 ).' ;
			% size(dd1), pause

			% find lowest error by shift & search;
			for h=1:max(size(scalec))/3
				delta=abs(scalec(h)*colk(xcolk)-scalec);
				dd1(h,:)=dd1(h,:)+min(delta);
			end
		end
		
		h=find(dd1==min(dd1));
		h=h(1);                           % ignore multiple min.s ;
		colk=scalec(h)*colk;
		% size( abs( colk( xcolk ) ) ), size( scalec ), pause
		
		for xcolk=2:max(size(colk))
			xscale=find(abs(colk(xcolk)-scalec)==min(abs(colk(xcolk)-scalec)));
			colk(xcolk)=scalec(xscale);    % select nearest values ;
		end
		
		colk(xsort)=colk;                 % sort back ;
		if f0 < 10                        % account for frequency range ;
			colk=100*colk;
		elseif f0 >= 10 & f0 <100
			colk=10*colk;
		elseif f0 > 10000
			colk=colk/10;
		end

		corr=cref*min(colk);              % correct R ;
		R(:,k)=R(:,k)/corr;
		
		if R(:,k) < 1                     % range of R ;
			xscale=find(abs(R(:,k)-scaler/10)==min(abs(R(:,k)-scaler/10)));
			R(:,k)=scaler(xscale)/10;
		elseif R(:,k) > 10
			xscale=find(abs(R(:,k)-10*scaler)==min(abs(R(:,k)-10*scaler)));
			R(:,k)=scaler(xscale)*10;
		else
			xscale=find(abs(R(:,k)-scaler)==min(abs(R(:,k)-scaler)));
			R(:,k)=scaler(xscale);
		end
		
		if isempty(xnan) ~= 1             % restore NaN's ;
			xones=find(orig==1);
			orig(xones)=colk;
			[mxn,nxn]=size(xzer);
			orig(xnan)=NaN*ones(mxn,nxn);
			colk=orig;
		end
		C(:,k)=colk;                      % store selected values ;
	end

	O=[R; C];
	O=O.';
	clear C ;
	% preserve C for display of values in kohms and nF ;
	C=O;
end

% compare approximation to ideal :

cmpx = 'n' ;
if opt == 'y' | opt == 'Y' 
	cmpx=input('Compare approximation to the ideal response ? [Y/N ; CR=Y] : ','s');
	if isempty(cmpx)==1
		cmpx='y';
	end
end
if cmpx == 'y' | cmpx == 'Y'
	% calculate frequency responses along each row of C and original poles
	% then multiply them
	
	fx=f0*logspace(-1,1,401);	% form frequency vector -/+1 decade about f0
	w=2*pi*fx;	% poles are normalized, so 2*pi is not needed !!!
	jw = j*w ;
	% jw = j*fx ;
	
	if layo == 1
		% Sallen-key layout
		[row, col]=size(C);

		if pass == 'h' | pass == 'H'
			% C(:,1) are capacitances in nF :
			% the rest are resistors in kohms :
			C = C * 1e+3 ;
			C(:,1) = C(:,1) * 1e-9 * 1e-3 ;
		else
			% C(:,1) are resistances in kohms :
			% the rest are capacitances in nF :
			C = C * 1e-9 ;
			C(:,1) = C(:,1) * 1e+9 * 1e+3 ;
		end
		O = C ;


		% check if last element in the first row is NaN :
		if isnan( C(1,4) )
			% --> rem( nord, 2 ) == 0  --> even order system :
			col = col - 1 ;
		end
		
		% first row :
		if col == 4
			if stage1 == 1
				% single RC stage response :
				A1 = 1 / ( C(1,1) * C(1,4) ) ;
				B1 = jw ;
				F1 = A1 + B1 ;
			else
				% 3rd-order response :
				A1 = 1 / ( ( C(1,1)^3 ) * C(1,2) * C(1,3) * C(1,4) ) ;
				if pass == 'h' | pass == 'H'
					F1 = A1 + jw * ( ( 2/C(1,1)^2 ) * ( 1 / ( C(1,2) * C(1,3) ) + 1 / ( C(1,2) * C(1,4) ) ) ) ;
					F1 = F1 + (jw.^2) * ( ( 1 / C(1,1) ) * ( 3 / C(1,2) + 1 / C(1,4) ) ) ;
				else
					F1 = A1 + jw * ( ( 1/C(1,1)^2 ) * ( 3 / ( C(1,4) * C(1,3) ) + 1 / ( C(1,2) * C(1,3) ) ) ) ;
					F1 = F1 + (jw.^2) * ( ( 2 / C(1,1) ) * ( 1 / C(1,4) + 1 / C(1,3) ) ) ;
				end
				B1 = jw.^3 ;
				F1 = F1 + B1 ;
			end
		elseif col == 3
			% 2nd-order response :
			A1 = 1 / ( ( C(1,1)^2 ) * C(1,2) * C(1,3) ) ;
			if pass == 'h' | pass == 'H'
				F1 = A1 + jw * ( 2 / ( C(1,1) * C(1,2) ) ) ;
			else
				F1 = A1 + jw * ( 2 / ( C(1,1) * C(1,3) ) ) ;
			end
			B1 = jw.^2 ;
			F1 = F1 + B1 ;
		end
	
		if pass == 'h' | pass == 'H'
			F1 =  B1 ./ F1 ;
		else
			F1 = A1 ./ F1 ;
		end

		if row > 1
			% all other rows are 2nd-order responses :
			for k = 2 : row
				A2 = 1 / ( ( C(k,1)^2 ) * C(k,2) * C(k,3) ) ;
				if pass == 'h' | pass == 'H'
					F2 = A2 + jw * ( 2 / ( C(k,1) * C(k,2) ) ) ;
				else
					F2 = A2 + jw * ( 2 / ( C(k,1) * C(k,3) ) ) ;
				end
				B2 = jw.^2 ;
				F2 = F2 + B2 ;
				
				if pass == 'h' | pass == 'H'
					F2 = B2 ./ F2 ;
				else
					F2 = A2 ./ F2 ;
				end

				F1 = F1 .* F2 ;
			end
		end
	elseif layo == 2
		disp('Sorry, this configuration has not been evaluated yet !')
		return ;
	elseif layo == 3
		% Multiple-Feedback layout
		[row, col]=size(C);

		% C(:,1) are resistances in kohms :
		% the rest are capacitances in nF :
		C = C * 1e-9 ;
		C(:,1) = C(:,1) * 1e+9 * 1e+3 ;

		% is correction of R for Ao needed here, or was it included in C calculation ???
		% C(:,1) = C(:,1)

		O = C ;

		% check if last element in the first row is NaN :
		if isnan( C(1,4) )
			% --> rem( nord, 2 ) == 0  --> even order system :
			col = col - 1 ;
		end
		
		% first row :
		if col == 4
			if stage1 == 1
				% single RC stage response :
				A1 = 1 / ( Ao * C(1,1) * C(1,4) ) ;
				B1 = jw ;
				F1 = (A1 + B1)/Ao ;
			else
				% 3rd-order response :
				A1 = 1 / ( Ao * ( C(1,1)^3 ) * C(1,2) * C(1,3) * C(1,4) ) ;
				F1 = A1 + jw * ( ( 1/C(1,1)^2 ) * ( (3+1/Ao) / ( C(1,4) * C(1,3) ) + 1 / ( 2*Ao * C(1,2) * C(1,3) ) ) ) ;
				F1 = F1 + (jw.^2) * ( ( 1 / C(1,1) ) * ( 2 / C(1,4) + (2+1/(2*Ao)) / C(1,3) ) ) ;
				B1 = jw.^3 ;
				F1 = (F1 + B1)/Ao ;
			end
		elseif col == 3
			% 2nd-order response :
			A1 = 1 / ( ( Ao * C(1,1)^2 ) * C(1,2) * C(1,3) ) ;
			F1 = A1 + jw * ( (2+1/Ao) / ( C(1,1) * C(1,3) ) ) ;
			B1 = jw.^2 ;
			F1 = (F1 + B1)/Ao ;
		end
	
		F1 = A1 ./ F1 ;

		if row > 1
			% all other rows are 2nd-order responses :
			for k = 2 : row
				A2 = 1 / ( Ao * ( C(k,1)^2 ) * C(k,2) * C(k,3) ) ;
				F2 = A2 + jw * ( (2+1/Ao) / ( C(k,1) * C(k,3) ) ) ;
				B2 = jw.^2 ;
				F2 = (F2 + B2)/Ao ;
				
				F2 = A2 ./ F2 ;

				F1 = F1 .* F2 ;
			end
		end
		
	elseif layo > 3
		disp('Sorry, this configuration has not been evaluated yet !')
		return ;
	end

	% form the ideal reference response in F2 :
	% denormalize poles and zeros :
	pporg=pporg*2*pi;
	zzorg=zzorg*2*pi;
	if pass == 'h' | pass == 'H'
		pporg=pporg.\1;
	end
	F2 = ones( size( jw ) ) ;
	for k = 1 : max( size( pporg ) )
		F2 = F2 ./ ( jw + f0 * pporg( k ) ) ;
		if pass == 'h' | pass == 'H'
			F2 = jw .* F2 ;
		else
			F2 = F2 * ( f0 * pporg( k ) ) ;
		end
	end
	% in the equations above f0 need not be multiplied by 2*pi, since poles are normalized !!!

	% set reference gain same as system :
	if layo == 3
		if Ao ~= 1
			F2=F2*Ao^nstages;
		end
	end
	
	% plot both responses :
	figure(figh+1)
	semilogx( fx, 20*log10(abs(F2)), '-k', fx, 20*log10(abs(F1)), '-r' )
	ylabel( 'Magnitude [dB]' )
	bigtitle( 'Frequency response compared to ideal' ) ;
	pause

	% plot error :
	figure(figh+2)
	semilogx( fx, 20*log10(abs(F1)) - 20*log10(abs(F2)), '-r' )
	ylabel( 'Magnitude [dB]' )
	bigtitle( 'Magnitude error' );
end

if opt == 'y' | opt == 'Y'
	peps=eps;
	eps=1e-6;
	if pass == 'h' | pass == 'H'
		% C(:,1) are capacitances in nF :
		% the rest are resistors in kohms :
		C = C * 1e-3 ;
		C(:,1) = C(:,1) * 1e+9 * 1e+3 ;
	else
		% C(:,1) are resistors in kohms,
		% the rest are capacitances in nF :
		C = C * 1e+9 ;
		C(:,1) = C(:,1) * 1e-9 * 1e-3 ;
	end
		
	% set number display format with 2 decimal digits :
	format bank ;
	disp(' ')
	disp('Results :')
	disp(' ')
	if layo == 1
		if pass == 'h' | pass == 'H'
			disp('        C[nF]        R1[kOhm]      R2[kOhm]      R3[kOhm]   ')
		else
			disp('        R[kOhm]       C1[nF]        C2[nF]        C3[nF]   ')
		end
	elseif layo == 2
		if pass == 'h' | pass == 'H'
			disp('        Ci[nF]       Ri[kOhm]       Re[kOhm]      Rf[kOhm]   ')
		else
			disp('        Ri[kOhm]      Ci[nF]        Re[kOhm]      Rf[kOhm]   ')
		end
	elseif layo == 3
			disp(['        Ao = ', num2str(Ao), '   for each stage'])
			disp('        R[kOhm]       C1[nF]        C2[nF]        C3[nF]        Rgain')
	elseif layo > 3
			disp('Sorry, no results for this layout!')
			clear C;
			C=NaN;
	end

	disp('------------------------------------------------------------------------------------------')
	if layo == 3
		[rows,cols]=size(C);
		Rgain=Ao*C(:,1);
		if ~isnan(C(1,4))
			Rgain(1,1)=2*Rgain(1,1);
		end
		disp([C+eps,Rgain.*ones(rows,1)])
	else
		disp(C+eps)
	end
	disp(' ')
	eps=peps;
end
% restore the default number format :
format ;
