function y=atdr(z,p,ta,q,gf)
%ATDR	Analytical Time-Domain Response, calculated as a sum of residues (simpified).
%	
%	y=atdr(z,p,t)		or
%	y=atdr(z,p,t,'s')	returns the step response of the system
%				specified by zeros (column vector z) and
%				poles (column vector p), in time (vector t).
%	
%	y=atdr(z,p,t,'i')	returns the system impulse response, normalized
%				in amplitude for an ideal infinite amplitude,
%				infinitely narrow input impulse.
%	
%	y=atdr(z,p,t,'n')	returns the system impulse response, normalized
%				to the unity gain.
%	y=atdr(z,p,t,'s',1) plots the result.
%	Specify the time vector like this :	t=(0:1:N-1)*deltat
%	where : N is the number of desired time samples, and
%	        deltat is the sampling period.
%	For time < 0, zeros are returned, asuming a causal system.
%
%	Example :		[z,p]=cheb1np(5,0.1) ;
%						t=(0:1:500)*0.01 ;
%						y=atdr(z,p,t,'s') ;
%
%	NOTE : ATDR does not work for systems with coincident poles.
%	       For such systems, split the poles in two or more non-coincident
%			 groups, then calculate the desired response from the first group
%			 and the impulse response from the second group (and others) and
%			 make a convolution (see CONV).
%
%	See FREQW, PHASE and GDLY. See also TRESP.

%	Erik Margan, Institute Jozef Stefan, Ljubljana, Slovenija
%	First editing     : 921008
%	Matlab-IV version : 940915
%	Last revision     : 000127
%	Matlab-V version  : 000805

if nargin < 5
   gf = 0 ;
end
if nargin < 4
	q='s';		% if unspecified, return the step response
end

np=length(p);		% number of poles
nz=length(z);		% number of zeros
				% test for coincident poles :
for k=1:np
	P=p;
	P(k)=[];	% elliminate current pole
				% check if there is another pole of the same value :
	if all(abs(P-p(k)))==0
		error('Sorry, ATDR can''t handle coincident-poles !  Type >>help atdr<< !')
	end
end

dc = 1 ;	% set low-pass system flag
if isempty(z)
	% no zeros
	Z=1 ;
else
	% zeros
	if any( abs( z ) < 1e-6 )
		dc = 0 ;
		% high-pass or band-pass system requried, clear the dc flag
	end

	if all( abs( real( z ) ) < 1e-6 )
		% all zeros on imaginar axis
		z = j * imag( z ) ;
	end
	Z=ones(size(p)) ;
		
	if dc
		for k = 1:np
			Z(k,:) = prod(p(k)-z)/prod(-z) ;
			% Z(k,:) = prod(p(k)+z)/prod(z) ;
		end
	else
		for k = 1:np
			for h = 1:nz
				if z(h) == 0
					% Z(k,:) = Z(k,:)*(-p(k)) ;
					Z(k,:) = Z(k,:)*p(k) ;
				else
					% Z(k,:) = Z(k,:)*(p(k)-z(h))/z(h) ;
					Z(k,:) = Z(k,:)*(p(k)+z(h))/z(h) ;
				end
			end
		end
	end
	
	Z=Z(:) ;		% column-wise orientation
end

if np == 1
	D=1 ;			% single pole case
else
	for k = 1:np
		d=p(k)-p ;	% difference, row orientation
		d(k)=[] ;	% elliminate zero from difference
		D(:,k)=d ;	% k-th column of D
	end
	if np > 2
		D=prod(D) ;	% column-wise product if D is a matrix
	end
	D=D.' ;		% column-wise orientation
end

P=prod(-p)*Z ./D ;	% impulse residues

if q == 's'
	P=P./p ;		% step response required, divide by p
end

nt=length(ta) ;	% total number of time samples required
x=find(ta>=0) ;		% make a positive-only time vector
t=ta(x) ;
t=t(:).' ;		% time vector, row orientation
no=nt-length(t) ;	% number of negative time samples

y=P(1)*exp(p(1)*t) ;	% response, first row
for k = 2:np
	y=[y; P(k)*exp(p(k)*t)] ;	% next row
	y=sum(y) ;	% sum column-wise, return single row
end

y=real(y) ;		% real only (imaginary part < 1e-14)

if ( q == 's' ) & ( isempty(z) | dc == 1 )
	y=y+1;		% for all-pole system step response and for a band-stop system,
					% add 1 for the pole at origin (0+j0)
end

if ( q == 'i' | q == 'n' ) & ( dc == 0 )
	y=-diff([0, y]);	% impulse response of a high-pass system
end

if q == 'n'
	y=y/abs(sum(y));	% normalize impulse to unity gain
end

% return zeros for negative time :
y=[zeros(1,no), y];

% plot if gf is nonzero :
if gf
   plot( t, y, '-b' )
   xlabel( 'Time' )
   ylabel( 'Amplitude')
end
