function [x, cnt] = fitx(ffun,x,fdata,tol,prnt)
%FITX	Minimum of a function of several variables (nonlinear optimization).
%	X = FITX('f',x,fdata) starts at x and produces a new vector x which
%	minimizes f(x,fdata).  'f' is a string containing the name of the
%	function to be minimized, normally an M-file and fdata is a vector
%	of data to which the function is fited. The variable returned
%	by 'f' must be a norm(y(x)-fdata).
%
%	There are two optional input arguments and one optional output
%	argument.
%	X = FITX(F,X,fdata,tol) uses tol for the stopping tolerance. The
%	default tol is 1.0e-3.
%	X = FITX(F,X,fdata,tol,1)  succinctly describes each step.
%	[X, cnt] = FITX(F,X,fdata,...)  also returns a step count.
%
%	This function uses the Nelder-Mead simplex algorithm.

%	C. Moler, 8-19-86
%	Copyright (c) 1986-88 by the MathWorks, Inc.

%	Ref: D. J. Woods, Report 85-5, Dept. Math. Sciences, Rice Univ.,
%	May, 1985.

%	Added variable "fdata" ; Erik Margan *911004* ;

[n,m] = size(x);
if m > n
   x = x';
   n = m;
end
if nargin < 4, tol = 1.e-3; end
if nargin < 5, prnt = 0; end
cnt = 0;

% Set up a simplex near the initial guess.
v = 0.9*x;
f = feval(ffun,v,fdata);
for j = 1:n
   y = x;
   if y(j) ~= 0
      y(j) = 1.1*y(j);
   else
      y(j) = 0.1;
   end
   v = [v y];
   f = [f feval(ffun,y,fdata)];
end
[f,j] = sort(f);
v = v(:,j);

if prnt
   clc
   format compact
   format short e
   home
   cnt
   disp('initial ')
   disp(' ')
   v
   f
end

% Iterate until the diameter of the simplex is less than tol.
while 1
   test = 0;
   for j = 2:n+1, test = max(test,norm(v(:,j)-v(:,1),1)); end
   if test <= tol, break, end
   [v,f,how] = fitstep(ffun,v,fdata,f);
   cnt = cnt + 1;
   if prnt
      home
      cnt
      disp(how)
      disp(' ')
      v
      f
      test
   end
end
x = v(:,1);
if prnt, format, end
