package algebras;

import terms.*;

/** An algebra over the natural numbers, yielding <code>Long</code> Objects
  * as values. Available operations are all
  * <code>long</code> numbers (as constants), the unary operations
  * <code>s</code> (successor), - (negation), and
  * <code>abs</code> (absolute value), and
  * the binary operations <code>max</code>, <code>min</code>,  + (plus),
  * - (minus), * (times), <code>div</code>
  * (integer division), and <code>mod</code> (remainder).
  * Since the semantics of symbols is fixed, the <code>parse</code>
  * method is meaningless. It simply does nothing.
  * @see algebra
  * @see symbol
  */
public class intAlgebra extends extensibleAlgebra {

  private static symbol neg   = new symbol("-", 1);
  private static symbol abs   = new symbol("abs", 1);
  private static symbol succ   = new symbol("s", 1);
  private static symbol max   = new symbol("max", 2);
  private static symbol min   = new symbol("min", 2);
  private static symbol plus  = new symbol("+", 2);
  private static symbol minus  = new symbol("-", 2);
  private static symbol times = new symbol("*", 2);
  private static symbol div = new symbol("div", 2);
  private static symbol mod = new symbol("mod", 2);
  
/** A symbol of rank 0 whose string representation is an integer yields that integer;
  * other operations (that behave as expected) are the unary operations
  * <code>s</code> (successor), - (negation), and
  * <code>abs</code> (absolute value), and
  * the binary operations <code>max</code>, <code>min</code>,  + (plus),
  * - (minus), * (times), <code>div</code>
  * (integer division), and <code>mod</code> (remainder).
  */
  protected Object apply(symbol op, Object[] args) {
    if (args.length == 0) return operationOf(op);
    long[] longArgs = new long[args.length];
    for (int i = 0; i < args.length; i++) {
      if (args[i] instanceof Long) longArgs[i] = ((Long)args[i]).longValue();
      else return null;
    }
    if (succ.equals(op)) return new Long(longArgs[0]+1);
    if (neg.equals(op)) return new Long(-longArgs[0]);
    if (abs.equals(op)) return new Long(Math.abs(longArgs[0]));
    if (max.equals(op)) return new Long(Math.max(longArgs[0], longArgs[1]));
    if (min.equals(op)) return new Long(Math.min(longArgs[0], longArgs[1]));
    if (plus.equals(op)) return new Long(longArgs[0] + longArgs[1]);
    if (minus.equals(op)) return new Long(longArgs[0] - longArgs[1]);
    if (times.equals(op)) return new Long(longArgs[0] * longArgs[1]);
    if (div.equals(op)) return new Long(longArgs[0] / longArgs[1]);
    if (mod.equals(op)) return new Long(longArgs[0] % longArgs[1]);
    return null;
  }
  
/** If a symbol has rank 0 and represents an integer (in the <code>long</code> range) the
  * corresponding <code>Long</code> object is returned.
  */
  protected Object extendBy(symbol s) {
    try { return new Long(s.toString()); }
    catch (NumberFormatException e) { return null; }
  }
  
}
