package applications.collages;

import java.awt.*;
import java.awt.geom.*;
import java.awt.image.*;
import java.awt.font.*;
import java.util.*;
import algebras.*;
import displays.*;
import util.*;
import gui.*;
import applications.geoWorld.geoWorldAlgebra;

/** A display for objects belonging to the class <code>collage</code>.
  * Collages are displayed on an object of type <code>canvas</code>,
  * which allows dragging and zooming by means of the mouse. Initially,
  * the origin is placed at the centre of the canvas.
  * @see canvas
  */
public class collageDisplay extends PSDisplay {

  private collage displayed = null;
  private displayCanvas theCanvas = new displayCanvas();
  private static double initialScaling = 10;
  private boolean autoZoom = false;
  private boolean clip = false;
  private static double autoZoomSize = .9;
  private static double zoomFactor = 1.5;
  private double width = 500;
  private double height = 500;
  private static String zoomIn = "zoom in";
  private static String zoomOut = "zoom out";
  private static String auto = "auto zoom";
  private static String manual = "manual zoom";
  private static String reset = "reset";
  private static String clipOn = "clipping on";
  private static String clipOff = "clipping off";
  private static String[][] manualCommands = {{zoomIn, zoomOut}, {auto, reset}};
  private static String[][] autoCommands = {{manual}};
  private static String[] clipOffCommands = {clipOn};
  private static String[] clipOnCommands = {clipOff};
  private static String noInput = "missing or undefined input";
  private static Font noInputFont = new Font("Monospaced", Font.BOLD, 14);
  
  public collageDisplay() { 
    theCanvas.setSize((int)width,(int)height);
    theCanvas.resetView();
  }

/** This type of display can deal only with subclasses of
  * <code>collageAlgebra</code>; other algebras will be rejected.
  * @see collageAlgebra
  */
  public boolean acceptable(algebra a) {
    return a instanceof collageAlgebra || a instanceof geoWorldAlgebra;
  }

/** Collages are displayed in the way described above. */
  protected void displayObject(Object obj) {
    if (obj instanceof collage) displayed = (collage)obj;
    else displayed = null;
    setAbs();
    theCanvas.repaint();
  }
  
  private void setAbs() {
    if (displayed != null) {
      Rectangle2D.Double bounds = displayed.bounds();
      bounds.x -= bounds.width * (1-autoZoomSize)/2;
      bounds.width /= autoZoomSize;
      bounds.y -= bounds.height * (1-autoZoomSize)/2;
      bounds.height /= autoZoomSize;
      theCanvas.setAbsSilently(bounds.x, bounds.x + bounds.width, -bounds.y-bounds.height, -bounds.y);
    }
    else theCanvas.setAbsSilently(0,100,0,100);
  }

/** Returns the <code>canvas</code> on which the collages will be shown.
  * @see canvas
  */
  public Component visualizer() {
    return theCanvas;
  }

/** Available commands are "zoom in", "zoom out", "auto zoom / manual zoom", "reset",
    and "clipping on/off". */
  public list commands() {
    list result = new list();
    String [][] commands = autoZoom ? autoCommands : manualCommands;
    for (int i = 0; i < commands.length; i++) result.append(commands[i]);
    if (super.PSOutputPossible()) {
      if (clip) result.append(clipOnCommands); else result.append(clipOffCommands);
    }
    result.concat(super.commands());
    return result;
  }
  
  public void execute(String command) {
    if (zoomIn.equals(command)) {
      theCanvas.scaleBy(zoomFactor);
      theCanvas.repaint();
    }
    else if (zoomOut.equals(command)) {
      theCanvas.scaleBy(1/zoomFactor);
      theCanvas.repaint();
    }
    else if (reset.equals(command)) {
      theCanvas.enableScaling(true);
      theCanvas.resetView();
    }
    else if (auto.equals(command) || manual.equals(command)) {
      autoZoom = auto.equals(command);
      theCanvas.enableScrolling(!autoZoom);
      theCanvas.enableScaling(!autoZoom);
      if (autoZoom) theCanvas.repaint();
    } 
    else if (clipOn.equals(command) || clipOff.equals(command)) {
      clip = clipOn.equals(command);
    }
    else {
      super.execute(command);
    }
  }
  

  protected final boolean PSOutputPossible() {
    return super.PSOutputPossible() && displayed != null;
  }
  
  protected final void PSOutput() {
    Enumeration parts = displayed.parts();
    while (parts.hasMoreElements()) {
      part p = (part)parts.nextElement();
      if (clip && !p.getOutline().intersects(theCanvas.visible)) continue;
      Color c = p.getColour();
      PSSetColour(((double)c.getRed())/255,
                           ((double)c.getGreen())/255,
                           ((double)c.getBlue())/255);
      double thickness = p.getThickness();
      PSOutputBezierPart(p, thickness);
    }
    PSOutputPreamble();
  }
  
// The following should hopefully handle even the situation where the
// outline of the part consists of several subsegments separated by
// SEG_MOVETO and/or SEG_CLOSE.
  private void PSOutputBezierPart(part p, double thickness) {
    PathIterator it = p.getOutline().getPathIterator(null);
    double[] coord = new double[6];
    double[] bounds = p.largeBounds();
    PSSetbbox(bounds[0], bounds[1],bounds[2], bounds[3]);
    while (!it.isDone()) {
      switch (it.currentSegment(coord)) {
        case PathIterator.SEG_MOVETO:
          PSMoveto(coord[0],coord[1]);
          break;
        case PathIterator.SEG_LINETO:
          PSNextPointOfPath(coord[0],coord[1]);
          break;
        case PathIterator.SEG_QUADTO:
          PSNextBezierSegment(coord[0],coord[1],coord[2],coord[3],coord[2],coord[3]);
          break;
        case PathIterator.SEG_CUBICTO:
          PSNextBezierSegment(coord[0],coord[1],coord[2],coord[3],coord[4],coord[5]);
          break;
        case PathIterator.SEG_CLOSE:
          PSWrite("  closepath\n");
          break;
      }
      it.next();
    }
    PSStrokeOrFill(p,thickness);
  }
  
  private void PSStrokeOrFill(part p, double thickness) {
    if (p.type() == part.FILLED) PSFillPath();
    else if (thickness >= 0) PSStrokeThickPath(thickness);
    else PSStrokePath();
  }
  
  private void PSOutputPreamble() {
    Rectangle2D.Double r = clip ? theCanvas.visible : displayed.bounds();
    PSPreamble(r.x,r.y,r.x+r.width,r.y+r.height);
  }
    
  private final class displayCanvas extends scrollCanvas {
  
    /**
	 * 
	 */
	private static final long serialVersionUID = 1393729918001416214L;
	public Rectangle2D.Double visible = new Rectangle2D.Double(0,0,0,0);
    private AffineTransform view = new AffineTransform();
    private BufferedImage buffer = null;
    
    private void resetView() {
      Dimension s = getSize();
      setViewSilently(-s.width / initialScaling, s.width / initialScaling,
                      -s.height / initialScaling, s.height / initialScaling);
      repaint();
    }

    public void paint(Graphics g) {
      super.paint(g);
      if (buffer != null) g.drawImage(buffer,0,0,this);
      else repaint();
    }
    
    public void update(Graphics g) {
      Dimension size = getSize();
      g.setClip(0,0,size.width,size.height);
      waiting(g);
      BufferedImage newBuffer = (BufferedImage)createImage(size.width,size.height);
      Graphics2D g2D = newBuffer.createGraphics();
      g2D.setBackground(getBackground());
      if (displayed != null) {
        if (autoZoom) setZoom();
        double scaling = size.width / (viewR - viewL);
        view.setToTranslation(size.width/2.0,size.height/2.0);
        view.scale(scaling,-scaling);
        view.translate(-(viewL + viewR)/2.0, (viewB + viewT)/2.0);
        visible.x = viewL; 
        visible.y = -viewB;
        visible.width = viewR - viewL;
        visible.height = viewB - viewT;
        g2D.setTransform(view);
        Enumeration parts = displayed.parts();
        while (parts.hasMoreElements()) drawPart((part)parts.nextElement(), g2D, scaling);
      }
      else {
        g2D.setColor(Color.red);
        g2D.translate(getSize().width/2, getSize().height/2);
        g2D.setFont(noInputFont);
        FontMetrics metrics = g2D.getFontMetrics(noInputFont);
        int width = metrics.stringWidth(noInput);
        int height = metrics.getMaxAscent();
        g2D.drawString(noInput, -width/2, height/2);
        int space = metrics.getMaxAdvance();
        g2D.drawRect(-(width+space)/2, -height, width+space, 2*height);
      }
      buffer = newBuffer;
      paint(g);
      adjustScrollBars();
    }
    
    private void waiting(Graphics g) {
      if (buffer != null  && displayed != null && displayed.numParts() > 5000) {
        String message = "Creating image...";
        Dimension size = getSize();
        Graphics2D g2D = buffer.createGraphics();
        g2D.clearRect(0,0,size.width,size.height);
        Rectangle2D.Float bounds = new Rectangle2D.Float(0,0,0,0);
        FontMetrics fm = g2D.getFontMetrics();
        LineMetrics lm = fm.getLineMetrics(message,g2D);
        bounds.setRect(fm.getStringBounds(message,g2D));
        g2D.drawString(message,(int)(size.width-bounds.width)/2,
                      (int)(size.height+bounds.height)/2-(int)lm.getDescent());
        paint(g);
      }
    }
    
    public void setZoom() {
      Dimension size = getSize();
      Rectangle2D.Double bounds = displayed.bounds();
      bounds.x -= bounds.width * (1-autoZoomSize)/2;
      bounds.y -= bounds.height * (1-autoZoomSize)/2;
      bounds.width /= autoZoomSize;
      bounds.height /= autoZoomSize;
      if (((double)size.width)/((double)size.height) >= bounds.width / bounds.height) {
        double w = bounds.height / size.height * size.width;
        bounds.x -= (w-bounds.width)/2;
        bounds.width = w;
      }
      else {
        double h = bounds.width / size.width * size.height;
        bounds.y -= (h-bounds.height)/2;
        bounds.height = h;
      }
      setViewSilently(bounds.x, bounds.x + bounds.width, -bounds.y - bounds.height, -bounds.y);
    }
    
    public void scaleBy(double factor) {
//      if (displayed != null) {
        double w = (viewR - viewL) / factor;
        double h = (viewB - viewT) / factor;
        setViewSilently((viewL + viewR - w) / 2, (viewL + viewR + w)/2,
                       (viewT + viewB - h) / 2, (viewT + viewB + h)/2);
//      }
    }
    
    private void drawPart(part p, Graphics2D g2D, double scaling) {
      g2D.setColor(p.getColour());
      GeneralPath outline = p.getOutline();
      if (p.type() == part.FILLED || p.getThickness() < 0) {
        g2D.setStroke(new BasicStroke((float)(1.0/scaling),BasicStroke.CAP_BUTT,BasicStroke.JOIN_MITER));
      }
      else g2D.setStroke(new BasicStroke((float)p.getThickness(),BasicStroke.CAP_BUTT,BasicStroke.JOIN_MITER));
      g2D.draw(outline);
      if (p.type() == part.FILLED) g2D.fill(outline);
    }
    
    protected void resizeView() {
      double newWidth = getSize().width;
      double newHeight = getSize().height;
      if (!autoZoom) {
        double d = (viewR - viewL) * (newWidth / width);
        double L = (viewL + viewR - d) / 2;
        double R = (viewL + viewR + d) / 2;
        d = (viewB - viewT) * (newHeight / height);
        setViewSilently(L, R, (viewT + viewB - d) / 2, (viewT + viewB + d) / 2);
      }
      width = newWidth;
      height = newHeight;
      invalidate();
      repaint();
    }
    
    protected void scalingRequest(double L, double R, double T, double B) {
//      if (displayed != null) {
        setViewSilently(L,R,T,B);
        resizeView();
//      }
    }

  }

}
