package applications.lineDrawings;

import java.awt.*;
import java.util.*;
import algebras.*;
import displays.*;

/** A display that can visualize objects of the class <code>lineDrawing</code>.
  * The drawings are automatically centred and scaled; no zooming or dragging
  * facilities are implemented yet.
  */
public class lineDrawingDisplay extends PSDisplay {

  private lineDrawing displayed = null;
  private displayCanvas theCanvas = new displayCanvas();
  private static int distance = 20;
  
  public lineDrawingDisplay() { 
    theCanvas.setSize(500,500);
  }

/** This type of display can deal only with subclasses of
  * <code>lineDrawingAlgebra</code>; other algebras will be rejected.
  * @see lineDrawingAlgebra
  */
  public boolean acceptable(algebra a) {
    return a instanceof lineDrawingAlgebra;
  }
  
/** Objects of the class <code>lineDrawing</code> are displayed as described above. */
  protected void displayObject(Object obj) {
    if (obj instanceof lineDrawing) displayed = (lineDrawing)obj;
    else displayed = null;
    theCanvas.invalidate();
    theCanvas.repaint();
  }
  
  protected final boolean PSOutputPossible() {
    return super.PSOutputPossible() && displayed != null;
  }
  
  protected void PSOutput() {
    PSDraw(displayed);
    double[] bounds = computeBounds();
    PSPreamble(bounds[0],bounds[1],bounds[2],bounds[3]);
  }
  
  private void PSDraw(lineDrawing draw) {
    Enumeration enm = draw.elements(lineDrawing.returnAsIs, lineDrawing.flatten);
    while (enm.hasMoreElements()) {
      Object elem = enm.nextElement();
      double xMin, yMin, xMax, yMax;
      if (elem instanceof line) {
        line l = (line)elem;
        if (!l.hidden) {
          xMin = Math.min(l.x1, l.x2);
          yMin = Math.min(l.y1, l.y2);
          xMax = Math.max(l.x1, l.x2);
          yMax = Math.max(l.y1, l.y2);
          PSSetColour(0,0,0);
          PSSetbbox(xMin, yMin, xMax, yMax);
          PSMoveto(l.x1, l.y1);
          PSNextPointOfPath(l.x2, l.y2);
          PSStrokePath();
        }
      }
      else { // Current element is filled polygon
        lineDrawing ld = (lineDrawing)elem;
        PSDraw(ld);
        Color col = (Color)ld.specialInfo;
        PSSetColour(((double)col.getRed())/255,
                    ((double)col.getGreen())/255,
                    ((double)col.getBlue())/255);
        Enumeration lines = ld.elements(lineDrawing.flatten, lineDrawing.skip);
        if (lines.hasMoreElements()) {
//
// User path may get too long for ghostscript, resulting in a stack overflow...
//          PSStartPath(tmpBounds[0], tmpBounds[1], tmpBounds[2], tmpBounds[3]);
          PSStartLongPath();
          
          line l = (line)lines.nextElement();
          PSMoveto(l.x1, l.y1);
          while (lines.hasMoreElements()) {
            l = (line)lines.nextElement();
            PSNextPointOfPath(l.x1, l.y1);
          }
          PSNextPointOfPath(l.x2, l.y2);
//
// see above...
//          PSFillPath();
          PSWrite("eofill\n");
        }
      }
    }
  }
  
  private double[] computeBounds() {
    double[] bounds = computeBranchBounds(displayed);
    if (bounds[0] == Double.POSITIVE_INFINITY) {
      bounds[0] = bounds[1] = bounds[2] = bounds[3] = 0;
    }
    return bounds;
  }
  
  private double[] computeBranchBounds(lineDrawing drawing) {
    Enumeration enm = drawing.elements(lineDrawing.returnAsIs, lineDrawing.flatten);
    double[] bounds = new double[4];
    bounds[0] = bounds[1] = Double.POSITIVE_INFINITY;
    bounds[2] = bounds[3] = Double.NEGATIVE_INFINITY;
    while (enm.hasMoreElements()) {
      Object elem = enm.nextElement();
      if (elem instanceof line) {
        line l = (line)elem;
        if (l.hidden) continue;
        if (l.x1 < l.x2) {
          bounds[0] = Math.min(bounds[0],l.x1);
          bounds[2] = Math.max(bounds[2],l.x2);
        }
        else {
          bounds[0] = Math.min(bounds[0],l.x2);
          bounds[2] = Math.max(bounds[2],l.x1);
        }
        if (l.y1 < l.y2) {
          bounds[1] = Math.min(bounds[1],l.y1);
          bounds[3] = Math.max(bounds[3],l.y2);
        }
        else {
          bounds[1] = Math.min(bounds[1],l.y2);
          bounds[3] = Math.max(bounds[3],l.y1);
        }
      }
      else {
        double[] subbounds = computePolygonBounds((lineDrawing)elem);
        bounds[0] = Math.min(bounds[0],subbounds[0]);
        bounds[1] = Math.min(bounds[1],subbounds[1]);
        bounds[2] = Math.max(bounds[2],subbounds[2]);
        bounds[3] = Math.max(bounds[3],subbounds[3]);
      }
    }
    return bounds;
  }
  
  private double[] computePolygonBounds(lineDrawing drawing) {
    Enumeration enm = drawing.elements(lineDrawing.flatten, lineDrawing.returnAsIs);
    double[] bounds = new double[4];
    bounds[0] = bounds[1] = Double.POSITIVE_INFINITY;
    bounds[2] = bounds[3] = Double.NEGATIVE_INFINITY;
    while (enm.hasMoreElements()) {
      Object elem = enm.nextElement();
      if (elem instanceof line) {
        line l = (line)elem;
        if (l.x1 < l.x2) {
          bounds[0] = Math.min(bounds[0],l.x1);
          bounds[2] = Math.max(bounds[2],l.x2);
        }
        else {
          bounds[0] = Math.min(bounds[0],l.x2);
          bounds[2] = Math.max(bounds[2],l.x1);
        }
        if (l.y1 < l.y2) {
          bounds[1] = Math.min(bounds[1],l.y1);
          bounds[3] = Math.max(bounds[3],l.y2);
        }
        else {
          bounds[1] = Math.min(bounds[1],l.y2);
          bounds[3] = Math.max(bounds[3],l.y1);
        }
      }
      else {
        double[] subbounds = computeBranchBounds((lineDrawing)elem);
        bounds[0] = Math.min(bounds[0],subbounds[0]);
        bounds[1] = Math.min(bounds[1],subbounds[1]);
        bounds[2] = Math.max(bounds[2],subbounds[2]);
        bounds[3] = Math.max(bounds[3],subbounds[3]);
      }
    }
    return bounds;
  }
  
/** Returns the canvas on which the line drawings will be shown. */
  public Component visualizer() {
    return theCanvas;
  }

  private class displayCanvas extends Canvas {
  
    /**
	 * 
	 */
	private static final long serialVersionUID = -8661511216981700468L;
	double winCentreX, winCentreY, xCentre, yCentre, unit;
  
    public void paint(Graphics g) {
      super.paint(g);
      g.setColor(Color.black);
      int type = Cursor.WAIT_CURSOR;
      Cursor wait = Cursor.getPredefinedCursor(type);
      theCanvas.setCursor(wait);
      if (displayed != null) {
        double[] b = computeBounds();
        double xMin = b[0]; double yMin = b[1]; double xMax = b[2]; double yMax = b[3];
        Rectangle bounds = getBounds();
        bounds.width -= 2*bounds.x;
        bounds.height -= 2*bounds.y;
        winCentreX = bounds.x + bounds.width / 2;
        winCentreY = bounds.y + bounds.height / 2;
        double winWidth = bounds.width - 2*distance;
        double winHeight = bounds.height - 2*distance;
        if (xMax == xMin) {
          if (yMax == yMin) unit = 1;
          else unit = winHeight / (yMax - yMin);
        }
        else if (yMax == yMin) unit = winWidth / (xMax - xMin);
        else unit = Math.min(winWidth / (xMax - xMin), winHeight / (yMax - yMin));
        xCentre = xMin + (xMax - xMin) / 2;
        yCentre = yMin + (yMax - yMin) / 2;
        paint(g, displayed);
      }
      int oldtype = Cursor.DEFAULT_CURSOR;
      Cursor normal = Cursor.getPredefinedCursor(oldtype);
      theCanvas.setCursor(normal);
    }
    
    public void paint(Graphics g, lineDrawing draw) {
      Enumeration enm = draw.elements(lineDrawing.returnAsIs, lineDrawing.flatten);
      while (enm.hasMoreElements()) {
        Object elem = enm.nextElement();
        if (elem instanceof line) {
          line l = (line)elem;
          if (!l.hidden) {
            g.drawLine((int)Math.round(winCentreX + (l.x1 - xCentre) * unit),
                       (int)Math.round(winCentreY - (l.y1 - yCentre) * unit),
                       (int)Math.round(winCentreX + (l.x2 - xCentre) * unit),
                       (int)Math.round(winCentreY - (l.y2 - yCentre) * unit));
          }
        }
        else {
          Polygon poly = new Polygon();
          Enumeration lines = ((lineDrawing)elem).elements(lineDrawing.flatten, lineDrawing.skip);
          line l = null;
          while (lines.hasMoreElements()) {
            l = (line)lines.nextElement();
            poly.addPoint((int)Math.round(winCentreX + (l.x1 - xCentre) * unit),
                          (int)Math.round(winCentreY - (l.y1 - yCentre) * unit));
          }
          if (l != null) poly.addPoint((int)Math.round(winCentreX + (l.x2 - xCentre) * unit),
                                       (int)Math.round(winCentreY - (l.y2 - yCentre) * unit));
          g.setColor((Color)((lineDrawing)elem).specialInfo);
          g.fillPolygon(poly);
          g.setColor(Color.black);
          paint(g, (lineDrawing)elem);
        }
      }
    }
  }
  
}
