package generators;

import java.util.*;
import terms.*;
import parsers.*;
import util.*;

/** A "parallel deterministic tree grammar".
  * For every non-terminal symbol there are two productions the second of which must be
  * terminating. A derivation step consists in applying rules to all non-terminal symbols in
  * parallel, and in such a way that either the first rule is chosen for all symbols or the second
  * one is taken for all of them. (Thus, the rules are actually divided in two tables.) 
  * In order to enhance efficiency, the implementation uses sharing of subterms. As a
  * consequence, <code>advance</code> commands are carried out in constant time and
  * space. 
  */ 
public class pdtGrammar extends treeGrammar {

//====================================================================
// First the 'interactive' part that deals with commands ...
//====================================================================
  private static String advance = "advance";
  private static String back = "back";
  private static String reset = "reset";
  private static String ntCom = "nonterminal results";
  private static String tCom = "terminal results";
  private static String[] basicCommands = {advance, back, reset};
  
/** Parallel deterministic tree grammars understand the commands
  * <code>advance</code>, <code>back</code>,
  * and <code>reset</code> as well as <code>terminal results</code>
  * and <code>nonterminal results</code>, which allow to switch
  * between terminal and nonterminal output terms.
  */
  public list commands() {
    list result = new list();
    result.append(basicCommands);
    String[] tnt = new String[1];
    result.append(tnt);
    if (currKind == terminal) tnt[0] = ntCom;
    else tnt[0] = tCom;
    return result;
  }
  
  public void execute(String command) {
    if (advance.equals(command)) advance();
    else if (back.equals(command)) back();
    else if (reset.equals(command)) reset();
    else if (ntCom.equals(command) || tCom.equals(command)) currKind = 1-currKind;
  }

//====================================================================
// Now the actual implementation ...
//====================================================================

  private static int terminal = 0;
  private static int nonterminal = 1;
  
  private term[][] rules;
  private int main;
  private Vector levels = new Vector();
  private int currLevel = 0;
  private int currKind = terminal;
  
  private term[][] level(int l) { return (term[][])levels.elementAt(l); }
  
  public term currentTerm() { return level(currLevel)[currKind][main]; }
  
  public void reset() { currLevel = 0; }
  
  public void advance() {
    if (++currLevel == levels.size()) {
      term[][] oldLevel = level(currLevel - 1);
      term[][] newLevel = new term[2][rules.length];
      for (int i = 0; i < rules.length; i++) {
        newLevel[terminal][i] = newLevel(oldLevel[terminal], rules[i][0]);
        newLevel[nonterminal][i] = newLevel(oldLevel[nonterminal], rules[i][0]);
      }
      levels.addElement(newLevel);
    }
  }
  
  public void back() { currLevel = Math.max(currLevel - 1, 0); }
  
  private term newLevel(term[] oldLevel, term t) {
    symbol top = t.topSymbol();
    if (top instanceof variable) return oldLevel[((variable)top).index()];
    term result = new term(top);
    for (int i = 0; i < top.rank(); i++) {
      result.defineSubterm(i, newLevel(oldLevel, t.subterm(i)));
    }
    return result;
  }
  
/** Initialize this <code>pdtGrammar</code> by reading its definition from a stream.
  * The syntax is defined by the class <code>pdtGrammarParser</code>.
  * @see pdtGrammarParser
  * @exception ParseException if an error occurs
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
    pdtGrammarParser parser = new pdtGrammarParser(stream);
    parser.pdtGrammar();
    rules = parser.rules;
    main = parser.startIndex;
    Vector nt = parser.nonterminals;
    term[][] bottomLevel = new term[2][rules.length];
    for (int i = 0; i < rules.length; i++) {
      bottomLevel[terminal][i] = rules[i][1];
      bottomLevel[nonterminal][i] = new term(new symbol((String)nt.elementAt(i), 0));
    }
    levels.addElement(bottomLevel);
  }
  
}

