package gui;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import javax.swing.undo.*;
import java.io.*;
import gui.nodeTypes.*;

public class editor extends JFrame implements ActionListener, DocumentListener {

  /**
	 * 
	 */
  private static final long serialVersionUID = -7969931654315423028L;
  private static String openCMD = "Open...";
  private static String saveCMD = "Save";
  private static String saveasCMD = "Save as...";
  private static String saveqCMD = "Save & quit";
  private static String quitCMD = "Quit";
  private static String undoCMD = "Undo";
  private static String redoCMD = "Redo";
  private static String copyCMD = "Copy";
  private static String cutCMD = "Cut";
  private static String pasteCMD = "Paste";
  private static String selectAllCMD = "Select all";
  private static String reloadCMD = "Auto (re)load";
  private static String indentCMD = "Auto indent";
  
  private UndoManager undo = null;

  private String fileName = null;
  private myJTextArea text = new myJTextArea(30,80);
  JCheckBoxMenuItem reload = new JCheckBoxMenuItem(reloadCMD);
  JCheckBoxMenuItem indent = new JCheckBoxMenuItem(indentCMD);
  private worksheetNode myNode;
  private boolean unsaved = false;
  private componentGraph graph = null;
  private Point nodePos = null;
  
  private editor() {
    super();
    reload.setState(true);
    indent.setState(true);
    setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
    enableEvents(AWTEvent.WINDOW_EVENT_MASK);
    getContentPane().add(new JScrollPane(text));
    text.setFont(new Font("Monospaced",Font.PLAIN,14));
    text.setLineWrap(true);
    text.setWrapStyleWord(true);
    text.setMargin(new Insets(5,7,5,7));
    setJMenuBar(menus());
  }

  public editor(String fileName, worksheetNode node) {
    this();
    myNode = node;
    node.addEditor(this);
    if (load(fileName)) {
      pack();
      show();
      text.getDocument().addDocumentListener(this);
      text.getDocument().addUndoableEditListener(undo = new UndoManager());
    }
    else {
      dispose();
      return;
    }
  }
  
  public editor(Point pos, componentGraph g) {
    this();
    nodePos = pos;
    graph = g;
    pack();
    show();
    text.getDocument().addDocumentListener(this);
    text.getDocument().addUndoableEditListener(undo = new UndoManager());
  }
  
  protected void processWindowEvent(WindowEvent e) {
    if (e.getID() == WindowEvent.WINDOW_CLOSING && confirm()) {
      if (myNode != null) myNode.removeEditor(this);
      dispose();
    }
  }
  
  private JMenuBar menus() {
    JMenuBar bar = new JMenuBar();
    JMenu m = new JMenu("File");
    bar.add(m);
    addItem(new JMenuItem(openCMD),m,KeyEvent.VK_O);
    m.addSeparator();
    addItem(new JMenuItem(saveCMD),m,KeyEvent.VK_S);
    addItem(new JMenuItem(saveasCMD),m,-1);
    m.addSeparator();
    addItem(new JMenuItem(saveqCMD),m,KeyEvent.VK_E);
    addItem(new JMenuItem(quitCMD),m,KeyEvent.VK_Q);
    m = new JMenu("Edit");
    bar.add(m);
    addItem(new JMenuItem(undoCMD),m,KeyEvent.VK_Z);
    addItem(new JMenuItem(redoCMD),m,KeyEvent.VK_R);
    m.addSeparator();
    addItem(new JMenuItem(copyCMD),m,KeyEvent.VK_C);
    addItem(new JMenuItem(cutCMD),m,KeyEvent.VK_X);
    addItem(new JMenuItem(pasteCMD),m,KeyEvent.VK_V);
    m.addSeparator();
    addItem(new JMenuItem(selectAllCMD),m,KeyEvent.VK_A);
    m = new JMenu("Options");
    bar.add(m);
    addItem(reload,m,-1);
    addItem(indent,m,-1);
    return bar;
  }
  
  private void addItem(JMenuItem i, JMenu m, int key) {
    if (key != -1) i.setAccelerator(KeyStroke.getKeyStroke(key, ActionEvent.CTRL_MASK));
    i.addActionListener(this);
    m.add(i);
  }
  
  private boolean load(String file) {
    try {
      text.read(new FileReader(file),null);
      setTitle(file);
      fileName = file;
      return true;
    } catch (IOException e) {
      errorDisplay.show("Failed to read file\n" + file);
      return false;
    } catch (java.security.AccessControlException ex) {
      errorDisplay.show("Failed to write file\n" + fileName);
      return false;
    }
  }
  
  public void actionPerformed(ActionEvent e) {
    if (e.getActionCommand().equals(saveCMD)) save();
    else if (e.getActionCommand().equals(saveasCMD)) saveAs();
    else if (e.getActionCommand().equals(openCMD) && confirm()) open();
    else if (e.getActionCommand().equals(saveqCMD)) saveQuit();
    else if (e.getActionCommand().equals(quitCMD) && confirm()) {
      if (myNode != null) myNode.removeEditor(this);
      dispose();
    }
    else if (e.getActionCommand().equals(undoCMD)) {
      try { undo.undo(); }
      catch (CannotUndoException ex) {}
    }
    else if (e.getActionCommand().equals(redoCMD)) {
      try { undo.redo(); }
      catch (CannotRedoException ex) {}
    }
    else if (e.getActionCommand().equals(copyCMD)) text.copy();
    else if (e.getActionCommand().equals(cutCMD)) text.cut();
    else if (e.getActionCommand().equals(pasteCMD)) text.paste();
    else if (e.getActionCommand().equals(selectAllCMD)) text.selectAll();
  }
  
  private boolean save() {
    if (fileName == null) return saveAs();
    else {
      try {
        text.write(new FileWriter(fileName));
        setUnsaved(false);
        if (reload.getState()) {
          if (myNode != null) return myNode.reload(fileName);
          else return mkNode();
        }
        return true;
      } catch (IOException ex) {
        errorDisplay.show("Failed to write file\n" + fileName);
        return false;
      } catch (java.security.AccessControlException ex) {
        errorDisplay.show("Failed to write file\n" + fileName);
        return false;
      }
    }
  }
  
  private boolean saveAs() {
    String name = fileChooser.selectFile("Save file",FileDialog.SAVE);
    if (name != null) {
      try {
        text.write(new FileWriter(name));
        setUnsaved(false);
        fileName = name;
        setTitle(fileName);
        if (reload.getState()) {
          if (myNode != null) return myNode.reload(fileName);
          else return mkNode();
        }
        return true;
      } catch (IOException ex) {
        errorDisplay.show("Failed to write file\n" + fileName);
        return false;
      } catch (java.security.AccessControlException ex) {
        errorDisplay.show("Failed to write file\n" + fileName);
        return false;
      }
    }
    else return false;
  }
  
  private void open() {
    String name = fileChooser.selectFile("Open file in editor",FileDialog.LOAD);
    if (name != null) {
      text.getDocument().removeDocumentListener(this);
      load(name);
      text.getDocument().addDocumentListener(this);
    }
  }
  
  private void saveQuit() {
    if (save()) dispose();
  }
  
  private boolean mkNode() {
    myNode = graph.newNode(nodePos,fileName);
    if (myNode != null) myNode.addEditor(this);
    return myNode != null;
  }
  
  private boolean confirm() {
    if (unsaved) {
      int choice = JOptionPane.showConfirmDialog(null,
        "Save modified file?",
        "Unsaved file!",
        JOptionPane.YES_NO_CANCEL_OPTION,
        JOptionPane.QUESTION_MESSAGE);
      if (choice == 0) {
        try {
          text.write(new FileWriter(fileName));
          if (reload.getState()) myNode.reload(fileName);
        } catch (IOException ex) {
          errorDisplay.show("Failed to write file\n" + fileName);
          return false;
        } catch (java.security.AccessControlException ex) {
          errorDisplay.show("Failed to write file\n" + fileName);
          return false;
        }
        return true;
      }
      else if (choice == 2) return false;
      setUnsaved(false);
    }
    return true;
  }
    
  
  private synchronized void setUnsaved(boolean b) {
    unsaved = b;
  }
  
  public synchronized boolean isUnsaved() {
    return unsaved;
  }
  
  public synchronized void changedUpdate(DocumentEvent e) {
    setUnsaved(true);
  }
  
  public synchronized void insertUpdate(DocumentEvent e) {
    setUnsaved(true);
  }
  
  public synchronized void removeUpdate(DocumentEvent e) {
    setUnsaved(true);
  }
    
  private class myJTextArea extends JTextArea {
      
    /**
	 * 
	 */
	private static final long serialVersionUID = 216835670162489493L;

	public myJTextArea(int rows, int cols) {
      super(rows,cols);
    }
    
    protected void processKeyEvent(KeyEvent e) {
      super.processKeyEvent(e);
      if (!indent.getState()) return;
      try {
        if (e.getID()==KeyEvent.KEY_RELEASED && e.getKeyCode()==KeyEvent.VK_ENTER) {
          insert(getWhiteSpace(new StringBuffer(),getCaretPosition()-2),getCaretPosition());
        }
      } catch (BadLocationException ex) { }
    }
    
    private String getWhiteSpace(StringBuffer result, int i) throws BadLocationException {
      while (true) {
        if (i < 0) return result.toString();
        char c = getText(i--,1).charAt(0);
        if (c==' ' || c=='\t') result.insert(0,c);
        else if (c=='\n') return result.toString();
        else result.setLength(0);
      }
    }
  }
  
}
