package algebras;

import terms.*;
import parsers.*;

/** The YIELD algebra - an algebra that yields terms as values.
  * The symbols it gives a special meaning to are those of the form
  * <ul>
  * <li> <code>const-&lt;sym&gt;-</code><em>n</em> of rank 0,
  * where <em>n</em> is a natural number,
  * <li> <code>proj-</code><em>n</em> of rank 0,
  * where <em>n</em> is a natural number &gt;0, and
  * <li> <code>subst</code> of rank <em>m</em>+1.
  * </ul>
  * The first symbol is interpreted as the term <code>&lt;sym&gt;[x1,...,x<em>n</em>]</code>,
  * the second as the term <code>x<em>n</em></code>, and the third, if applied to terms
  * <em>t</em>0,...,<em>tm</em>, yields the term obtained from <em>t</em>0 by substituting
  * <em>ti</em> for each variable <code>x</code><em>i</em>
  * (<em>i</em>=1,...,<em>m</em>). All symbols which are not of one of these kinds are
  * interpreted as themselves (i.e., as in the free term algebra).
  */
public class YIELDAlgebra extends extensibleAlgebra {

  private static String substitution = "subst";
  private static String projection = "proj-";
  private static String constant = "const-";
  
/** Apply an operation to argument terms as described above. */
  protected Object apply(symbol op, Object[] args) {
    Object operation;
    if ((operation = operationOf(op)) == null) {
      term result = new term(op);
      for (int i = 0; i < op.rank(); i++) {
        if (args[i] instanceof term) result.defineSubterm(i, (term)args[i]);
        else return null;
      }
      return result;
    }
    if (operation instanceof constant) {
      symbol s = ((constant)operation).sym;
      term result = new term(s);
      for (int i = 0; i < s.rank(); i++) result.defineSubterm(i, new term(new variable(i+1)));
      return result;
    }
    if (operation instanceof projectionConstant) {
      return new term(new variable(((projectionConstant)operation).index));
    }
    if (operation instanceof substitutionOperation) {
      term result;
      if (args[0] instanceof term) result = (term)args[0];
      else return null;
      term[] subst = new term[args.length];
      for (int i = 1; i < args.length; i++) {
        if (args[i] instanceof term) subst[i] = (term)args[i];
        else return null;
      }
      return result.substitute(subst);
    }
    return null;
  }

/** Yields objects that represent the operations associated with the three types of symbols
  * described above.
  */
  public Object extendBy(symbol s) {
    if (s.rank() > 0 && substitution.equals(s.toString())) return new substitutionOperation();
    if (s.rank() > 0) return null;
    if (s.toString().startsWith(projection)) {
      int index;
      try { index = Integer.parseInt(s.toString().substring(projection.length())); }
      catch (NumberFormatException e) { return null; }
      if (index < 1) return null;
      else return new projectionConstant(index);
    }
    if (s.toString().startsWith(constant)) {
      String info = s.toString().substring(constant.length());
      int pos = info.lastIndexOf('-');
      if (pos < 0) return null;
      String name = info.substring(0, pos);
      int rank;
      try { rank = Integer.parseInt(info.substring(pos+1)); }
      catch (NumberFormatException e) { return null; }
      if (rank < 0) return null;
      return new constant(name, rank);
    }
    return null;
  }
  
/** Initialize a <code>YIELDAlgebra</code> by reading its definition from a stream.
  * @see YIELDAlgebraParser
  * @exception ParseException if an error occurs
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
    YIELDAlgebraParser parser = new YIELDAlgebraParser(stream);
    parser.YIELDAlgebra(this);
  }

  private class substitutionOperation { }

  private class projectionConstant {
    public int index;
    public projectionConstant(int index) { this.index = index; }
  }

  private class constant {
    public symbol sym;
    public constant(String name, int rank) { sym = new symbol(name, rank); }
  }

}
