package algebras;

import java.util.*;
import terms.*;

/** A type of algebras that allows to add new operations dynamically when "new" symbols are
  * encountered.
  * This is useful to implement algebras with a large (and perhaps even infinite) number of
  * symbols whose semantics is fixed. In a concrete situation only a rather small number of
  * these operations will actually be used. Thus, for efficiency reasons it is not a good idea
  * to compute the parameters of the required operations from a symbol every time that
  * symbol is encountered. Instead, the meaning should be computed only once (namely
  * when the symbol is encountered the first time) and memorized for future re-use. The
  * computation of the meaning of a symbol must be implemented by the method
  * <code>extendBy</code>, which should return an object representing the respective
  * operation (or <code>null</code> if the symbol has no meaning). Then,
  * <code>operationOf</code> yields the operation associated with a symbol. (This method
  * will call <code>extendBy</code> if the symbol is unknown so far, and will also store
  * the result in order to avoid repeated calls to <code>extendBy</code> with the same
  * argument).
  */
public abstract class extensibleAlgebra extends algebra {

  private Hashtable op = new Hashtable();

/** Get the operation associated with a symbol (represented by an object that has once
  * been returned as a result of a call to <code>extendBy</code>).
  */
  protected Object operationOf(symbol s) {
    Object o = op.get(s);
    if (o == null) {
      o = op.get(s.toString());
      if (o == null && (o = extendBy(s)) == null) return null;
    }
    op.put(s, o);
    return o;
  }

/** This should figure out whether a symbol has a meaning and, if so, it should return an
  * object representing that meaning; otherwise, <code>null</code> must be returned as
  * an indication that this symbol has no meaning.
  */
  protected abstract Object extendBy(symbol s);

/** This operation can be used to add symbol-operation pairs directly (which usually
  * should not be necessary, however, since it is done automatically when the method
  * <code>operationOf</code> is called).
  * @returns the old operation bound to the symbol (<code>null</code> if the symbol
                    was undefined until that moment)
  */
  public Object defineSymbol(symbol s, Object operation) { return op.put(s, operation); }
  
/** The same as the method above, except that no rank is considered.
  * <code>operationOf</code> will return this operation as a standard operation
  * for symbols with the given name, regardless of the rank. If the symbol/rank
  * combination is known, however, that one is taken.
  */
  public Object defineSymbol(String name, Object operation) {
    return op.put(name, operation);
  } 
    
}
