package displays;

import java.io.*;
import parsers.*;
import util.*;

/** A display that is able to write the displayed objects onto PostScript files.
  */
public abstract class PSDisplay extends display {
	
  private static String stop = "no ps-output";
  private static String resume = "ps-output";
  private static String reset = "reset ps-output";
  private boolean outputRunning = false;
  
  private double currentRed;
  private double currentGreen;
  private double currentBlue;
  private int currentCap;
  private double currentWidth;
  protected double lineWidth;
  protected int lineCap = 1;
  protected int lineJoin = 0;
  
  private boolean PSOutput = false;
  private String dirName;
  private String[] fileName;
  private RandomAccessFile file = null;
  private int number = 0;
  private static int fileOffset = 160; // space that will be left for preamble
  private static byte[] blanks = new byte[fileOffset]; { for (int i = 0; i < fileOffset; i++) blanks[i] = (byte)' '; }
  
/** The available commands are <code>no ps-output</code>
  * to turn PostScript output off,
  * <code>ps-output</code> to turn
  * it on again, and <code>reset ps-output</code> to set the
  * counter back to 1.
  */
  public list commands() {
    list l = new list();
    if (PSOutput) {
      String[] result = new String[2];
      if (outputRunning) result[0] = stop; else result[0] = resume;
      result[1] = reset;
      l.append(result);
    }
    return l;
  }
  
  public void execute(String command) {
    if (stop.equals(command)) outputRunning = false;
    else if (resume.equals(command)) {
      outputRunning = true;
      writePSFile();
    }
    else if (reset.equals(command)) number = 0;
  }
  
  private String composeName() {
    StringBuffer result = new StringBuffer(50);
    for (int i = 0; i < fileName.length-1; i++) {
      result.append(fileName[i]);
      result.append(number);
    }
    result.append(fileName[fileName.length-1]);
    return result.toString();
  }
  
/** The method that actually produces the PostScript output.
  */
  protected abstract void PSOutput();
  
/** In any subclass, this method should yield <code>true</code> if (and only if)
  * the display is currently able to produce PostScript output.
  */
  protected boolean PSOutputPossible() {
    return PSOutput;
  }
  
  private void writePSFile() {
    {
      if (PSOutput && outputRunning && PSOutputPossible()) {
        try {
          number++;
          File f = new File(dirName + File.separator + composeName());
          f.delete();
          file = new RandomAccessFile(f, "rw");
          file.write(blanks); // Leave enough space for preamble
        }
        catch (IOException e) { 
          System.err.println("Error: could not open file "
                                       + dirName + File.separator + composeName() + " (" + e + ")");
        }
        catch (SecurityException e) {
          System.err.println("Error: could not open file " +
                                        dirName + File.separator + composeName() + " (" + e + ")");
        }
        PSDefinitions();
        PSOutput();
        if (file != null) {
          try { file.close(); file = null; } catch (IOException e) { 
            System.err.println("Error: could not close file "
                                         + dirName + File.separator + composeName() + " (" + e + ")");
          }
        }
      }
    }
  }
  
  private final void PSDefinitions() {
//    PSWrite("/line { /toY exch def /toX exch def newpath moveto toX toY lineto stroke } def\n");
    PSDouble(lineWidth); PSWrite(" setlinewidth 0 setcolor " + lineCap + " setlinecap " + lineJoin + " setlinejoin true setstrokeadjust\n");
    currentRed = 0; currentGreen = 0; currentBlue = 0;
    currentCap = lineCap; currentWidth = -1;
  }
  
  public final void PSSetColour(double red, double green, double blue) {
    if (currentRed != red || currentGreen != green || currentBlue != blue) {
      currentRed = red;
      currentGreen = green;
      currentBlue = blue;
      PSDouble(red); PSWrite(" ");
      PSDouble(green); PSWrite(" ");
      PSDouble(blue); PSWrite(" setrgbcolor\n");
    }
  }
  
  private final void PSSetCap(int cap) {
    if (currentCap != cap) {
      currentCap = cap;
      PSWrite(cap + " setlinecap\n");
    }
  }
  
  private final void PSSetWidth(double width) {
    if (currentWidth != width) {
      currentWidth = width;
      if (width == -1) PSDouble(lineWidth);
      else PSCoord(width);
      PSWrite(" setlinewidth\n");
    }
  }
  
  public void PSSetbbox(double x1, double y1, double x2, double y2) {
    PSWrite("{ ");
    PSCoords(x1, y1); PSWrite(" "); PSCoords(x2, y2); PSWrite(" setbbox\n");
  }
  
  public void PSMoveto(double x, double y) {
    PSWrite("  ");
    PSCoords(x,y);
    PSWrite(" moveto\n");
  }
  
  public void PSStartLongPath() {
    PSWrite("newpath\n");
  }
  
  public void PSNextPointOfPath(double x, double y){
    PSWrite("  ");
    PSCoords(x,y);
    PSWrite(" lineto\n");
  }
    
  public void PSNextBezierSegment(double x1, double y1,
                                  double x2, double y2,
                                  double x3, double y3){
    PSCoords(x1, y1); PSWrite("  ");
    PSCoords(x2, y2); PSWrite("  ");
    PSCoords(x3, y3); PSWrite("  ");
    PSWrite(" curveto\n");
  }
  
  public void PSClosePath() {
    PSWrite("  closepath");
  }
  
  public void PSStrokePath() {
    PSWrite(" }\n");
    PSSetCap(1);
    PSSetWidth(-1);
    if (lineWidth == 0) PSWrite("ustroke\n");
    else PSWrite("matrix defaultmatrix matrix currentmatrix matrix invertmatrix matrix concatmatrix ustroke\n");
  }
  
  public void PSStrokeThickPath(double width) {
    PSWrite(" }\n");
    PSSetCap(0);
    PSSetWidth(width);
    PSWrite("ustroke\n");
  }
  
  public void PSFillPath() {
    PSWrite("} ueofill\n");
  }
  
  public void display(Object obj) {
    super.display(obj);
    if (obj != null) writePSFile();
  }
  
  protected final void PSWrite(String s) {
    try { file.writeBytes(s); }
    catch (IOException e) {
      System.err.println("Error: could not write to file "
                                   + dirName + File.separator + composeName() + " (" + e + ")");
    }
  }
  
  protected final void PSCoords(double x, double y) {
    PSCoord(x);
    PSWrite(" ");
    PSCoord(y);
  }

  protected final void PSCoord(double c) {
    PSDouble(makeCoord(c));
  }
  
  private final double makeCoord(double c) {
    return 72D/2.54D*c;
  }

  protected final void PSDouble(double d) {
    PSDouble(d, false);
  }
  
  private final void PSDouble(double d, boolean avoidExponent) {
    try {
      if (!avoidExponent) {
        if (d > Float.MAX_VALUE) file.writeBytes(Float.toString(Float.MAX_VALUE));
        else if (d < -Float.MAX_VALUE) file.writeBytes(Float.toString(-Float.MAX_VALUE));
        else file.writeBytes(Float.toString((float)d));
      }
      else {
        if (d < 0) file.writeByte('-');
        long l = Math.round(Math.abs(d*10000));
        file.writeBytes(Long.toString(l / 10000));
        l %= 10000;
        String rest = Long.toString(l);
        while (rest.length() < 4) rest = "0" + rest;
        int k = 4;
        while (--k >= 0 && rest.charAt(k) == '0') {}
        if (k >= 0) {
          file.writeByte('.');
          for (int i = 0; i <= k; i++) file.writeByte(rest.charAt(i));
        }
      }
    } catch (IOException e) { 
      System.err.println("Error: could not write to file "
               + dirName + File.separator + composeName() + " (" + e + ")");
    }
  }

  protected final void PSLine(double x1, double y1, double x2, double y2) {
    try {
      PSCoords(x1, y1); file.writeByte(' '); 
      PSCoords(x2, y2); file.writeBytes(" line\n");
    } catch (IOException e) { 
      System.err.println("Error: could not write to file "
                                   + dirName + File.separator + composeName() + " (" + e + ")");
    }
  }
  
  protected final void PSPreamble(double x1, double y1, double x2, double y2) {
    try {
      file.seek(0);
      file.writeBytes("%!PS-Adobe-2.0\n");
      file.writeBytes("%%Creator: TREEBAG\n");
      file.writeBytes("%%BoundingBox: ");
//      PSDouble(Math.floor(makeCoord(x1)), true); file.writeByte(' ');
//      PSDouble(Math.floor(makeCoord(y1)), true); file.writeByte(' ');
//      PSDouble(Math.ceil(makeCoord(x2)), true); file.writeByte(' ');
//      PSDouble(Math.ceil(makeCoord(y2)), true);
      PSDouble(makeCoord(x1), true); file.writeByte(' ');
      PSDouble(makeCoord(y1), true); file.writeByte(' ');
      PSDouble(makeCoord(x2), true); file.writeByte(' ');
      PSDouble(makeCoord(y2), true);
      file.writeBytes("\n%%EndComments");
      file.seek(fileOffset-1);
      file.writeByte('\n');
    }
    catch (IOException e) {
      System.err.println("Error: could not write to file " + dirName +
        File.separator + composeName() + " (" + e + ")");
    }
  }
  
/** Initialize a <code>PSDisplay</code> by reading its definition from a stream.
  * @see PSDisplayParser
  * @exception ParseException if an error occurs
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
    PSDisplayParser parser = new PSDisplayParser(stream);
    PSOutput = parser.initialize();
    if (parser.background != null) visualizer().setBackground(parser.background);
    if (PSOutput) {
      lineWidth = parser.lineWidth;
      dirName=parser.dirName.toString();
      fileName = new String[parser.fileName.length];
      for (int i = 0; i < parser.fileName.length; i++) fileName[i]=parser.fileName[i].toString();
    }
  }
  
}
