/** Class for rules with syncronization symbols.
 *  This class is required by BSTGrammar.java
 */
package generators;

import java.util.*;
import terms.*;

/** A SyncedRule is very similar to a rule used by eg
  * ET0LGrammar.java where it consists of a Object[3]
  * vector containing lhs, rhs and weight which are
  * the same as the three public attributes of this class.
  * The main structural difference is that in this class, 
  * the tree rhs can contain, in addition to "symbol":s, 
  * "synchronizedSymbol":s which are nonterminal with a vector of integers.
  * The vector is called a synchorization symbol or syncsymbol.
  */
public class SyncedRule {

  /*These are the attributes associated with one rule*/
  public symbol lhs;
  public term rhs;
  public double weight;

  /*Some private attributes that basically are used to avoid recomputing
    expensive operations during translation */
  private Vector syncStrings = null;  //Will be set to a Vector of int[] objects
  public SyncedRule(symbol _lhs, term _rhs)
  {
    this(_lhs, _rhs, 1.0);
  }
  public SyncedRule(symbol _lhs, term _rhs, double w)
  {
    lhs = _lhs;
    rhs = _rhs;
    weight = w;

    //Extract all sync-strings from the rhs and store in a vector. This vector
    //will be used when creating the tree grammar and transducers except last one.
    updateSyncStringVector();
  }
/*
  //Tv ytterligare alternativa konstruktorer, fr baktkompatibilitet
  //Dessa fyller inte lngre ngon funktion d vektorn syncStrings nd kommer att berknas utifrn rhs.
  public SyncedRule(symbol _lhs, term _rhs, Vector _syncStrings)
  {
    this(_lhs, _rhs);
  }
  public SyncedRule(symbol _lhs, term _rhs, Vector _syncStrings, double w)
  {
    this(_lhs, _rhs, w);
  }
*/

  //Help function to constructor
  //Creates an int vector containing all sync symbols in the order
  //they appeared in the tree.
  private Vector checkRhs(term rhs, Vector sync)
  {
    symbol top = rhs.topSymbol();
    if (top instanceof synchronizedSymbol) {
        sync.addElement(((synchronizedSymbol)top).getSync());
    }
    else
    {
      for (int i = 0; i < top.rank(); i++) checkRhs(rhs.subterm(i), sync);
    }
    return sync;
  }

  public void setWeight(double w) {
    weight = w;
  }
  public double getWeight() {
    return weight;
  }
  public symbol getLhs()
  {
    return lhs;
  }
  public term getRhs()
  {
    return rhs;
  }
  public Vector getSyncStrings()
  {
    if(syncStrings == null)
    {
      updateSyncStringVector();
    }
    return syncStrings;
  }
  private void updateSyncStringVector()
  {
    syncStrings = new Vector();
    checkRhs(getRhs(), syncStrings);
  }

  //example: if only <00> appears in the rule, only first subtree in sync tree is used, so {0} is returned.
  //if <00> and <10> appears and d=2, first and third subtree is used, so {0,2} is returned.
  //transducer and tree grammar will be calling this to determine where to put
  //sub trees and where to put bottom symbols in the sync.trees.
  //note: returned vector may contain duplicates.
  public Vector getUsedSyncBranchNumbers(int d)
  {
    Vector ret = new Vector();
    for(int i=0; i<syncStrings.size(); i++)
    {
      int[] s = (int[])syncStrings.elementAt(i);
      int baseD = 0;
      for (int j = 0; j < s.length; j++) baseD = baseD * d + s[j];
      ret.addElement(new Integer(baseD));
    }
    return ret;
  }

  //Same as getUsedSyncBranchNumbers except that it looks only on the 
  //first l numbers in the sync-symbol.
  //If <10> appears {1} is returned. If <01>,<11>,<10>,<00> all appears,
  //{0,1} is returned.
  public Vector getUsedSyncBranchNumbersForSyncLevel(int d, int l)
  {
    Vector ret = new Vector();
    for(int i=0; i<syncStrings.size(); i++)
    {
      int[] s = (int[])syncStrings.elementAt(i);
      int baseD = 0;
      for (int j = 0; j < l; j++) baseD = baseD * d + s[j];
      ret.addElement(new Integer(baseD));
    }
    return ret;
  }

  //Transform a syncsymbol to an integer, interpreting it as a number
  //on base d.
  public static int syncStringToNumberOfBaseD(int[] syncInfo, int d)
  {
    int factor = 1;
    int sum = 0;
    for(int ii=0;ii<syncInfo.length;ii++)
    {
      sum += factor * syncInfo[syncInfo.length-ii-1];
      factor *= d;
    }
    return sum;
  }
  //Transform the first n numbers in a syncsymbol to an integer, interpreting 
  //it as a number on base d.
  public static int syncStringFirstNToNumberOfBaseD(int[] syncInfo, int n, int d)
  {
    int factor = 1;
    int sum = 0;
    for(int ii=0;ii<syncInfo.length && ii<n;ii++)
    {
      sum += factor * syncInfo[n-ii-1];
      factor *= d;
    }
    return sum;
  }

  //transform to final td rule
  //Example. original rule in table 1,1: S -> o[S<00>]
  //should give: S[t11[x1,x2,x3,x4]] -> o[S[x1]],
  public String getFinalTDRule(SuperTable myTable, int myTablesRank, int d)
  {
    String ret = lhs + "[" + myTable.getName() + "[";
    for(int i=0; i<myTablesRank; i++)
    {
      ret = ret + "x" + (i+1);
      if(i+1<myTablesRank)
        ret = ret + ",";
    }
    ret = ret + "]] -> "; //end of created lhs

    ret = ret + transformOriginalRhsToFinalTransducerRhs(rhs, d);

    return ret;
  }//method getFinalTDRule

  //Help function to getFinalTDRule
  private String transformOriginalRhsToFinalTransducerRhs(term rhs, int d)
  {
    String ret = "";
    ret = ret + rhs.topSymbol(); //start of created rhs

    //special case for single nonterminal in rhs
    if(rhs.topSymbol() instanceof synchronizedSymbol)
    {
        synchronizedSymbol sym = (synchronizedSymbol)rhs.topSymbol();
        int syncInfo[] = sym.getSync();
        int correspondingSubTree = syncStringToNumberOfBaseD(syncInfo, d);
        ret = ret + "[x" + (1+correspondingSubTree) + "]";
        return ret;
    }
    if(rhs.topSymbol().rank() == 0) //special case for single terminal symbol rhs.
    {
    }
    else //rhs is tree, not symbol, do recursive calls on all subtrees
    {
      ret = ret + "[";

      for (int i=0; i < rhs.topSymbol().rank(); i++) { //loop for all symbols in rhs
        ret = ret + transformOriginalRhsToFinalTransducerRhs(rhs.subterm(i),d);
        if(i+1 < rhs.topSymbol().rank())
        {
          ret = ret + ",";
        }
      }//loop for all symbols in rhs
      ret = ret + "]"; //end of created right hand side

    }
    return ret;
  }

  //use instead of method getFinalTDRule when using yield algebra
  public String getFinalTDRuleWithYield(SuperTable myTable, int myTablesRank, int d, Vector origSyncStrings)
  {
    String ret = lhs + "[" + myTable.getName() + "[";
    for(int i=0; i<myTablesRank; i++)
    {
      ret = ret + "x" + (i+1);
      if(i+1<myTablesRank)
        ret = ret + ",";
    }
    ret = ret + "]] -> "; //end of created lhs

    ret = ret + transformOriginalRhsToFinalTransducerRhsWithYield(rhs, d, origSyncStrings);

    return ret;
  }
  private String transformOriginalRhsToFinalTransducerRhsWithYield(term rhs, int d, Vector origSyncStrings)
  {
    String ret = "";

    //special case for single nonterminal in rhs
    if(rhs.topSymbol() instanceof synchronizedSymbol)
    {
        synchronizedSymbol sym = (synchronizedSymbol)rhs.topSymbol();
        int syncInfo[] = sym.getSync();
        int correspondingSubTree = syncStringToNumberOfBaseD(syncInfo, d);
        ret = ret + "subst[";
        ret = ret + rhs.topSymbol();
        ret = ret + "[x" + (1+correspondingSubTree) + "]";
        for(int i=0; i<syncInfo.length; i++)
        {
          ret = ret + ", "+ origSyncStrings.elementAt(syncInfo[i]) + "[proj-" + (i+1) + "]";
        }
        ret = ret + "]";
        return ret;
    }
    ret = ret + rhs.topSymbol();
    if(rhs.topSymbol().rank() == 0) //special case for single terminal symbol rhs.
    {
    }
    else //rhs is tree, not symbol, do recursive calls on all subtrees
    {
      ret = ret + "[";

      for (int i=0; i < rhs.topSymbol().rank(); i++) { //loop for all symbols in rhs
        ret = ret + transformOriginalRhsToFinalTransducerRhsWithYield(rhs.subterm(i), d, origSyncStrings);
        if(i+1 < rhs.topSymbol().rank())
        {
          ret = ret + ",";
        }
      }//loop for all symbols in rhs
      ret = ret + "]"; //end of created right hand side

    }
    return ret;
  }

  //Returns a signature containing nonterminals that needs a particular subtree.
  //syncNumber is the number of this subtree. d and currentN are needed to transform
  //all syncsymbols we find on nonterminals to integers and see if these intergers
  //match syncNumber. currentN are how many numbers in a syncsymbol we should look at
  //and d is the base we should consider the syncsymbol as a number in.
  public fixedRankSignature nonterminalsThatUseGivenSyncNumber(term rhs, int syncNumber, int d, int currentN)
  {
    fixedRankSignature ret = new fixedRankSignature(0);
    symbol top = rhs.topSymbol();
    if (top instanceof synchronizedSymbol) {
      int[] syncInfo = ((synchronizedSymbol)top).getSync();
      if(syncStringFirstNToNumberOfBaseD(syncInfo, currentN+1, d) == syncNumber)
        ret.addSymbol(top.toString());
    }
    else
    {
      for (int i = 0; i < top.rank(); i++)
      {
      	fixedRankSignature s = nonterminalsThatUseGivenSyncNumber(rhs.subterm(i), syncNumber, d, currentN);
        ret.unionWith(s);
      }
    }
    return ret;
  }//nonterminalsThatUseGivenSyncNumber

  //Used when increasing grammar depth by one.
  //Add the specified integer to the end of each syncsymbol.
  public void increaseLengthOfSyncSymbolTuplesWithOne(int padSyncSymbol)
  {
    increaseTuplesInTerm(padSyncSymbol, getRhs());
    updateSyncStringVector();
  }
  //help method to increaseLengthOfSyncSymbolTuplesWithOne
  private void increaseTuplesInTerm(int padSyncSymbol, term rhs)
  {
    symbol top = rhs.topSymbol();
    if (top instanceof synchronizedSymbol)
    {
      synchronizedSymbol s = (synchronizedSymbol)top;
      int syncInfo[] = s.getSync();
      int n = syncInfo.length+1;
      int[] newSyncInfo = new int[n];
      newSyncInfo[0] = padSyncSymbol;
      for(int i=1; i<n; i++)
      {
        newSyncInfo[i] = syncInfo[i-1];
      }
      s.setSync(newSyncInfo);
    }
    else
    {

      for (int i = 0; i < top.rank(); i++){
        increaseTuplesInTerm(padSyncSymbol, rhs.subterm(i));
      }

    }
  }//method increaseTuplesInTerm

  //When the settings are such that all nonterminals, with names
  //equal to some terminal should be renamed, this method is used.
  //from and to are old and new name of one nonterminal. 
  public void relabelSymbol(symbol from, symbol to)
  {
    if(lhs.toString().equals(from.toString()))
      lhs = new synchronizedSymbol(to.toString(),0);
    relabelSymbolInRhs(from.toString(), to.toString(), getRhs());
  }
  private void relabelSymbolInRhs(String from, String to, term rhs)
  {
    symbol top = rhs.topSymbol();
    if (top.toString().equals(from))
    {
      if (top instanceof synchronizedSymbol)
      {
        synchronizedSymbol s = (synchronizedSymbol)top;
        int[] syncInfo = s.getSync();
        rhs.relabel(new synchronizedSymbol(to, syncInfo.length));
        s = (synchronizedSymbol)rhs.topSymbol();
        s.setSync(syncInfo);//needed?
      }
      else
      {
//        rhs.relabel(to);
      }
    }
    for (int i = 0; i < top.rank(); i++){
      relabelSymbolInRhs(from, to, rhs.subterm(i));
    }
  }//method relabelSymbolInRhs

}

