package generators;

import java.util.*;
import util.*;
import terms.*;
import parsers.*;

/**
 * Macro Tree Transducer class for TreeBag.
 * 
 * @version     0.01
 */
public class mtTransducer extends treeTransducer {

    private boolean io = true;
    private static String ioText = "switch to IO";
    private static String oiText = "switch to OI";
    private static String nextStep = "single step";
    private static String parallelStep = "parallel step";
    private static String back = "back";
    private static String singleSteps = "derive stepwise";
    private static String completeRuns = "results only";
    private static String newSeedText = "new random seed";
    private static String safeText = "make safe derivations";
    private static String unSafeText = "make blind derivations";
    //private boolean isDeterministic;    
    private boolean safeMode = true;
    protected boolean performance = false;
    
    //====================================================================
    // User Interface
    //====================================================================

    public list commands() {
        list result = new list();
        String[] com;
        boolean stepwise = false;
        if (currCompTree != null && !performance)
            stepwise = currCompTree.getStepwise();
        
        if (stepwise && !performance) {
            com = new String[3];
            com[0] = nextStep;
            com[1] = parallelStep;
            com[2] = back;
            result.append(com);
        }
        
        if(!deterministic) {
            com = new String[2];
            com[0] = safeMode ? unSafeText : safeText;
            com[1] = newSeedText;
            result.append(com);
        }

        if(performance)
            com = new String[1];
        else {
            com = new String[2];
            com[0] = stepwise ? completeRuns : singleSteps;
        }
        if(io) {
            com[performance?0:1] = oiText;
        } else {
            com[performance?0:1] = ioText;
        }
        result.append(com);
        

        return result;
    }

    /**
     * Reaction
     */
    public void execute(String command) {
        if (lastArgument == null)
            return;
        
        if (command.equals(ioText)) {

            io = true;
            if (ioDerivation == null && safeMode) {
                ioDerivation =
                    new mtComputationTree(lastArgument, io, states, rule,
                            weight, initialState, randGen.nextLong(), true,
                            this);
            }
            else if (ioBlindDerivation == null && !safeMode) {
                ioBlindDerivation =
                    new mtComputationTree(lastArgument, io, states, rule,
                            weight, initialState, randGen.nextLong(), false,
                            this);
            }
            if (safeMode)
                currCompTree = ioDerivation;
            else
                currCompTree = ioBlindDerivation;
        }

        if(command.equals(oiText)) {
            io = false;
            if (oiDerivation == null && safeMode) {
                oiDerivation =
                    new mtComputationTree(lastArgument, io, states, rule,
                            weight, initialState, randGen.nextLong(), true,
                            this);
            }
            else if (oiBlindDerivation == null && !safeMode) {
                oiBlindDerivation =
                    new mtComputationTree(lastArgument, io, states, rule,
                            weight, initialState, randGen.nextLong(), false,
                            this);
            }
            if (safeMode)
                currCompTree = oiDerivation;
            else
                currCompTree = oiBlindDerivation;
        }
        else if (command.equals(completeRuns)) {
            currCompTree.setResultsOnly();
        }
        else if (command.equals(singleSteps)) {
            currCompTree.setStepwise();
        }
        else if (command.equals(nextStep)) {
            currCompTree.singleStep();
        }
        else if (command.equals(parallelStep)) {
            currCompTree.parallelStep();
        }
        else if (command.equals(back)) {
            currCompTree.back();
        }
        else if (command.equals(newSeedText)) {
            newSeed();
        }
        else if (command.equals(safeText)) {
            safeMode = true;
            if (io)
                execute(ioText);
            else
                execute(oiText);
        }
        else if (command.equals(unSafeText)) {
            safeMode = false;
            if (io)
                execute(ioText);
            else
                execute(oiText);
        }
    }


    /**
     * The input tree to the mtt.
     */    
    public term apply(term t) {
        ioDerivation = null;
        ioBlindDerivation = null;
        oiDerivation = null;
        oiBlindDerivation = null;

        lastArgument = new extendedTerm(t);

        currCompTree = new mtComputationTree(lastArgument, io, states, rule,
                weight, initialState, randGen.nextLong(), safeMode, this);
        currTerm = currCompTree.getOutput();

        if (io && safeMode)
            ioDerivation = currCompTree;
        else if (io)
            ioBlindDerivation = currCompTree;
        else if (safeMode)
            oiDerivation = currCompTree;
        else
            oiBlindDerivation = currCompTree;
        
        return currTerm;
    }
    
    public void newSeed() {
        currCompTree = new mtComputationTree(lastArgument, io, states, rule,
                weight, initialState, randGen.nextLong(), safeMode, this);
        if(io)
            ioDerivation = currCompTree;
        else
            oiDerivation = currCompTree;
    }

    public term currentTerm() {
        if (currCompTree != null)
            return currCompTree.getOutput();
        else
            return null;
    }

    //====================================================================
    // Implementation
    //====================================================================

    private finiteSignature states;
    private finiteSignature inputSignature;
    private symbol initialState;
    private term[][] rule;
    private double[] weight;

    private extendedTerm currTerm = null;
    private extendedTerm lastArgument = null; 
    private Random randGen = new Random();
    protected boolean deterministic = false;
    
    private mtComputationTree ioDerivation = null;
    private mtComputationTree ioBlindDerivation = null;
    private mtComputationTree oiDerivation = null;
    private mtComputationTree oiBlindDerivation = null;
    private mtComputationTree currCompTree = null;
    public static int stateSize = 0;
    
    /**
     * Parses the input stream describing the desired Macro Tree Transducer.
     *
     * @param stream Input stream of ASCII characters
     * @exception ParseException 
     */
    public void parse(ASCII_CharStream stream) throws ParseException {
        mtTransducerParser parser = new mtTransducerParser(stream);
        // Call parser function and fetch parsed definitions
        parser.mtTransducer();
        states = parser.states;
        if(stateSize < states.size())
            stateSize = states.size();
        inputSignature = parser.in;
        rule = parser.rule;
        weight = parser.weight;
        initialState = parser.initial;
        deterministic = checkDeterminism();
        performance = parser.performance;
        if(deterministic){
            safeMode = false;
        }
        else if (checkTotality()) {
            safeMode = false;
        }
    }

    public boolean checkDeterminism() {
        for (int i = 0; i < rule.length; i++)
            for (int j = i + 1; j < rule.length; j++) {
                symbol iState = rule[i][0].topSymbol();
                symbol jState = rule[j][0].topSymbol();
                symbol iSigma = rule[i][0].subterm(0).topSymbol();
                symbol jSigma = rule[j][0].subterm(0).topSymbol();
                if(iState.equals(jState) && iSigma.equals(jSigma)) {
                    return false;
                }
            }
        return true;
    }

    private boolean checkTotality() {
        BitSet[] totSet = new BitSet[states.size()];
        for (int i = 0; i < states.size(); i++)
            totSet[i] = new BitSet();

        for (int i = 0; i < rule.length; i++) {
            symbol state = rule[i][0].topSymbol();
            symbol sigma = rule[i][0].subterm(0).topSymbol();
            int ruleNo = states.indexOf(state);
            int symbolNo = inputSignature.indexOf(sigma);
            totSet[ruleNo].set(symbolNo);
        }

        for (int i = 0; i < states.size(); i++)
            for (int j = 0; j < inputSignature.size(); j++)
                if (!totSet[i].get(j))
                    return false;
        return true;
    }

    public void allowExit() {
        super.allowExit();
    }

// mtComputationTree, private class of mtTransducer

private class mtComputationTree {
    private extendedTerm input = null;
    private finiteSignature states = null;
    private term rule[][] = null;
    private double[] weight = null;
    private symbol initialState = null;
    private boolean io = false;
    private boolean stepwise = false;
    private Random randGen = null;
    private boolean safeMode = true;
    private boolean unSafeAndError = false;
    private mtTransducer exitRef = null;
    private boolean performance = false;
    private boolean deterministic = false;
    private int RUNS_BEFORE_ALLOWEXIT = 20;
    
    /**
     * Initialization requires input tree and derivation mode, corresponding
     * output tree is generated.
     */
    public mtComputationTree(extendedTerm input, boolean io,
            finiteSignature states, term rule[][], double[] weight,
            symbol initialState, long seed, boolean safeMode,
            mtTransducer exitRef) {

        this.initialState = initialState;
        this.input = new extendedTerm( input );
        this.io = io;
        this.states = states;
        this.rule = rule;
        this.weight = weight;
        if(!deterministic)
            randGen = new Random(seed);
        this.safeMode = safeMode;
        this.exitRef = exitRef;
        this.performance = exitRef.performance;
        this.deterministic = exitRef.deterministic;
        
        if(safeMode) {
            if(io) {
                calcIOGamma(this.input);
            }
            else {
                calcOIGamma(this.input);
            }
        }

        this.input = addInitialState(this.input);
        try {
            this.input = run(this.input);
        } catch (UnsolvableStateException e) {
            if (safeMode)
                this.input = null;
            else
                unSafeAndError = true;
        }
    }

    /**
     * Returns the final output tree
     */
    public extendedTerm getOutput() {
        if (unSafeAndError && !stepwise)
            return null;
        
        if(input != null) {
            if(performance)
                return input;
            else
                return getOutput(input);
        }
        else
            return null;
    }

    private extendedTerm getOutput(extendedTerm currRoot) {
        //The calculated result should be returned if (1) results-only
        //derivations are active or (2) "stepping" have come this far
        extendedTerm result = currRoot.getResult();
        if (result != null) {
            if ( (currRoot.getCounter() > -1) || !stepwise) {
                return getOutput( result );
            }
        }
        //Otherwise, return this node with its children
        //Stop, if at leaf
        if(currRoot.topSymbol().rank() == 0) { 
            return new extendedTerm( currRoot.topSymbol() );
        }
        result = new extendedTerm( currRoot.topSymbol() );
        //Continue with this node's children, if at intermediate node
        for (int i = 0; i < result.topSymbol().rank(); i++)
            result.defineSubterm(i,
                    getOutput((extendedTerm)currRoot.subterm(i)));

        return result;
    }        

   
    public void setStepwise() {
        stepwise = true;
    }
    public boolean getStepwise() {
        return stepwise;
    }
    public void setResultsOnly() {
        stepwise = false;
    }

    /**
     * Make a parallel step
     */
    public void parallelStep() {
        singleStepMode = false;
        step(input);
    }
    
    /**
     * Make a single step
     */
    public void singleStep() {
        singleStepMode = true;
        singleStepPerformed = false;
        step(input);
    }
    private boolean singleStepPerformed;
    private boolean singleStepMode;
   
    /**
     * Goes through the computation tree and marks which computations
     * should be shown.
     * @param currRoot Root of the active subtree
     * @return true - if the step has changed a state on a level below this
     *          (only for IO derivations)
     */
    private boolean step(extendedTerm currRoot) {
        if (currRoot == null)
            return false;
        
        if (io) {
            if (currRoot.getResult() != null && currRoot.getCounter() >= 0) {
                currRoot.increaseCounter();
                return step( currRoot.getResult() );
            }

            boolean applied = false;
            //First, make steps on the node's children
            for (int i = 0; i < currRoot.topSymbol().rank(); i++) {
                if (step( (extendedTerm)currRoot.subterm(i) ))
                    applied = true;
            }

            if (singleStepPerformed && singleStepMode)
                return true;

            //If this node has a result, increase its counter
            extendedTerm result = currRoot.getResult();
            if (result != null && !applied) {
                currRoot.increaseCounter();

                //If this is the "frontier", then stop
                if (currRoot.getCounter() == 0) {
                    singleStepPerformed = true;
                    return true;
                }
                //Otherwise, continue adding the result's counter
                return false;
            }
            return applied;

        }
        //OI case
        else {
            if (singleStepPerformed && singleStepMode)
                return false;

            //If this node has a result, increase its counter
            extendedTerm result = currRoot.getResult();
            if (result != null) {
                currRoot.increaseCounter();

                //If this is the "frontier", then stop
                if (currRoot.getCounter() == 0) {
                    singleStepPerformed = true;
                    return false;
                }
                //Otherwise, continue adding the result's counter
                step( result );
                return false;
            }

            //No result: Continue stepping on this node's children
            for (int i = 0; i < currRoot.topSymbol().rank(); i++)
                step( (extendedTerm)currRoot.subterm(i) );
            return false;
        }
    }

    public void back() {
        if (input != null)
            back(input);
    }
    private void back(extendedTerm currRoot) {
        if (io) {
            //First, which path should be followed (result or this)?
            if (currRoot.getResult() != null && currRoot.getCounter() >= 0) {
                back(currRoot.getResult());
                currRoot.decreaseCounter();
                if (currRoot.getCounter() == -1)
                    back(currRoot);
            }
            else {
                for (int i = 0; i < currRoot.topSymbol().rank(); i++)
                    back( (extendedTerm)currRoot.subterm(i) );
            }
        }
        //OI case
        else {
            currRoot.decreaseCounter();

            if (currRoot.getResult() != null)
                back(currRoot.getResult());
            else
                for (int i = 0; i < currRoot.topSymbol().rank(); i++)
                    back( (extendedTerm)currRoot.subterm(i) );
        }
    }

    /**
     * Simply returns the tree corresponding to the last steps
     */
    public extendedTerm currentTerm() {
        return null;
    }

    int goExit = 0;
    /**
     * Calculate all subcomputations 
     */
    private extendedTerm run(extendedTerm currRoot)
        throws UnsolvableStateException {
        if (goExit++ % RUNS_BEFORE_ALLOWEXIT == 0)
            exitRef.allowExit();
            
        if (unSafeAndError)
            return currRoot;
            
        //Leaf
        if (currRoot.topSymbol().rank() == 0)
            return currRoot;

        //IO-derivations
        else if(io) {
            //First calculate this node's children
            if (!performance || !currRoot.bottomMostState())
                for(int i = 0; i < currRoot.topSymbol().rank(); i++)
                    currRoot.defineSubterm(i,
                            run((extendedTerm)currRoot.subterm(i)));

            //If a rule can be applied at this state, apply it and make it this
            //node's result
            if (states.contains( currRoot.topSymbol() ) && !unSafeAndError) {
                extendedTerm applied = applyRuleAt(currRoot);
                applied = run(applied);
                if(performance)
                    return applied;
                currRoot.setResult( applied );
            }
            return currRoot;
        }
        //OI-derivations
        else {
            //If a rule can be applied at this state, apply it and make it this
            //node's result
            if (states.contains( currRoot.topSymbol() ) && !unSafeAndError) {
                extendedTerm applied = applyRuleAt(currRoot);
                applied = run(applied);
                if(performance)
                    return applied;
                currRoot.setResult( applied );
                return currRoot;
            }
            //Otherwise, continue by calculating this node's children
            else {
                for(int i = 0; i < currRoot.topSymbol().rank(); i++)
                    currRoot.defineSubterm(i,
                            run((extendedTerm)currRoot.subterm(i)));
                return currRoot;
            }
        }
    }

    /**
     * Chooses a "suitable" rule and applies it.
     * @param state State where a rule will be applied.
     */
    private extendedTerm applyRuleAt(extendedTerm state)
        throws UnsolvableStateException {
     
        //First, choose a suitable rule
        if(!performance)
            state = state.getCurrentResultTree();
        int ruleNumber;
        
        if (safeMode)
            ruleNumber = chooseRule(state);
        else {
            try {
                ruleNumber = chooseRule(state);
            } catch (UnsolvableStateException e) {
                unSafeAndError = true;

                //"Mark" the error state
                extendedTerm errorReturn = new extendedTerm(state);
                String oldName = errorReturn.topSymbol().toString();
                String newName = new String("-- ");
                newName = (newName.concat(oldName)).concat(" --");
                symbol newSymbol =
                    new symbol(newName, errorReturn.topSymbol().rank() );
                errorReturn.relabel(newSymbol);
                return errorReturn;
            }
        }

        
        
        //Bind x,y to cloned subtrees
        Hashtable xList = new Hashtable();
        Hashtable yList = new Hashtable();
        substituteBind(rule[ruleNumber][0], state, xList, yList);

        //Replace this tree with the rhs
        term rhs;
        if (performance)
            rhs = rule[ruleNumber][1];
        else
            rhs = (term)rule[ruleNumber][1].clone();
        extendedTerm result = new extendedTerm( rhs );

        //Replace x,y in tree with the cloned subtrees.
        result = substituteReplace(result, xList, yList,
                new BitSet(), new BitSet());
        
        //If in OI, calculate the new safe-parameter sets
        if(!io && safeMode)
            calcNewSafeParameters(rule[ruleNumber][1], result);

        return result;
    }

    /**
     * Calculates the safe parameters (T,F) for all new states in the tree t.
     */
    private void calcNewSafeParameters(term rhs, extendedTerm t) {
        if (rhs.topSymbol().rank() == 0)
            return;
        
        for (int i = 0; i < t.topSymbol().rank(); i++)
            calcNewSafeParameters(rhs.subterm(i), t.extendedSubterm(i));
        
        if(states.contains( rhs.topSymbol() )) {
            //Calculate safe parameters
            BitSet b = new BitSet();
            for (int i = 1; i < t.topSymbol().rank(); i++)
                if (OK_OI(t.extendedSubterm(i), null))
                    b.set(i-1);
            t.setSafeParameters(b);
        }
        

    }

    
    /**
     * Visit entire tree, replace all occurances of x:s and y:s with
     * subtrees found in hashtables.
     */
    private extendedTerm substituteReplace(extendedTerm tree,
            Hashtable xList, Hashtable yList, BitSet dirtyX, BitSet dirtyY) {
        symbol thisSymbol = tree.topSymbol();
        //Is this a variable or parameter
        if (thisSymbol instanceof variable) {
            int hashIndex = ((variable)thisSymbol).index();
            extendedTerm result =
                (extendedTerm)xList.get(new Integer(hashIndex));
            if(performance && !dirtyX.get(hashIndex)) {
                dirtyX.set(hashIndex);
            }
            else
                result = new extendedTerm( result );
            
            result.setBottomMostState();
            return result;
        }
        else if (thisSymbol instanceof parameter) {
            if (yList == null)
                return tree;
            int hashIndex = ((parameter)thisSymbol).index();
            extendedTerm result =
                (extendedTerm)yList.get(new Integer(hashIndex));
            if (performance && !dirtyY.get(hashIndex))
                dirtyY.set(hashIndex);
            else
                result = new extendedTerm( result );
            return result;
        }
        else {
            //Continue checking each subtree
            for (int i = 0; i < tree.topSymbol().rank(); i++)
                tree.defineSubterm(i,
                        substituteReplace((extendedTerm)tree.subterm(i),
                            xList, yList, dirtyX, dirtyY));
            return tree;
        }

    }

    /**
     * Goes through a lhs and a real tree. Binds occurances of x and y
     * to subtrees and stores this in the provided hashtables.
     */
    private void substituteBind(term lhs, extendedTerm tree,
            Hashtable xList, Hashtable yList) {
        symbol thisSymbol = lhs.topSymbol();
        //Is this a variable or parameter
        if (thisSymbol instanceof variable) {
            int hashIndex = ((variable)thisSymbol).index();
            extendedTerm copy;
            if(performance)
               copy = tree;
            else
               copy = new extendedTerm( tree );
            xList.put(new Integer(hashIndex), copy);
        }
        else if (thisSymbol instanceof parameter) {
	    if (yList == null)
	        return;
            int hashIndex = ((parameter)thisSymbol).index();
            extendedTerm copy;
            if(performance)
               copy = tree;
            else
               copy = new extendedTerm( tree );
            yList.put(new Integer(hashIndex), copy);
        }
        else {
            //Else, call recursively to find x,y:s deeper in the tree
            for (int i = 0; i < thisSymbol.rank(); i++)
            {
                substituteBind(lhs.subterm(i), tree.extendedSubterm(i),
                        xList, yList);
            }
        }
    }

    /**
     * Choose a "suitable" rule, return its index
     * @exception UnsolvableStateException No rule could be applied,
     *                   tree is in an unsolvable state
     */
    private int chooseRule(extendedTerm t) throws UnsolvableStateException {
        int temp = -1;
        double sumWeight = 0.0;
        BitSet applicableRules = new BitSet(rule.length);
        //Find a rule of the form
        //q(a(x1,...,xm),y1,...,ym) which matches term state
        for (int i = 0; i < rule.length; i++) {
            term lhs = rule[i][0];
            //q matches?
            if (!lhs.topSymbol().equals(t.topSymbol()) || weight[i] <= 0.0)
                continue;

            //a matches?
            symbol lhsSigma = lhs.subterm(0).topSymbol();
            symbol tSigma = t.subterm(0).topSymbol();
            if (lhsSigma.equals(tSigma)) {
                //Lhs matches, does rhs satisfy "solvable states" requirements?
                if (solvableRule(i, t) || !safeMode) {
                    if(deterministic) //performance
                        return i;
                    applicableRules.set(i);
                    sumWeight += weight[i];
                    if (temp == -1)
                        temp = i;
                }
            }
        }
        double rand = randGen.nextDouble() * sumWeight;
        //All rules found, now randomly pick one.
        for(int i = 0; i < rule.length; i++) {
           if( !applicableRules.get(i) ) 
               continue;
           
           rand -= weight[i];
           if (rand <= 0.0) {
               return i;
           }
        }

        //Raise exception, should be caught at the UI-part, so null is returned
        //to Treebag
        throw new UnsolvableStateException();
    } private class UnsolvableStateException extends Exception {

		/**
		 * 
		 */
		private static final long serialVersionUID = 631877830901645571L; }


    /**
     * If OK_t(rhs[[b]]) then it can be used as a rule.
     */
    private boolean solvableRule( int i, extendedTerm et ) {

        if(deterministic)//for performance
            return true;
        
        //Replace x_i in rhs with real trees.
        Hashtable xList = new Hashtable();
        term lhs = rule[i][0];
        term rhs = rule[i][1];
        if(io && !lhs.topSymbol().equals( et.topSymbol() ))
            substituteBind(lhs.subterm(0), et, xList, null);
        else
            substituteBind(lhs, et, xList, null);
        extendedTerm result = new extendedTerm( rhs );
        result = substituteReplace(result, xList, null,
                new BitSet(), new BitSet());

        if (io) {
            return OK_IO(result);
        }
        else {
            boolean ret = OK_OI(result, et.getSafeParameters());
            return ret;
        }
    }
    
    /**
     * Check all occurances of q'(tau(x1,...),...) in rhs, if tau is a
     * child's symbol then q must already be in that childs
     * solvableStates-set (if not, return false)
     */
    private boolean OK_IO( extendedTerm et) {
        if (!states.contains(et.topSymbol())) {
            for(int i = 0; i < et.topSymbol().rank(); i++)
                if(!OK_IO( et.extendedSubterm(i) )) {
                    return false;
                }
            return true;
        }
        //States must be checked
        else if (states.contains( et.topSymbol() )) {
            symbol qPrime = et.topSymbol();
            extendedTerm node = et.extendedSubterm(0);

            if (!node.get(states.indexOf(qPrime))) {
                return false;
            }
            
            //Each parameter-child must be OK
            for (int i = 1; i < et.topSymbol().rank(); i++) {
                if(!OK_IO(et.extendedSubterm(i))) {
                    return false;
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Calculates which (q,a)-transformations can be used at each node
     * in tree t.
     * @param t Input term
     */
    private void calcIOGamma(extendedTerm t) {
        //If this is a a leaf
        if (t.topSymbol().rank() == 0) {
            t.defineSolvableStates( findTransformations(t.topSymbol()) );
        }
        //This is the root of a subtree
        else {
            //The children's bitsets must be calculated before the parent's
            for (int i = 0; i < t.topSymbol().rank(); i++)
                calcIOGamma( (extendedTerm)t.subterm(i) );

            //Find which rules may "satisfy" the node's children
            for (int i = 0; i < rule.length; i++)
            {
                //If this rule does not apply to this node's symbol, skip it
                if ( !rule[i][0].subterm(0).topSymbol().equals(t.topSymbol()) )
                    continue;

                //Check rhs if it "satisfies" the children
                if ( solvableRule(i, t) ) {
                    symbol q = rule[i][0].topSymbol();
                    t.set(states.indexOf(q));
                }
            }
        }
    }

    /**
     * Find all (q,a)-transformations
     * @param a Symbol to search for in the lhs
     * @return BitSet identifying solvable states
     */
    private BitSet findTransformations(symbol a) {
        BitSet solvable = new BitSet(states.maxIndex());

        //For each rule: Check if lhs is of the correct form
        for (int i = 0; i < rule.length; i++)
            if(rule[i][0].subterm(0).topSymbol().equals(a)) {
                symbol q = rule[i][0].topSymbol();
                solvable.set(states.indexOf(q));
            }

        return solvable;
    }

    private extendedTerm addInitialState(extendedTerm et) {
        extendedTerm etInit = new extendedTerm(initialState);
        if (!io) {
            BitSet b = new BitSet();
            b.set(0);
            etInit.setSafeParameters( b );
        }
        etInit.defineSubterm(0, et);
        return etInit;
    }

    private boolean OK_OI(extendedTerm n, BitSet b) {
        if (!safeMode)
            return true;
        
        if(n.topSymbol() instanceof parameter){
            int paraNum = ((parameter)n.topSymbol()).index();
            return b.get(paraNum - 1);
        }
        else if(!states.contains( n.topSymbol() )) {
            for(int x = 0; x < n.topSymbol().rank(); x++)
                if(!OK_OI(n.extendedSubterm(x),b))
                    return false;
            return true;
        }
        else {
            symbol qPrime = n.topSymbol();
            extendedTerm delta = n.extendedSubterm(0);
            int qPIndex = states.indexOf(qPrime);
            for(int x = 0; x < delta.getBSize(qPIndex); x++) {
                boolean okRhs = true;
                BitSet currB = delta.getB(qPIndex,x);
                for(int i = 0; i < qPrime.rank(); i++)
                    if(currB.get(i) && !OK_OI(n.extendedSubterm(i+1),b))
                        okRhs = false;
                if(okRhs)
                    return true;
            }
            return false;
        }
    }

    private void calcOIGamma(extendedTerm n) {
        for(int i = 0; i < n.topSymbol().rank(); i++)
            calcOIGamma(n.extendedSubterm(i));

        for(int i = 0; i < rule.length; i++) {
            symbol q = rule[i][0].topSymbol();
            symbol sigma = rule[i][0].subterm(0).topSymbol();
            if(!sigma.equals(n.topSymbol()))
                continue;

            long limit = Math.round(Math.pow(2.0,(double)q.rank()-1));
            for(int j = 0; j < limit; j++) {
                BitSet b = new BitSet(q.rank());
                for(int x = 1, y = 0; x < limit; x*=2, y++) {
                    if ((j & x) > 0)
                        b.set(y);

                }
                //Replace x_i in rhs with real trees.
                Hashtable xList = new Hashtable();
                term lhs = rule[i][0];
                substituteBind(lhs.subterm(0), n, xList, null);
                term rhs = (term)rule[i][1].clone();
                extendedTerm result = new extendedTerm( rhs );
                result = substituteReplace(result, xList, null,
                        new BitSet(), new BitSet());

                if(OK_OI(result, b) || limit == 1) {
                    n.addB(states.indexOf(q),b);
                }
            }
        }
    }

/*    private void printIOGamma(extendedTerm n) {
        for (int x = 0; x < n.topSymbol().rank(); x++)
            printIOGamma(n.extendedSubterm(x));

        Enumeration myStates = states.elements();
        while(myStates.hasMoreElements()){
            symbol currSymbol = (symbol)myStates.nextElement();
        }
    }

    private void printOIGamma(extendedTerm n) {
        for (int x = 0; x < n.topSymbol().rank(); x++)
            printOIGamma(n.extendedSubterm(x));

        Enumeration myStates = states.elements();
        while(myStates.hasMoreElements()){
            symbol currSymbol = (symbol)myStates.nextElement();
        }
    }*/
}

}
