function [m_margin,e_margin,err_num,rej_num,err_idx,rej_idx, theta] = fnfp(Output, Target, silent, sim_name)
% [m_margin,e_margin,err_num,rej_num,err_idx,rej_idx, theta] = fnfp(Output, Target, silent, sim_name)
% False negative/false positive curves
% Inputs:
% Output    -- matrix m x c of outputs for each of the c classes
% Target    -- +- corresponding target values
% silent    -- a flag not to display the curve is 1
% sim_name  -- a string to identify the figure
% Returns:
% m_margin   -- median margin
% e_margin   -- extremal margin
% err_num    -- number of errors at zero reject
% rej_num    -- number of rejected at zero error
% err_idx    -- indices of errors
% rej_idx    -- indices of rejected
% theta      -- optimum threshold for best balanced error rate

% Isabelle Guyon -- March 2003 -- isabelle@clopinet.com

if nargin<3, silent=1; end
if nargin<4, sim_name=''; end

m=length(Output);

Posidx = find(Target>0);
Negidx = find(Target<0);
Posout = Output(Posidx);
Negout = Output(Negidx);
Sposout = sort(Posout);
Snegout = sort(Negout);

% Median margin
medneg=median(Snegout);
medpos=median(Sposout);
maxpos=max(Sposout);
minneg=min(Snegout);
maxall=max(Output);
minall=min(Output);
%m_margin=(medpos-medneg)/abs(maxpos-minneg);
if(maxall==minall)
   m_margin=0;
else
   m_margin=(medpos-medneg)/abs(maxall-minall);
end



% Extremal margin
maxneg=max(Snegout);
minpos=min(Sposout);
%e_margin=(minpos-maxneg)/abs(maxpos-minneg);
if(maxall==minall)
   e_margin=0;
else
   e_margin=(minpos-maxneg)/abs(maxall-minall);
end

% Number of errors at zero rejection
err_idx = union(Posidx(find(Posout<=0)),Negidx(find(Negout>=0)));
err_num = length(err_idx);

% Number of rejections at zero error
if err_num==0, 
   rej_num=0; 
   rej_idx=[];
elseif e_margin>0,
   theta = min(abs(maxneg),abs(minpos));
   rej_idx = union(Posidx(find(Posout<=theta)),Negidx(find(Negout>=-theta)));
   rej_num = length(rej_idx);
else
   theta = max(abs(maxneg),abs(minpos));
   rej_idx = union(Posidx(find(Posout<=theta)),Negidx(find(Negout>=-theta)));
   rej_num = length(rej_idx);
end
   
   
if ~silent 
   figure('Name', [sim_name ' fn/fp curves']);
 	plot(Snegout,length(Snegout):-1:1,'r-',Sposout,1:length(Sposout),'b-');
   xlabel('threshold');
   ylabel('number of false negative (blue) / false positive (red)');
   title(['tot. trial=', num2str(m), ', err. num.(0 rej)=',num2str(err_num), ', rej. num.(0 err)=',num2str(rej_num),...
            ', ext. marg.= ', num2str(round(100*e_margin)/100), ...
            ', med. marg.=', num2str(round(100*m_margin)/100)]);
    figure('Name', [sim_name ' sensitivity and specificity']);
    ln=length(Snegout);
    lp=length(Sposout);
    plot(Snegout,1/ln:1/ln:1,'r-',Sposout,1:-1/lp:1/lp,'b-');
    ylabel('Sensitivity (blue) and Specificity (red)');
    xlabel('threshold');
end

% Algorithm to compute berrate
soutval=sort(unique(Output));
midval=0.5*(soutval(2:length(soutval))+soutval(1:length(soutval)-1));
midval=[soutval(1); midval; soutval(length(soutval))];
poserr=zeros(size(midval));
negerr=zeros(size(midval));
ln=length(Snegout);
lp=length(Sposout);
j=1;
poserr(1)=0;
for i=2:length(midval)
    poserr(i)=poserr(i-1);
    while(Sposout(j)<midval(i))
        poserr(i)=poserr(i)+1;
        j=j+1;
    end
end
poserr(i)=lp;
j=length(Snegout);
negerr(length(negerr))=0;
for i=length(midval)-1:-1:1
    negerr(i)=negerr(i+1);
    while(Snegout(j)>midval(i))
        negerr(i)=negerr(i)+1;
        j=j-1;
    end
end
negerr(i)=ln;

sensitivity=1-(poserr/lp);
specificity=1-(negerr/ln);
berrate=1-0.5*(sensitivity+specificity);

figure('Name', [sim_name ' fn/fp curves']);
 	plot(Snegout,length(Snegout):-1:1,'r-',Sposout,1:length(Sposout),'b-');
   xlabel('threshold');
   ylabel('number of false negative (blue) / false positive (red)');
hold on; plot(midval, poserr, 'b--', midval, negerr, 'r--');

figure('Name', [sim_name ' sensitivity and specificity']);
    ylabel('Sensitivity (blue), Specificity (red), and balanced error rate (black)');
    xlabel('threshold');
hold on; plot(midval, sensitivity, 'b-', midval, specificity, 'r-', midval, berrate, 'k-');

% Find the best threshold
bm=min(berrate);
good_idx=find(berrate==bm);
theta=median(midval(good_idx));
title(['Theta = ' num2str(theta)]);
return


