function my_model = model_examples(model_name, data_name)
%my_model = model_examples(model_name, data_name)
% We show a number of examples of models, which are valid
% model constructions for the challenge.
% Use [> type model_examples] to view the code.
% Use [> model_examples] to view the choice of model names
% Input:
% model_name -- The name of the model chosen.
% data_name -- The name of the dataset on which the model will be used.
%              This last input is rendered necessary so that all examples
%              work on all datasets because some combinations of
%              preprocessings or hyperparameters are computationally
%              infeasible or use too much memory resources.
% Output:
% my_model -- A chain of algorithm including a svc, ridge, naive, rf, a
%             neural object

% Isabelle Guyon -- February 2006 -- isabelle@clopinet.com

global UsePixelRep;

if nargin<1 | isempty(model_name)
    %%% IG: new
    if ~(exist('svc.m') ==2)
        my_model = {'zarbi'};
    else
        my_model= ...
              {
                'zarbi'
                'Prepro+linearSVC'
                'Prepro+naiveBayes'
                'PCA+kernelRidge'
                'Prepro+nonlinearSVC'
                'GS+kernelRidge'
                'S2N+RFFS+RF'
                'Relief+neuralNet'
                'RFE+SVC'
                'linearSVC_Ensemble'
                'naiveBayes_Ensemble'
                'kridge'
                'naive'
                'neural'
                'rf'
                'svc'
                's2n'
                'gs'
                'relief'
                'rffs'
                'svcrfe'
                'standardize'
                'normalize'
                'shift_n_scale'
                'pc_extract'
                'subsample'
                'baseline'
                'optimized'
                'noselect'
                'pixelGisette_exp_conv'
                'pixelGisette_gauss_conv'
              };
    end
    %%% End new
    return;
end

switch model_name
    case 'zarbi'
        my_model=zarbi
    case 'Prepro+linearSVC'
        % Support vector classifier 
        % with linear kernel (the default.)
        if strcmp(data_name, 'gisette')
            my_model=chain({subsample('p_max=3000'), normalize, svc('shrinkage=0')})
        elseif strcmp(data_name, 'dorothea')
            my_model=svc('shrinkage=100') % Standardizing dorothea poses memory problems
        elseif strcmp(data_name, 'dexter')
            my_model=chain({normalize, svc('shrinkage=0')})
        else
            my_model=chain({standardize, svc('shrinkage=0')})
        end
    case 'Prepro+naiveBayes'
        % A complex chain of preprocessing followed by a Naive Bayes
        % classifier.
        my_prepro={standardize, s2n('f_max=200'), shift_n_scale('take_log=1'), normalize};
        if strcmp(data_name, 'dexter') | strcmp(data_name, 'dorothea')
            my_model=naive % The preprocessing is inadequate for dexter and dorothea
        else
            my_model=chain({my_prepro{:}, naive})
        end
    case 'PCA+kernelRidge'
        % Preprocessing with standardization,
        % followed by the extraction of the first 100 principal
        % components and by kernel ridge regression.
        my_classif={standardize, pc_extract('f_max=100'), kridge};
        if strcmp(data_name, 'dorothea')
            my_model=kridge % pca is long to perform on dorothea
        elseif  strcmp(data_name, 'dexter') 
            % balancing the training set helps pca for dexter
            my_model=chain({subsample({'p_max=200', 'balance=1'}), my_classif{:}})
        else
            my_model=chain(my_classif)
        end
    case 'Prepro+nonlinearSVC'
        % Preprocessing with standardization,
        % support vector classifier with kernel 
        % k(x,y) = (x.y+coef0)^degree * exp(-gamma |x-y|^2)
        my_svc=svc({'coef0=1', 'degree=1', 'gamma=0.00001', 'shrinkage=0.1'});
        if strcmp(data_name, 'dorothea')
            my_model=my_svc % Standardizing dorothea poses memory problems
        elseif strcmp(data_name, 'gisette')
            % gisette is very big for nonlinearSVC (takes long to train)
            my_model=chain({subsample('p_max=3000'), standardize, my_svc})
        else
            my_model=chain({standardize, my_svc})
        end
    case 'GS+kernelRidge'
        % Feature standardization followed by,
        % Gram-Schmidt feature selection (down to 10 features),
        % and by a ridge regression with kernel 
        % k(x,y) = (x.y+coef0)^degree * exp(-gamma |x-y|^2)
        % The example is a linear ridge regression (the default.)
        my_classif=kridge({'shrinkage=0.01', 'balance=1'});
        if strcmp(data_name, 'dorothea')
            my_model=my_classif
        else
            % This does not work for large sparse matrices like dorothea:
            my_model=chain({standardize, gs('f_max=10'), my_classif})
        end
    case 'S2N+RFFS+RF'
        % Feature selection with S2N (reduce only to 200 features)
        % further reduction to 10 features with Random Forest feature
        % selection, followed by Random Forest classification.
        %%%% Note: The present interface to RF is pleagued with several
        %%%% bugs: segmentation faults occur for large datasets
        %%%% data balancing and the use of categorical variables
        %%%% does not work yet.
        my_classif={s2n('f_max=200'), rffs('f_max=10'), rf({'units=100', 'mtry=6'})};
        if strcmp(data_name, 'gisette')
            my_model=chain({subsample('p_max=3000'), my_classif{:}})
        elseif strcmp(data_name, 'dexter') | strcmp(data_name, 'dorothea') 
            my_model=naive; % We did not get RF to work on categorical variables so far
        else
            my_model=chain(my_classif)
        end
    case 'Relief+neuralNet'
        % Feature standardization, selection of 40 features
        % by relief, followed by Neural Network classification.
        my_classif=neural({'units=10', 'shrinkage=0.01', 'balance=1', 'maxiter=100'});
        if strcmp(data_name, 'gisette')
            % gisette is too big for Relief
            my_model=chain({standardize, my_classif})
        elseif strcmp(data_name, 'dorothea')
            % Standardization in impractical for dorothea
            my_model=chain({relief('f_max=40'), my_classif})
        else
            my_model=chain({standardize, relief('f_max=40'), my_classif})
        end
    case 'RFE+SVC'
        % Feature standardization, selection of 40 features
        % by RFE SVC, followed by a linear SVC.
        my_classif={svcrfe({svc('shrinkage=0.1'), 'f_max=40'}), svc('shrinkage=0.1')};
        if strcmp(data_name, 'dorothea')
            % Standardization in impractical for dorothea
            my_model=chain(my_classif)
        else
            my_model=chain({standardize, my_classif{:}})
        end
    case 'linearSVC_Ensemble'
        % Ensemble of support vector classifiers 
        % having several different hyperparameters.
        for k=1:3
            if strcmp(data_name, 'dorothea')
                base_model{k}=svc(['shrinkage=' num2str(10^-(k-1))]); % Standardizing dorothea poses memory problems
            else
                base_model{k}=chain({standardize, svc(['shrinkage=' num2str(10^-(k-1))]) });
            end
        end
        my_model=ensemble(base_model, 'signed_output=1')
        my_model=chain({probe(relief,{'p_num=2000', 'pval_max=0'}), standardize, my_classif})
    case 'naiveBayes_Ensemble'
        % Ensemble of Naive Bayes classifiers 
        % build from data subsamples (bagging ensemble).
        for k=1:10
            if strcmp(data_name, 'dorothea')
                base_model{k}=chain({subsample({'p_max=1500', 'balance=1'}), naive}); ; % Standardizing dorothea poses memory problems
            else
                base_model{k}=chain({subsample({'p_max=3000', 'balance=1'}), standardize, naive});
            end
        end
        my_model=ensemble(base_model, 'signed_output=1')
    case {'kridge', 'naive', 'neural', 'rf', 'svc'}
        % These are the raw models. They do not necessarily work on all
        % datasets and may need preprocessing.
        my_model=eval(model_name);
    case {'s2n', 'gs', 'relief', 'rffs', 'svcrfe', ...
            'standardize', 'normalize', 'shift_n_scale', ...
            'pc_extract', 'subsample'};
        % These are the the feature extraction and preprocessing functions
        % coupled with naive Bayes. This deso not necessarily work on all
        % datasets and may need hyperpameter tuning.
        my_model=chain({eval(model_name), naive})
    case 'baseline'
        switch data_name
            case 'arcene'
                my_svc=svc({'coef0=1', 'degree=3', 'gamma=0', 'shrinkage=0.1'});
                my_model=chain({standardize, s2n('f_max=1100'), normalize, my_svc})
                % TIP#1: train on both validation and test set 
                % TIP#2: use ensemble methods  
            case 'dexter'
                my_classif=svc({'coef0=1', 'degree=1', 'gamma=0', 'shrinkage=0.5'});
                my_model=chain({s2n('f_max=300'), normalize, my_classif})
                % TIP#1: train on both validation and test set  
                % TIP#2: vary the number of features fmax=???  
            case 'gisette'
                my_classif=svc({'coef0=1', 'degree=3', 'gamma=0', 'shrinkage=1'});
                %my_model=chain({subsample({'p_max=1000'}), normalize, s2n('f_max=1000'), my_classif});
                % Use subsample if you want to try some ideas fast. 
                % Because of subsampling, the training error is not equal to the
                % error on all the training data.
                my_model=chain({ normalize, s2n('f_max=1000'), my_classif})
                % TIP#1: swap s2n and normalize
                % TIP#2: use the pixel representation and pixelGisette_exp_conv
            case 'madelon'
                my_classif=svc({'coef0=1', 'degree=0', 'gamma=1', 'shrinkage=1'});
                my_model=chain({probe(relief,{'p_num=2000', 'pval_max=0'}), standardize, my_classif})
                % TIP#1: replace pval_max=0 by f_max=20 (the number of features may vary because 
                % of the limited precision of the probe method and cause variance in the BER)
                % TIP#2: vary the number of features fmax=???  
            case 'dorothea'
                my_model=chain({TP('f_max=1000'), naive, bias}); %Reference
                % TIP#0: see how much worse it gets without "bias"
                % TIP#1: try to normalize
                % TIP#2: try to keep more features with TP and chain with
                % another feature selection method to get overall fewer
                % features.
        end
    case 'optimized'
        switch data_name
            case 'arcene'
                my_svc=svc({'coef0=2', 'degree=3', 'gamma=0', 'shrinkage=0.1'});
                my_model=chain({relief('f_max=1400'),normalize,my_svc}) 
            case 'dexter'
                my_classif=svc({'coef0=1', 'degree=1', 'gamma=0', 'shrinkage=0.5'});
                my_model=chain({s2n('f_max=4500'), normalize, my_classif})
            case 'gisette'
                my_classif=svc({'coef0=1', 'degree=3', 'gamma=0', 'shrinkage=1'});
                my_model=chain({ s2n('f_max=1000'), normalize, my_classif})
            case 'madelon'
                my_classif=svc({'coef0=1', 'degree=0', 'gamma=0.3', 'shrinkage=0.3'});
                my_model=chain({relief('f_max=20'), standardize, my_classif})
            case 'dorothea'
                my_model=chain({TP('f_max=15000'),normalize,relief('f_max=700'),naive, bias})
        end
     case 'noselect'
        switch data_name
            case 'arcene'
                my_svc=svc({'coef0=2', 'degree=3', 'gamma=0', 'shrinkage=0.1'});
                my_model=chain({normalize,my_svc}) 
            case 'dexter'
                my_classif=svc({'coef0=1', 'degree=1', 'gamma=0', 'shrinkage=0.5'});
                my_model=chain({normalize, my_classif})
            case 'gisette'
                my_classif=svc({'coef0=1', 'degree=3', 'gamma=0', 'shrinkage=1'});
                my_model=chain({normalize, my_classif})
            case 'madelon'
                my_classif=svc({'coef0=1', 'degree=0', 'gamma=0.3', 'shrinkage=0.3'});
                my_model=chain({standardize, my_classif})
            case 'dorothea'
                my_model=chain({normalize, naive, bias})
        end
    case 'pixelGisette_exp_conv'
        if ~UsePixelRep, 
            error('Does not work if the representation is not pixels');
        end
        % Purposely designed for Gisette, smoothe pixels
        my_classif=svc({'coef0=1', 'degree=4', 'gamma=0', 'shrinkage=0.1'});
        my_model=chain({convolve(exp_ker({'dim1=9', 'dim2=9'})), normalize, my_classif})
        %my_model=chain({subsample('p_max=200'), normalize, my_classif})
    case 'pixelGisette_gauss_conv'
        if ~UsePixelRep, 
            error('Does not work if the representation is not pixels');
        end
        % Purposely designed for Gisette, smoothe pixels
        my_classif=svc({'coef0=1', 'degree=4', 'gamma=0', 'shrinkage=0.5'});
        my_model=chain({convolve(gauss_ker({'dim1=5', 'dim2=5'})), normalize, my_classif})

    otherwise
        error('No such model!');
end