package algebras;

import terms.*;
import parsers.*;

/** The boolean algebra with <code>true</code> and <code>false</code>
  * as constants and <code>not</code> (negation), <code>and</code>
  * (binary and) and <code>or</code> (binary or) as further operations.
  * Since the semantics of symbols is fixed, the <code>parse</code>
  * method is meaningless. It simply does nothing.
  * @see algebra
  * @see symbol
  */
public class booleanAlgebra extends extensibleAlgebra {

  private static Integer True = new Integer(0);
  private static Integer False = new Integer(1);
  private static Integer not = new Integer(2);
  private static Integer and = new Integer(3);
  private static Integer or = new Integer(4);
  private static Integer xor = new Integer(5);
  private static Integer impl = new Integer(6);
  
  private static Boolean t = new Boolean(true);
  private static Boolean f = new Boolean(false);
  
/** Make the algebra recognise the standard symbols true, false, and, or, xor, not. **/

  public Object extendBy(symbol s) {
    if ("true".equalsIgnoreCase(s.toString()) && s.rank() == 0) return True;
    if ("false".equalsIgnoreCase(s.toString()) && s.rank() == 0) return False;
    if ("not".equalsIgnoreCase(s.toString()) && s.rank() == 1) return not;
    if ("implies".equals(s.toString()) && s.rank() == 2) return impl;
    if ("and".equalsIgnoreCase(s.toString())) return and;
    if ("or".equalsIgnoreCase(s.toString())) return or;
    if ("xor".equalsIgnoreCase(s.toString())) return xor;
    return null;
  }

/** This implements the operations <code>true</code>, <code>false</code>, <code>not</code>,
  * <code>and</code>, and </code>or</code> so that they behave as expected.
  */
  protected Object apply(symbol op, Object[] args) {
    Object o = operationOf(op);
    if (True.equals(o)) return t;
    if (False.equals(o)) return f;
    boolean[] boolArgs = new boolean[args.length];
    for (int i = 0; i < args.length; i++) {
      if (args[i] instanceof Boolean) boolArgs[i] = ((Boolean)args[i]).booleanValue();
      else return null;
    }
    if (not.equals(o)) {
      if (boolArgs[0]) return f; else return t;
    }
    if (and.equals(o)) {
      for (int i = 0; i < boolArgs.length; i++) {
        if (!boolArgs[i]) return f;
      }
      return t;
    }
    if (or.equals(o)) {
      for (int i = 0; i < boolArgs.length; i++) {
        if (boolArgs[i]) return t;
      }
      return f;
    }
    if (xor.equals(o)) {
      boolean result = false;
      for (int i = 0; i < boolArgs.length; i++) {
        if (boolArgs[i]) result = !result;
      }
      if (result) return t; else return f;
    }
    if (impl.equals(o)) {
      if (boolArgs[0] && !boolArgs[1]) return f; else return t;
    }
    return null;
  }
  
/** Initialize a <code>booleanAlgebra</code> by reading its definition from a stream.
  * @see booleanAlgebraParser
  * @exception ParseException if an error occurs
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
    booleanAlgebraParser parser = new booleanAlgebraParser(stream);
    parser.booleanAlgebra(this);
  }
  
}
