/** Class for managing tables in a branching tree grammar.
 *  This class is required by BSTGrammar.java
 *  This class requires SyncedRule.java
 */
package generators;

import java.util.*;
import terms.*;

/** According to definitions in the article, a more appropriate name for this class
  * would be Table, as supertables are tables containing tables, wheresas a table
  * can contain either other tables or rules, which is what this class implements.
  *
  * You can add rules and subtables to the table, by using the methods addRule
  * and addSubTable. But do not use both with the same table object, since in
  * the current implementation only tables OR rules are allowed in a table,
  * and Exception will be thrown if used otherwise.
  */
public class SuperTable  {

  private Vector subTables;
  private Vector rules;

  //at most one of these will be set to true later.
  private boolean containsSubTables,containsRules;

  private double weight;
  private String name;
  
  public SuperTable()
  {
    this(false);
  }
  public SuperTable(boolean c)
  {
    subTables = new Vector();
    rules = new Vector();
    containsSubTables = false;
    containsRules = false;
    weight = 1.0;
  }

  //returns true if at least one rule in at least one subtable has a right hand
  //side consisting of terminals only.
  public boolean isTerminable(finiteSignature termSig, finiteSignature nontermSig)
  {
    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        if(t.isTerminable(termSig, nontermSig))
          return true;
      }
    }//if(containsSubTables)
    else
    {
      for (int i=0; i < getNumRules(); i++)
      {
        term t = getRule(i).getRhs();
        if(isTerminal(t, termSig, nontermSig))
          return true;
      }
      return false;
    }//did not contain subtables
    return false;  //did not find terminating rule anywhere
  }//method isTerminable
  //Help function to isTerminable
  private boolean isTerminal(term t, finiteSignature termSig, finiteSignature nontermSig) {
/*
    symbol top = t.topSymbol();
    if (nontermSig.contains(top)) return termSig.contains(new symbol(top.toString(),0));
    for (int i=0; i < top.rank(); i++) {
      if (!isTerminal(t.subterm(i), termSig, nontermSig)) return false;
    }
    return true;
*/
    //code above was the old code, but it did no longer work when tranforming
    //BST so that terminals and nonterminals was disjoint, but then making the
    //suffix added to nonterminals in that process be "" to hide that fact.
    //if(BSTGrammar.getNontermSuffix().equals("")){
    symbol top = t.topSymbol();
    if (nontermSig.contains(top)) return false;//termSig.contains(new symbol(top.toString(),0));
    for (int i=0; i < top.rank(); i++) {
      if (!isTerminal(t.subterm(i), termSig, nontermSig)) return false;
    }
    return true;
    //}
  }

  public void setName(String _name)
  {
    name = _name;
  }
  public String getName()
  {
    return name;
  }
  public double getWeight()
  {
    return weight;
  }
  public void setWeight(double w)
  {
    weight = w;
  }
  public int getNumTables()
  {
    return subTables.size();
  }
  public SuperTable getSubTable(int index)
  {
    return (SuperTable)subTables.elementAt(index);
  }
  public int getNumRules()
  {
    return rules.size();
  }
  public SyncedRule getRule(int index)
  {
    return (SyncedRule)rules.elementAt(index);
  }
  public void addSubTable(SuperTable t)
  {
    if(containsRules)
    {
      throw new RuntimeException("Errnous usage of SuperTable.java. \n"+
        "Currently no support for having subtables and rules in same table.\n");
      //An alternative would be to add the subtable and move all rules to it.
      //But this action must be remembered because other subtables created later
      //should also get those rules.
    }
    subTables.addElement(t);
    containsSubTables = true;
  }
  public void removeSubTable(SuperTable t)
  {
    subTables.remove(t);
  }
  public void addRule(SyncedRule r)
  {
    if(containsSubTables)
    {
      throw new RuntimeException("Errnous usage of SuperTable.java. \n"+
        "Currently no support for having subtables and rules in same table.\n");
      //An alternative would be to add this rule to all subtables instead.
      //But if doing so this action must be remembered when more subtables are created.
    }
    rules.addElement(r);
    containsRules = true;
  }

  //Generate a vector containing all tables at a certain depth.
  public Vector getTablesAtDepth(int depth)
  {
    Vector v = new Vector();
    if(depth==0)  //base case
    {
      v.addElement(this);
    }//base case
    else
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        Vector v2 = t.getTablesAtDepth(depth-1);
        v.addAll(v2);
      }
    }//not base case
    return v;
  }//method getTablesAtDepth

  //Add implicit rules to all tables, according to the convention
  //that if in a table no rule exists with A as left hand side,
  //the rule A->A<0,0...,0> is added to that table.
  public void addImplicitRules(fixedRankSignature nonterminals, int d)
  {
    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        t.addImplicitRules(nonterminals, d);
      }
    }//if(containsSubTables)
    else
    {
      Vector collect = new Vector();
      Enumeration enum_ = nonterminals.elements();
      search:
      while (enum_.hasMoreElements()) {
        symbol s = (symbol)enum_.nextElement();
        for (int i=0; i < getNumRules(); i++) {
          if (s./*toString().*/equals( getRule(i).getLhs()/*.toString()*/ )) continue search;
        }
        collect.addElement(s);
      }
      for (int i=0; i < collect.size(); i++) {
        symbol lhs = (symbol)collect.elementAt(i);
        synchronizedSymbol rhs = new synchronizedSymbol(lhs.toString(), d);
        for(int j=0; j<d; j++)
        {
          rhs.setSync(j, 0);
        }
        SyncedRule rule = new SyncedRule(lhs, new term(rhs) );
        addRule(rule);
      }
    }//did not contain subtables

  }//addImplicitRules()


  //example: if only <00> appears in the rule, only first subtree in sync tree is used, so {0} is returned.
  //if <00> and <10> appears and d=2, first and third subtree is used, so {0,2} is returned.
  //transducer and tree grammar will be calling this to determine where to put
  //sub trees and where to put bottom symbols in the sync.trees.
  //note: returned vector may contain duplicates.
  public Vector getUsedSyncBranchNumbers(int d)
  {
    Vector ret = new Vector();

    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        Vector v = t.getUsedSyncBranchNumbers(d);
        ret.addAll(v);
      }
    }//if(containsSubTables)
    else
    {
      for(int i=0; i<rules.size(); i++)
      {
        SyncedRule r = (SyncedRule)rules.elementAt(i);
        Vector v = r.getUsedSyncBranchNumbers(d);
        ret.addAll(v);
      }
    }//did not contain subtables
    return ret;
  }

  //Same as getUsedSyncBranchNumbers except that it looks only on the
  //first l numbers in the sync-symbol.
  //If <10> appears {1} is returned. If <01>,<11>,<10>,<00> all appears,
  //{0,1} is returned.
  public Vector getUsedSyncBranchNumbersForSyncLevel(int d, int l)
  {
    Vector ret = new Vector();

    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        Vector v = t.getUsedSyncBranchNumbersForSyncLevel(d,l);
        ret.addAll(v);
      }
    }//if(containsSubTables)
    else
    {
      for(int i=0; i<rules.size(); i++)
      {
        SyncedRule r = (SyncedRule)rules.elementAt(i);
        Vector v = r.getUsedSyncBranchNumbersForSyncLevel(d,l);
        ret.addAll(v);
      }
    }//did not contain subtables
    return ret;
  }

  public String toString() {
    return toString("");
  }

  private String toString(String indent) {
    String result = indent + name;
    result = result + " {\n";
    for (int i = 0; i < rules.size(); ) {
      SyncedRule r = (SyncedRule)rules.elementAt(i);
      result = result + indent + "  " + r.lhs + " -> " + r.rhs
                 + " weight " + r.weight;
      Vector v = r.getSyncStrings();
      if (v.size() > 0) {
        result = result + " sync ";
        for (int j = 0; j < v.size(); ) {
          int[] s = (int[])v.elementAt(j);
          result = result + "<";
          for (int k = 0; k < s.length; ) {
            result = result + s[k++];
            if (k < s.length) result = result + " ";
          }
          result = result + ">";
          if (++j < v.size()) result = result + " ";
        }
      }
      if (++i < rules.size()) result = result + ",\n";
    }
    for (int i = 0; i < subTables.size(); ) {
      result = result + ((SuperTable)subTables.elementAt(i)).toString(indent+"  ");
      if (++i < subTables.size()) result = result + ",\n";
    }
    result = result + "\n" + indent + "} weight " + weight;
    return result;
  }

  public fixedRankSignature nonterminalslnRhsThatUseGivenSyncNumber(finiteSignature lhs, int syncNumber, int d, int currentN)
  {
    fixedRankSignature ret = new fixedRankSignature(0);
    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        fixedRankSignature s = t.nonterminalslnRhsThatUseGivenSyncNumber(lhs, syncNumber, d, currentN);
        ret.unionWith(s);
      }
    }//if(containsSubTables)
    else
    {
      for(int i=0; i<rules.size(); i++)
      {
        SyncedRule r = (SyncedRule)rules.elementAt(i);
        term rhs = r.getRhs();
        if(lhs==null || lhs.contains(r.getLhs()))
        {
          fixedRankSignature s = r.nonterminalsThatUseGivenSyncNumber(rhs, syncNumber, d, currentN);
          ret.unionWith(s);
        }
      }
    }//did not contain subtables
    return ret;
  }//nonterminalslnRhsThatUseGivenSyncNumber

  public void increaseLengthOfSyncSymbolTuplesWithOne(int padSyncSymbol)
  {
    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        t.increaseLengthOfSyncSymbolTuplesWithOne(padSyncSymbol);
      }
    }//if(containsSubTables)
    else
    {
      for(int i=0; i<rules.size(); i++)
      {
        SyncedRule r = (SyncedRule)rules.elementAt(i);
        r.increaseLengthOfSyncSymbolTuplesWithOne(padSyncSymbol);
      }
    }//did not contain subtables
  }//method increaseLengthOfSyncSymbolTuplesWithOne

  public void relabelSymbol(symbol from, symbol to)
  {
    if(containsSubTables)
    {
      for(int i=0; i<subTables.size(); i++)
      {
        SuperTable t = (SuperTable)subTables.elementAt(i);
        t.relabelSymbol(from, to);
      }
    }//if(containsSubTables)
    else
    {
      for(int i=0; i<rules.size(); i++)
      {
        SyncedRule r = (SyncedRule)rules.elementAt(i);
        r.relabelSymbol(from, to);
      }
    }//did not contain subtables
  }//method relabelSymbol

}//class SuperTable

