package generators;

import util.*;
import terms.*;
import parsers.*;

/** This class provides a sort of hull for <code>treeTransducer</code>s that adds a
  * command named <code>iterate</code>, which can be used to iterate the application
  * of the internal <code>treeTransducer</code>, i.e., to apply it to its own output term again
  * and again.
  */
public class iterator extends treeTransducer {

  private static String iterate = "iterate";
  private static String[] commands = { iterate };
  private treeTransducer mod = null;
  private term currTerm = null;

/** The list of commands is the original one (i.e., the one provided by the
  * <code>treeTransducer</code> object which is captured by this one), plus
  * the additional command named <code>iterate</code>.
  */
  public list commands() {
    list com = (list)mod.commands().clone();
    com.prepend(commands);
    return com;
  }

/** Except for the <code>iterate</code> command (which just causes
  * the internal <code>treeTransducer</code> to be applied to its last output term),
  * commands are passed through to the wrapped object.
  * In this case, the results of <code>currentTerm()</code> before and after are
  * compared by == <em>and</em> the <code>equal</code> method. If the test
  * indicates that a change has taken place, the term is taken as the new current
  * term. Otherwise, the current term does not change. (Note that, depending on
  * the implementation of the internal <code>treeTransducer</code> this may sometimes
  * lead to an unexpected behaviour. In most cases, however, the result should
  * be the expected one.)
  */ 
  public void execute(String command) {
    if (iterate.equals(command)) {
      if (currTerm != null) currTerm = mod.apply(currTerm);
    }
    else {
      term old = mod.currentTerm();
      term oldCopy = old==null ? null : (term)old.clone();
      mod.execute(command);
      term newCurr = mod.currentTerm();
      if (newCurr != old || old != null && !oldCopy.equals(newCurr)) {
        currTerm = newCurr;
      }
    }
  }

/** Apply the internal <code>treeTransducer</code> to a term. */
  public term apply(term t) {
    currTerm = mod.apply(t);
    return currTerm;
  }
  
  public term currentTerm() { return currTerm; }

/** The syntax consists of an object definition that can be read by an
  * <code>objectParser</code>; it is required to yield an instance of
  * <code>treeTransducer</code>.
  * @see objectParser
  * @exception ParseException if an error occurs (in particular, if the defined
  *                                      object is not a<code>treeTransducer</code>.
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
     objectParser parser = new objectParser(stream);
     parsable parsed = parser.parse();
     if (parsed instanceof treeTransducer) mod = (treeTransducer)parsed;
     else throw new ParseException(parsed.getClassName() + " not a subclass of termModifer");
   }

}
