package gui;

import java.awt.*;
import java.awt.event.*;

/** A <code>node</code> whose graphical representation consists of a bullet
  * with an arbitrary number of labels (i.e., strings) underneath.
  */
public abstract class visibleNode extends node implements MouseListener,
                                               MouseMotionListener, ComponentListener {

  public static int radius = 15;
  
  protected Color col = new Color(.75f,1,.75f);
  protected int[] figureX = null;
  protected int[] figureY = null;

  private static int textGap = 5;
  private Point pressedAt   = null;
  private boolean clickedOnce = false;
  private textCanvas labels = new textCanvas();

  public visibleNode() {
    super();
    addMouseListener(this);
    addMouseMotionListener(this);
    setTextLocation();
    add(labels);
  }
  
  public void mouseClicked(MouseEvent e) {
    if (e.getClickCount() == 1) clickedOnce = true;
    else if (e.getClickCount() == 2) {
      clickedOnce = false;
      open();
    }
  }
  
  protected abstract void open();
    
  public void mousePressed(MouseEvent e) {
    pressedAt = new Point(e.getPoint());
  }
  
  public void mouseReleased(MouseEvent e) {
    pressedAt = null;
  }
  
  public void mouseEntered(MouseEvent e) {
  }
  
  public void mouseExited(MouseEvent e) {
    if (clickedOnce) {
      clickedOnce = false;
      parent().createDanglingEdge(this, e.getPoint());
    }
  }
  
  public void mouseDragged(MouseEvent e) {
    clickedOnce = false;
    if (pressedAt != null) {
      setPosition(position.x + e.getPoint().x - pressedAt.x,
                  position.y + e.getPoint().y - pressedAt.y);
    }
  }
  
  public void mouseMoved(MouseEvent e) {
  }
  
  public void componentHidden(ComponentEvent e) {
  }
  
  public void componentMoved(ComponentEvent e) {
  }
  
  public void componentResized(ComponentEvent e) {
    setPosition(position);
  }
  
  public void componentShown(ComponentEvent e) {
  }
  
  private void setPosition(int x, int y) {
    Dimension d = getParent().getSize();
    x = Math.min(x, d.width - radius);
    y = Math.min(y, d.height - radius);
    x = Math.max(radius, x);
    y = Math.max(radius, y);
    if (x != position.x || y != position.y) {
      position.x = x;
      position.y = y;
      updateBounds();
      setTextLocation();
      setChanged();
      notifyObservers();
    }
  }
  
  public void setPosition(Point p) {
    setPosition(p.x, p.y);
  }
  
  private void updateBounds() {
    setBounds(position.x-Math.max(radius,labels.getSize().width/2),
              position.y-radius,
              Math.max(2*radius,labels.getSize().width),
              2*radius+labels.getSize().height+textGap);
  }
  
/** Add a label. */
  public void addLabel(String line) {
    labels.addTextLine(line);
    setTextLocation();
    updateBounds();
    invalidate();
    repaint();
  }
  
/** Remove all labels. */
  public void removeLabels() {
    labels.removeTextLines();
    setTextLocation();
    updateBounds();
    invalidate();
    repaint();
  }
  
  public void paint(Graphics g) {
    super.paint(g);
    g.setColor(Color.black);
    g.translate(getSize().width/2, radius);
    g.fillArc(-radius,-radius,2*radius,2*radius,0,360);
    if (figureX != null) {
      g.setColor(col);
      g.fillPolygon(figureX, figureY, figureX.length);
      g.drawPolygon(figureX, figureY, figureX.length);
      g.setColor(Color.black);
    }
  }
  
  public boolean contains(int x, int y) {
    x -= getSize().width/2;
    y -= radius;
    return Math.sqrt(x*x + y*y) <= radius;
  }
  
  public boolean contains(Point p) {
    return contains(p.x, p.y);
  }
  
  public Point getBorderPoint(Point seenFrom) {
    int deltaX = seenFrom.x - position.x;
    int deltaY = seenFrom.y - position.y;
    double len = Math.sqrt(deltaX * deltaX + deltaY * deltaY);
    if (len <= radius) return null;
    else return new Point((int)(position.x + deltaX * radius / len),
                          (int)(position.y + deltaY * radius / len));
  }
  
  private void setTextLocation() {
    labels.setLocation(Math.max(0,radius - labels.getSize().width/2), 2*radius + textGap);
  }
  
}

