package displays;

import java.awt.*;
import java.util.*;
import terms.*;
import util.*;
import gui.*;
import parsers.*;

/** A display for terms, which are shown as trees. */
public class treeDisplay extends display {

  private static int horizontalPixels = 500;
  private static int verticalPixels = 500;
  private static int inset = 20;
  private int distanceX;
  private int distanceY;
  private int initialFontSize = 14;
  private int fontSize = initialFontSize;
  private static int fontStyle = Font.BOLD;
  private Hashtable colours = new Hashtable(10);
  private term inputTerm = null;
  private layoutTerm displayed = null;
  private displayCanvas theCanvas = new displayCanvas();
  private FontMetrics metrics;
  private static String noTree = "missing or undefined input";
  private static Font noTreeFont = new Font("Monospaced", fontStyle, 14);
  private static String larger = "larger";
  private static String smaller = "smaller";
  private static String smallest = "smallest";
  private static String reset = "reset view";
  private static String[] fontCommands = { larger, smaller, smallest };
  private static String[] otherCommands = { reset };
  
  public treeDisplay() {
    theCanvas.setSize(horizontalPixels,verticalPixels);
    theCanvas.setView(-horizontalPixels / 2, horizontalPixels / 2, 0, verticalPixels);
    initFont();
  }    
  
  private void initFont() {
    Font f = new Font("Serif", fontStyle, fontSize);
    theCanvas.setFont(f);
    theCanvas.enableScaling(false);
    metrics = theCanvas.getFontMetrics(f);
    distanceX = fontSize;
    distanceY = 2 * fontSize;
  }
  
  public void setColour(symbol s, Color c) {
    colours.put(s, c);
  }
  
/** If the argument is a term, it is displayed as a tree. */
  protected void displayObject(Object obj) {
    if (obj != null && obj instanceof term) {
      inputTerm = (term)obj;
      displayed = new layoutTerm(inputTerm);
    }
    else inputTerm = displayed = null;
    setAbs();
    theCanvas.invalidate();
    theCanvas.repaint();
  }
  
  private void setAbs() {
    if (displayed != null) {
      int w = Math.max(displayed.width+2*inset,theCanvas.getSize().width)/2;
      int h = displayed.height*(distanceY+metrics.getMaxAscent())+2*inset+metrics.getMaxAscent();
      theCanvas.setAbs(-w, w, 0, Math.max(h,theCanvas.getSize().height));
    }
    else {
      theCanvas.setAbs(0,1,0,1);
    }
  }
  
/** The component on which the trees will be drawn. */
  public Component visualizer() { return theCanvas; }

/** This class provides the commands "larger", "smaller", and "smallest",
  * which affect the font size used
  * for the nodes of the tree.
  */
  public list commands() {
    list result = new list();
    result.append(fontCommands);
    result.append(otherCommands);
    return result;
  }
  
  public void execute(String command) {
    if (reset.equals(command)) {
      fontSize=initialFontSize;
      theCanvas.setView(-horizontalPixels / 2, horizontalPixels / 2, 0, verticalPixels);
    }
    else if (larger.equals(command)) fontSize++;
    else if (smaller.equals(command) && fontSize > 1) fontSize--;
    else if (smallest.equals(command)) fontSize = 1;
    initFont();
    if (inputTerm != null) displayed = new layoutTerm(inputTerm);
    setAbs();
    theCanvas.invalidate();
    theCanvas.repaint();
  }
  
/** Initialize a <code>treeDisplay</code> by reading its definition from a stream.
  * @see treeDisplayParser
  * @exception ParseException if an error occurs
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
    setParser parser = new setParser(stream);
    parser.set(new treeDisplayParser(stream, this));
  }
  
  private class layoutTerm extends term {
  
    public int width;       // The width of the tree when layed out
    public int subWidth;    // The total width of the direct subtrees
    public int height;      // The (abstract) height of the tree
    
    public layoutTerm(term t) {
      super(t.topSymbol());
      int rank = topSymbol().rank();
      subWidth = (rank > 0) ? (rank - 1) * distanceX : 0;
      height = 0;
      for (int i = 0; i < rank; i++) {
        layoutTerm s = new layoutTerm(t.subterm(i));
        defineSubterm(i, s);
        subWidth += s.width;
        height = Math.max(height, s.height + 1);
      }
      width = Math.max(subWidth, metrics.stringWidth(topSymbol().toString()));
    }
  
  }
  
  private class displayCanvas extends scrollCanvas {
  
    /**
	 * 
	 */
	private static final long serialVersionUID = -1303828560222021564L;

	public void paint(Graphics g) {
      super.paint(g);
      if (displayed != null) {
        g.setColor(Color.black);
        g.translate((getSize().width-(int)(viewL + viewR))/2, -(int)viewT);
        paint(displayed, 0, inset + metrics.getMaxAscent(), g);
      }
      else paintNoTree(g);
    }
    
    private void paint(layoutTerm t, int x, int y, Graphics g) {
      symbol symb = t.topSymbol();
      String label = symb.toString();
      Object c = colours.get(symb);
      if (c != null) g.setColor((Color)c);
      g.drawString(symb.toString(), x - metrics.stringWidth(label)/2, y);
      if (c != null) g.setColor(Color.black);
      int currX = x - t.subWidth/2;
      int currY = y + distanceY + metrics.getMaxAscent();
      for (int current = 0; current < symb.rank(); current++) {
        layoutTerm sub = (layoutTerm)t.subterm(current);
        int mid = currX + sub.width/2;
        int y1 = y + metrics.getMaxDescent();
        int y2 = currY - metrics.getMaxAscent();
        g.drawLine(x, y1, mid, y2);
        paint(sub, mid, currY, g);
        currX = currX + sub.width + distanceX;
      }
    }
    
    private void paintNoTree(Graphics g) {
      g.setColor(Color.red);
      g.translate(getSize().width/2, getSize().height/2);
      g.setFont(noTreeFont);
      FontMetrics metrics = g.getFontMetrics(noTreeFont);
      int width = metrics.stringWidth(noTree);
      int height = metrics.getMaxAscent();
      g.drawString(noTree, -width/2, height/2);
      int space = metrics.getMaxAdvance();
      g.drawRect(-(width+space)/2, -height, width+space, 2*height);
    }
    
    protected void resizeView() {
      double deltaX = getSize().width - (viewR - viewL);
      double deltaY = getSize().height - (viewB - viewT);
      setView(viewL - deltaX/2, viewR + deltaX/2, viewT, viewB + deltaY);
    }
  }

}
