package generators;

import java.util.*;
import terms.*;
import parsers.*;
import util.*;

/** A top-down tree transducer.
  * Currently, non-determinism is implemented by random choice (evenly distributed, if
  * <code>Math.random()</code> yields such a distribution). This is not really the most
  * appropriate choice, but it is simple...<br>
  * The basic behaviour of an object of this class can be controlled by the commands
  * <code>single steps</code> and <code>complete runs</code>. Normally, if an input
  * term is given to the transducer, rules will be applied as long as possible and, upon
  * successful termination, the resulting term is returned (otherwise, the result is
  * <code>null</code>). However, using <code>single steps</code> one can switch to
  * a mode where single derivation steps are applied in a breadth-first manner. In this
  * case, there is an additional command <code>next step</code> which acts in the
  * obvious way. Using the command <code>complete runs</code> one can switch back
  * to the default behaviour. In case of non-determinism, the sequences of random choices
  * underlying successive derivations will be based on one and the same random seed
  * unless the command <code>new random seed</code> is applied, which randomly
  * selects a new seed. 
  */
public class tdTransducer extends treeTransducer {

//====================================================================
// First the 'interactive' part that deals with commands ...
//====================================================================
  private static String nextStep = "single step";
  private static String parallelStep = "parallel step";
  private static String back = "back";
  private static String singleSteps = "derive stepwise";
  private static String completeRuns = "results only";
  private static String newSeed = "new random seed";
  private static String variable = "variable random seed";
  private static String fixed = "fixed random seed";
  private boolean stepwise = false;
  protected boolean fixedSeed = true;
  private boolean isDeterministic, isTotal;
  
/** Returns the (names of the) commands described above. */
  public list commands() {
    list result = new list();
    String[] com;
    if (stepwise) {
      com = new String[3];
      com[0] = nextStep;
      com[1] = parallelStep;
      com[2] = back;
      result.append(com);
    }
    if (isDeterministic) {
      com = new String[1];
      com[0] = stepwise ? completeRuns : singleSteps;
      result.append(com);
    }
    else {
      com = new String[2];
      com[0] = stepwise ? completeRuns : singleSteps;
      com[1] = newSeed;
      result.append(com);
      com = new String[1];
      com[0] = fixedSeed ? variable : fixed;
      result.append(com);
    }
    return result;
  }
  
  public void execute(String command) {
    if (nextStep.equals(command) && ct != null) {
      singleStep();
    }
    else if (parallelStep.equals(command)) {
      parallelStep();
    }
    else if (back.equals(command)) {
      back();
    }
    else if (newSeed.equals(command)) {
      newSeed();
      apply(lastArgument);
    }
    else {
      if (variable.equals(command) || fixed.equals(command)) fixedSeed = !fixedSeed;
      else if (singleSteps.equals(command) || completeRuns.equals(command)) {
        stepwise = !stepwise;
        if (ct != null && stepwise) ct.stepCount = -1;
        currTerm = getTerm(ct);
      }
    }
  }

//====================================================================
// Now the actual implementation of top-down tree transducers ...
//====================================================================

  private term[][] rule;
  private double[] weight;
  private symbol[][] rhsQ;
  private int[][] rhsIndex;
  private term[] rhs;
  private fixedRankSignature states;
  private finiteSignature inputSignature;
  private symbol initialState;
  private term lastArgument = null;
  private term currTerm = null;
  private computationTree ct;
  private long randomSeed;
  private Random random = new Random();
  private int decompositionIndex;
  private lookAheadTree lookAhead;
  
  public tdTransducer() {
    newSeed();
  }
  
  private void newSeed() {
    randomSeed = (long)(((double)Long.MAX_VALUE) * Math.random());
    random.setSeed(randomSeed);
  }
  
  private void resetSeed() {
    if (fixedSeed) random.setSeed(randomSeed);
    else newSeed();
  }
  
/** Compute the output for a given input term. */
  public term apply(term t) {
    lastArgument = t;
    if (t == null) {
      ct = null;
      lookAhead = null;
    }
    else {
      if (isDeterministic || isTotal) lookAhead = null;
      else {
        lookAhead = lookAhead(t);
        if (!lookAhead.reachable.get(states.indexOf(initialState))) lookAhead = null;
      }
      resetSeed();
      run();
      if (stepwise) ct.stepCount = -1;
    }
    currTerm = getTerm();
    return currTerm;
  }
  
  private void run() {
    boolean useLookAhead = lookAhead != null;
    list thisLevel = new list(); 
    if (useLookAhead) thisLevel.prepend(lookAhead);
    thisLevel.prepend(ct = new computationTree());
    thisLevel.prepend(lastArgument);
    thisLevel.prepend(initialState);
    list nextLevel = new list();
    while (true) {
      allowExit();
      if (thisLevel.isEmpty()) {
        if (nextLevel.isEmpty()) {
          return;
        }
        thisLevel = nextLevel;
        nextLevel = new list();
      }
      symbol state = (symbol)thisLevel.head(); thisLevel = thisLevel.tail();
      term t = (term)thisLevel.head(); thisLevel = thisLevel.tail();
      computationTree currCt = (computationTree)thisLevel.head(); thisLevel = thisLevel.tail();
      lookAheadTree la;
      if (useLookAhead) {
        la = (lookAheadTree)thisLevel.head();
        thisLevel = thisLevel.tail();
      }
      else la = null;
      term initial = new term(state);
      initial.defineSubterm(0,t);
      int rule = chooseRule(initial,la);
      currCt.initialize(initial, rule);
      if (rule != -1) {
        for (int i = 0; i < rhsQ[currCt.rule].length; i++) {
          if (useLookAhead) nextLevel.prepend(la.sub[rhsIndex[currCt.rule][i]]);
          nextLevel.prepend(currCt.subcomputation[i] = new computationTree());
          nextLevel.prepend(t.subterm(rhsIndex[currCt.rule][i]));
          nextLevel.prepend(rhsQ[currCt.rule][i]);
        }
      }
    }
  }
  
  boolean singleStepPerformed;
  
  private void singleStep() {
    singleStepPerformed = false;
    singleStep(ct);
    currTerm = getTerm();
  }
  
  private void singleStep(computationTree ct) {
    if (ct.stepCount != -1) {
      ct.stepCount++;
      for (int i = 0; i < ct.subcomputation.length; i++) singleStep(ct.subcomputation[i]);
    }
    else if (!singleStepPerformed) {
      singleStepPerformed = true;
      ct.stepCount = 0;
      for (int i = 0; i < ct.subcomputation.length; i++) ct.subcomputation[i].stepCount = -1;
    }
  }
  
  private void parallelStep() {
    parallelStep(ct);
    currTerm = getTerm();
  }
  
  private void parallelStep(computationTree ct) {
    if (ct.stepCount++ != -1) {
      for (int i = 0; i < ct.subcomputation.length; i++) parallelStep(ct.subcomputation[i]);
    }
    else {
      for (int i = 0; i < ct.subcomputation.length; i++) ct.subcomputation[i].stepCount = -1;
    }
  }
  
  private void back() {
    back(ct);
    currTerm = getTerm();
  }
  
  private void back(computationTree ct) {
    if (ct.stepCount != -1) {
      ct.stepCount--;
      for (int i = 0; i < ct.subcomputation.length; i++) back(ct.subcomputation[i]);
    }
  }
  
  private int chooseRule(term t, lookAheadTree lookAhead) {
    symbol state = t.topSymbol();
    symbol root = t.subterm(0).topSymbol();
    double sumWeights = 0;
    double zeroes = 0;
    int ruleNumber = -1;
    ruleloop: for (int i = 0; i < rule.length; i++) {
      if (state.equals(rule[i][0].topSymbol()) &&
          root.equals(rule[i][0].subterm(0).topSymbol())) {
        if (lookAhead != null) {
          for (int j = 0; j < rhsQ[i].length; j++) {
            if (!lookAhead.sub[rhsIndex[i][j]].reachable.get(states.indexOf(rhsQ[i][j]))) continue ruleloop;
          }
        }
        sumWeights += weight[i];
        if (sumWeights > 0) {
          if (weight[i] > 0 && random.nextFloat() <= weight[i]/sumWeights) ruleNumber = i;
        }
        else if (random.nextFloat() <= 1.0/++zeroes) ruleNumber = i;
      }
    }
    return ruleNumber;
  }
  
  private term getTerm() {
    if (ct == null) return null;
    return getTerm(ct);
  }
  
  private term getTerm(computationTree ct) {
    if (stepwise && ct.stepCount == -1) return ct.initial;
    if (ct.rule == -1) return null;
    return constructTerm(rhs[ct.rule],ct.subcomputation);
  }
  
  private term constructTerm(term t, computationTree[] subcomp) {
    symbol top = t.topSymbol();
    if (top instanceof variable) return getTerm(subcomp[((variable)top).index()]);
    term result = new term(top);
    for (int i = 0; i < top.rank(); i++) {
      term s = constructTerm(t.subterm(i),subcomp);
      if (s == null) return null;
      result.defineSubterm(i,s);
    }
    return result;
  }
    
  
  public term currentTerm() {
    return currTerm;
  }
  
  protected boolean isDeterministic() { return isDeterministic; }
  
  private void checkDeterminism() {
    isDeterministic = false;
    for (int i = 0; i < rule.length; i++) {
      for (int j = i+1; j < rule.length; j++) {
        if (rule[i][0].topSymbol().equals(rule[j][0].topSymbol()) &&
            rule[i][0].subterm(0).topSymbol().equals(rule[j][0].subterm(0).topSymbol())) {
          return;
        }
      }
    }
    isDeterministic = true;
  }
  
  private void checkTotality() {
    isTotal = true;
    for (Enumeration stateEnum = states.elements(); stateEnum.hasMoreElements(); ) {
      symbol q = (symbol)stateEnum.nextElement();
      l: for (Enumeration inEnum = inputSignature.elements(); inEnum.hasMoreElements(); ) {
        symbol f = (symbol)inEnum.nextElement();
        for (int i = 0; i < rule.length; i++) {
          if (rule[i][0].topSymbol().equals(q) && rule[i][0].subterm(0).topSymbol().equals(f)) continue l;
        }
        isTotal = false;
        return;
      }
    }
  }

/** Initialize this object by parsing the definition of a top-down tree transducer.
  * The syntax is defined by the class <code>tdTransducerParser</code>.
  * @see tdTransducerParser
  * @exception ParseException if an error occurs
  */
  public void parse(ASCII_CharStream stream) throws ParseException {
    tdTransducerParser parser = new tdTransducerParser(stream);
    parser.tdTransducer();
    initialState = parser.initial;
    rule = parser.rule;
    weight = parser.weight;
    states = parser.states;
    inputSignature = parser.in;
    rhsQ = new symbol[rule.length][];
    rhsIndex = new int[rule.length][];
    rhs = new term[rule.length];
    for (int i = 0; i < rule.length; i++) {
      Vector Q = new Vector();
      Vector I = new Vector();
      decompositionIndex = 0;
      rhs[i] = decompose(Q,I,rule[i][1]);
      rhsQ[i] = new symbol[Q.size()];
      Q.copyInto(rhsQ[i]);
      rhsIndex[i] = new int[I.size()];
      for (int j = 0; j < I.size(); j++) rhsIndex[i][j] = ((Integer)I.elementAt(j)).intValue();
     }
    checkDeterminism();
    checkTotality();
  }
  
  private term decompose(Vector Q, Vector I, term t) {
    symbol top = t.topSymbol();
    if (states.contains(top)) {
      Q.addElement(top);
      I.addElement(new Integer(((variable)t.subterm(0).topSymbol()).index()-1));
      return new term(new variable(decompositionIndex++));
    }
    else {
      term result = new term(top);
      for (int i = 0; i < top.rank(); i++) result.defineSubterm(i,decompose(Q,I,t.subterm(i)));
      return result;
    }
  }
  
  private class computationTree {
    
    public term initial;
    public int rule;
    public int stepCount;
    public computationTree[] subcomputation = null;
    
    public void initialize(term initial, int rule) {
      this.initial = initial;
      this.rule = rule;
      this.stepCount = -1;
      if (rule != -1) this.subcomputation = new computationTree[rhsQ[rule].length];
      else this.subcomputation = new computationTree[0];
    }
    
  }
  
  private class lookAheadTree {
    
    public BitSet reachable;
    public lookAheadTree[] sub;
    
    public lookAheadTree(int rank) {
      sub = new lookAheadTree[rank];
      reachable = new BitSet(states.maxIndex());
    }
    
//    public String toString() {
//      if (sub.length == 0) return reachable.toString();
//      String result = reachable.toString() + "[" + sub[0].toString();
//      for (int i = 1; i < sub.length; i++) result = result + "," + sub[i];
//      return result + "]";
//    }
  }
  
  private lookAheadTree lookAhead(term t) {
    symbol top = t.topSymbol();
    lookAheadTree result = new lookAheadTree(top.rank());
    for (int i = 0; i < top.rank(); i++) result.sub[i] = lookAhead(t.subterm(i));
    l: for (int i = 0; i < rule.length; i++) {
      symbol q = rule[i][0].topSymbol();
      if (result.reachable.get(states.indexOf(q)) || !rule[i][0].subterm(0).topSymbol().equals(top)) continue;
      for (int j = 0; j < rhsQ[i].length; j++) {
        int stateIndex = states.indexOf(rhsQ[i][j]);
        if (!result.sub[rhsIndex[i][j]].reachable.get(stateIndex)) continue l;
      }
      result.reachable.set(states.indexOf(q));
    }
    return result;
  }
  
}
