package gui;

import java.awt.*;
import javax.swing.*;
import java.awt.event.*;

/** A canvas that allows the user to drag the shown image to another place using the mouse.
  * The <code>paint</code> method of subclasses of this class should take into account the
  * intended translation of the image that can be inquired using <code>getTranslation</code>,
  * in order to implement the expected behaviour.
  */
public abstract class scrollCanvas extends Canvas implements AdjustmentListener, ComponentListener {

  private boolean scalingEnabled = true;
  private Point lastPressed = null;
  private Point firstPressed = null;
  private JScrollBar hScroll = null;
  private JScrollBar vScroll = null;
  protected double absL, absR, absT, absB, viewL, viewR, viewT, viewB;
  private boolean scrollingEnabled = true;
  
  private static final int smax = 10000;
  
  public scrollCanvas() {
    super();
    enableEvents(AWTEvent.MOUSE_EVENT_MASK);
    enableEvents(AWTEvent.MOUSE_MOTION_EVENT_MASK);
    hScroll = new JScrollBar(Adjustable.HORIZONTAL);
    vScroll = new JScrollBar(Adjustable.VERTICAL);
    hScroll.addAdjustmentListener(this);
    vScroll.addAdjustmentListener(this);
    addComponentListener(this);
  }
  
  public JScrollBar getHScrollBar() {
    return hScroll;
  }
  
  public JScrollBar getVScrollBar() {
    return vScroll;
  }
  
  public void setAbsSilently(double left, double right, double top, double bottom) {
    absL = left;
    absR = right;
    absT = top;
    absB = bottom;
  }
  
  public void setViewSilently(double left, double right, double top, double bottom) {
    viewL = left;
    viewR = right;
    viewT = top;
    viewB = bottom;
  }
  
  public void setAbs(double left, double right, double top, double bottom) {
    setAbsSilently(left, right, top, bottom);
    adjustScrollBars();
  }
  
  public void setView(double left, double right, double top, double bottom) {
    setViewSilently(left, right, top, bottom);
    adjustScrollBars();
  }
  
  public void enableScrolling(boolean b) {
    scrollingEnabled = b;
    adjustScrollBars();
  }
  
  protected abstract void resizeView();
  
  private double absL() { return absL <= viewL ? absL:viewL; }
  private double absR() { return absR >= viewR ? absR:viewR; }
  private double absT() { return absT <= viewT ? absT:viewT; }
  private double absB() { return absB >= viewB ? absB:viewB; }
  
  protected void adjustScrollBars() {
    hScroll.removeAdjustmentListener(this);
    vScroll.removeAdjustmentListener(this);
    if (scrollingEnabled) {
      double deltaH = Math.abs(absL()-absR());
      double deltaV = Math.abs(absT()-absB());
      double valH = Math.abs(viewL - absL());
      double valV = Math.abs(viewT - absT());
      double extH = Math.min(deltaH-valH,Math.abs(viewL-viewR));
      double extV = Math.min(deltaV-valV,Math.abs(viewT-viewB));
      hScroll.setValues((int)(valH * ((double)smax / deltaH)),
                        (int)(extH * ((double)smax / deltaH)),
                        0, smax);
      vScroll.setValues((int)(valV * ((double)smax / deltaV)),
                        (int)(extV * ((double)smax / deltaV)),
                        0, smax);
      hScroll.setBlockIncrement(hScroll.getVisibleAmount() * 19 / 20);
      vScroll.setBlockIncrement(vScroll.getVisibleAmount() * 19 / 20);
    }
    else {
      hScroll.setValues(0, 100, 0, 100);
      vScroll.setValues(0, 100, 0, 100);
    }
    hScroll.addAdjustmentListener(this);
    vScroll.addAdjustmentListener(this);
  }
  
  public boolean between(double x, double a, double b) {
    return a <= x && x <= b || b <= x && x <= a;
  }
  
  public void adjustmentValueChanged(AdjustmentEvent e) {
    if (e.getAdjustable()==hScroll) hValueChanged(e.getValue());
    else if (e.getAdjustable()==vScroll) vValueChanged(e.getValue());
    invalidate();
    repaint();
    adjustScrollBars();
  }
  
  protected void hValueChanged(int val) {
    double hSize = viewR - viewL;
    viewL = absL() + (absR() - absL()) * (((double)val) / (double)smax);
    viewR = viewL + hSize;
  }
  
  protected void vValueChanged(int val) {
    double vSize = viewB - viewT;
    viewT = absT() + (absB() - absT()) * (((double)val) / (double)smax);
    viewB = viewT + vSize;
  }
  
  public void componentResized(ComponentEvent e) {
    resizeView();
  }
  
  public void componentMoved(ComponentEvent e) {}
  public void componentShown(ComponentEvent e) {}
  public void componentHidden(ComponentEvent e) {}

  public void enableScaling(boolean b) {
    scalingEnabled = b;
  }
  
  protected void processMouseEvent(MouseEvent e){
    super.processMouseEvent(e);
    if (e.getID() == MouseEvent.MOUSE_PRESSED) {
      if (!(e.isMetaDown() || e.isAltDown())) lastPressed = e.getPoint();
      else if (scalingEnabled) {
        firstPressed = lastPressed = e.getPoint();
        drawSquare(getGraphics());
      }
    }
    else if (e.getID() == MouseEvent.MOUSE_RELEASED && lastPressed != null) {
      if (!(e.isMetaDown() || e.isAltDown())) lastPressed = null;
      else if (scalingEnabled && firstPressed != null) {
        drawSquare(getGraphics());
        int xMin = Math.min(firstPressed.x, lastPressed.x);
        int xMax = Math.max(firstPressed.x, lastPressed.x);
        int yMin = Math.min(firstPressed.y, lastPressed.y);
        int yMax = Math.max(firstPressed.y, lastPressed.y);
        Dimension d = getSize();
        firstPressed = lastPressed = null;
        scalingRequest(viewL + ((double)xMin)/d.width * (viewR - viewL),
                       viewL + ((double)xMax)/d.width * (viewR - viewL),
                       viewT + ((double)yMin)/d.height * (viewB - viewT),
                       viewT + ((double)yMax)/d.height * (viewB - viewT));
      }
    }
  }
  
  protected void scalingRequest(double L, double R, double T, double B) {}
  
  protected void processMouseMotionEvent(MouseEvent e){
    super.processMouseMotionEvent(e);
    if (e.getID() == MouseEvent.MOUSE_DRAGGED && lastPressed != null) {
      if (!(e.isMetaDown() || e.isAltDown())) {
        Point p = e.getPoint();
        double deltaX = (p.x - lastPressed.x) * (viewR - viewL) / getSize().width;
        double deltaY = (p.y - lastPressed.y) * (viewB - viewT) / getSize().height;
        lastPressed = p;
        setView(viewL - deltaX, viewR - deltaX, viewT - deltaY, viewB - deltaY);
        adjustScrollBars();
        repaint();
      }
      else if (scalingEnabled && firstPressed != null) {
        Graphics g = getGraphics();
        drawSquare(g);
        lastPressed = e.getPoint();
        drawSquare(g);
      }
    }
  }
  
  private void drawSquare(Graphics g) {
    g.setColor(Color.black);
    g.setXORMode(Color.white);
    g.drawRect((int)Math.min(firstPressed.x, lastPressed.x),
                    (int)Math.min(firstPressed.y, lastPressed.y),
                    (int)Math.abs(lastPressed.x - firstPressed.x),
                    (int)Math.abs(lastPressed.y - firstPressed.y));
  }
  
}
