/**
    \file Chamfer2D_7x7.cpp
    Implementation of Chamfer2D_7x7 distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
/*
     no OUT OF BOUNDS checks!
*/
#include <iostream>
#include <stdlib.h>
#include "Chamfer2D_7x7.h"

using namespace std;
//----------------------------------------------------------------------
/*
Input : (C,I) - a 2D binary scene of domain size X by Y
Output: (C,d) - a 2D grey scene of domain size X and Y representing the
                distance scene
*/
void Chamfer2D_7x7::doTransform ( const unsigned char* const I,
    const int A, const int B, const int C, const int D, const int E,
    const bool halfIsZero )
{
    const int dx=A, dy=A, dxy=B;

    borderCheck(I);
    cleanUp();
    int x, y;
    //initialize d
    int* d = (int*)malloc( ySize*xSize*sizeof(int) );    assert(d!=NULL);
    for (y=0; y<ySize; y++) {
        for (x=0; x<xSize; x++) {
            d[sub(x,y)] = DistanceTransform::IntInfinity;
        }
    }
    //initialize immediate interior & exterior elements
    if (halfIsZero)    initImmediate(I, d, 0,    0);
    else               initImmediate(I, d, dx/2, dy/2);

    //perform the first pass
    for (y=3; y<ySize-3; y++) {
        for (x=3; x<xSize-3; x++) {
            const int center = sub(x,y);

            check( d, center, x-1, y-1, dxy );
            check( d, center, x,   y-1, dy  );
            check( d, center, x+1, y-1, dxy );
            check( d, center, x-1, y,   dx  );

            check( d, center, x-1, y-2, C );
            check( d, center, x+1, y-2, C );
            check( d, center, x-2, y-1, C );
            check( d, center, x+2, y-1, C );

            check( d, center, x-1, y-3, D );
            check( d, center, x+1, y-3, D );
            check( d, center, x-3, y-1, D );
            check( d, center, x+3, y-1, D );

            check( d, center, x-2, y-3, E );
            check( d, center, x+2, y-3, E );
            check( d, center, x-3, y-2, E );
            check( d, center, x+3, y-2, E );
        }
    }

    //perform the final pass
    for (y=ySize-4; y>=3; y--) {
        for (x=xSize-4; x>=3; x--) {
            const int center = sub(x,y);

            check( d, center, x+1, y,   dx  );
            check( d, center, x-1, y+1, dxy );
            check( d, center, x,   y+1, dy  );
            check( d, center, x+1, y+1, dxy );

            check( d, center, x-2, y+1, C );
            check( d, center, x+2, y+1, C );
            check( d, center, x-1, y+2, C );
            check( d, center, x+1, y+2, C );

            check( d, center, x-1, y+3, D );
            check( d, center, x+1, y+3, D );
            check( d, center, x-3, y+1, D );
            check( d, center, x+3, y+1, D );

            check( d, center, x-2, y+3, E );
            check( d, center, x+2, y+3, E );
            check( d, center, x-3, y+2, E );
            check( d, center, x+3, y+2, E );
        }
    }

    finish(I, d, dx, dy);
}
//----------------------------------------------------------------------
void Chamfer2D_7x7::borderCheck ( const unsigned char* const I ) {

    for (int x=0; x<xSize; x++) {
        assert( I[sub(x,0)]==0 );        //check the first row
        assert( I[sub(x,1)]==0 );        //check the second row
        assert( I[sub(x,2)]==0 );        //check the third row

        assert( I[sub(x,ySize-2)]==0 );  //check the next-to-next-to-last row
        assert( I[sub(x,ySize-2)]==0 );  //check the next-to-last row
        assert( I[sub(x,ySize-1)]==0 );  //check the last row
    }

    for (int y=0; y<ySize; y++) {
        assert( I[sub(0,y)]==0 );        //check the first col
        assert( I[sub(1,y)]==0 );        //check the second col
        assert( I[sub(2,y)]==0 );        //check the third col

        assert( I[sub(xSize-3,y)]==0 );  //check the next-to-next-to-last col
        assert( I[sub(xSize-2,y)]==0 );  //check the next-to-last col
        assert( I[sub(xSize-1,y)]==0 );  //check the last col
    }
}
//----------------------------------------------------------------------

