/**
    \file DijkstraVectors.cpp

    Implementation of DijkstraVectors distance transform class which, given
    and input binary image, calculates the corresponding distance transform
    using the DijkstraVectors method.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */

#include "DijkstraVectors.h"
#include <iostream>
//----------------------------------------------------------------------
void DijkstraVectors::doTransform ( const unsigned char* const I ) {
    //initialize
    borderCheck(I);
    cleanUp();

    //used as a 2d array.  stores the current minimum distance assignment.
    double* d = (double*)malloc( ySize*xSize*sizeof(double)   );
    assert( d!=NULL );
    for (int i=0; i<xSize*ySize; i++)
        d[i] = DistanceTransform::FloatInfinity;

    //used as a 2d array.  each one is a vector of SetElement's that
    //correspond to the border points and distance assignments (in no
    //particular order) for that particular location.
    this->p = new vector<SetElement*> [ySize*xSize];
    assert( this->p!=NULL );

    //s is a set of SetElements ordered (primarily) by distance values
    set< SetElement*, ltSetElement >  s;

    //initially put the immediate interior & exterior elements on s
    for (int y=1; y<ySize-1; y++) {
        for (int x=1; x<xSize-1; x++) {
            const int c=sub(x,y);
            if (I[sub(x-1,y)] != I[c] || I[sub(x+1,y)] != I[c] ||
                I[sub(x,y-1)] != I[c] || I[sub(x,y+1)] != I[c]) {
                d[c] = 0.0;
                SetElement* se = new SetElement(0.0, x, y);
                s.insert( se );
                SetElement* const pe = new SetElement(0.0, x, y);
                this->p[c].push_back( pe );
            }
        }
    }

    // cout << "here: " << this->p[sub(26,31)].size() << endl;

    set< SetElement*, ltSetElement >::iterator  it;
    for (it=s.begin(); it!=s.end(); it=s.begin()) {
        SetElement* v = *it;
        const int c = sub(v->x, v->y);

        //checking the 4-connected neighbors is sufficient
        // check( d, c, v->x,   v->y-1, s );
        // check( d, c, v->x,   v->y+1, s );
        // check( d, c, v->x-1, v->y,   s );
        // check( d, c, v->x+1, v->y,   s );
#if 1
        //8-connected neighbors
        check( d, c, v->x+1, v->y,   s );
        check( d, c, v->x-1, v->y,   s );
        check( d, c, v->x,   v->y+1, s );
        check( d, c, v->x,   v->y-1, s );

        check( d, c, v->x+1, v->y+1, s );
        check( d, c, v->x+1, v->y-1, s );
        check( d, c, v->x-1, v->y+1, s );
        check( d, c, v->x-1, v->y-1, s );
#endif
        s.erase(v);  //remove the first element
                     //i.e., the one w/ min d
    }

    // cout << "here2: " << this->p[sub(26,31)].size() << endl;
    finish(I, d);
}
//----------------------------------------------------------------------
void DijkstraVectors::check ( double* d, const int center, const int X,
    const int Y, set< SetElement*, ltSetElement > &s )
{
    //bounds check
    if (X<0 || X>=xSize || Y<0 || Y>=ySize)    return;

    for (int i=0; i<(int)this->p[center].size(); i++) {
        const double D = sqrt( (double)
            ((X - this->p[center][i]->x)*(X - this->p[center][i]->x) +
             (Y - this->p[center][i]->y)*(Y - this->p[center][i]->y)) );
        const int n=sub(X, Y);

        //first time encountering this neighbor?
        if (d[n] == DistanceTransform::FloatInfinity) {
            d[n] = D;
            //add to the ordered set
            SetElement* se = new SetElement(D, X, Y);
            s.insert(se);
            //add to parent vector
            assert(this->p[n].size() == 0);  //should be currently empty
            SetElement* pe = new SetElement( D, this->p[center][i]->x,
                                                this->p[center][i]->y );
            this->p[n].push_back(pe);
        } else if (D < d[n]) {
            //not the first encounter for this neighbor.
            //but it's better than what's already been assigned!

            d[n] = D;

            //add it (if it's not already there).
#if 0
            static int longest=0;
            if (this->p[n].size() > longest) {
                longest = this->p[n].size();
                cout << "longest: " << longest << endl;
            }
            for (int j=0; j<(int)this->p[n].size(); j++) {
                cout << "there" << endl;
                // assert( this->p[n][j]->x != X && this->p[n][j]->y != Y );
            }
#endif
            //add to the ordered set
            SetElement* se = new SetElement(D, X, Y);
            s.insert(se);
            //add to parent vector
            SetElement* pe = new SetElement( D, this->p[center][i]->x,
                                                this->p[center][i]->y );
            this->p[n].push_back(pe);
        }
    }  // end for
}
//----------------------------------------------------------------------

