/**
    \file DijkstraVectors.h
    Header file for DijkstraVectors distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
#ifndef DijkstraVectors_h
#define DijkstraVectors_h

#ifdef WIN32
#pragma warning(disable:4786)  //necessary because stl generates longer
                               // names than bill's compiler can handle!
#endif

#include "DistanceTransform.h"
#include <set>
#include <vector>

using namespace std;
//----------------------------------------------------------------------
/// DijkstraVectors distance transform class
class DijkstraVectors : public DistanceTransform {

public:
    /// construct a distance transform object for an image of size (xSize,ySize).
    /**
     * if 'unload' is true, transforms that use ratios of integers should convert
     * their integer results to the actual double values (of the ratios).
     */
    DijkstraVectors ( const int xSize, const int ySize,
                      const bool unload=true )
        : DistanceTransform(xSize, ySize, unload)
    {
    }

    void doTransform ( const unsigned char* const I );

    virtual inline bool getP ( const int x, const int y, int& px, int& py )
    const { return getP(x, y, px, py, 0); }

    inline bool getP ( const int x, const int y, int& px, int& py,
                       const int which ) const
    {
        if (this->p == NULL) {
            px = py = -1;
            return false;
        }

        const int i = sub(x,y);
        if (which < (int)p[i].size()) {
            px = p[i][which]->x;
            py = p[i][which]->y;
            // for (int j=0; j<(int)p[i].size(); j++) {
            //     cout << "P(" << x << "," << y << ")=(" << p[i][j].x << ","
            //          << p[i][j].y << ")" << endl;
            // }
            return true;
        }

        px = py = -1;
        return false;
    }

private:
    /** \brief Elements of a set class.
     *
     *  SetElement contains (x,y) location (of a point - not it's parent)
     *  and the current distance assignment, d.
     */
    class SetElement {
        public:
            double d;  ///< current distance assignment
            int    x;  ///< 2d location (x)
            int    y;  ///< 2d location (y)
        SetElement ( const double d, const int x, const int y ) {
            this->d = d;
            this->x = x;
            this->y = y;
        }
    };

    /// strict weak ordering on SetElement's.
    struct ltSetElement {
        /// implements the strict weak ordering on SetElement's.
        /// \returns true if the first precedes the second; false otherwise.
        bool operator() ( const SetElement* s1, const SetElement* s2 ) const {
            if (s1->d < s2->d)  return true;
            if (s1->d > s2->d)  return false;
            //both must have the same 'd' value
            if (s1->x < s2->x)  return true;
            if (s1->x > s2->x)  return false;
            //both must have the same 'd' and 'x' values
            if (s1->y < s2->y)  return true;
            if (s1->y > s2->y)  return false;
            return false;  //d, x, and y are equal
        }
    };

    /// stores vectors of parent locations (border points) & the corresponding
    /// distance assignments.  used as a 2d array.
    vector<SetElement*>* p;

    inline void check ( double* d, const int center, const int X, const int Y,
        set< SetElement*, ltSetElement > &s );

};

#endif
//----------------------------------------------------------------------

