/**
    \file EightSED.cpp
    Implementation of EightSED distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
/*
     no OUT OF BOUNDS checks!
*/
#include <iostream>
#include <stdlib.h>
#include "EightSED.h"

using namespace std;
//----------------------------------------------------------------------
void EightSED::init ( double *d ) {
    for (int y=0; y<ySize; y++) {
        for (int x=0; x<xSize; x++) {
            const int i=sub(x,y);
            d[i] = DistanceTransform::FloatInfinity;
            //pointer to immediate interior or exterior i.e.
            //p(x,y).x is the x coordinate of the closest boundary
            //point (and similarly for p(x,y).y)
            this->p[i].x = this->p[i].y = -1;
        }
    }

}
//----------------------------------------------------------------------
void EightSED::initImmediate ( const unsigned char* const I, double *d ) {

    //initialize immediate interior & exterior elements
    for (int y=1; y<ySize-1; y++) {
        for (int x=1; x<xSize-1; x++) {
            if (I[sub(x-1,y)] != I[sub(x,y)] || I[sub(x+1,y)] != I[sub(x,y)]) {
                const int i=sub(x,y);
                d[i] = 0.0;
                this->p[i].x = x;
                this->p[i].y = y;
            }
            if (I[sub(x,y-1)] != I[sub(x,y)] || I[sub(x,y+1)] != I[sub(x,y)]) {
                const int i=sub(x,y);
                d[i] = 0.0;
                this->p[i].x = x;
                this->p[i].y = y;
            }
        }
    }

}
//----------------------------------------------------------------------
/*
Input : (C,I) - a 2D binary scene of domain size X by Y
Output: (C,d) - a 2D grey scene of domain size X and Y representing the
                distance scene
*/
void EightSED::doTransform ( const unsigned char* const I ) {
    const double  dx  = 1.0;
    const double  dy  = 1.0;
    const double  dxy = sqrt(dx*dx + dy*dy);  //diagonal distance

    borderCheck(I);
    cleanUp();
    int  x, y;

    //initialize d
    double*  d = (double*)malloc( ySize*xSize*sizeof(double)   );
    assert( d!=NULL );
    this->p = (P*)malloc( ySize*xSize*sizeof(P) );
    assert( this->p!=NULL );

    init(d);
    initImmediate(I, d);

    #define _check(X,Y,D) {                               \
        const int  i1=sub((X),(Y));                       \
        const int  i2=sub(x,y);                           \
        if (d[i1]+(D) < d[i2]) {                          \
            this->p[i2] = this->p[i1];                    \
            const double  deltaX = (p[i1].x - x)*dx;      \
            const double  deltaY = (p[i1].y - y)*dy;      \
            d[i2] = sqrt(deltaX*deltaX + deltaY*deltaY);  \
        }                                                 \
    }

    //perform the first pass ("first picture scan")
    for (y=1; y<=ySize-1; y++) {
        for (x=0; x<=xSize-1; x++) {
            if (x>0) {  //** not checked in original but needed
                _check( x-1, y-1, dxy );
            }
            _check( x,   y-1, dy  );
            //_check( x-1, y,   dx  );
            if (x<xSize-1) {  //** not checked in original but needed
                _check( x+1, y-1, dxy );
            }
        }
        for (x=1; x<=xSize-1; x++) {
            _check( x-1, y,   dx  );
        }
        for (x=xSize-2; x>=0; x--) {
            _check( x+1, y,   dx  );
        }
    }

    //perform the final pass ("second picture scan")
    for (y=ySize-2; y>=0; y--) {
        for (x=0; x<=xSize-1; x++) {
            if (x>0) {  //** not checked in original but needed
                _check( x-1, y+1, dxy );
            }
            _check( x,   y+1, dy  );
            //_check( x+1, y,   dx  );
            if (x<xSize-1) {  //** not checked in original but needed
                _check( x+1, y+1, dxy );
            }
        }
        for (x=1; x<=xSize-1; x++) {
            _check( x-1, y, dx );
        }
        for (x=xSize-2; x>=0; x--) {
            _check( x+1, y, dx );
        }
    }

    #undef _check

    finish(I, d);
}
//----------------------------------------------------------------------
