/**
    \file Euclidean2D.cpp
    Implementation of Euclidean2D distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
/*
     no OUT OF BOUNDS checks!
*/
#include <iostream>
#include <stdlib.h>
#include "Euclidean2D.h"

using namespace std;
//----------------------------------------------------------------------
/*
Input : (C,I) - a 2D binary scene of domain size X by Y
Output: (C,d) - a 2D grey scene of domain size X and Y representing the
                distance scene
*/
void Euclidean2D::doTransform ( const unsigned char* const I,
                                const double dx, const double dy,
                                const bool halfIsZero )
{
    borderCheck(I);
    cleanUp();
    //initialize d
    double* d = (double*)malloc( ySize*xSize*sizeof(double) );
    assert(d!=NULL);
    int x, y;
    for (y=0; y<ySize; y++) {
        for (x=0; x<xSize; x++) {
            d[sub(x,y)] = DistanceTransform::FloatInfinity;
        }
    }
    //initialize immediate interior & exterior elements
    if (halfIsZero)    initImmediate(I, d, 0.0,    0.0);
    else               initImmediate(I, d, dx/2.0, dy/2.0);
    
    const double dxy = sqrt(dx*dx +  dy*dy);  //diagonal distance

    //perform the first pass
    for (y=1; y<ySize-1; y++) {
        for (x=1; x<xSize-1; x++) {
            const int center=sub(x,y);
            check( d, center, x-1, y-1, dxy );
            check( d, center, x,   y-1, dy  );
            check( d, center, x+1, y-1, dxy );
            check( d, center, x-1, y,   dx  );
        }
    }
    
    //perform the final pass
    for (y=ySize-2; y>=1; y--) {
        for (x=xSize-2; x>=1; x--) {
            const int center=sub(x,y);
            check( d, center, x+1, y,   dx  );
            check( d, center, x-1, y+1, dxy );
            check( d, center, x,   y+1, dy  );
            check( d, center, x+1, y+1, dxy );
        }
    }

    finish(I, d);
}
//----------------------------------------------------------------------

