/**
    \file ModifiedDijkstra.h
    Header file for ModifiedDijkstra distance transform class which, given
    and input binary image, calculates the corresponding distance transform.

    \author George J. Grevera, Ph.D., ggrevera@sju.edu

    Copyright (C) 2002, George J. Grevera

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307
    USA or from http://www.gnu.org/licenses/gpl.txt.

    This General Public License does not permit incorporating this
    code into proprietary programs.  (So a hypothetical company such
    as GH (Generally Hectic) should NOT incorporate this code into
    their proprietary programs.)
 */
#ifndef ModifiedDijkstra_h
#define ModifiedDijkstra_h

#include "DistanceTransform.h"
#ifdef WIN32
#pragma warning(disable:4786)  //necessary because stl generates longer
                               // names than bill's compiler can handle!
#endif
#include <set>

using namespace std;
//----------------------------------------------------------------------
/// ModifiedDijkstra class
class ModifiedDijkstra : public DistanceTransform {

public:
    ModifiedDijkstra ( const int xSize, const int ySize,
                       const bool unload=true )
        : DistanceTransform(xSize, ySize, unload)
    {
    }

    /**
        This method performs the distance transform.  It is the same as
        calling modifiedDijkstra(I) directly.
     */
    void doTransform ( const unsigned char* const I ) {
        modifiedDijkstra(I);
    }

    /**
        This method performs the distance transform using an 
        implementation/modification of Dijkstra's algorithm
        using the elements of the boundary as multiple starting
        points.  The distance is propagated as the summation
        of edges in the graph that is 8-connected.
     */
    void modifiedDijkstra ( const unsigned char* const I );
    /**
        This method performs the distance transform using an
        modification of Dijkstra's algorithm.  The distance
        is propagated as the actual Euclidean distance (i.e., "as
        the crow flies") propagated among 8-connected neighbors.
     */
    void modifiedDijkstra_DeadReckoning ( const unsigned char* const I );
    /**
        This method performs the distance transform using an
        modification of Dijkstra's algorithm.  The distance
        is propagated as the actual Euclidean distance (i.e., "as
        the crow flies") propagated among 4-connected neighbors.
     */
    void modifiedDijkstra_ModifiedDeadReckoning ( const unsigned char* const I );

    /// return the parent (border point/nearest element of the immediate
    /// interior or exterior)
    virtual inline bool getP ( const int x, const int y, int& px, int& py ) const {
        if (p==NULL)    return false;
        const int s = sub(x,y);
        px = p[s].x;
        py = p[s].y;
        return true;
    }

protected:
    /// stores the parent location (border point)
    P *p;

private:
    /** \brief Elements of a set class.
     *
     *  SetElement contains (x,y) location (of a point - not it's parent)
     *  and the current distance assignment, d.
     */
    typedef struct {
        double d;  ///< current distance assignment
        int    x;  ///< 2d location (x)
        int    y;  ///< 2d location (y)
    } SetElement;

    /// strict weak ordering on SetElement's.
    struct ltSetElement {
        /// implements the strict weak ordering on SetElement's.
        /// \returns true if the first precedes the second; false otherwise.
        bool operator() ( const SetElement* s1, const SetElement* s2 ) const {
            if (s1->d < s2->d)  return true;
            if (s1->d > s2->d)  return false;
            //both must have the same 'd' value
            if (s1->x < s2->x)  return true;
            if (s1->x > s2->x)  return false;
            //both must have the same 'd' and 'x' values
            if (s1->y < s2->y)  return true;
            if (s1->y > s2->y)  return false;
            return false;  //d, x, and y are equal
        }
    };

    /**
     * @param I is the input 2d binary image.
     * @param d is a reference to a pointer for which memory will be allocated
     *          and returned.
     * @param se is a reference to what will be used as a 2D array of all
     *          possible SetElement's - one for each (x,y).
     * @param s is a set which initially consists of elements of the II and IE.
     */
    void init ( const unsigned char* const I, double* &d, SetElement* &se,
        set< const SetElement *, ltSetElement > &s );

    inline void check ( double* d, const int center, const int X,
        const int Y, SetElement* se, set< const SetElement*, ltSetElement > &s );
};

#endif
//----------------------------------------------------------------------

