#include <sstream>
#include <vector>
#include "shader.h"
#include "geoshader.h"
extern "C" {
#include "miaux.h"
}

typedef struct {
    miInteger index;
    miTag type;
    miTag filename;
} indexed_framebuffer;

struct make_indexed_framebuffers {
    miTag option_block_name;
    int i_fb;
    int n_fb;
    indexed_framebuffer fb[1];
};

extern "C" DLLEXPORT
int make_indexed_framebuffers_version(void) { return 1; }

// Struct to store evaluated parameters:
typedef struct {
    int index;
    char* type;
    char* filename;
    char* ext;
    char* name;
} fb_params;

std::vector<fb_params> get_framebuffer_params(
    miState *state, struct make_indexed_framebuffers *params)
{
    int n_fb = *mi_eval_integer(&params->n_fb);
    int i_fb = *mi_eval_integer(&params->i_fb);    

    mi_info("Making %i user framebuffers:", n_fb);

    std::vector<fb_params> buffers(n_fb);
    for (int i = 0; i < n_fb; ++i) {
	indexed_framebuffer fb = params->fb[i + i_fb];
	buffers[i].index = *mi_eval_integer(&fb.index);
	buffers[i].type = miaux_tag_to_string(*mi_eval_tag(&fb.type), NULL);
	buffers[i].filename = miaux_tag_to_string(*mi_eval_tag(&fb.filename), NULL);
	buffers[i].ext = strrchr(buffers[i].filename, '.') + 1; // String after "."
	std::ostringstream fb_name;
	fb_name << "fb" << buffers[i].index;
	buffers[i].name = mi_mem_strdup(fb_name.str().c_str());
	mi_progress("   Buffer %d: filename '%s', type '%s', ext '%s', name '%s'",
		    buffers[i].index, buffers[i].filename, buffers[i].type,
		    buffers[i].ext, buffers[i].name);
    }
    return buffers;
}

extern "C" DLLEXPORT
miBoolean make_indexed_framebuffers (
    miTag *result, miState *state, struct make_indexed_framebuffers *params)
{
    std::vector<fb_params> fb = get_framebuffer_params(state, params);

    miOptions* options = const_cast<miOptions*>(state->options);
    char* option_block_name = 
	miaux_tag_to_string(*mi_eval_tag(&params->option_block_name), "opt");

    mi_api_options_begin(mi_mem_strdup(option_block_name));
    for (unsigned int i = 0; i < fb.size(); ++i)
	mi_api_framebuffer(options, fb[i].index, mi_mem_strdup(fb[i].type));
    mi_api_options_end();

    for (unsigned int i = 0; i < fb.size(); ++i) {
	if (fb[i].filename) {  // Allow framebuffers without an associated file.
	    mi_api_framebuffer_add(
		state->camera->buffertag, mi_mem_strdup(fb[i].name), 
		mi_mem_strdup(fb[i].ext), 0, mi_mem_strdup(fb[i].filename));
	}
    }
    return miTRUE;
}

