/*
   Arithmetic operator shaders for use in mental ray Phenomena
   See http://www.writingmentalrayshaders.com/arithmetic_operators.html
*/

#include "math.h"
#include "shader.h"


static miScalar min(miScalar A, miScalar B)
{
    return A < B ? A : B;
}

static miScalar max(miScalar A, miScalar B)
{
    return A > B ? A : B;
}

static miScalar mix(miScalar A, miScalar B, miScalar F)
{
    return A * F + B * (1.0 - F);
}


/* op_add_cc - Add a color to a color to define a color */

struct op_add_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_add_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_cc(miColor *result, miState *state, struct op_add_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = A->r + B->r;
    result->g = A->g + B->g;
    result->b = A->b + B->b;
    result->a = A->a + B->a;
    return miTRUE;
}


/* op_add_cv - Add a color to a vector to define a color */

struct op_add_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_add_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_cv(miColor *result, miState *state, struct op_add_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = A->r + B->x;
    result->g = A->g + B->y;
    result->b = A->b + B->z;
    return miTRUE;
}


/* op_add_cs - Add a color to a scalar to define a color */

struct op_add_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_add_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_cs(miColor *result, miState *state, struct op_add_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = A->r + B;
    result->g = A->g + B;
    result->b = A->b + B;
    result->a = A->a + B;
    return miTRUE;
}


/* op_add_vc - Add a vector to a color to define a vector */

struct op_add_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_add_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_vc(miVector *result, miState *state, struct op_add_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = A->x + B->r;
    result->y = A->y + B->g;
    result->z = A->z + B->b;
    return miTRUE;
}


/* op_add_vv - Add a vector to a vector to define a vector */

struct op_add_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_add_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_vv(miVector *result, miState *state, struct op_add_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = A->x + B->x;
    result->y = A->y + B->y;
    result->z = A->z + B->z;
    return miTRUE;
}


/* op_add_vs - Add a vector to a scalar to define a vector */

struct op_add_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_add_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_vs(miVector *result, miState *state, struct op_add_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = A->x + B;
    result->y = A->y + B;
    result->z = A->z + B;
    return miTRUE;
}


/* op_add_ss - Add a scalar to a scalar to define a scalar */

struct op_add_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_add_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_add_ss(miScalar *result, miState *state, struct op_add_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = A + B;
    return miTRUE;
}


/* op_sub_cc - Subtract a color from a color to define a color */

struct op_sub_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_sub_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_cc(miColor *result, miState *state, struct op_sub_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = A->r - B->r;
    result->g = A->g - B->g;
    result->b = A->b - B->b;
    result->a = A->a - B->a;
    return miTRUE;
}


/* op_sub_cv - Subtract a color from a vector to define a color */

struct op_sub_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_sub_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_cv(miColor *result, miState *state, struct op_sub_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = A->r - B->x;
    result->g = A->g - B->y;
    result->b = A->b - B->z;
    return miTRUE;
}


/* op_sub_cs - Subtract a color from a scalar to define a color */

struct op_sub_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_sub_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_cs(miColor *result, miState *state, struct op_sub_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = A->r - B;
    result->g = A->g - B;
    result->b = A->b - B;
    result->a = A->a - B;
    return miTRUE;
}


/* op_sub_vc - Subtract a vector from a color to define a vector */

struct op_sub_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_sub_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_vc(miVector *result, miState *state, struct op_sub_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = A->x - B->r;
    result->y = A->y - B->g;
    result->z = A->z - B->b;
    return miTRUE;
}


/* op_sub_vv - Subtract a vector from a vector to define a vector */

struct op_sub_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_sub_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_vv(miVector *result, miState *state, struct op_sub_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = A->x - B->x;
    result->y = A->y - B->y;
    result->z = A->z - B->z;
    return miTRUE;
}


/* op_sub_vs - Subtract a vector from a scalar to define a vector */

struct op_sub_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_sub_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_vs(miVector *result, miState *state, struct op_sub_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = A->x - B;
    result->y = A->y - B;
    result->z = A->z - B;
    return miTRUE;
}


/* op_sub_ss - Subtract a scalar from a scalar to define a scalar */

struct op_sub_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_sub_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_sub_ss(miScalar *result, miState *state, struct op_sub_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = A - B;
    return miTRUE;
}


/* op_mul_cc - Multiply a color by a color to define a color */

struct op_mul_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_mul_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_cc(miColor *result, miState *state, struct op_mul_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = A->r * B->r;
    result->g = A->g * B->g;
    result->b = A->b * B->b;
    result->a = A->a * B->a;
    return miTRUE;
}


/* op_mul_cv - Multiply a color by a vector to define a color */

struct op_mul_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_mul_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_cv(miColor *result, miState *state, struct op_mul_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = A->r * B->x;
    result->g = A->g * B->y;
    result->b = A->b * B->z;
    return miTRUE;
}


/* op_mul_cs - Multiply a color by a scalar to define a color */

struct op_mul_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_mul_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_cs(miColor *result, miState *state, struct op_mul_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = A->r * B;
    result->g = A->g * B;
    result->b = A->b * B;
    result->a = A->a * B;
    return miTRUE;
}


/* op_mul_vc - Multiply a vector by a color to define a vector */

struct op_mul_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_mul_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_vc(miVector *result, miState *state, struct op_mul_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = A->x * B->r;
    result->y = A->y * B->g;
    result->z = A->z * B->b;
    return miTRUE;
}


/* op_mul_vv - Multiply a vector by a vector to define a vector */

struct op_mul_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_mul_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_vv(miVector *result, miState *state, struct op_mul_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = A->x * B->x;
    result->y = A->y * B->y;
    result->z = A->z * B->z;
    return miTRUE;
}


/* op_mul_vs - Multiply a vector by a scalar to define a vector */

struct op_mul_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_mul_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_vs(miVector *result, miState *state, struct op_mul_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = A->x * B;
    result->y = A->y * B;
    result->z = A->z * B;
    return miTRUE;
}


/* op_mul_ss - Multiply a scalar by a scalar to define a scalar */

struct op_mul_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_mul_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_mul_ss(miScalar *result, miState *state, struct op_mul_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = A * B;
    return miTRUE;
}


/* op_div_cc - Divide a color by a color to define a color */

struct op_div_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_div_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_cc(miColor *result, miState *state, struct op_div_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = A->r / B->r;
    result->g = A->g / B->g;
    result->b = A->b / B->b;
    result->a = A->a / B->a;
    return miTRUE;
}


/* op_div_cv - Divide a color by a vector to define a color */

struct op_div_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_div_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_cv(miColor *result, miState *state, struct op_div_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = A->r / B->x;
    result->g = A->g / B->y;
    result->b = A->b / B->z;
    return miTRUE;
}


/* op_div_cs - Divide a color by a scalar to define a color */

struct op_div_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_div_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_cs(miColor *result, miState *state, struct op_div_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = A->r / B;
    result->g = A->g / B;
    result->b = A->b / B;
    result->a = A->a / B;
    return miTRUE;
}


/* op_div_vc - Divide a vector by a color to define a vector */

struct op_div_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_div_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_vc(miVector *result, miState *state, struct op_div_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = A->x / B->r;
    result->y = A->y / B->g;
    result->z = A->z / B->b;
    return miTRUE;
}


/* op_div_vv - Divide a vector by a vector to define a vector */

struct op_div_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_div_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_vv(miVector *result, miState *state, struct op_div_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = A->x / B->x;
    result->y = A->y / B->y;
    result->z = A->z / B->z;
    return miTRUE;
}


/* op_div_vs - Divide a vector by a scalar to define a vector */

struct op_div_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_div_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_vs(miVector *result, miState *state, struct op_div_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = A->x / B;
    result->y = A->y / B;
    result->z = A->z / B;
    return miTRUE;
}


/* op_div_ss - Divide a scalar by a scalar to define a scalar */

struct op_div_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_div_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_div_ss(miScalar *result, miState *state, struct op_div_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = A / B;
    return miTRUE;
}


/* op_min_cc - Find the minimum of a color and a color to define a color */

struct op_min_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_min_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_cc(miColor *result, miState *state, struct op_min_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = min(A->r, B->r);
    result->g = min(A->g, B->g);
    result->b = min(A->b, B->b);
    result->a = min(A->a, B->a);
    return miTRUE;
}


/* op_min_cv - Find the minimum of a color and a vector to define a color */

struct op_min_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_min_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_cv(miColor *result, miState *state, struct op_min_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = min(A->r, B->x);
    result->g = min(A->g, B->y);
    result->b = min(A->b, B->z);
    return miTRUE;
}


/* op_min_cs - Find the minimum of a color and a scalar to define a color */

struct op_min_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_min_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_cs(miColor *result, miState *state, struct op_min_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = min(A->r, B);
    result->g = min(A->g, B);
    result->b = min(A->b, B);
    result->a = min(A->a, B);
    return miTRUE;
}


/* op_min_vc - Find the minimum of a vector and a color to define a vector */

struct op_min_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_min_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_vc(miVector *result, miState *state, struct op_min_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = min(A->x, B->r);
    result->y = min(A->y, B->g);
    result->z = min(A->z, B->b);
    return miTRUE;
}


/* op_min_vv - Find the minimum of a vector and a vector to define a vector */

struct op_min_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_min_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_vv(miVector *result, miState *state, struct op_min_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = min(A->x, B->x);
    result->y = min(A->y, B->y);
    result->z = min(A->z, B->z);
    return miTRUE;
}


/* op_min_vs - Find the minimum of a vector and a scalar to define a vector */

struct op_min_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_min_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_vs(miVector *result, miState *state, struct op_min_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = min(A->x, B);
    result->y = min(A->y, B);
    result->z = min(A->z, B);
    return miTRUE;
}


/* op_min_ss - Find the minimum of a scalar and a scalar to define a scalar */

struct op_min_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_min_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_min_ss(miScalar *result, miState *state, struct op_min_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = min(A, B);
    return miTRUE;
}


/* op_max_cc - Find the maximum of a color and a color to define a color */

struct op_max_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_max_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_cc(miColor *result, miState *state, struct op_max_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = max(A->r, B->r);
    result->g = max(A->g, B->g);
    result->b = max(A->b, B->b);
    result->a = max(A->a, B->a);
    return miTRUE;
}


/* op_max_cv - Find the maximum of a color and a vector to define a color */

struct op_max_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_max_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_cv(miColor *result, miState *state, struct op_max_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = max(A->r, B->x);
    result->g = max(A->g, B->y);
    result->b = max(A->b, B->z);
    return miTRUE;
}


/* op_max_cs - Find the maximum of a color and a scalar to define a color */

struct op_max_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_max_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_cs(miColor *result, miState *state, struct op_max_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = max(A->r, B);
    result->g = max(A->g, B);
    result->b = max(A->b, B);
    result->a = max(A->a, B);
    return miTRUE;
}


/* op_max_vc - Find the maximum of a vector and a color to define a vector */

struct op_max_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_max_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_vc(miVector *result, miState *state, struct op_max_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = max(A->x, B->r);
    result->y = max(A->y, B->g);
    result->z = max(A->z, B->b);
    return miTRUE;
}


/* op_max_vv - Find the maximum of a vector and a vector to define a vector */

struct op_max_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_max_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_vv(miVector *result, miState *state, struct op_max_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = max(A->x, B->x);
    result->y = max(A->y, B->y);
    result->z = max(A->z, B->z);
    return miTRUE;
}


/* op_max_vs - Find the maximum of a vector and a scalar to define a vector */

struct op_max_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_max_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_vs(miVector *result, miState *state, struct op_max_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = max(A->x, B);
    result->y = max(A->y, B);
    result->z = max(A->z, B);
    return miTRUE;
}


/* op_max_ss - Find the maximum of a scalar and a scalar to define a scalar */

struct op_max_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_max_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_max_ss(miScalar *result, miState *state, struct op_max_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = max(A, B);
    return miTRUE;
}


/* op_pow_cc - Raise a color to the power of a color to define a color */

struct op_pow_cc {
    miColor A;
    miColor B;
};

DLLEXPORT
int op_pow_cc_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_cc(miColor *result, miState *state, struct op_pow_cc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->r = pow(A->r, B->r);
    result->g = pow(A->g, B->g);
    result->b = pow(A->b, B->b);
    result->a = pow(A->a, B->a);
    return miTRUE;
}


/* op_pow_cv - Raise a color to the power of a vector to define a color */

struct op_pow_cv {
    miColor A;
    miVector B;
};

DLLEXPORT
int op_pow_cv_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_cv(miColor *result, miState *state, struct op_pow_cv *params)
{
    miColor *A = mi_eval_color(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->r = pow(A->r, B->x);
    result->g = pow(A->g, B->y);
    result->b = pow(A->b, B->z);
    return miTRUE;
}


/* op_pow_cs - Raise a color to the power of a scalar to define a color */

struct op_pow_cs {
    miColor A;
    miScalar B;
};

DLLEXPORT
int op_pow_cs_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_cs(miColor *result, miState *state, struct op_pow_cs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->r = pow(A->r, B);
    result->g = pow(A->g, B);
    result->b = pow(A->b, B);
    result->a = pow(A->a, B);
    return miTRUE;
}


/* op_pow_vc - Raise a vector to the power of a color to define a vector */

struct op_pow_vc {
    miVector A;
    miColor B;
};

DLLEXPORT
int op_pow_vc_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_vc(miVector *result, miState *state, struct op_pow_vc *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miColor *B = mi_eval_color(&params->B);
    result->x = pow(A->x, B->r);
    result->y = pow(A->y, B->g);
    result->z = pow(A->z, B->b);
    return miTRUE;
}


/* op_pow_vv - Raise a vector to the power of a vector to define a vector */

struct op_pow_vv {
    miVector A;
    miVector B;
};

DLLEXPORT
int op_pow_vv_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_vv(miVector *result, miState *state, struct op_pow_vv *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miVector *B = mi_eval_vector(&params->B);
    result->x = pow(A->x, B->x);
    result->y = pow(A->y, B->y);
    result->z = pow(A->z, B->z);
    return miTRUE;
}


/* op_pow_vs - Raise a vector to the power of a scalar to define a vector */

struct op_pow_vs {
    miVector A;
    miScalar B;
};

DLLEXPORT
int op_pow_vs_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_vs(miVector *result, miState *state, struct op_pow_vs *params)
{
    miVector *A = mi_eval_vector(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    result->x = pow(A->x, B);
    result->y = pow(A->y, B);
    result->z = pow(A->z, B);
    return miTRUE;
}


/* op_pow_ss - Raise a scalar to the power of a scalar to define a scalar */

struct op_pow_ss {
    miScalar A;
    miScalar B;
};

DLLEXPORT
int op_pow_ss_version(void) { return 1; }

DLLEXPORT
miBoolean op_pow_ss(miScalar *result, miState *state, struct op_pow_ss *params)
{
    miScalar A = *mi_eval_scalar(&params->A);
    miScalar B = *mi_eval_scalar(&params->B);
    *result = pow(A, B);
    return miTRUE;
}


/* op_mix_ccs - Mix a color with a color using a scalar to define a color */

struct op_mix_ccs {
    miColor A;
    miColor B;
    miScalar F;
};

DLLEXPORT
int op_mix_ccs_version(void) { return 1; }

DLLEXPORT
miBoolean op_mix_ccs(miColor *result, miState *state, struct op_mix_ccs *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    miScalar F = *mi_eval_scalar(&params->F);
    result->r = mix(A->r, B->r, F);
    result->g = mix(A->g, B->g, F);
    result->b = mix(A->b, B->b, F);
    result->a = mix(A->a, B->a, F);
    return miTRUE;
}


/* op_mix_ccc - Mix a color with a color using a color to define a color */

struct op_mix_ccc {
    miColor A;
    miColor B;
    miColor F;
};

DLLEXPORT
int op_mix_ccc_version(void) { return 1; }

DLLEXPORT
miBoolean op_mix_ccc(miColor *result, miState *state, struct op_mix_ccc *params)
{
    miColor *A = mi_eval_color(&params->A);
    miColor *B = mi_eval_color(&params->B);
    miColor *F = mi_eval_color(&params->F);
    result->r = mix(A->r, B->r, F->r);
    result->g = mix(A->g, B->g, F->g);
    result->b = mix(A->b, B->b, F->b);
    result->a = mix(A->a, B->a, F->a);
    return miTRUE;
}


