c     Program to compare the predictions of clear-sky radiation models
c
c     Calculates clear-sky direct, diffuse and global irradiances from 
c   measured meteorological/atmospheric data with different models.
c
c     Written by 
c   C.A. Gueymard, Solar Consulting Services, Sept 2002. 
c   
c   Revised and augmented, Nov. 2007.
c
c+++++++++++++++++++++++++++++ USER'S MANUAL ++++++++++++++++++++++++++
c
c   Static Inputs (one set for each run):
c   ==============
c
c   Elev: Elevation (km)

c   Latit: Latitude (+ North; - South)

c   Longit: Longitude (+ East; - West)

c   shift: time lag (minutes, positive or 0) between mid-measurement 
c      period and time stamp

c   Zmax: maximum zenith angle allowed for tests (0 to 90 deg.)

c   Zone: Time zone in hours from Greenwich, + for E, - for W of Greenwich
c
c
c   Dynamic (sequential) Inputs:
c   ============================
c
c************
c*   NOTE: Mandatory inputs are marked with ***; Defaultable inputs are unmarked
c************

c    alfa1, alfa2: Angstrom's wavelength exponents for wavelengths <700nm
c       and >700nm, resp.  Use <0 value if unknown. Default: alpha.
c
c*** alpha: Angstrom's wavelength exponents for all shortwave wavelengths

c*** beta: Angstrom's turbidity coefficient for all shortwave wavelengths

c    beta2: Angstrom's turbidity coefficient for wavelengths >700nm.
c        Use <0 value if unknown. Default: beta.

c*** Day: day of month (1-31)

c*** hr: decimal hour of day in _Local Standard Time_ (0.0000 to 23.9999)

c*** Month: month of year (1-12)

c    p: station pressure (mb or hPa).  Use <0 value if unknown. Default: 1013.25 mb.

c    piar1, piar2: aerosol single-scattering albedo for wavelengths <700nm
c       and >700nm, resp. Use <0 value if unknown. Defaults: .92 and .84, resp.

c*** rhog: ground albedo (0 to 1)

c    T: air temperature (deg. C).  Use <0 value if unknown. Default: 10 C

c    un: total NO2 vertical pathlength for the vetical column (atm-cm).
c        Use <0 value if unknown. Default: .0002 atm-cm.

c*** uo: total ozone vertical pathlength (atm-cm)

c*** w: precipitable water (cm)

c    Year (4 digits). Default: 2007 if value <0.
c
c
c   Outputs:
c   ========
c
c  DNI(i): Direct normal irradiance predicted by Model i (W/m2)
c  Dif(i): Diffuse horizontal irradiance predicted by Model i (W/m2)
c  Glo(i): Global horizontal irradiance predicted by Model i (W/m2)
c
c
c   Key to Models (in alphabetical order):
c   =============
c
c   1   ASHRAE 
c   2   Bashahu & Laplaze
c   3   BCLSM
c   4   Biga & Rosa
c   5   Bird
c   6   CEM
c   7   Chandra
c   8   CLS
c   9   CPCR2
c   10  Dogniaux
c   11  DPP
c   12  ESRA2
c   13  ESRA3
c   14  ESRA4
c   15  HLJ
c   16  Ideriah
c   17  Iqbal A
c   18  Iqbal B
c   19  Iqbal C
c   20  Josefsson
c   21  KASM
c   22  Kasten
c   23  Maxwell (METSTAT)
c   24  MAC
c   25  MRM4
c   26  MRM5
c   27  Nijegorodov et al.
c   28  Paltridge & Platt
c   29  PR
c   30  PSIM
c   31  REST2
c   32  Rodgers et al.
c   33  RSC
c   34  Santamouris
c   35  Schulze
c   36  Sharma & Pal
c   37  Wesely & Lipschutz
c   38  Yang et al.
c
c
c     The inputs required for each model are listed in he header of their
c     respective subroutines. Each subroutine has the same output:
c
c     Ebn: direct normal irradiance (W/m2)
c     Ed: diffuse horiontal irradiance (W/m2)
c     E: global horizontal irradiance (W/m2)
c
c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
c
      Double Precision Longit,Latit,Radius,decli
      Real DNI(38),Dif(38),Glo(38)
      Integer Year,Month,Day,yeard

      Character*96 Comment1, Comment2
      Character*16 Text1
      Character*50 Source1,Source2,Source3,Source4
            
      open (unit=12,file='Models_perf_exp.dat.txt',status='old')
      open (unit=15,file='Models_perf_exp.out.txt',status='new')
c
      Esc=1366.1
      En0=Esc
      p0=1013.25
      Td=10.
      und=.0002
      yeard=2007
      pi=3.1415927
      rhog1=.15
      wvl=.745
      Piar1d=.92
      Piar2d=.84
c
      Text1='****************'
c
      Read(12,199)Comment1
 199  Format(A96)
      Read(12,*)Latit,Longit,Elev,zone,shift,Zmax
      Read(12,*)Comment2
      
      N=0
      Write(15,109)Comment1
 109  FORMAT('Performance comparison of clear-sky radiation models,',
     1 ' C. Gueymard, April 2007',/,a96,/,'    #   Z     m     ',
     1 '  w   alpha   beta   tau    TL      DNI_1  DNI_2  DNI_3  DNI_4',
     3 '  DNI_5  DNI_6  DNI_7  DNI_8  DNI_9 DNI_10 DNI_11 DNI_12 DNI_',
     4 '13 DNI_14 DNI_15 DNI_16 DNI_17 DNI_18 DNI_19 DNI_20 DNI_21 ',
     5 'DNI_22 DNI_23 DNI_24 DNI_25 DNI_26 DNI_27 DNI_28 DNI_29 DNI_30',
     6 ' DNI_31 DNI_32 DNI_33 DNI_34 DNI_35 DNI_36 DNI_37 DNI_38',
     7 '    Dif_1  Dif_2  Dif_3  Dif_4  ',
     3 'Dif_5  Dif_6  Dif_7  Dif_8  Dif_9 Dif_10 Dif_11 Dif_12 Dif_13',
     4 ' Dif_14 Dif_15 Dif_16 Dif_17 Dif_18 Dif_19 Dif_20 Dif_21 ',
     5 'Dif_22 Dif_23 Dif_24 Dif_25 Dif_26 Dif_27 Dif_28 Dif_29 Dif_30',
     6 ' Dif_31 Dif_32 Dif_33 Dif_34 Dif_35 Dif_36 Dif_37 Dif_38',
     7 '   Glo_1  Glo_2  Glo_3  Glo_4  ',
     3 'Glo_5  Glo_6  Glo_7  Glo_8  Glo_9 Glo_10 Glo_11 Glo_12 Glo_13',
     4 ' Glo_14 Glo_15 Glo_16 Glo_17 Glo_18 Glo_19 Glo_20 Glo_21 ',
     5 'Glo_22 Glo_23 Glo_24 Glo_25 Glo_26 Glo_27 Glo_28 Glo_29 Glo_30',
     6 ' Glo_31 Glo_32 Glo_33 Glo_34 Glo_35 Glo_36 Glo_37 Glo_38')
     
 1    continue
      N=N+1
      Nmax=N
 2    continue
      Read(12,*,end=999)year,month,day,hr,p,w,uo,un,beta2,
     1 alfa1,alfa2,piar1,piar2,rhog,alpha,beta,T
c 
c   Check data for consistency and missing data
c
      if(alpha.lt.0.0.or.beta.lt.0.)goto 2
      if(uo.le.0.0.or.w.lt.0.0.or.rhog.lt.0.)goto 2
      if(piar1.le.0.)piar1=piar1d
      if(piar2.le.0.)piar2=piar2d
      if(p.lt.600.)p=p0*(1.-.1*Elev)
      if(T.lt.-50.)T=Td
      if(alfa1.lt.0.)alfa1=alpha
      if(alfa2.lt.0.)alfa2=alpha
      if(beta2.lt.0.)beta2=beta
      if(un.le.0.)un=und
      if(year.le.0.)year=yeard
c
c   Calculate Universal Time
c
      HourUT=hr-Zone-shift/60.
      if(HourUT.le.24.)goto 601
      HourUT=HourUT-24.
      Day=Day+1
      goto 602
 601  continue
      if(HourUT.ge.0.)goto 602
      HourUT=HourUT+24.
      Day=Day-1
 602  continue
      
      TdegK=T+273.15
      CALL SunPSA(HourUT,Latit,Longit,Z,Radius,Decli,
     1 p,TdegK,Year,Month,Day)
      En0=Esc/(Radius*Radius)
      if(Z.gt.Zmax) goto 2
 
      zenit=Z
      ZR=Zenit*0.017453293
      cosZ=COS(ZR)      
      presc=p/p0
      E0=En0*cosZ      

c
c     Original Kasten + revised (Kasten & Young) air masses
c 
      amk=1./(cosZ+.15/(93.885-zenit)**1.253)
      amky=1./(cosZ+.50572/(96.07995-zenit)**1.6364)
c
c     Effective aerosol wavelength
c     Spectral to broadband AOD conversion (Molineaux et al., 1998)
c     for Shettle & Fenn's rural aerosol mix
c
      tau70=beta/(.7**alpha)
      wvlef=.695+amky*(.0160+.066*tau70)
      tauA=beta/(wvlef**alpha)
c
c   Linke turidity coefficient
c   TL=f(beta), Hinzpeter-type correlation TL=a0+a1*beta
c   Coefficients obtained here from an average of 4 beta=f(TL) by
c   Hinzpeter, Katz, Abdelrahman and Grenier.
c   This method is used for the Chandra, ESRA3, and Kasten models only.
c   Other TL-based models (ESRA2, ESRA4 and Dogniaux) use other
c   estimates of TL.
c
      TL=2.1331+19.0204*beta
c            
c   Call all 38 model subroutines
c
c   Model 1
      Call ASHRAE (DNI(1),Dif(1),Glo(1),month,cosZ)

c   Model 2
      Call Basha (DNI(2),Dif(2),Glo(2),En0,beta,cosZ,presc,w,amk)

c   Model 3
      Call BCLSM (DNI(3),Dif(3),Glo(3),En0,cosZ,w,amk)

c   Model 4
      Call Biga (DNI(4),Dif(4),Glo(4),cosZ)
    
c   Model 5
      Call Bird (DNI(5),Dif(5),Glo(5),E0,En0,rhog,presc,uo,amk,w,alpha,
     1 beta,TrB,ToB,TuB,TwB,TaB)

c   Model 6
      Call CEM (DNI(6),Dif(6),Glo(6),En0,E0,cosZ,presc,w,rho,taua)
      
c   Model 7     
      Call Chandr (DNI(7),Dif(7),Glo(7),Esc,cosZ,TL)

c   Model 8
      Call CLS (DNI(8),Dif(8),Glo(8),En0,E0,cosZ,rhog,presc,w,amk)

c   Model 9
      Call CPCR2 (DNI(9),Dif(9),Glo(9),E0,En0,rhog,z,cosZ,uo,w,presc,
     1 alfa1,alfa2,beta2,piar1,piar2)

c   Model 10
      Call  Dognio (DNI(10),Dif(10),Glo(10),En0,presc,z,cosZ,w,amk,
     1 beta,decli)
    
c   Model 11
      Call DPP (DNI(11),Dif(11),Glo(11),Z,cosZ)

c   Model 12
      Call ESRA2 (DNI(12),Dif(12),Glo(12),En0,presc,cosz,amky,w,beta)

c   Model 13
      Call ESRA3 (DNI(13),Dif(13),Glo(13),En0,presc,cosz,amky,w,alpha,
     1 beta)

c   Model 14
      Call ESRA4 (DNI(14),Dif(14),Glo(14),En0,presc,cosz,amky,TL)
      
c   Model 15
      Call HLJ (DNI(15),Dif(15),Glo(15),En0,cosZ,Elev)

c   Model 16
      Call Ideria (DNI(16),Dif(16),Glo(16),E0,En0,alpha,beta,presc,w,
     1 cosZ,amk)

c   Model 17
      Call IqbalA (DNI(17),Dif(17),Glo(17),En0,cosZ,uo,rhog,presc,w,
     1 alpha,beta,amk,TdegK)

c   Model 18
      Call IqbalB (DNI(18),Dif(18),Glo(18),En0,E0,cosZ,uo,rhog,presc,w,
     1 beta,amk,TdegK)

c   Model 19
      Call IqbalC (DNI(19),Dif(19),Glo(19),E0,En0,rhog,presc,uo,
     1 amk,w,alpha,beta,TdegK)

c   Model 20
      Call Josefs (DNI(20),Dif(20),Glo(20),En0,E0,cosZ,Z,rhog,presc,
     1 w,amk)

c   Model 21
      Call KASM (DNI(21),Dif(21),Glo(21),E0,En0,cosZ,presc,w,amk)

c   Model 22
      Call Kasten (DNI(22),Dif(22),Glo(22),E0,En0,cosZ,presc,TL,amk)
 
c   Model 23
      Call Maxwel (DNI(23),Dif(23),Glo(23),E0,En0,rhog,presc,uo,
     1 amky,w,tauA)    

c   Model 24
      Call MAC (DNI(24),Dif(24),Glo(24),En0,cosZ,rhog,presc,w,TdegK)

c   Model 25
      Call MRM4 (DNI(25),Dif(25),Glo(25),E0,En0,presc,uo,amky,w)
    
c   Model 26
      Call MRM5 (DNI(26),Dif(26),Glo(26),E0,cosZ,amky,presc,w,uo,rhog
     1 ,TdegK,EdM,EM)
    
c   Model 27
      Call Nijego (DNI(27),Dif(27),Glo(27),En0,Elev,cosZ,
     1 TrB,ToB,TuB,TwB,TaB,TdegK)
     
c   Model 28
      Call Paltri (DNI(28),Dif(28),Glo(28),z,cosZ)

c   Model 29
      Call PR (DNI(29),Dif(29),Glo(29),En0,E0,rhog,presc,uo,cosZ,
     1 amky,w,beta)

c   Model 30
      Call PSIM (DNI(30),Dif(30),Glo(30),En0,E0,rhog,presc,beta,w,cosZ)

c   Model 31
      Call REST2 (DNI(31),Dif(31),Glo(31),En0,zenit,cosZ,uo,un,w,
     1 presc,rhog,alfa1,alfa2,beta2,Piar1,Piar2)

c   Model 32
      Call Rodger (DNI(32),Dif(32),Glo(32),En0,Esc,presc,z,cosZ,w,taua)

c   Model 33
      Call RSC (DNI(33),Dif(33),Glo(33),En0,rhog,presc,cosZ,amk,w,beta)

c   Model 34
      Call Santam(DNI(34),Dif(34),Glo(34),En0,E0,rhog,presc,uo,
     1 cosZ,amky,w)
     
c   Model 35
      Call Schulz (DNI(35),Dif(35),Glo(35),cosZ)
      
c   Model 36
      Call Sharma (DNI(36),Dif(36),Glo(36),En0,cosZ)

c   Model 37
      Call Wesely (DNI(37),Dif(37),Glo(37),cosZ,amk,presc)

c   Model 38
      Call Yang (DNI(38),Dif(38),Glo(38),En0,E0,beta,presc,uo,w,amk)
c     
c   Print results
c
      Write(15,118)N,Zenit,amky,w,alpha,beta,tauA,TL,
     1 (DNI(i),i=1,38),(Dif(j),j=1,38),(Glo(k),k=1,38)
     
 118  Format(i5,1x,f4.1,1x,6(f6.3,1x),
     1 2x,3(38(f6.1,1x),2x),8(f10.5,1x))
      goto 1
            
 999  continue
      Stop
      End   
c
c
c
c=======================================================================
c
c
c
      Subroutine ASHRAE (Ebn,Ed,E,Mth,cosZ)
c
c
c     Same as original ASHRAE model of 1972, but with the revised 
c     coefficients that appeared in ASHRAE's Handbook of Fundamentals, 
c     2005 (chap. 31).
c
c     Necessary inputs:
c
c     cosZ: cosine of zenith angle
c     Mth: Month
c
c
c---------------------------------------------------------------------
c
      Real AA(12),BB(12),CC(12)
      Data AA/1202.,1187.,1164.,1130.,1106.,1092.,1093.,1107.,1136.,
     1 1166.,1190.,1204./
      Data BB/.141,.142,.149,.164,.177,.185,.186,.182,.165,.152,
     1 .144,.141/
      Data CC/.103,.104,.109,.12,.13,.137,.138,.134,.121,.111,.106,.103/
      
      A=AA(Mth)
      B=BB(Mth)
      C=CC(Mth)
      Ebn=A*exp(-B/cosZ)
      Ed=C*Ebn
      E=Ebn*cosZ+Ed
c 
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine Basha (Ebn,Ed,E,En0,beta,cosZ,presc,w,amk)
c
c
c     Composite model of Bashahu & Laplaze (Renew. Energy, 4: 455-458, 1994)
c   based on direct irradiance model of King & Buckius, Solar Energy, 22: 297-301, 1979.
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     beta: Angstrom turbidity for alpha=1.3
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      amb=amk*beta
c     
      Tscat=(5.228/am)*(exp(-.0002254*am)-exp(-.1409*am))*
     1 exp(-amb/.6489**1.3)+.00165*am+.2022*(1.-amb/2.875**1.3)
      Tabs=(.1055+.07053*log10(amk*w+.07854))*exp(-amb/1.519**1.3)
      Ebn=En0*(Tscat-Tabs)
      Ed=.5*En0*(1.-Tscat)*cosZ**1.3333
      E=Ebn*cosZ+Ed
c 
      Return
      End
C
c
c
c=======================================================================
c
c
c
      Subroutine BCLSM (Ebn,Ed,E,En0,cosZ,w,amk)
c
c     Model originally proposed by Barbaro et al. (Solar Energy, 22: 225-228, 1979),
c   and modified by Badescu (1987, 1997) and Davies et al. (1988). 
c!!!
c    Problem: none of these revisions yield realistic diffuse results.
c
c   Original Barbaro equations used here, with three modifications.
c   1. Conversion from cal/cm2/min to W/m2 in Barbaro's Eq (10), using
c   1 cal = 4.184 J., hence 1 cal/cm2/min = 41840/60 W/m2.
c   2. A multiplier of 0.1 was apparently missing in the 2nd part of Eq (10).
c   3. Barbaro's Eq (11) has been corrected for the missing cosZ.
c   These modifications now yield fair results.
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     w: precip. water (cm)
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      wm=10.*w*amk
      d=200.
      a1=-.13491
      b1=-.00428
      a2=.13708
      b2=.00261
      a3=3.68e-5
      b3=1.131e-4
      Ebn=En0*exp(a1+b1*w-a3*(d-400.))*exp(-amk*(a2+b2*w+b3*(d-400.)))
      Ewz0=En0*(.938*exp(-.0154*wm))
      Ewz1=.1*(41840./60.)*(.004*(wm**2.1)
     1 -1.1086e-5*(wm**3)+121.948*(1.+wm)/(1.+10.*(wm**2)))
      Ewz=Ewz0+Ewz1
      Ed=Max(0.,.5*(Ewz-Ebn)*(cosZ**1.33333))
      E=(Ed+Ebn*cosZ)/.96
      Ed=E-Ebn*cosZ
      Ex=Ebn
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine Biga (Ebn,Ed,E,cosZ)
c
c     Biga & Rosa model (Solar Energy, 23: 61-67, 1979).
c
c     Necessary inputs:
c
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      Ebn=926.*cosZ**.29
      Ed=131.*cosZ**.6
      E=Ebn*cosZ+Ed
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine Bird (Ebn,Ed,E,E0,En0,rhog,presc,uo,amk,
     1 w,alpha,beta,Tr,To,Tu,Tw,Ta)
c
c
c     Broadband transmittances and turbidity--Bird model
c     according to papers by Bird & Hulstrom, 
c     e.g., Trans. ASME JSEE, 103: 182-192, 1981.
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     alpha: Angstrom wavelength exponent
c     beta: Angstrom turbidity coeff
c     rhog: ground albedo
c     amk: Kasten's air mass
c     z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      amkp=amk*presc
c     
      Tr=exp(-.0903*(amkp**.84)*(1.+amkp-amkp**1.01))
      ako3=amk*uo
      To=1.-.1611*ako3/((1+139.48*ako3)**.3035)
     2 -.002715*ako3/(1.+.044*ako3+.0003*ako3*ako3)
      Tu=exp(-.0127*amkp**.26)
      TCDA=Tr*Tu*To
      akw=amk*w
      Tw=1.-2.4959*akw/((1.+79.034*akw)**.6828+6.385*akw)
      tauA=beta*(.2758*(1./(.38**alpha))+.35*(1./(.5**alpha)))
      Ta=exp(-(tauA**.873)*(1.+tauA-tauA**.7088)*amk**.9108)
      Tn=.9662*Tr*To*Tu*Tw*Ta
      Ebn=En0*Tn
      Eb=E0*Tn

      omeg=.9
      Ba=.84      
      Taa=1.-(1.-omeg)*(1.-amk+amk**1.06)*(1.-Ta)
      Tas=Ta/Taa
      Tabs=To*Tu*Taa*Tw
      Ts0=.79*Tabs*(.5*(1.-Tr)+Ba*(1.-Tas))/(1.-amk+amk**1.02)
      rhos=.0685+(1.-Ba)*(1.-Tas)
      Ed0=Ts0*E0
      E=(Eb+Ed0)/(1.-rhog*rhos)
      Ed=E-Eb
c
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine CEM (Ebn,Ed,E,En0,E0,cosZ,presc,w,rho,taua)
c
c
c     Broadband Atwater & Ball model
c     according to their paper (Solar Energy, 21: 163-170, 1978) and
c     Errata (SE, 23: 275, 1979).
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     taua: Unsworth-Monteith broadband turbidity coeff
c     rho: ground albedo
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      am=35./(1224.*cosZ*cosZ+1.)**.5
      amp=am*presc
      p=presc*101.325
c     
      X1=(am*(949.*p*1e-5+.051))**.5
      Trb=1.041-.16*X1
      Trg=1.021-.0824*X1
      Tg=1.
      To=1.
      absw=.077*(am*w)**.3
      Trw=1.-absw
      Tra=exp(-amp*tauA)
      Ebn=En0*Tra*(Trb*Tg-absw)
      E=E0*Tra*(Trg*Tg-absw)/(1.-rho*.0685)
      Ed=E-Ebn*cosZ
c    
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine Chandr (Ebn,Ed,E,Esc,cosZ,TL)
c
c     From M. Chandra's paper:
c   "Dependence of solar radiation availability on atmospheric turbidity", 
c   Proc. ISES Conf., 430-434, 1978.
c!!!
c   The original model uses TL as the turbidity input.
c   TL is calculated here from beta using TL=a0+a1*beta, with a0 and a1 
c   same as in Kasten's model.
c   Chandra reported his results as ABSOLUTE irradiances in cal/cm2/min 
c   and used a solar constant of 1.94 cal/cm2/min or 1353 W/m2.
c   His results are therefore divided by 1.94 to obtain transmittances.
c   This empirical model is based on measurements that certainly used 
c   IPS56, hence the addition of a correction factor of 1.022 to comply 
c   with the WRR scale.
c  
c
c     Necessary inputs:
c
c     Esc: solar constant
c     TL: Linke turbidity coeff
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      cosZ2=cosZ*cosZ
      cosZ3=cosZ*cosZ2
      cosZ4=cosZ*cosZ3
      
      if(TL.ge.3.0)goto 10
      Ebn=3.27*cosZ-3.17*cosZ2+1.03*cosZ3
      Ed=.745*cosZ-1.56*cosZ2+1.85*cosZ3-.767*cosZ4
      goto 30
 10   continue
      if(TL.ge.4.0)goto 20
      Ebn=2.58*cosZ-2.68*cosZ2+1.10*cosZ3
      Ed=.816*cosZ-1.30*cosZ2+1.45*cosZ3-.657*cosZ4
      goto 30
 20   continue
      Ebn=1.64*cosZ-1.10*cosZ2+2.64*cosZ3
      Ed=1.64*cosZ-5.13*cosZ2+7.31*cosZ3-3.41*cosZ4
 30   continue
      Ebn=Ebn*Esc*1.022/1.94
      Ed=Ed*Esc*1.022/1.94
      E=Ebn*cosZ+Ed
c
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine CLS (Ebn,Ed,E,En0,E0,cosZ,rho,presc,w,amk)
c
c     Cloud Layer Sunshine model developed by Suckling & Hay:
c   (Atmosphere, 14: 298-308, 1976; 15: 194-207, 1977).
c!!!
c   w must be in cm, contrarily to what is indicated in the CLS papers!
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     rho: ground albedo
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      am2=am*am
      am3=am2*am
      am4=am3*am
      wm=w*am
      wmb=w*1.66
c     
      TrR=.972-.08262*am+.00933*am2-.00095*am3+4.37e-5*am4
      Trrb=.85655
      Tws=1.-.0225*wm
      Twsb=1.-.0225*wmb
      Twa=1.-.077*(wm**.3)
      Twab=1.-.077*(wmb**.3)
      Tda=.975**am
      Tdab=.975**1.66
      Tds=Tda
      Tdsb=Tdab
      Ebn=En0*TrR*Tda*Tds*Twa*Tws
      Eds=.6*E0*Twa*Tda*(1.-Tws*Tds*TrR)
      Es=Ebn*cosZ+Eds
      Edb=rho*Es*.4*Twab*Tdab*(1.-Twsb*Tdsb*Trrb)
      Ed=Eds+Edb
      E=Es+Edb
      Ebn1=Ebn
      
      return
      end
c
c
c=======================================================================
c
c
c
      SUBROUTINE CPCR2 (Ebn,Ed,E,E0,En0,rog,z,zcos,uo,w,presc,
     1 alfa1,alfa2,beta,piar1,piar2)
C
c     Original CPCR2 model (Solar Energy, 43: 253-265,1989) with revised 
c     optical masses per Gueymard (Solar Energy, 51: 121-138, 1993).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     alfa1: Angstrom wavelength exponent for <700nm
c     alfa2: Angstrom wavelength exponent for >700nm
c     beta: Angstrom turbidity coeff
c     rog: ground albedo
c     piar1: aerosol single-scattering albedo, <700nm
c     piar2: aerosol single-scattering albedo, >700nm
c     z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      REAL Al0(2),Al1(2),Al2(2)
      REAL X(2),S1(2),ALM1(2),ALM2(2),Y(2),XB(2)
      REAL BETAM1(2),BETAM2(2),WVLTH(2,2),XX(2),TR(2),YY(2)
      REAL AWV(2),AMS(2),AMG(2),YW(2),A0(2),A1(2),A2(2)     
C
C
C            CONSTANTS
C                                                                      
      AMd=1.66
      AMdP=AMd*presc
      F1=0.4604
      F2=0.5057
      F12=.9661
      bet=beta
      if(beta.gt.0.5)bet=0.5
      beta2=bet
      beta1=bet
c
c     Turbidity functions
c
      ALFA12=ALFA1*ALFA1
      ALFA22=ALFA2*ALFA2
      if(abs(alfa1-1.3).gt.1e-3)goto 10
      Al0(1)=.48530305
      Al1(1)=.05603278
      Al2(1)=.00104599
      goto 20
 10   continue
      Al0(1)=.510941-.028607*ALFA1+.006835*ALFA12
      Al1(1)=-.026895+0.054857*ALFA1+.006872*ALFA12
      Al2(1)=.009649+.005536*ALFA1-.009349*ALFA12
 20   continue
      if(abs(alfa2-1.3).gt.1e-3)goto 11
      Al0(2)=1.0534924
      Al1(2)=.02265614
      Al2(2)=.07569785
      goto 21
 11   continue          
      Al0(2)=1.128036-0.0642*ALFA2+0.005276*ALFA22
      Al1(2)=-0.032851+0.036112*ALFA2+0.005066*ALFA22
      Al2(2)=0.027787+0.064655*ALFA2-0.021385*ALFA22
 21   continue
C
C     OPTICAL MASSES
C 
      H=90.-Z
      FMR=(H+4.37515)**1.21563
      FMW=(H+2.24849)**1.2529
      AM=1./(zcos+.00176759*Z/FMR)
      AM2=AM*AM
      AMp=AM*presc
      AMW=1./(zcos+.000429452*Z/FMW)
      AMZ=13.5/(181.25*zcos*zcos+1.)**.5
C                                                                             
C     Transformed OPTICAL MASSES
C   
      X(1)=LOG(1.+AMp)
      X(2)=LOG(1.+AMdp)
      XX(1)=AMp
      XX(2)=AMdp
      Y(1)=AMW                                                          
      Y(2)=AMd
      YY(1)=LOG(AMp)
      YY(2)=LOG(AMdp)
      IF(W.LT.0.001)GOTO 30
      YW(1)=LOG(AMW*W)
      YW(2)=LOG(AMd*W)
 30   CONTINUE
C
C     TURBIDITY AND OTHER FUNCTIONS
C
      BETAM1(1)=AMW*BETA1
      BETAM1(2)=AMd*BETA1
      BETAM2(1)=AMW*BETA2
      BETAM2(2)=AMd*BETA2
C                                                                       
      DO 50 J=1,2                                                             
      S1(J)=1.-EXP(-0.253565+0.061685*X(J)-0.003997*X(J)*X(J)) 
      TR(J)=0.999523-0.010274*XX(J)+7.375E-5*XX(J)*XX(J)
      AMG(J)=0.0224+0.0094*YY(J)+0.0019*YY(J)*YY(J)
      ALM1(J)=LOG(1.+BETAM1(J))
      ALM2(J)=LOG(1.+BETAM2(J))
      IF(W.LT.0.001)GOTO 42
      AWV(J)=0.1779+0.0519*YW(J)+0.0033*YW(J)*YW(J)
 42   CONTINUE
C
C     EQUIVALENT WAVELENGTHS
C
      WVLTH(1,J)=Al0(1)+Al1(1)*ALM1(J)+Al2(1)*ALM1(J)*ALM1(J)
      WVLTH(2,J)=Al0(2)+Al1(2)*ALM2(J)+Al2(2)*ALM2(J)*ALM2(J)
 50   CONTINUE 
C
C     TRANSMITTANCES
C
      ALFAZ1=0.
      IF(Uo.gt.0.)ALFAZ1=EXP(-2.5686+0.6706*LOG(Uo*AMZ))
      TAUZ1=1.-ALFAZ1
c     TAUZ=1.-ALFAZ1*F1/F12
      TAUR1=EXP(-AMp*S1(1))
      TAURM1=EXP(-AMdp*S1(2))
      TAUR2=TR(1)
      TAURM2=TR(2)
c     TAUR=(F1*TAUR1+F2*TAUR2)/F12
      TAUW2=1.
      TAUWP2=1.
      IF(W.LT.0.001)GOTO 60
      TAUW2=1.-AWV(1)
      TAUWP2=1.-AWV(2)
 60   CONTINUE
c     TAUW=1.-AWV(1)*F2/F12
      TAUG2=1.
      TAUGP2=1.
      TAUG2=1.-AMG(1)
      TAUGP2=1.-AMG(2)
c     TAUG=1.-AMG(1)*F2/F12
C                                                                             
C       AEROSOL DIRECTIONAL FACTOR BA
C                                                                             
      BA=1.-EXP(-0.6931-1.8326*zcos)
      BAM=0.8342
      IF(BETA.GT.0.)GOTO 65
      BA=1.
      BAM=1.
 65   CONTINUE
C
C       AEROSOL TRANSMITTANCES AND ALBEDOS FOR THE 2 BANDS
C   
      TAUA1=1.
      TAUA2=1.
      TAUD1=1.
      TAUD2=1.
      TMD1=1.
      TMD2=1.
      TAUM1=1.
      TAUM2=1.
      IF(BETA.LE.0.)GOTO 70
      TAUA1=MAX(EXP(-BETAM1(1)/(WVLTH(1,1)**ALFA1)),0.0001)
      IF(TAUA1.GE.1.)TAUA1=1.
      SIGMA1=-LOG(TAUA1)/AMW                                            
      TAUD1=EXP(-AMW*SIGMA1*PIAR1)                                      
      TMD1=MIN(EXP(-AMd*SIGMA1*PIAR1),1.)                                   
      TAUM1=EXP(-BETAM1(2)/(WVLTH(1,2)**ALFA1))
      TAUM2=EXP(-BETAM2(2)/(WVLTH(2,2)**ALFA2))
      TAUA2=MAX(EXP(-BETAM2(1)/(WVLTH(2,1)**ALFA2)),0.0001)
      IF(TAUA2.GE.1.)TAUA2=1.
      SIGMA2=-LOG(TAUA2)/AMW                                            
      TMD2=MIN(EXP(-AMd*SIGMA2*PIAR2),1.)                               
      TAUD2=exp(-AMW*SIGMA2*PIAR2) 
 70   CONTINUE
      TAUAA1=TAUA1/TAUD1
      TAUAA2=TAUA2/TAUD2
      TMA1=MIN((TAUM1/TMD1),1.)
      TMA2=MIN((TAUM2/TMD2),1.)
c      TranA=(F1*TauA1+F2*TauA2)/F12
c      taua=-log(TranA)/AMw
C
C     SKY ALBEDO (for backscattered radiation)
C
      RO1A=(1.-TMD1)*(1.-BAM)*TMA1*TAURM1                                     
      ROR1=0.5*(1.-TAURM1)*TMA1                                               
      RO1=RO1A+ROR1                                                           
      RO2=(1.-TMD2)*(1.-BAM)*TMA2*TAURM2*TAUWP2*TAUGP2                        
C
C      BAND1 : VISIBLE PART OF THE SPECTRUM (0.29-0.7 m)
C
      ESC1=En0*F1
      EHOR1=ESC1*zcos
      TAU1=TAUZ1*TAUR1                                                        
      ENORM1=TAU1*TAUA1*ESC1                                                  
      EDIFR1=.5*TAUZ1*(1.-TAUR1)*TAUAA1*EHOR1                                
      EDIFA1=BA*TAU1*(1.-TAUD1)*TAUAA1*EHOR1                                  
      EDIF10=EDIFR1+EDIFA1                                                    
      EDIR1=ENORM1*zcos                                                       
      R1=Ro1*Rog                                                  
      EDIFD1=R1*(EDIR1+EDIF10)/(1.-R1)                            
      EDIF1=EDIF10+EDIFD1                                                     
      ETOT1=EDIF1+EDIR1                                                       
C                                                                             
C     BAND2 : INFRA-RED PART OF THE SPECTRUM (0.7-4 m)
C                                                                             
      ESC2=En0*F2
      EHOR2=ESC2*zcos
      TAU2=TAUR2*TAUW2*TAUG2                                                  
      ENORM2=TAU2*TAUA2*ESC2                                                  
      EDIFR2=0.5*(1.-TAUR2)*TAUAA2*TAUW2*TAUG2*EHOR2                          
      EDIFA2=BA*TAU2*(1.-TAUD2)*TAUAA2*EHOR2                                  
      EDIF20=EDIFR2+EDIFA2                                                    
      EDIR2=ENORM2*zcos                                                 
      R2=Ro2*Rog                                                  
      EDIFD2=R2*(EDIR2+EDIF20)/(1.-R2)                         
      EDIF2=EDIF20+EDIFD2                                                     
      ETOT2=EDIF2+EDIR2                                                       
C
C     WHOLE SPECTRUM RESULTS
C
      Ebn=ENORM1+ENORM2                                                       
      DIF0=EDIF10+EDIF20                                                      
      GLOB0=EB+ED0                                                            
      Ed=EDIF1+EDIF2                                                         
      E=ETOT1+ETOT2  
c
c     PAR calculations
c
c      Bpar=Edir1*(.87375+.04031*AM-.00358*AM2)
c      Gpar=Etot1*(.86225+.02084*AM-.00234*AM2)   
c
c     Illuminance calculations
c
c      Blum=Edir1*(.22249+.01614*AM-.0016*AM2)
c      Glum=Etot1*(.21815+.00581*AM-.0008*AM2)
c
      RETURN
      END
C
c
c=======================================================================
c
c
      Subroutine Dognio (Ebn,Ed,E,En0,presc,z,cosZ,w,amk,beta,Decli)
c
c     From B. Dogniaux's reports:
c   "Programme de calcul des eclairements solaires energetiques et lumineux
c   de surfaces orientees et inclinees" Rep. C14, IRM, Brusels, 1976 for direct 
c   and 
c   "Variations qualitatives et quantitatives des composantes du rayonnement 
c   solaire sur ubne surace horizntale par ciel serein en fonction du trouble
c   atmospherique" Rep. B62, IRM, 1968 for diffuse.
c!!!
c   His expression for TL=f(beta) is taken here from:
c   "Variations geographiques et climatiques des expositions energetiques 
c   solaires sur des surfaces receptrices horizontales et verticales"
c   Rep. B38, IRM, 1975.
c 
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     beta: Angstrom turbidity coeff
c     amk: Kasten's air mass
c     z: zenith angle (deg.)
c     Decli: sun declination (deg.)
c
c
c---------------------------------------------------------------------
c

      Double Precision Decli
      
      degrad=.0174532925
      h=90.-z
      hr=h*degrad
      sh=cosZ
      s2h=sin(2.*hr)
      s3h=sin(3.*hr)
      ch=cos(hr)
      c2h=cos(2.*hr)
      c3h=cos(3.*hr)
      am=amk*presc
      tCDA=1.4899-2.1099*ch+.6322*c2h+.0252*c3h-1.0022*sh+1.0077*s2h
     1  -.2606*s3h
      TLD=.1+((h+85.)/(39.5*exp(-w)+47.4))+beta*(16.+.22*w)
      Ebn=En0*exp(-tCDA*am*TLD)
      XK=(.051+.018*am)*TLD-.0016*Decli-.0895
      Eb=Ebn*cosZ
      E=Eb/(1.-XK)
      Ed=E-Eb
c      
      return
      end
c
c
c
c=======================================================================
c
c
c
      Subroutine DPP (Ebn,Ed,E,Z,cosZ)
c
c     Daneshyar-Paltridge-Proctor model tested by T.K. Goswami & D.E. Klett,
c   "Comparison of several models for long term monthly average daily 
c   insolation on horizontal suraces and the estimation of horizontal 
c   surface insolation for 16 U.S. locations", ASME paper 80-WA/Sol-28 (1980)
c     and  Badescu (Solar Energy, 61: 251-264, 1997), and also reviewed 
c   in R. Festa & C.F. Ratto, "Solar radiation statistical properties",
c   Rep. IEA-SCHP-9E-4,IEA Task IX, International Energy Agency (1993). 
c   The DPP acronym is from Badescu.
c!!!
c     Ebn is here from the original paper by
c     Paltridge & Proctor (Solar Energy, 18: 235-243, 1976), with corrected
c     unit for Z. (It is _degrees_
c     rather than radians as they, Badescu or Daneshyar suggested!)
c     Coefficients for Ed are given by Daneshyar as 
c     0.218 and 0.299 cal/cm2/hr for the USA, as cited from Kreider & Kreith
c     (1975) [who are possibly the true original authors of this diffuse model...]
c     Daneshyar also suggests 0.123 and 0.181 for Iran. However, their
c     equation (5a) suggests that Z is in radians, which is not correct.
c     Goswami & Klett used the US values, which are used here too, after
c     conversion from cal/cm2/hr to W/m2 [1 cal = 4.184 J]. They used the 
c     correct unit for Z.
c     Festa & Ratto (1993) mention the Iran values, but with Z in 
c     radians rather than degrees, due to the confusion in the original papers.
c     Badescu (1997) also used radians, but with considerably larger values 
c     for the ocefficients, without explanation.
c
c
c     Required Inputs:
c
c     Z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      h=90.-Z
      Ebn=950.0*(1.-exp(-.075*h))
      Ed=2.534+3.475*h
      E=Ebn*cosZ+Ed
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine ESRA2 (Ebn1,Ed1,E1,En0,presc,cosz,amky,w,beta)
c
c
c     Broadband transmittances and turbidity--ESRA model
c     according to Rigollier et al. (Solar Energy, 68: 33-48, 2000) and 
c     ESRA's book (http://www.helioclim.net/publications/index.html#esra).
c!!!
c     v2 (my nomenclature) according to this revision:
c     TL(am=2) obtained by new Page's formula, Eq. (19) of Remund et al.,
c     "Worldwide Linke turbidity information", Proc. ISES Conf., 2003.
c   This revision was suggested by Jan Remund (pers. commm., 2006).
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     beta: Angstrom turbidity coeff
c     w: precip. water (cm)
c     amky: Kasten & Young's air mass
c     cosZ: cosine of zenith angle
c
c---------------------------------------------------------------------
c
      am=amky*presc
      am2=am*am
      am3=am*am2
      am4=am2*am2
c
c     Clean-dry atmospheric optical depth
c
      sigma=1./(6.62960+1.7513*am-.1202*am2+.0065*am3-.00013*am4)
      if(am.gt.20.)sigma=1. / (10.4 + 0.718 * am)
c
c
      TLam2=(1.8494+.2425*w-.0203*w*w)+beta*(15.427+.3153*w-.0254*w*w)
      TL = 0.8662 * TLam2
      TL2=TL*TL
      Tb=exp(-am*TL*sigma)
      Ebn1=En0*Tb
      A0=.26463-.061581*TL+.0031408*TL2
      A1=2.0402+.018945*TL-.01161*TL2
      A2=-1.3025+.039231*TL+.0085079*TL2
      Trd=-.015843+.030543*TL+.0003797*TL2
      if((A0*Trd).lt.0.002)A0=.002/Trd
      Fd=A0+A1*cosz+A2*cosz*cosz
      Ed1=Trd*Fd*En0
      E1=Ebn1*cosz+Ed1
c
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine ESRA3 (Ebn2,Ed2,E2,En0,presc,cosz,amky,w,alpha,beta)
c
c
c     Broadband transmittances and turbidity--ESRA model
c     according to Rigollier et al. (Solar Energy, 68: 33-48, 2000) and 
c     ESRA's book (http://www.helioclim.net/publications/index.html#esra).
c!!!
c     v3 (my nomenclature) according to this revision I suggest:
c     TL obtained using Ineichen (Solar Energy, 80: 468-478, 2006) and
c     Molineaux et al (Appl. Opt., 37: 7008-7018, 1998) formulae
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     beta: Angstrom turbidity coeff
c     w: precip. water (cm)
c     amky: Kasten & Young's air mass
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      am=amky*presc
      am2=am*am
      am3=am*am2
      am4=am2*am2
c
c     Clean-dry atmospheric optical depth
c
      sigma=1./(6.62960+1.7513*am-.1202*am2+.0065*am3-.00013*am4)
      if(am.gt.20.)sigma=1. / (10.4 + 0.718 * am)

c     
      dR=-.101+.235/(2.**.16)
      dW=.112*(w**.34)/(2.**.55)
      t700=beta/(.7**alpha)
      tautot=dR+dW+t700
      TLam2=(9.4+.9*2.)*tautot
      TL = 0.8662 * TLam2
      TL2=TL*TL
      Tb=exp(-am*TL*sigma)
      Ebn2=En0*Tb
      A0=.26463-.061581*TL+.0031408*TL2
      A1=2.0402+.018945*TL-.01161*TL2
      A2=-1.3025+.039231*TL+.0085079*TL2
      Trd=-.015843+.030543*TL+.0003797*TL2
      if((A0*Trd).lt.0.002)A0=.002/Trd
      Fd=A0+A1*cosz+A2*cosz*cosz
      Ed2=Trd*Fd*En0
      E2=Ebn2*cosz+Ed2
c
      Return
      End
c
C
c=======================================================================
c
c
c
      Subroutine ESRA4 (Ebn3,Ed3,E3,En0,presc,cosz,amky,TL)
c
c
c     Broadband transmittances and turbidity--ESRA model
c     according to Rigollier et al. (Solar Energy, 68: 33-48, 2000) and 
c     ESRA's book (http://www.helioclim.net/publications/index.html#esra).
c!!!
c     v4 (my nomenclature) according to this revision I suggest:
c     TL obtained from an average of four inear relationships
c     beta=f(TL) by Hinzpeter, Katz, Abdelrahman and Grenier
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     beta: Angstrom turbidity coeff
c     w: precip. water (cm)
c     amky: Kasten & Young's air mass
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      am=amky*presc
      am2=am*am
      am3=am*am2
      am4=am2*am2
c
c     Clean-dry atmospheric optical depth
c
      sigma=1./(6.62960+1.7513*am-.1202*am2+.0065*am3-.00013*am4)
      if(am.gt.20.)sigma=1./(10.4 + 0.718 * am)
c
c
c      TL=2.1331+19.0204*beta
c     
      TL2=TL*TL
      Tb=exp(-am*TL*sigma)
      Ebn3=En0*Tb
      A0=.26463-.061581*TL+.0031408*TL2
      A1=2.0402+.018945*TL-.01161*TL2
      A2=-1.3025+.039231*TL+.0085079*TL2
      Trd=-.015843+.030543*TL+.0003797*TL2
      if((A0*Trd).lt.0.002)A0=.002/Trd
      Fd=A0+A1*cosz+A2*cosz*cosz
      Ed3=Trd*Fd*En0
      E3=Ebn3*cosz+Ed3
c
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine HLJ (Ebn,Ed,E,En0,cosZ,Alt)
c
c
c     Broadband Hottel model for direct irradiance
c     according to his Solar Energy paper, vol.18, p.129-134 (1976).
c     Modified by De Carli (1986), Jafarpur & Yaghoubi (1989), Aziz (1990),
c     Khalil (1995), and Togrul (2000), who all added the diffuse 
c     transmittance formula of Liu & Jordan.
c!!!
c     I added reasonable default values for high-altitude cases.
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     Alt: site's elevation (km), must be <6.5
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      a0=.4237-.00821*(6.-Alt)**2
      if(Alt.gt.6.0)a0=.6*(1.-exp(-.214*(Alt+1.12)))
      a1=.5055+.00595*(6.5-Alt)**2
      if(Alt.gt.6.5)a1=.4954
      ak=.2826+.01895*(2.5-Alt)**2
      if(Alt.gt.2.5)ak=-.02173+.33693*Alt
      Tranb=a0+a1*exp(-ak/cosZ)
      Trand=.271-.2939*Tranb
      Ebn=En0*Tranb
      Ed=En0*cosZ*Trand
      E=Ebn*cosZ+Ed
      
c
      Return
      End
c
c
c=======================================================================
c
c
C
      Subroutine Ideria (Ebn,Ed,E,E0,En0,alpha,beta,presc,w,cosZ,amk)
c
c
c     Ideriah's model (Solar Energy, 26, 447-452, 1981) based on the 
c   direct irradiance model by King & Buckius (Solar Energy, 22: 297-301, 1979).
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     beta: Angstrom turbidity for alpha=1.3
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      amb=amk*beta
c     
      Tscat=(5.228/am)*(exp(-.0002254*am)-exp(-.140875*am))*
     1 exp(-amb/.6489**alpha)+.00165*am+.2022*(1.-amb/2.875**alpha)
      Tabs=(.1055+.07053*log10(amk*w+.07854))*exp(-amb/1.519**alpha)
      Ebn=En0*(Tscat-Tabs)
      Ed=.5*Tscat*E0*(cosZ**.33333)
      E=Ebn*cosZ+Ed
c 
      Return
      End
c
c
c=======================================================================
c
c
c
c
      Subroutine IqbalA (Ebn,Ed,E,En0,cosZ,uo,rhog,presc,w,
     1 alpha,beta,amk,TdegK)
c
c     Iqbal's Model A from his book (An introducion to solar radiation, 
c   Academic Press, 1983). This model is adapted from the McMaster model
c   by Davies et al. (Bound. Lay. Meteorol., 9: 33-52, 1975).
c!!!
c     The original formulation for the Rayleigh transmittance
c     is used here, rather than Iqbal's Eq. (7.4.8), which contains a typo.
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     uo: reduced ozone vertical pathlength (atm-cm)
c     alpha: Angstrom wavelength exponent
c     beta: Angstrom turbidity coeff
c     rhog: ground albedo
c     amk: Kasten's air mass
c     TdegK: air temperature, dry-bulb (K)
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      am2=am*am
      am3=am2*am
      am4=am3*am
      am66=1.66*presc
      w1=10.*w
      omeg=.9
c     
      TrR=.972-.08262*am+.00933*am2-.00095*am3+4.37e-5*am4
      x=amk*uo
      aov=.002118*x/(1.+.0042*x+.00000323*x*x)
      aou=.1082*x/(1.+13.86*x)**.805 +.00658*x/(1.+(10.36*x)**3)
      Tro=1.-aov-aou
      wp=w1*amk*(presc**.75)*(273./TdegK)**.5
      aw=.29*wp/((1.+14.15*wp)**.635+.5925*wp)
      Tabs=TrR*Tro-aw
      Tra=.12445*alpha-.0162+(1.003-.125*alpha)*
     1 exp(-am*beta*(1.089*alpha+.5123))
      Trab=.12445*alpha-.0162+(1.003-.125*alpha)*
     1 exp(-am66*beta*(1.089*alpha+.5123))
c
      Ebn=En0*Tabs*Tra
      Eb=Ebn*cosZ
      Fa=.93-.21*log(amk)
      Fab=.83
      rhos=.0685+(1.-Trab)*omeg*(1.-Fab)
      Ed0=En0*cosZ*(.5*Tro*Trab*(1.-TrR)+omeg*Fa*Tabs*(1.-Tra))
      Edb=rhog*rhos*(Eb+Ed0)/(1.-rhog*rhos)
      Ed=Ed0+Edb
      E=Eb+Ed
      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine IqbalB (Ebn,Ed,E,En0,E0,cosZ,uo,rhog,presc,w,
     1 beta,amk,TdegK)
c
c     Iqbal's Model B from his book (An introducion to solar radiation, 
c   Academic Press, 1983). This model is adapted from the Hoyt model 
c   (Solar Energy, 21, 27-35, 1978).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     uo: reduced ozone vertical pathlength (atm-cm)
c     beta: Angstrom turbidity coeff
c     rhog: ground albedo
c     amk: Kasten's air mass
c     TdegK: air temperature, dry-bulb (K)
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      am66=1.66*presc
      omeg=.95
      if(beta.ge.0.5)betac=.5
      wp=w*amk*(presc**.75)*(273./TdegK)**.5
      U3=amk*uo
c     
      TrR=.615958+.375566*exp(-.0221185*am)
      g=-.914+1.909267*exp(-.667023*beta)
      
      aw=.11*(wp+.000631)**.3-.0121
      ag=.002351*(126.*am+.0129)**.26-.00075+.0075*am**.875
      ao3=.045*(U3+.000834)**.38-.0031
      aa=(1.-omeg)*g**am
      Tras=g**am
      Tabs=1.-aa-ao3-aw-ag
c
      Ebn=En0*Tabs*Tras*TrR
      Eb=Ebn*cosZ
      TrRb=.615958+.375566*exp(-.0221185*am66)
      Trasb=g**am66
      rhos=Tabs*(.5*(1.-TrRb)+.25*(1.-Trasb))
      Ed0=E0*Tabs*(.5*(1.-TrR)+.75*(1.-Tras))
      Edb=rhog*rhos*(Eb+Ed0)
      Ed=Ed0+Edb
      E=Eb+Ed
      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine IqbalC (Ebn,Ed,E,E0,En0,rhog,presc,uo,
     1 amk,w,alpha,beta,TdegK)
c
c
c     Iqbal's Model C from his book (An introducion to solar radiation, 
c   Academic Press, 1983). This model is adapted from Bird's model (#5).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     uo: reduced ozone vertical pathlength (atm-cm)
c     alpha: Angstrom wavelength exponent
c     beta: Angstrom turbidity coeff
c     rhog: ground albedo
c     amk: Kasten's air mass
c     TdegK: air temperature, dry-bulb (K)
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      amkp=amk*presc
c
c     Transmittances
c
      Tr=exp(-.0903*(amkp**.84)*(1.+amkp-amkp**1.01))
      ako3=amk*uo
      To=1.-.1611*ako3/((1+139.48*ako3)**.3035)
     2 -.002715*ako3/(1.+.044*ako3+.0003*ako3*ako3)
      Tu=exp(-.0127*amkp**.26)
      TCDA=Tr*Tu*To
      w0=w*(presc**.75)*(273./TdegK)**.5
      akw=amk*w0
      Tw=1.-2.4959*akw/((1.+79.034*akw)**.6828+6.385*akw)
      tauA=beta*(.2758*(1./(.38**alpha))+.35*(1./(.5**alpha)))
      Ta=exp(-(tauA**.873)*(1.+tauA-tauA**.7088)*amkp**.9108)
      Tn=.9751*TCDA*Tw*Ta
c
c     Direct irradiance
c
      Ebn=En0*Tn
      Eb=E0*Tn    
c
c     Diffuse and global irradiance
c
      omeg=.9
      Ba=.84      
      Taa=1.-(1.-omeg)*(1.-amkp+amkp**1.06)*(1.-Ta)
      Tas=Ta/Taa
      Tabs=To*Tu*Taa*Tw
      Ts0=.79*Tabs*(.5*(1.-Tr)+Ba*(1.-Tas))/(1.-amkp+amkp**1.02)
      rhos=.0685+(1.-Ba)*(1.-Tas)
      Ed0=Ts0*E0
      E=(Eb+Ed0)/(1.-rhog*rhos)
      Ed=E-Eb

      Return
      End
c
C
c=======================================================================
c
c
c
      Subroutine Josefs (Ebn,Ed,E,En0,E0,cosZ,Z,rhog,presc,w,amk)
c
c     Model by Josefsson (1985, unpublished)
c     Described, used and/or modified by 
c     Davies et al. (1984), Davies et al. (1988), and Davies & McKay (1989)
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     rhog: ground albedo
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      am2=am*am
      am3=am2*am
      am4=am3*am
      am5=am4*am
      am66=1.66*presc
      h=90.-Z
      w1=10.*w
      omeg=.75
c     
      wp=w1*amk
      aw=.29*wp/((1.+14.15*wp)**.635+.5925*wp)
      g=.5248+.007981*h
      if(h.gt.45.)g=.856+.000734*h
      Tra=.95**amk
      Trab=.95**am66
      Traa=1.-(1.-omeg)*(1.-Tra)
      Tras=1.-omeg*(1.-Tra)
      TrO3=.95545
      TrR=.9768-.0874*am+.010607552*am2-.000846205*am3+3.57246e-5*am4
     1 -6.0176e-7*am5
      Tabs=TrO3*Traa
      Ebn=En0*(Tabs*Tras*TrR-aw)
      Eb=Ebn*cosZ
      Fa=.93-.21*log(amk)
      Fab=.93-.21*log(1.66)
      rhos=.0685+(1.-Trab)*omeg*(1.-Fab)
      Ed0=E0*(.5*Tabs*Tras*(1.-TrR)+g*(TrR*Tabs-aw)*(1.-Tras))
      Edb=rhog*rhos*(Eb+Ed0)/(1.-rhog*rhos)
      Ed=Ed0+Edb
      E=Eb+Ed
      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine KASM (Ebn,Ed,E,E0,En0,cosZ,presc,w,amk)
c
c     Modified Kasten model (1983) according to Badescu (SE, 61: 251-264, 1997).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      am2=am*am
      am3=am2*am
      am4=am3*am
      am5=am4*am
      w1=10.*w
c     
      TrR=.9768-.0874*am+.010607552*am2-.000846205*am3+3.57246e-5*am4
     1 -6.0176e-7*am5
      x=amk*3.5
      aov=.002118*x/(1.+.0042*x+.00000323*x*x)
      aou=.1082*x/(1.+13.86*x)**.805 +.00658*x/(1.+(10.36*x)**3)
      TroP=1.-aov-aou
      wp=w1*am
      aw=.29*wp/((1.+14.15*wp)**.635+.5925*wp)
      Tra=.9**am
      Ebn=En0*(TrR*TroP-aw)*Tra
      Ed=E0*(Trop*(1.-TrR)*.5 + 
     1 (TroP*TrR-aw)*(1.-Tra)*.75*(.93-.21*log(am)))
      E=Ebn*cosZ+Ed     
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine Kasten (Ebn,Ed,E,E0,En0,cosZ,presc,TL,amk)
c
c     Modified Kasten model (1983) according to Davies & McKay (SE 1989) 
c   and Krarti (ASHRAE 2006).
c!!!
c   TL (Linke turbidity) obtained here from an average of four linear
c   relationships beta=f(TL) by
c   Hinzpeter, Katz, Abdelrahman and Grenier.
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     TL: Linke turbiity coefficient
c     amk: Kasten's air mass
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
c      TL=2.1331+19.0204*beta
c     
      tCDA=1./(9.4+.9*am)
      Ebn=En0*exp(-tCDA*TL*am)
      E=E0*.84*exp(-.027*TL*am)
      Ed=E-Ebn*cosZ
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine MAC (Ebn,Ed,E,En0,cosZ,rhog,presc,w,TdegK)
c
c     McMaster model by Davies et al. (Bound. Lay. Meteorol., 9: 33-52, 1975).
c     Used and/or modified by Davies & Uboegbulam (1979), 
c     Davies & Hay (1980), McLaren et al. (1980), 
c     Davies et al. (1984), Davies et al. (1988), and Davies & McKay (1989)
c!!!
c     The formulation (particularly for the Rayleigh and aerosol transmittances)
c     used here are from Davies et al., IEA Task IX report (1988).
c     The Rayleigh transmittance formula in this report contained a typo;
c     it is corrected here!
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     rhog: ground albedo
c     TdegK: air temperature, dry-bulb (K)
c     cosZ: cosine of zenith angle (needs to be strictly >0)
c
c
c---------------------------------------------------------------------
c
      amr=35./(1.+1224.*cosZ*cosZ)**.5
      am=amr*presc
      w1=10.*w
      omeg=.98
c     
      amexp=.0279286*log(am)-.806955
      X=8.688237*am**amexp
      TrR=X/(1.+X)
      x=amr*3.5
      aov=.002118*x/(1.+.0042*x+.00000323*x*x)
      aou=.1082*x/(1.+13.86*x)**.805 +.00658*x/(1.+(10.36*x)**3)
      Tro=1.-aov-aou
      wp=w1*amr*(presc**.75)*(273./TdegK)**.5
      aw=.29*wp/((1.+14.15*wp)**.635+.5925*wp)
      Tabs=TrR*Tro-aw
      Tra=.95**am
      Trab=.95**1.66
      Ebn=En0*Tabs*Tra
      Eb=Ebn*cosZ
      Fa=.93-.21*log(amr)
      Fab=.93-.21*log(1.66)
      rhos=.0685+(1.-Trab)*omeg*(1.-Fab)
      Ed0=En0*cosZ*(.5*Tro*Trab*(1.-TrR)+omeg*Fa*Tabs*(1.-Tra))
      Edb=rhog*rhos*(Eb+Ed0)/(1.-rhog*rhos)
      Ed=Ed0+Edb
      E=Eb+Ed
      
      return
      end
c
c
c=======================================================================
c
c
c     
      Subroutine Maxwel (Ebn,Ed,E,E0,En0,rhog,presc,uo,amky,w,tauA)
c
c
c     Broadband transmittances and turbidity--METSTAT model
c     according to paper by Maxwell (Solar Energy, 62: 263-279, 1998)
c     and NSRDB Final Rept vol.2, NREL 1995.
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
C     E0: extraterrestrial irradiance on a horizontal plane
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     rhog: ground albedo
c     w: precip. water (cm)
c     tauA: Unsworth-Monteith broadband aerosol optical depth
c     amky: Kasten & Young's air mass
c
c
c---------------------------------------------------------------------
c
c     amky=1./(zcos+.50572/(96.07995-z)**1.6364)
      amkp=amky*presc
c     
      Tr=exp(-.0903*amkp**.84*(1.+amkp-amkp**1.01))
      ako3=amky*uo
      To=1.-.1611*ako3/((1+139.48*ako3)**.3035)
     2 -.002715*ako3/(1.+.044*ako3+.0003*ako3*ako3)
      Tu=exp(-.0127*amkp**.26)
      TCDA=Tr*Tu*To
      akw=amky*w
      Tw=1.-1.668*akw/((1.+54.6*akw)**.637+4.042*akw)
      Ta=exp(-amky*tauA)
      Tn=.9751*TCDA*Tw*Ta
      Ebn=En0*Tn  
      omeg=.9
      Ba=.84
      Taa=1.-(1.-omeg)*(1.-amky+amky**1.06)*(1.-Ta)
      Tas=Ta/Taa
      Tabs=To*Tu*Taa
      Tsr=.5*(1.-Tr)*Tabs
      Tsa=Ba*(1.-Ta)*Tabs
      Fam=.38+.925*exp(-.851*amky)
      rhos=.0685+(1.-Ba)*(1.-Tas)
      Ts0=(Tsr+Tsa)*Fam
      Tsg=(Tn+Ts0)*(rhog-.2)*rhos
      Td=Ts0+Tsg
      XKt=Tn+Td
      Ed=Td*E0
      E=XKt*E0

      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine MRM4 (Ebn,Ed,E,E0,En0,presc,uo,amky,w)
c
c
c     Broadband transmittances and turbidity--MRM model
c     according to paper by Muneer et al. (Energy Convers. Mgmt., 39: 
c     303-317, 1998)
c     and Fortran code, v.4 from Kambezidis (pers. comm. 2002).
c   Numerical coefficients AF are for USA and southern Europe, as
c   given in Muneer's book (Elsevier, 2004), p. 73.
c   The ka value (Fortran variable here: t) is also for the USA.
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     E0: same as En0, but for a horizontal surface
c     amky: air mass according to Kasten & Young
c     presc: p/1013.25
c     uo: total ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c
c
c---------------------------------------------------------------------
c
      Real AF(14)
      DATA AF/.873,.7088,.9108,.972,.08262,.00933,.00095,.000437,
     1 2.4959,79.034,.6828,6.385,.0127,.26/     

      t=.387
      am=amky*presc
      am2=am*am
      am3=am2*am
      am4=am3*am
      AAM=am
      RAM=amky
      RAM102=RAM**1.02
      ERAD=E0
c     
C     CALCULATE ATMOSPHERIC TRANSMITTANCE DUE TO OZONE
      XLO3=uo
      XO3=XLO3*RAM
      TO1=0.1611*XO3/(1.0+139.48*XO3)**.3035
      TO2=-0.002715*XO3/(1.0+0.044*XO3+0.0003*XO3**2)
      TO3=1.0-(TO1+TO2)
 
C     CALCULATE ATMOSPHERIC TRANSMITTANCE DUE TO MIXED GASES 
      TMG3=EXP(-AF(13)*am**AF(14))      
     
C     CALCULATE ATMOSPHERIC TRANSMITTANCE DUE TO RAYLEIGH SCATTERING
      TR3=AF(4)-AF(5)*am+AF(6)*am2-AF(7)*am3+AF(8)*am4
      
C     CALCULATE ATMOSPHERIC TRANSMITTANCE DUE TO WATER VAPOR
      XH2O=RAM*W
      TH3=1.-AF(9)*XH2O/((1.+AF(10)*XH2O)**AF(11)+(AF(12)*XH2O))       
      
C     CALCULATE ATMOSPHERIC TRANSMITTANCE DUE TO AEROSOL SCATTERING

      TA3=EXP(-t**AF(1)*(1.+t-t**AF(2))*am**AF(3))      
      
C     CALCULATE BEAM RADIATION
      Tabs=TO3*TH3*TMG3
      XKb=TA3*TR3*Tabs
      Ebn=En0*XKb
      Eb=E0*XKb
      
C     CALCULATE TAA & TAS
      TAA2=1.0-0.1*(1.-TA3)*(1.-RAM+RAM**1.06)   
      TAS=10.0**(-0.045*AAM**0.7)    

C     CALCULATE DIFFUSE RADIATION
      GC=(ERAD*Tabs*TAA2/(1.-RAM+RAM102))*(0.84*(1.-TAS)+0.5*(1.-TR3))    
      DC=GC     

      IF (XKB.GE.0.015)THEN
      
C     CALCULATE GLOBAL RADIATION 
      
      DBR=.285/XKb**1.006
      DC=DBR*Eb
      GC=Eb+DC    
      ENDIF
      Ed=DC
      E=GC
c     write(15,911)Trr,Trg,Tro,Trw,Tra,amky
 911  format('MRM Tr Tg To Tw Ta m = ',8(f7.4,1x))

      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine MRM5 (Ebn,Ed,E,E0,cosZ,amky,presc,w,uo,rhog,TdegK
     1 ,Ed1,E1)
c

c     Broadband irradiances--MRM model version 5.
c     Adapted from Fortran code, v5, from Kambezidis & Psiloglou 
c   (pers. comm. 2007) and Badescu's book (2007).
c
c     Necessary inputs:
c
c     E0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance and for a
c           horizontal surface
c     amky: air mass according to Kasten & Young
c     cosZ: cosine of zenith angle
c     presc: p/1013.25
c     rhog: ground albedo (0-1); defaults to 0.2 if <0 (e.g., because of
c           missing value)
c     TdegK: air temperature (K)
c     uo: total ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c
c
c---------------------------------------------------------------------
c
      am=amky*presc
      AAM=am
      RAM=amky
      RAM102=RAM**1.02
      ERAD=E0
      XLW=w*(presc**.75)*(273.15/TdegK)**.5
      XO3=uo
      Rs=rhog
      if(Rs.le.0.)Rs=.2
c     
C     ------------------------------------------------------------------
C     EXTINCTION EQUATIONS FROM "PSILOGLOU" MODEL
C     ------------------------------------------------------------------
C
C     ------------------------------------------------------------------
C     Calculation of atmospheric transmittance (TMGAS) due to mixed gases
C     ------------------------------------------------------------------
        UCO2=330.0
        ACO2=AAM*UCO2
        XXA=((1.0+(377.89*ACO2))**0.5855)+(3.1709*ACO2)
        ACO2=(0.0721*ACO2)/XXA
        TCO2=1.0-ACO2
 
        UCO=0.075
        ACO=AAM*UCO
        XXB=((1.0+(243.67*ACO))**0.4246)+(1.7222*ACO)
        ACO=(0.0062*ACO)/XXB
        TCO=1.0-ACO
 
        UN2O=0.28
        AN2O=AAM*UN2O
        XXC=((1.0+(107.413*AN2O))**0.5501)+(0.9093*AN2O)
        AN2O=(0.0326*AN2O)/XXC
        TN2O=1.0-AN2O
 
        UCH4=1.60
        ACH4=AAM*UCH4
        XXD=((1.0+(166.095*ACH4))**0.4221)+(0.7186*ACH4)
        ACH4=(0.0192*ACH4)/XXD
        TCH4=1.0-ACH4
 
        UO2=2.095*100000.0
        AO2=AAM*UO2
        XXE=((1.0+(476.934*AO2))**0.4892)+(0.1261*AO2)
        AO2=(0.0003*AO2)/XXE
        TO2=1.0-AO2
 
        TMGAS=TCO2*TCO*TN2O*TCH4*TO2
C
C     ------------------------------------------------------------------
C     Calc.of atmospheric transmittance (TRAYL) due to Rayleigh scattering
C     ------------------------------------------------------------------
        XXH=0.9341-(AAM**0.9868)+(0.9391*AAM)
        XXH=0.1128*(AAM**0.8346)*XXH
        TRAYL=EXP(-XXH)
C
C     ------------------------------------------------------------------
C     Calculation of atmospheric transmittance (TWTR) due to water vapour
C     ------------------------------------------------------------------
        XH2O=RAM*XLW
 
        XXF=((1.0+(119.3*XH2O))**0.644)+(5.814*XH2O)
        Aw=(3.014*XH2O)/XXF
        TWTR=1.0-Aw
C
C     ------------------------------------------------------------------
C     Calculation of atmospheric transmittance (TO3) due to OZONE
C     ------------------------------------------------------------------
        XXG=((1.0+(6107.26*XO3))**0.204)+(0.471*XO3)
        AO3=(0.2554*XO3)/XXG
        TO3=1.0-AO3
C
C     ------------------------------------------------------------------
C     Calc.of atmospheric transmittance (TAER) due to aerosol scattering
C     ------------------------------------------------------------------
C --- YANG et al. SE,2001 - AEROSOL TOTAL EXTINCTION
C
C        HEIGHT=500.0
C        BHTA=(0.025+0.1*COS(YLAT))*EXP(-(0.7*HEIGHT)/1000.0)
 
        BHTA=0.10
        A3=0.677+0.1464*(AAM*BHTA)-0.00626*((AAM*BHTA)**2.0)
        A3=(AAM*BHTA)/(A3**1.3)
        TAER=EXP(-A3)
C
C --- AEROSOL ABSORPTION ONLY
C --- FOR URBAN AREAS -> OMEGA=0.6
C     FOR RURAL AREAS -> OMEGA=0.9
        OMEGA=0.9
        TAAB=1.0-((1.0-OMEGA)*(1.0-AAM+(AAM**1.06))*(1.0-TAER))
 
C
C --- AEROSOL SCATTERING ONLY
        TASC=TAER/TAAB
 
C
C     ------------------------------------------------------------------
C     Calculation of direct-beam radiation (BHOR) and DNI (Ebn).
C     ------------------------------------------------------------------
         Tabs=TO3*TWTR*TMGAS
         BHOR=ERAD*TAER*TRAYL*Tabs
         Ebn=BHOR/cosZ
C
C
C     ------------------------------------------------------------------
C     Calculation of diffuse and global radiation
C     ------------------------------------------------------------------
C
c   Sky albedo
c
c         Rs=0.2 [Variable ground albedo used here--CG]
c
         AM166=1.66
         A3M166=0.677+0.1464*(AM166*BHTA)-0.00626*((AM166*BHTA)**2.0)
         A3M166=(AM166*BHTA)/(A3M166**(1.3))
         TAERM166=EXP(-A3M166)
         Ra=0.0685 + 0.17*(1.0-TAERM166)
         Rsa = Rs * Ra
c         
C --- DIFFUSE MODEL 1 (Bird) ---
c          
         HD1=ERAD*Tabs*TAAB*(0.5*(1.0-TRAYL)+0.84*(1.0-TASC))/
     $                                     (1.0-RAM+RAM102)
      HDb1=(BHOR+HD1)*Rsa/(1.0-Rsa)
      Ed1=HD1+HDb1
      E1=BHOR+Ed1
c         
C --- DIFFUSE MODEL 3 (???) ---
c          
         HDR2=ERAD*Tabs*TAAB*(0.5*(1.0-TASC*TRAYL)) 
         HDA2=(BHOR+HDR2)*Rsa/(1.0-Rsa)
         DHOR3 = HDR2 + HDA2
         Ed=DHOR3
         E=BHOR+Ed
C
      Return
      End
c
c
c
c=======================================================================
c
c
c
      Subroutine Nijego (Ebn,Ed,E,En0,Alt,cosZ,Tr,To,Tu,Tw,Ta,TdegK)
c
c
c     Broadband transmittances and turbidity--Nijegorodov et al. model
c     (Renew. Energy, 11: 341-350, 1997).
c   Uses the transmittances from Bird's model (model #5).
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     alpha: Angstrom wavelength exponent
c     beta: Angstrom turbidity coeff
c     TdegK: air temperature (K)
c     amk: Kasten's air mass
c     z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      RsH=6372./40.
      amN=(((1.+2.*RsH+RsH*RsH*cosZ*cosZ)**.5)-RsH*cosZ)
     1 *exp(-.034169*Alt/TdegK)
c     
      Tn=Tr*To*Tu*Tw*Ta
      Ebn=En0*Tn

      xKs=-log(Tr)
      xKst=-log(Ta)
      xKt=-log(Tn)
      Ed=En0*(xKt*exp(-(amN-1.)*xKt))*(.5*xKs+.7*xKst)/(exp(xKt)-1.)
      E=Ed+Ebn*cosZ
c      
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine Paltri (Ebn,Ed,E,z,cosZ)
c
c     Paltridge & Platt empirical model from their book
c   "Radiative processes in meteorology and cimatology",
c   Elsevier, 1976.
c
c     Necessary inputs:
c
c     z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      h=90.-z
      Ebn=1000.*(1.-exp(-.06*h))
      Ed=5.+96.*(1.-exp(-.05*h))
      E=Ebn*cosZ+Ed
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine PR (Ebn,Ed,E,En0,E0,rog,presc,uo,cosZ,amky,w,beta)
c
c
c     Broadband transmittances and turbidity--Psiloglou model
c     as described in Psiloglou et al. (P. Appl. Geophys., 157: 829-860, 2000).
c!!!
c     Taer expression is here from Gueymard's REST model (Solar Energy,
c     74: 355-379, 2003) per Psiloglou's request (pers. comm. 2006), to
c     improve the model's performance.
c     The new acronym stands for "Psiloglou revised".
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25 is the pressure correction
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     beta: Angstrom's turbidity coeff.
c     rog: ground albedo
c     amky: Kasten & Young's air mass function
c
c
c---------------------------------------------------------------------
c
      Real MasA
c
      amkyp=amky*presc
c
c     Transmittances from Psiloglou et al, Pageoph 2000
c***  I added the pressure correction where needed for consistency
c     
      Trr=exp(-.1128*(amkyp**.8346)*(.9341-amkyp**.9868+.9391*amkyp))
      ako3=amky*uo      
      Tro=1.-.2554*ako3/((1.+6107.26*ako3)**.204+.471*ako3)
      akco2=amkyp*330.
      TrCO2=1.-.0721*akco2/((1.+377.89*akco2)**.5855+3.1709*akco2)
      akco=amkyp*.075
      TrCO=1.-.0062*akco/((1.+243.67*akco)**.4246+1.7222*akco)
      akch4=amkyp*1.6
      TrCH4=1.-.0192*akch4/((1.+166.095*akch4)**.4221+.7186*akch4)
      akn2o=amkyp*.28
      TrN2O=1.-.0326*akn2o/((1.+107.413*akn2o)**.5501+.9093*akn2o)
      ako2=amkyp*2.095e+05
      TrO2=1.-.0003*ako2/((1.+476.934*ako2)**.4892+.1261*ako2)
      Trmg=TrCO2*TrCO*TrCH4*TrN2O*TrO2
      akw=amky*w
      Trw=1.-3.014*akw/((1.+119.3*akw)**.6440+5.814*akw)
c
c==================================
c     Aerosol transmittance from REST
c
      
      Masa=amky
      beta2=beta*beta
      b0=1.6933
      bx=1.+.42003*beta
      b1=(-.013029+.13126*beta)/bx
      b2=(-.0083581+.40323*beta+.123*beta2)/bx
      TauA=beta*(b0+MasA*b1)/(1.+b2*MasA)       
      Tra=exp(-MasA*TauA)
c==================================
c
      Ebn=En0*Trr*Trmg*Tro*Trw*Tra
      Eb=Ebn*cosZ
      Traa=1.0-1.405e-3*amky-9.013e-5*amky**2+2.2e-6*amky**3
      Tras=Tra/Traa
      ros=.08503
      Ed1=E0*Trmg*Tro*Trw*Traa*(1.-Tras*Trr)/2.
      Ed2=(Ed1+Eb)*rog*ros/(1.-rog*ros)
      Ed=Ed1+Ed2
      E=Eb+Ed
c
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine PSIM (Ebn,Ed,E,En0,E0,rog,presc,beta,w,cosZ)
c
C
C     PSIM = PSI Model. Original code: 11 dec 1990
c   Described in Gueymard (Solar Energy, 50: 385-397, 1993).
C
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     beta: Angstrom turbidity coeff
c     cosZ: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      ATM=Presc
      Fbgw=EXP(.155*(1.-W**.24))
      F2PB=1.+(.1594-.226*BETA)*(1.-ATM)
      F2PG=1.+(.0752-.107*BETA)*(1.-ATM)
      ROS=0.061+0.072*BETA**.5
      F3RO=(1.-0.2*ROS)/(1.-ROG*ROS)
      S=Max(cosZ,.0285)
      S2=S*S
      S3=S2*S
      S4=S2*S2
      S5=S3*S2      
C
      X=log(1.+10.*beta)
      X2=X*X
      X3=X2*X
      IF(beta.GE.0.175)GOTO 24
      A0=EXP(-1.62364-6.8727*X)
      A1=2.94298+5.23504*X-18.23861*X2+11.1652*X3
      A2=-8.12158-15.8*X+69.2345*X2-45.1637*X3
      A3=12.5571+25.44*X-123.3933*X2+83.1014*X3
      A4=-9.8044-20.3172*X+103.9906*X2-71.3091*X3
      A5=3.00487+6.3176*X-33.3891*X2+23.1547*X3      
      GOTO 25
 24   CONTINUE      
      a0=0.
      a1=4.41547-5.09266*X+1.471865*X2
      a2=-18.45187+38.3584*X-22.7449*X2+4.3189*X3
      a3=31.2506-74.53835*X+48.355*X2-9.8657*X3
      a4=-25.18762+64.3575*X-43.6586*X2+9.23151*X3
      a5=7.64179-20.41687*X+14.20502*X2-3.06053*X3
 25   CONTINUE
      Ebn=En0*Fbgw*F2pb*(A0+A1*S+A2*S2+A3*S3+A4*S4+A5*S5)
c
      g0=.006
      g1=.387021-.386246*X+.09234*X2
      g2=1.35369+1.533*X-1.07736*X2+.23728*X3
      g3=-1.59816-1.903774*X+1.631134*X2-.3877*X3
      g4=.66864+.80172*X-.75795*X2+.18895*X3
      E=En0*Fbgw*F2PG*(g0+g1*S+g2*S2+g3*S3+g4*S4)*F3ro
      Ed=Max(0.,E-Ebn*cosZ)
c
      return
      END
c
c
c=======================================================================
c
c
c
      SUBROUTINE REST2 (Ebn,Ed,E,En0,z,zcos,uo,un,w,
     1 presc,rog,alfa1,alfa2,beta,piar1,piar2)
C
c     Gueymard's REST2 model (Proc. ASES Conf., 2004; Solar Energy, 
c     in press, 2007).
c     Latest version (v6) used here (Nov. 2007).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     un: total NO2 vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     alfa1: Angstrom wavelength exponent for <700nm
c     alfa2: Angstrom wavelength exponent for >700nm
c     beta: Angstrom turbidity coeff
c     rog: ground albedo
c     piar1: aerosol single-scattering albedo, <700nm
c     piar2: aerosol single-scattering albedo, >700nm
c     z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      REAL Al0(2),Al1(2),Al2(2),Al3(2)
      REAL BETAM1(2),BETAM2(2),ALM1(2),ALM2(2),ALM12(2),ALM22(2)
      REAL MASR,MASO,MASW,MasR2,MasO2,MasW2,MasA,MasA2,MasA15,WVLTH(2,2)      
C
C            CONSTANTS
C                                                                      
      AMd=1.66
      AMd2=AMd*AMd
      F1=0.46512
      F2=0.51951
      F12=.98463
c
c==============
c     This part is not useful here because Piar is measured.
c
c     Determine what aerosol type to use
c
c     alpha=(alfa1+alfa2)/2.
c     iaer=1
c     if(alpha.ge.alphax(1))goto 5
c     iaer=5
c     if(alpha.le.alphax(5))goto 5
c     do 1 i=2,5
c     if(alpha.lt.0.5*(alphax(i-1)+alphax(i)))iaer=i-1
c 1   continue
c 5   continue
c
c     Determine single-scattering albedo
c
c     Piar1=Pia1(Iaer)
c     Piar2=Pia2(Iaer)
c     Piar1=.92
c     Piar2=.84
c
c==============
c
c
      bet=beta
      if(beta.gt.1.1)bet=1.1
      beta2=bet
      beta1=bet
      difal=alfa1-alfa2
      if(abs(difal).gt.1e-3)beta1=beta*0.7**difal
c
c     Optical masses from SMARTS model
c
      AM=1./(zcos+.48353*(Z**.095846)/(96.741-Z)**1.754)
      MasR=presc*AM
      MasW=1./(zcos+.10648*(Z**.11423)/(93.781-Z)**1.9203)
      MasO=1./(zcos+1.0651*(Z**.6379)/(101.8-Z)**2.2694)
      MasA=1./(zcos+.16851*(Z**.18198)/(95.318-Z)**1.9542)
      MasR2=MasR*MasR
      MasO2=MasO*MasO
      MasW2=MasW*MasW
      MasA2=MasA*MasA
      MasA15=MasA**1.5
      uo2=uo*uo
      un2=un*un
      w2=w*w
      AM2=AM*AM
C
C     TRANSMITTANCES, Band 1
C
      TranR1=(1.+1.8169*MasR-.033454*MasR2)/
     1  (1.+2.063*MasR+.31978*MasR2)
      TranG1=(1.+.95885*MasR+.012871*MasR2)/(1.+.96321*MasR
     1  +.015455*MasR2)
      TGP1=(1.+.95885*AMd+.012871*AMd2)/(1.+.96321*Amd
     1  +.015455*AMd2)
     
      TranO1=1.
      IF(Uo.LE.0.01)GOTO 707
      a1=uo*(10.979-8.5421*uo)/(1.+2.0115*uo+40.189*uo2)
      a2=uo*(-.027589-.005138*uo)/(1.-2.4857*uo+13.942*uo2)
      a3=uo*(10.995-5.5001*uo)/(1.+1.6784*uo+42.406*uo2)
      TranO1=(1.+a1*MasO+a2*MasO2)/(1.+a3*MasO)
 707  CONTINUE    
      b1=(.17499+41.654*un-2146.4*un2)/(1.+22295.0*un2)
      b2=un*(-1.2134+59.324*un)/(1.+8847.8*un2)
      b3=(.17499+61.658*un+9196.4*un2)/(1.+74109.0*un2)
      TranN1=Min(1.,(1.+b1*MasW+b2*MasW2)/(1.+b3*MasW))
      TNP1=Min(1.,(1.+b1*AMd+b2*AMd2)/(1.+b3*AMd))
      
      c1=w*(.065445+.00029901*w)/(1.+1.2728*w)
      c2=w*(.065687+.0013218*w)/(1.+1.2008*w)
      TranW1=(1.+c1*MasW)/(1.+c2*MasW)
      TWP1=(1.+c1*AMd)/(1.+c2*AMd)
C
C     TRANSMITTANCES, Band 2
C
      TranR2=(1.-.010394*MasR)/(1.-.00011042*MasR2)
      TranG2=(1.+.27284*MasR-.00063699*MasR2)/(1.+.30306*MasR)
      TGP2=(1.+.27284*AMd-.00063699*AMd2)/(1.+.30306*AMd)
      TranO2=1.
      TranN2=1.
      TNP2=1.
      
      TranW2=1.
      TWP2=1.
      if(w.le.0.)goto 708
      c1=w*(19.566-1.6506*w+1.0672*w2)/(1.+5.4248*w+1.6005*w2)
      c2=w*(.50158-.14732*w+.047584*w2)/(1.+1.1811*w+1.0699*w2)
      c3=w*(21.286-.39232*w+1.2692*w2)/(1.+4.8318*w+1.412*w2)
      c4=w*(.70992-.23155*w+.096514*w2)/(1.+.44907*w+.75425*w2)
      TranW2=(1.+c1*MasW+c2*MasW2)/(1.+c3*MasW+c4*MasW2)
      TWP2=(1.+c1*AMd+c2*AMd2)/(1.+c3*AMd+c4*AMd2)
 708  continue
c     
c     Coefficients of Alpha
c     
      ALFA12=ALFA1*ALFA1
      ALFA22=ALFA2*ALFA2
      if(abs(alfa1-1.3).gt.1e-3)goto 10
      Al0(1)=0.544474
      Al1(1)=0.00877874
      Al2(1)=0.196771
      Al3(1)=0.294559
      goto 20
 10   continue
      Al0(1)=.57664-.024743*ALFA1
      Al1(1)=(.093942-.2269*ALFA1+.12848*ALFA12)/(1.+.6418*alfa1)
      Al2(1)=(-.093819+.36668*ALFA1-.12775*ALFA12)/(1.-.11651*alfa1)
      Al3(1)=alfa1*(.15232-.087214*ALFA1+.012664*ALFA12)/
     1 (1.-.90454*alfa1+.26167*alfa12)
 20   continue
      if(abs(alfa2-1.3).gt.1e-3)goto 11
      Al0(2)=1.038076
      Al1(2)=-0.105559
      Al2(2)=0.0643067
      Al3(2)=-0.109243
      goto 21
 11   continue    
      Al0(2)=(1.183-.022989*alfa2+.020829*alfa22)/(1.+.11133*alfa2)
      Al1(2)=(-.50003-.18329*alfa2+.23835*alfa22)/(1.+1.6756*alfa2)
      Al2(2)=(-.50001+1.1414*alfa2+.0083589*alfa22)/(1.+11.168*alfa2)
      Al3(2)=(-.70003-.73587*alfa2+.51509*alfa22)/(1.+4.7665*alfa2)
 21   continue
C
C     TURBIDITY FUNCTIONS and EQUIVALENT Aerosol WAVELENGTHS
C
      BETAM1(1)=MasA*BETA1
      BETAM1(2)=AMd*BETA1
      BETAM2(1)=MasA*BETA2
      BETAM2(2)=AMd*BETA2
C           
      DO 50 J=1,2             
      ALM1(J)=LOG(1.+BETAM1(J))
      ALM2(J)=LOG(1.+BETAM2(J))
      ALM12(j)=ALM1(j)*ALM1(j)
      ALM22(j)=ALM2(j)*ALM2(j)
      WVLTH(1,j)=(Al0(1)+Al1(1)*ALM1(J)+Al2(1)*ALM12(j))/
     1 (1.+Al3(1)*ALM12(j))
      WVLTH(2,J)=(Al0(2)+Al1(2)*ALM2(J)+Al2(2)*ALM22(j))/
     1 (1.+Al3(2)*ALM2(j))
 50   continue
C           
C AEROSOL DIRECTIONAL FACTOR BA
C           
      BA=1.
      IF(BETA.GT.0.)BA=1.-EXP(-0.6931-1.8326*zcos)
c
c     Rayleigh Multiple-scattering correction 
c
      BR1=.5*(.89013-.0049558*AM+.000045721*AM2) 
      BR2=0.5
C
C AEROSOL TRANSMITTANCES AND ALBEDOS FOR THE 2 BANDS
C   
      TranA1=1.
      TranA2=1.
      TranD1=1.
      TranD2=1.
      tauA1=0.
      tauA2=0.
      Ratio1=1.
      Ratio2=1.
      IF(BETA.LE.0.)GOTO 70
      TranA1=MAX(EXP(-BETAM1(1)/(WVLTH(1,1)**ALFA1)),0.00001)
      IF(TranA1.GE.1.)TranA1=1.
      tauA1=-LOG(TranA1)/MasA     
      TranD1=EXP(-MasA*tauA1*PIAR1)      
      TranA2=MAX(EXP(-BETAM2(1)/(WVLTH(2,1)**ALFA2)),0.00001)
      IF(TranA2.GE.1.)TranA2=1.
      tauA2=-LOG(TranA2)/MasA 
      TranD2=exp(-MasA*tauA2*PIAR2) 
      g0=(3.715+.368*MasA+.036294*MasA2)/(1.+.0009391*MasA2)
      g1=(-.164-.72567*MasA+.20701*MasA2)/(1.+.0019012*MasA2)
      g2=(-.052288+.31902*MasA+.17871*MasA2)/(1.+.0069592*MasA2)
      Ratio1=(g0+g1*tauA1)/(1.+g2*tauA1)
      h0=(3.4352+.65267*MasA+.00034328*MasA2)/(1.+.034388*MasA15)
      h1=(1.231-1.63853*MasA+.20667*MasA2)/(1.+.1451*MasA15)
      h2=(.8889-.55063*MasA+.50152*MasA2)/(1.+.14865*MasA15)
      Ratio2=(h0+h1*tauA2)/(1.+h2*tauA2)
 70   CONTINUE
C
C     Broadband SKY ALBEDO (for backscattered radiation)
Cc
      Tau550=beta/0.55**alfa2
      Rsky1=(.13363+.00077358*alfa1+beta1*(.37567+.22946*alfa1)/
     1 (1.-.10832*alfa1))/(1.+beta1*(.84057+.68683*alfa1)/
     2 (1.-.08158*alfa1))
      Rsky2=(.010191+.00085547*alfa2+beta2*(.14618+.062758*alfa2)/
     1 (1.-.19402*alfa2))/(1.+beta2*(.58101+.17426*alfa2)/
     2 (1.-.17586*alfa2))
C
C      BAND1 : VISIBLE PART OF THE SPECTRUM (0.29-0.7 m)
C
      ESC1=En0*F1
      EHOR1=ESC1*zcos
      Tran1=TranO1*TranG1 
      Tranp1=TNP1*TWP1        
      ENORM1=Tran1*TranN1*TranW1*TranR1*TranA1*ESC1  
      EDIFR1=BR1*Tran1*Tranp1*(1.-TranR1)*(TranA1**.25)*EHOR1      
      EDIFA1=BA*Tranp1*(1.-TranD1**.25)*Tran1*(TranR1*Ratio1)*EHOR1 
      EDIF10=EDIFR1+EDIFA1 
      EDIR1=ENORM1*zcos             
C           
C     BAND2 : INFRA-RED PART OF THE SPECTRUM (0.7-4 m)
C           
      ESC2=En0*F2
      EHOR2=ESC2*zcos
      Tran2=TranO2*TranG2 
      Tranp2=TNP2*TWP2        
      ENORM2=Tran2*TranN2*TranW2*TranR2*TranA2*ESC2  
      EDIFR2=BR2*Tran2*Tranp2*(1.-TranR2)*(TranA2**.25)*EHOR2      
      EDIFA2=BA*Tranp2*(1.-TranD2**.25)*Tran2*(TranR2*Ratio2)*EHOR2     
      EDIF20=EDIFR2+EDIFA2 
      EDIR2=ENORM2*zcos
      
c      if(z.lt.50.)write(15,199)trana2,trand2,trand2**.25
c 199  format(10(f12.5,1x))
        
C
C     WHOLE SPECTRUM RESULTS
C
      Ed=Edif10+Edif20
c      Ebh=Edir1+Edir2 
      R1=RSky1*Rog            
      EDIFD1=R1*(EDIR1+EDIF10)/(1.-R1)    
      EDIF1=EDIF10+EDIFD1           
      ETOT1=EDIF1+EDIR1 
      R2=RSky2*Rog            
      EDIFD2=R2*(EDIR2+EDIF20)/(1.-R2)    
      EDIF2=EDIF20+EDIFD2           
      ETOT2=EDIF2+EDIR2 
      Ebn=ENORM1+ENORM2   
            
      Ed=EDIF1+EDIF2         
      E=ETOT1+ETOT2  
c 
      RETURN
      END
c
c
c=======================================================================
c
c
c     
      Subroutine Rodger (Ebn,Ed,E,En0,Esc,presc,z,zcos,w,taua)
c
c     John Page's team method based on Unsworth-Monteith turbidity.
c     Described in  Rodgers et al., "The development of an interactive 
c   computer program SUN1 for the calculation of solar irradiances and 
c   daily irradiations on horizontal surfaces on cloudless days for given
c   conditions of sky clarity and atmsopheric waer content", Rep. BS28,
c   Univ. of Sheffield (1978).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     taua: Unsworth-Monteith broadband turbidity coeff.
c     z: zenith angle (deg.)
c
c
c---------------------------------------------------------------------
c
      real a(3),bi0(3),bi1(3),bi2(3),ci0(8),ci1(6),ap0(8),ap1(7)
      data bi0/-.0642111,-.0046883,.000844097/
      data bi1/-.00801046,.00220414, -.000191442/
      data bi2/.000153069,-.0000429818,.00000374176/
      data ci0/0.,47.382,29.671,-15.8621,4.3463,-.57764,
     1 .03472,-.0007362/
      data ci1/297.,1.8313,-3.7082,4.1233,-.6409,.02855/
c
      zcos2=zcos*zcos
      zcos3=zcos2*zcos
      zcos4=zcos3*zcos
      zcos5=zcos4*zcos
      zcos6=zcos5*zcos
      
      amR=1/zcos  
      if(z.gt.80.)amR=exp(3.67985-24.4465*zcos+154.017*zcos2
     1  -742.181*zcos3+2263.36*zcos4-3804.89*zcos5+2661.05*zcos6)
      am=amR*presc
      am2=am*am
      am3=am2*am
      wR=10.*w
      wR2=wR*wR
      h10=(90.-z)/10.
c     
      a0=-.129641+.00412828*wR-.000112096*wR2
      do 1 i=1,3
      a(i)=bi0(i)+bi1(i)*wR+bi2(i)*wR2
 1    continue
c
      ap1(1)=0.
      do 2 i=2,7
      ap1(i)=ap1(i-1)+ci1(i-1)*h10**(i-2)
 2    continue
      ap11=1e-3*ap1(7)
      ap0(1)=0.
      do 3 i=2,8
      ap0(i)=ap0(i-1)+ci0(i)*h10**(i-1)
 3    continue
      ap00=2.+ap0(8)
c
      Ew=exp(a0+a(1)*am+a(2)*am2+a(3)*am3)
      Ebn=Min(En0,En0*Ew*exp(-am*tauA))
      Ed=(En0/Esc)*ap00-ap11*Ebn*zcos
      E=Ebn*zcos+Ed
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine RSC (Ebn,Ed,E,En0,rhog,presc,cosZ,amk,w,beta)
c
c     Composite model described by Carroll (Solar Energy, 35: 105-118, 1985)
c     based on algorithms by Robinson and Sellers.
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     w: precip. water (cm)
c     beta: Angstrom turbidity coeff
c     rog: ground albedo
c     cosz: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      amkp=amk*presc
      am2=amkp*amkp
      am3=am2*amkp
c     
      Tr=1./10.**(.054-.0088*amkp+.00108*am2-.000051*am3)
      Tw=1./10.**((.04*w**.1+.01*w)*amkp)
      Tg=1./10.**(.02*amkp)
      Ta=1./10.**(.666*amkp*beta)
      Tabs=Tg*Tw
      Ebn=En0*Tabs*Tr*Ta
      Eb=Ebn*cosZ
      Ew=En0*cosZ*Tabs
      Edf=(.5+.3*beta)*(cosZ**.33333)*(Ew-Eb)
      f3=(.1*rhog+.36*beta*(rhog-.25)-.02)*(Eb+Edf)
      Ed=Edf+f3
      E=Eb+Ed
      
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine Santam (Ebn,Ed,E,En0,E0,rog,presc,uo,cosZ,amky,w)
c
c
c     Santamouris model
c!!!
c     Transmittances are from Psiloglou et al. (Pageoph 2000), see model #29,
c     per Santamouris advice (pers. comm., 2002).
c     I also added the pressure correction where needed for consistency.
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25 is the pressure correction
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     rog: ground albedo
c     amky: Kasten & Young's air mass function
c
c
c---------------------------------------------------------------------
c
      amkyp=amky*presc
c     
      Trr=exp(-.1128*(amkyp**.8346)*(.9341-amkyp**.9868+.9391*amkyp))
      ako3=amky*uo      
      Tro=1.-.2554*ako3/((1.+6107.26*ako3)**.204+.471*ako3)
      akco2=amkyp*330.
      TrCO2=1.-.0721*akco2/((1.+377.89*akco2)**.5855+3.1709*akco2)
      akco=amkyp*.075
      TrCO=1.-.0062*akco/((1.+243.67*akco)**.4246+1.7222*akco)
      akch4=amkyp*1.6
      TrCH4=1.-.0192*akch4/((1.+166.095*akch4)**.4221+.7186*akch4)
      akn2o=amkyp*.28
      TrN2O=1.-.0326*akn2o/((1.+107.413*akn2o)**.5501+.9093*akn2o)
      ako2=amkyp*2.095e+05
      TrO2=1.-.0003*ako2/((1.+476.934*ako2)**.4892+.1261*ako2)
      Trmg=TrCO2*TrCO*TrCH4*TrN2O*TrO2
      akw=amky*w
      Trw=1.-3.014*akw/((1.+119.3*akw)**.6440+5.814*akw)
c     aka=amky
      Tra=1.-.2579*amky/((1./(1.+.04001*amky)**2.8451)+.2748*amky)
      Ebn=En0*Trr*Trmg*Tro*Trw*Tra
      Eb=Ebn*cosZ
      Traa=1.0-1.405e-3*amky-9.013e-5*amky**2+2.2e-6*amky**3
      Tras=Tra/Traa
      ros=.08503
      Ed1=E0*Trmg*Tro*Trw*Traa*(1.-Tras*Trr)/2.
      Ed2=(Ed1+Eb)*rog*ros/(1.-rog*ros)
      Ed=Ed1+Ed2
      E=Eb+Ed
c
      Return
      End
c
c
c=======================================================================
c
c
c
      Subroutine Schulz (Ebn,Ed,E,cosZ)
c
c     Schulze model (Agric. Meteorol., 16: 85-101, 1976),
c    based on Archer (1964).
c
c
c     Necessary inputs:
c
c     cosz: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      Ebn=1127.*(.808**(1./cosZ))
      Ed=94.23*(cosZ**.5)
      E=Ebn*cosZ+Ed
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine Sharma (Ebn,Ed,E,En0,cosZ)
c
c     Sharma & Pal empirical model (Solar Energy, 9: 183-192, 1965).
c!!!
c     They used a solar constant of 2 cal/cm2/min.
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     cosz: cosine of zenith angle
c
c
c---------------------------------------------------------------------
c
      Ebn=1.842*(En0/2.)*cosZ/(.3135+cosZ)
      E=4.5*(En0/(2.*60.))+1.071*Ebn*cosZ
      Ed=E-Ebn*cosZ
c      
      return
      end
c
c
c=======================================================================
c
c
c
      Subroutine Wesely (Ebn,Ed,E,cosZ,amk,presc)
c
c     Wesely & Lipschutz model (Atmos. Envir., 10: 981-987, 1976).
c
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25 is the pressure correction
c     amk: Kasten's air mass function
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      Ebn=(1050./am)*exp(-.1*am)
      Ed=.9*Ebn*(1.-exp(-.1*am))
      E=Ebn*cosZ+Ed
c      
      return
      end
c
c
c=======================================================================
c
c
C
      Subroutine Yang (Ebn,Ed,E,En0,E0,beta,presc,uo,w,amk)
c
c
c     Broadband transmittances and irradiance
c     according to paper by Yang et al. (Solar Energy, 70: 13-22, 2001).
c!!!
c     I made corrections based on personal communications with Yang (2002),
c     see Gueymard (Solar Energy, 74: 355-379, 2003).
c     Revised version eventually published by Yang (Agr. For. Met., 
c     137: 43-55, 2006).
c
c     Necessary inputs:
c
c     En0: extraterrestrial irradiance (W/m2), ie the solar constant
c           corrected for the actual sun-earth distance
c     presc: p/1013.25
c     uo: reduced ozone vertical pathlength (atm-cm)
c     w: precip. water (cm)
c     beta: Angstrom turbidity coeff.
c     cosZ: cos of zenith angle Z
c     amk: Kasten's air mass, i.e.
c                 amk=1./(cosZ+.15/(93.885-Z)**1.253)
c
c
c---------------------------------------------------------------------
c
      am=amk*presc
      am2=am*am
      am3=am2*am
      
      xko3=.0365/(amk*uo)**.2864
c     cw=0.
c     if(w.gt.0.)cw=-log(min(1.0,-0.036*log(amk*w)+0.909))
      cg=.0117*am**.3139
      wvlr=.547+.014*am-.00038*am2+.0000046*am3
c     
      TrrY=exp(-.008735*am/(wvlr**4.08))
      TroY=exp(-amk*uo*xko3)
      TrumY=exp(-cg)
      TrwY=Min(1.,.909-.036*log(amk*w))
C
      amb=amk*beta
      wvla=.6777+.1464*amb-.00626*amb*amb
      tauA=beta/(wvla**1.3)
      TraY=exp(-amk*tauA)
      Tn=Max(0.,TrrY*TroY*TrumY*TrwY*TraY-.013)
      Td=Max(0.,.5*(TroY*TrumY*TrwY*(1.-TraY*TrrY)+.013))
      Ebn=En0*Tn
      Eb=E0*Tn
      Ed=E0*Td
      E=Eb+Ed
c                 
      Return
      End
C
c
c=======================================================================
C
      Function Nday(Year,Month,Day)
c
c     To obtain the calendar day from (Month, Day of month)
c
      Integer Day,Year
c
      Leap1=Mod(Year,4)
      Leap2=Mod(Year,100)
      Leap3=Mod(Year,400)
      Leap=Leap1-Leap2+Leap3
      Xdm=32.8
      If(Month.le.2)Xdm=30.6
      If(Leap.eq.0.and.Month.gt.2)Xdm=31.8
      Cday=Int(30.6*Month + Day - Xdm + 0.5)
      Nday=Ifix(Cday)
      Return
      End
c     
c
c=======================================================================
c
      Subroutine SunPSA(dHour,dLat,dLong,Z,Radius,Decli,
     1 P,TdegK,Year,Month,Day)
c
c     'PSA Algorithm' by Blanco-Muriel et al. (Solar Energy 2001)
c     Translated to Fortran from their original code in C++
c
c***  Inputs  ***
c
c     Year
c     Month
c     Day
c     dHour: decimal Hour (Universal Time, UT)
c     dLat: decimal Latitude (deg., positive North)
c     dLong: decimal Longitude (deg., positive East)
c
      Integer Year,Month,Day
      Double Precision Julian,pi,twopi,rad,LMST,GMST,dRLat,HourAngl,
     1 cosHA,Elapsd,dX,dY,EclipL,Paralx,Decli,dLong,dLat,AUnit,Radius,
     2 EclipO,RightA,Anomly,Omega,sinELg,SunLng,zenith,azimu,cosLat,
     3 sinLat
c
c
c      Declaration of some constants
c
      pi=3.14159265358979323846
      twopi=2.*pi
      rad=pi/180.
      Radius=6371.01
      AUnit=149597890.0
c
c     Calculate current Julian Day
c
      liAux1 =(Month-14)/12
      LiAux2=(1461*(Year + 4800 + liAux1))/4 
     1 + (367*(Month- 2-12*liAux1))/12
     2 - (3*((Year + 4900+ liAux1)/100))/4+Day-32075
      Julian=Float(LiAux2)-0.5+dHour/24.
c
c     Calculate difference in days between the current Julian Day 
c     and JD 2451545.0, which is noon 1 January 2000 Universal Time 
c
      Elapsd = Julian-2451545.0
c 
c     Calculate ecliptic coordinates (ecliptic longitude and obliquity of the 
c     ecliptic in radians) but without limiting the angle to be less than 2*Pi 
c     (i.e., the result may be greater than 2*Pi)
c
      Omega=2.1429-0.0010394594*Elapsd
      SunLng = 4.8950630+ 0.017202791698*Elapsd
      Anomly = 6.2400600+ 0.0172019699*Elapsd
      EclipL = SunLng + 0.03341607*sin(Anomly) + 0.00034894*
     1 sin(2.*Anomly)-0.0001134-0.0000203*sin(Omega)
      EclipO = 0.4090928 - 6.2140e-9*Elapsd+0.0000396*cos(Omega)
c 
c     Calculate celestial coordinates (right ascension and declination) in radians 
c     but without limiting the angle to be less than 2*Pi (i.e., the result may be 
c     greater than 2*Pi)
c
      SinELg= sin(EclipL)
      dY = cos(EclipO) * SinELg
      dX = cos(EclipL)
      RightA = atan2(dY,dX)
      if(RightA.lt.0.0) RightA = RightA + twopi
      Decli = asin(sin(EclipO)*SinELg)
c 
c     Calculate local coordinates (azimuth and zenith angle) in degrees
c           
      GMST = 6.6974243242 + 0.0657098283*Elapsd + dHour
      LMST = (GMST*15. + dLong)*rad
      HourAngl = LMST - RightA
      dRLat = dLat*rad
      cosLat = cos(dRLat)
      sinLat = sin(dRLat)
      cosHA= cos(HourAngl)
      Zenith = acos(cosLat*cosHA*cos(Decli) + sin(Decli)*sinLat)
      
c     write(15,911)float(year),float(month),float(day),dHour,Julian,LMST,
c     1 ZENITH/rad
 911  format(9(f16.6,1x))
 
      dY = -sin(HourAngl)
      dX = tan(Decli)*cosLat - sinLat*cosHA
      Azimu = atan2(dY, dX)
      if (Azimu.lt. 0.0)Azimu = Azimu + twopi
      Azimu = Azimu/rad
c
c     Parallax Correction
c
      Paralx=(Radius/AUnit)*sin(Zenith)
      Zenith=(Zenith + Paralx)/rad
c
c     Sun-Earth actual distance in AU (from Michalsky's paper)
c
      Radius=1.00014-.01671*cos(Anomly)-.00014*cos(2.*Anomly)
c
c     Equation of Time (in min, from Michalsky's paper)
c
      RightA=RightA/rad
      SunLng=SunLng/rad
      xsun=-aint(abs(SunLng)/360.)
      if(Sunlng.lt.0.)xsun=-xsun+1.
      SunLng=SunLng+xsun*360.
      EOT=(SunLng-RightA)*4.
C
C     REFRACTION CORRECTION FOR ACTUAL ATMOSPHERIC CONDITIONS (P,T)
C
      ELD=90.-Zenith
      ELD2=ELD*ELD
      REFR=0.
      PT=P/TdegK
      IF(ELD.LT.15.0.AND.ELD.GE.-2.5)REFR=PT*(.1594+.0196*ELD+
     # 2E-5*ELD2)/(1.+.505*ELD+.0845*ELD2)
      IF(ELD.GE.15.0.AND.ELD.LT.90.)REFR=.00452*PT/TAN(ELD*rad)
      Zenith=90.-(ELD+REFR)
c
c     Declination in degrees
c
      Decli=Decli/rad
      z=real(zenith)
      return
      end
c
