c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######                    RADIATION                         ######
c     ######                                                      ######
c     ######                 Developed by Kun Yang                ######
c     ######     River and Environmental Engineering Laboratory   ######
c     ######                University of Tokyo                   ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
      PROGRAM RADIATION
c
c#######################################################################
c
c     PURPOSE:
c
c     This program is an example using a subroutine to calculate 
c     clear-sky solar radiation 
c
c     Please cite the references:
c     Yang, K., Huang, G.-W., & Tamai, N. (2001). A hybrid model for 
c           estimating global solar radiation. Solar Energy, 70, 13-22.
c     Yang, K., Koike, T., and Ye, B. (2006). Improving estimation of 
c           hourly, daily, and monthly solar radiation by importing global  
c           data sets. Agricultural and Forest Meteorology, 137:43-55.
c
c#######################################################################
c      
c
c     AUTHOR: Kun Yang
c     17/08/2005 
c                                                                               
c     MODIFICATION HISTORY:
c
c#######################################################################
c
c
c#######################################################################
c
c     Variable Declarations.
c
c#######################################################################
c
      implicit none 
c
c#######################################################################
c
c     temporary variables
c
c#######################################################################
c
      character*200 infile
      character*200 outfile

c     input data record
      integer no
      real    lon0       ! the longitude where the standard time is defined. 
                         ! If input time is UTC, lon0 = 0
                         ! If input time is BST, lon0 = 120
                         ! If input time is JST, lon0 = 135
      real    lon        ! longitude of each site (deg.)
                         ! > 0 in East Hemesphere
                         ! < 0 in West Hemesphere
      real    lat        ! latitdue of each site (deg.)
                         ! > 0 in North Hemesphere
                         ! < 0 in South Hemesphere
      real    alt        ! Local surface elevation (m)
      integer	yy,mm,dd   ! calendar year:month:day		
      integer	hh,mn,ss   ! time     hour:minute:second		
      real	pa         ! air pressure (Pa)
      real	ta         ! air temperature (K)
      real	rh         ! realtive humidity (%)
c    end data record

c     output
      real    rad0       ! instantaneous TOA solar irradiance (W m^-2)
      real    radb       ! instantaneous clear-sky solar beam (W m^-2)
      real    radd       ! instantaneous clear-sky solar diffuse (W m^-2)
      real    rad        ! instantaneous clear-sky solar irradiance (W m^-2)

c     temp   
      integer k      
c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
!
!#############################################################################
!
!  Set site information
!
!#############################################################################
!
!    input file
      OPEN(1, file = 'Lhasa.input.txt', status = 'old')
!    output file
      OPEN(2, file = 'Lhasa.output.txt', status = 'unknown')
      write(2,'(A5,3(1x,A7))')'k','rad','radb','radd'

      read(1,*)
      k =0 
      DO WHILE (.NOT. EOF(1))
         k = k + 1
         read(1,*)no,lon0,lon,lat,alt,yy,mm,dd,hh,mn,ss,pa,ta,rh

         CALL Radclr(rad,radb,radd,lon0,lon,lat,alt,
     :               yy,mm,dd,hh,mn,ss,pa,ta,rh)

         write(2,'(i5,4(1x,f7.1))')k,rad,radb,radd
      END DO
         
      END PROGRAM RADIATION

c
c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######               SUBROUTINE  Radclr                     ######
c     ######                                                      ######
c     ######                     Developed by Kun Yang            ######
c     ######     River and Environmental Engineering Laboratory   ######
c     ######                University of Tokyo                   ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
c
      SUBROUTINE  Radclr(rad,radb,radd,lon0,lon,lat,alt,
     :                   yy,mm,dd,hh,mn,ss,pa,ta,rh)
c
c#######################################################################
c
c     PURPOSE:
c
c     This subroutine is used to calculate clear-sky downward solar radiation
c
c     Please cite the references:
c     Yang, K., Huang, G.-W., & Tamai, N. (2001). A hybrid model for 
c           estimating global solar radiation. Solar Energy, 70, 13-22.
c     Yang, K., Koike, T., and Ye, B. (2006). Improving estimation of 
c           hourly, daily, and monthly solar radiation by importing global  
c           data sets. Agricultural and Forest Meteorology, 137:43-55.
c 
c#######################################################################
c      
c
c     AUTHOR: Kun Yang
c     17/08/2005 
c                                                                               
c     MODIFICATION HISTORY:
c
c#######################################################################
c
c
c#######################################################################
c
c     Variable Declarations.
c
c#######################################################################
c
      implicit none
c
c  INPUT:
c
      real    lon0       ! the longitude where the standard time is defined. 
                         ! If input time is UTC, lon0 = 0
                         ! If input time is BST, lon0 = 120
                         ! If input time is JST, lon0 = 135
      real    lon        ! longitude of each site (deg.)
                         ! > 0 in East Hemesphere
                         ! < 0 in West Hemesphere
      real    lat        ! latitdue of each site (deg.)
                         ! > 0 in North Hemesphere
                         ! < 0 in South Hemesphere
      real    alt        ! Local surface elevation (m)
      integer	yy,mm,dd   !calendar year:month:day		
      integer	hh,mn,ss   !standard time hour:minute:second in long0
      real    pa         ! surface air pressure (Pa)
      real    ta         ! air temperature (K)
      real    rh         ! relative humidity
c
c  OUTPUT:
c
      real    radb       ! instantaneous clear-sky solar beam (W m^-2)
      real    radd       ! instantaneous clear-sky solar diffuse (W m^-2)
      real    rad        ! instantaneous clear-sky solar irradiance (W m^-2)
c
c  Temporary:
c
      integer jday   ! julian day
      real hsun      ! the height of the sun (90-zenith angle)
      real rad0       ! instantaneous TOA solar irradiance (W m^-2)
      real taub      ! Beam radiative transmittance
      real taud      ! Diffuse radiative transmittance
c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
      rad  = 0
      radb = 0
      radd = 0

c     Calculate TOA (top of atmosphere) solar irradiance on horizontal level
      CALL solarR0(Rad0,hsun,lon0,lon,lat,yy,mm,dd,hh,mn,ss)

      IF(Rad0.gt.0)THEN 
         CALL JULDAY( yy, mm, dd, jday )
         CALL  trans(jday,lat,lon,alt,hsun,pa,ta,rh,
     :               taub,taud)

         Radb = Rad0*taub 
         Radd = Rad0*taud
         Rad  = Radb + Radd 
      ELSE
         Rad  = 0 
         Radb = 0
         Radd = 0
      END IF

      END SUBROUTINE  Radclr

c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######                SUBROUTINE trans                      ######
c     ######                                                      ######
c     ######                     Developed by                     ######
c     ######     River and Environmental Engineering Laboratory   ######
c     ######                University of Tokyo                   ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
      SUBROUTINE  trans(jday,lat,lon,alt,hsun,pa,ta,rh,taub,taud)
c
c#######################################################################
c
c     PURPOSE:
c
c     Calculate the transmittance in clear skies
c
c#######################################################################
c      
c
c     AUTHOR: Kun Yang
c     10/11/2004 
c                                                                               
c     MODIFICATION HISTORY:
c
c#######################################################################
c
c#######################################################################
c
c     Variable Declarations.
c
c#######################################################################
c
      implicit none
c
c
c     INPUT:
c
      integer jday    ! julian day
      real lat        ! latitdue of each site (deg.)
                      ! > 0 in North Hemesphere
                      ! < 0 in South Hemesphere
      real lon        ! longitude of each site (deg.)
                      ! > 0 in East Hemesphere
                      ! < 0 in West Hemesphere
      real alt        ! Local surface elevation (m)
      real hsun       ! the height of the sun (90-zenith angle)
      real pa         ! surface air pressure (Pa)
      real ta         ! air temperature (K)
      real rh         ! relative humidity
c
c  OUTPUT:
c
      real taub     ! Beam radiation transmittance  Eq. (9a)
      real taud     ! Diffuse radiation transmittance	Eq. (9b)
c
c#######################################################################
c
c     Misc. local variables:
c
c#######################################################################
c
      real mass     ! air mass 
      real beta        !  Angstrom turbidity coefficient           
      real beta_annual ! Annual mean Angstrom turbidity coefficient           
                       ! =0.0-0.15               
      real beta_season ! Seasonal variability of turbidity coefficient 
                       ! =0.02-0.04               
      real water    ! precipitable water (cm)
      real pp0      ! pa/p0
      real loz      ! the thickness of ozone layer (cm)

      real mp       ! mass * pa/p0
      real mb       ! mass * beta
      real moz      ! mass * loz
      real mw       ! mass * water

      real koz	  ! Parameter of ozone absorption,		   Eq. (8a)
      real wv		  ! Parameter of water vapor absorption,   Eq. (8b)
      real lamr     ! Parameter of Rayleigh acttering,	   Eq. (8d)
      real lama     ! Parameter of aerosol extinction,	   Eq. (8e)
      real gas	  ! Parameter of permanent gas absorption, Eq. (8c)

      real tauoz	  ! transmittance due to ozone absorption		   Eq. (7a)
      real tauw	  ! transmittance due to water vapor absorption	   Eq. (7b)
      real taur     ! transmittance due to Rayleigh acttering		   Eq. (7c)
      real taua     ! transmittance due to aerosol extinction		   Eq. (7d)
      real taug	  ! transmittance due to permanent gas absorption  Eq. (7e)

      real tema, temb    

      real p0       ! standard atmospheric pressure (Pa) 
      parameter (p0=101300.)
      real pi
      parameter (pi=3.1415927/180)
c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
c     calculate air mass
      mass =1/(sin(hsun)+0.15*(57.3*hsun+3.885)**(-1.253))
      mass=max(0.0,mass)

c     calculate relative air mass
      pp0    = pa / p0

c     calculate turbidity
      CALL MPI_Beta(jday,lat,lon,rh,beta)      

c     calculate ozone thickness
      CALL NASA_ozone(jday,lat,lon,loz)

c     calculate precipitable water amount
      water = 0.493*rh/100/ta*exp(26.23-5416/ta) 
		
      
c     calculate transmittance

      mp  = mass * pp0
      mb  = mass * beta
      moz = mass * loz
      mw  = mass * water

      lamr = 0.5474+0.01424*mp-0.0003834*mp**2+4.59e-6*mp**3
      lama = 0.6777+0.1464*mb-0.00626*mb**2
      koz  = 0.0365*moz**(-0.2864)
      gas  = 0.0117*mp**0.3139
      wv   = -log(min(1.0,-0.036*log(mw)+0.909))

      taur = exp(-0.008735*mp*lamr**(-4.08))
      taua = exp(-mb*lama**(-1.3))
      tauoz= exp(-moz*koz)
      taug = exp(-gas)
      tauw = exp(-wv)					

      taub = max (0.0,taur*taua*tauoz*taug*tauw-0.013)
      taud = 0.5 * (tauoz*taug*tauw*(1-taur*taua)+0.013)

      RETURN
      END SUBROUTINE trans
c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######                SUBROUTINE MPI_Beta                   ######
c     ######                                                      ######
c     ######                     Developed by                     ######
c     ######     River and Environmental Engineering Laboratory   ######
c     ######                University of Tokyo                   ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
      SUBROUTINE  MPI_Beta(jday,lat,lon,rh,beta)
c
c#######################################################################
c
c     PURPOSE:
c
c     Calculate the turbidity coef. according to the dataset of 
c     Hess et al. (1998, BAMS)
c
c#######################################################################
c      
c
c     AUTHOR: Kun Yang
c     24/11/2004 
c                                                                               
c     MODIFICATION HISTORY:
c
c#######################################################################
c
c#######################################################################
c
c     Variable Declarations.
c
c#######################################################################
c
      implicit none
c
c
c     INPUT parameters:
c
      real betaw(-18:18,-36:36,8) ! turbidity for winter ! read from 'beta_mpi.txt'
      real betas(-18:18,-36:36,8) ! turbidity for summer ! read from 'beta_mpi.txt'
      real rhbeta(8)              ! Standard relative humidity for calculating 
     :                            ! turbidity, see Hess et al. (1998, BAMS)  
      common /turbidity_info/betaw,betas,rhbeta
c
c     INPUT:
c
      integer jday    ! julian day
      real lat        ! Local latitude, positive for northern hemisphere (deg.)
      real lon        ! longitude of each site (deg.)
                      ! > 0 in East Hemesphere
                      ! < 0 in West Hemesphere
      real rh         ! relative humidity
c
c  OUTPUT:
c
      real beta        !  Angstrom turbidity coefficient           
c
c#######################################################################
c
c     Misc. local variables:
c
c#######################################################################
c
      real x1,x2,y1,y2,z1,z2    

      integer i,j,k
      integer index 
      integer lati,loni   
      integer firstcall
      save firstcall
      data firstcall /1/

c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
      IF(firstcall.eq.1)THEN

         OPEN(10, file = 'beta_mpi.txt')
         READ(10,*)
         READ(10,*)((rhbeta(i)),i=1,8)
         DO WHILE(.NOT.EOF(10))
           READ(10,*)index,lati,loni,
     :              (betaw(lati/5,loni/5,i),i=1,8),
     :              (betas(lati/5,loni/5,i),i=1,8)
         END DO         
         CLOSE(10)

         DO j = -18, 18
         DO i = 1, 8
           betaw(j,36,i) = betaw(j,-36,i) ! value at -180 = value at 180 
           betaw(j,36,i) = betaw(j,-36,i)
         END DO 
         END DO 
         firstcall = 0
      END IF
!
       
      i = lat/5
      IF(lat.lt.0)THEN
         i = max(-18,i-1) 
      ELSE
         i = min(17,i) 
      END IF  

      j = lon/5
      IF(lon.lt.0)THEN
         j = max(-36,j-1) 
      ELSE
         j = min(35,j) 
      END IF  

      DO k=1,7
         IF(rh.ge.rhbeta(k).and.rh.le.rhbeta(k+1))THEN
            EXIT
         END IF
      END DO
      k = min(k,7)
          
      x1 = lat/5.0-i
      x2 = 1 - x1
      y1 = lon/5.0-j
      y2 = 1 - y1
      z1 = (rh-rhbeta(k))/(rhbeta(k+1)-rhbeta(k))
      z2 = 1 - z1

      IF(jday.ge.60.and.jday.le.240)THEN 
        beta = betas(i, j,   k  )*x2*y2*z2 + betas(i+1,j,  k  )*x1*y2*z2
     :       + betas(i, j+1, k  )*x2*y1*z2 + betas(i,  j,  k+1)*x2*y2*z1
     :       + betas(i+1,j+1,k  )*x1*y1*z2 + betas(i+1,j,  k+1)*x1*y2*z1
     :       + betas(i,  j+1,k+1)*x2*y1*z1 + betas(i+1,j+1,k+1)*x1*y1*z1
      ELSE
        beta = betaw(i, j,   k  )*x2*y2*z2 + betaw(i+1,j,  k  )*x1*y2*z2
     :       + betaw(i, j+1, k  )*x2*y1*z2 + betaw(i,  j,  k+1)*x2*y2*z1
     :       + betaw(i+1,j+1,k  )*x1*y1*z2 + betaw(i+1,j,  k+1)*x1*y2*z1
     :       + betaw(i,  j+1,k+1)*x2*y1*z1 + betaw(i+1,j+1,k+1)*x1*y1*z1
      END IF           

      RETURN 
      END SUBROUTINE MPI_Beta 

c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######                SUBROUTINE NASA_ozone                 ######
c     ######                                                      ######
c     ######                     Developed by                     ######
c     ######     River and Environmental Engineering Laboratory   ######
c     ######                University of Tokyo                   ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
      SUBROUTINE  NASA_ozone(jday,lat,lon,loz)
c
c#######################################################################
c
c     PURPOSE:
c
c     Calculate the ozone thickness according to the monthly dataset of 
c     NASA TOMS
c
c#######################################################################
c      
c
c     AUTHOR: Kun Yang
c     09/12/2004 
c                                                                               
c     MODIFICATION HISTORY:
c
c#######################################################################
c
c#######################################################################
c
c     Variable Declarations.
c
c#######################################################################
c
      implicit none
c
c
c     INPUT parameter:
c
      real  ozone_nasa(-18:18,12) ! ozone thickness derived from NASA data set   
                                  ! read from ozone_monthly_nasa.txt            

      common /ozone_info/ozone_nasa
c
c
c     INPUT:
c
      integer jday    ! julian day
      real lat        ! Local latitude, positive for northern hemisphere (deg.)
      real lon        ! longitude of each site (deg.)
                      ! > 0 in East Hemesphere
                      ! < 0 in West Hemesphere
c
c  OUTPUT:
c
      real loz        !  ozone thickness           
c
c#######################################################################
c
c     Misc. local variables:
c
c#######################################################################
c
      integer year, month, day 
      integer i,j,k
      integer index 
      integer lati,loni   
	character*100	infile
      real     lat1,lat2
      integer firstcall
      save firstcall
      data firstcall /1/
c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
      IF(firstcall.eq.1)THEN

         OPEN(10, file = 'ozone_monthly_nasa.txt')
         DO WHILE(.NOT.EOF(10))
            READ(10,*)(lat1,lat2,(ozone_nasa(j,k),k=1,12),j=18,-17,-1)
         END DO         
         CLOSE(10)

         DO k = 1, 12
            ozone_nasa(-18,k) = ozone_nasa(-17,k)
         END DO 

         firstcall = 0
      END IF

      CALL CALDAY( jday, year, month, day )

      j = lat/5
      IF(lat.lt.0)THEN
         j = max(-17,j) 
      ELSE
         j = min(18,j+1) 
      END IF  

      loz = ozone_nasa(j,month)       

      RETURN 
      END SUBROUTINE NASA_ozone 

c     ########################################################################
c     ########################################################################
c     #####                                                              #####
c     #####                                                              #####
c     #####                   SUBROUTINE solarR0                         #####
c     #####                                                              #####
c     #####                                                              #####
c     ########################################################################
c     ########################################################################

      SUBROUTINE solarR0(R0,hsun,lon0,lon,lat,yy,mm,dd,hh,mn,ss)

c#############################################################################
c
c     PURPOSE:
c	Calculate horizontal extraterrestial solar insolation (w/s)
c	
c#############################################################################
	implicit none

c	INPUT:
      real    lon0       ! the longitude where the standard time is defined. 
                         ! If input time is UTC, lon0 = 0
                         ! If input time is BST, lon0 = 120
                         ! If input time is JST, lon0 = 135
      real    lon        ! longitude of each site (deg.)
                         ! > 0 in East Hemesphere
                         ! < 0 in West Hemesphere
      real    lat        ! latitdue of each site (deg.)
                         ! > 0 in North Hemesphere
                         ! < 0 in South Hemesphere
      integer	yy,mm,dd   !calendar year:month:day		
      integer	hh,mn,ss   !standard time hour:minute:second in lon0
c
c	OUTPUT:
	real	R0	       !horizontal extraterrestial solar insolation
                         !(w/m^2)	
	real	hsun       ! the height of the sun (rad)
c
c	WORK:
	real, parameter::	i00=1353		 ! solar constant
	real, parameter::   pi=3.1415926
	integer	julian		! Julian day
	integer jday,jday0  ! day number starting with 1 on Jan.1	
	real	delta		! solar declination(rad)
	real	hrangle		! hour angle(rad)
	real	d0d2        !(a^2/r^2):earth-sun distance factor
	real    eta,ts,t	
	real	sinh	
	real	w	
c
c@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
!	Begining of executable code......
      CALL JULDAY( yy, mm, dd, jday )
      CALL JULDAY( yy, 12, 31, jday0)
	w    = 2*pi*jday/jday0

c     Calculate square distance between the earth and the sun: (d0/d)^2
	d0d2 = 1.00011+0.034221*cos(w)+0.00128*sin(w)+
     :	   0.000719*cos(2*w)+0.000077*sin(2*w)	  

c	Calculate solar declination angle delta from calendar
	delta= 0.3622133-23.24763*COS(w+0.153231)-0.3368908
     :	   *COS(2.*w+0.2070988)-0.1852646*COS(3.*w+0.6201293)
      delta= delta * pi / 180

c	Calculate daily averaged time error due to elliptic shape of earth 
c	orbit (min)
	eta  = 60.*(-0.0002786409+0.1227715*COS(w+1.498311)
     :	   -0.1654575*COS(2.*w-1.261546)-0.00535383	 
     :	   *COS(3.*w-1.1571))                   !daily averaged time error(min) 
      ts   = FLOAT(hh)+FLOAT(mn)/60.+FLOAT(ss)/3600 !convert standard time

c     convert hour and longitude difference to hour angle (rad)
	t    = 15.*(ts-12.+eta/60.)+lon-lon0 		
	hrangle=t	* pi /180		

c	Calculate the height of the sun (rad)
	sinh  = sin(lat*pi/180)*sin(delta)
     :      + cos(lat*pi/180)*cos(delta)*cos(hrangle)
	if(sinh.le.0)then	
		hsun = 0    	        ! rad
	else	
		hsun = asin(sinh)       ! rad
	end if

c	Calculate TOA solar radiation at horizontallevel (W m^-2) 

	R0 = i00 * d0d2 * sin(hsun)

      RETURN
      END	SUBROUTINE solarR0



c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######                SUBROUTINE JULDAY                     ######
c     ######                                                      ######
c     ######                     Developed by                     ######
c     ######     Center for Analysis and Prediction of Storms     ######
c     ######                University of Oklahoma                ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
      SUBROUTINE JULDAY( year, month, day, jday )
c
c#######################################################################
c
c     PURPOSE:
c
c     Compute Julian day from year, month, and day
c
c     Start from 1 (Jan. 1) to 365, or 366 for leap year (Dec. 31)
c
c     The rule is that a year will be a leap year if
c
c       the year can be divided by 400, or
c       the year can by divided by 4, but not by 100
c
c     Form this rule year 1972, 1984, 1996, and 2000 are leap years,
c     but 1700, 1800 and 1900 are not.
c
c#######################################################################
c
c     AUTHOR: Yuhe Liu
c     07/29/93
c
c     MODIFICATIONS:
c
c     05/06/1998 (Yuhe Liu)
c     Corrected the leap year calculation.
c
c#######################################################################
c
c     INPUT:
c
c       year       Reference calendar year
c       month      Reference monthe of the year
c       day        Reference day of the month
c
c    OUTPUT:
c
c       jday       Julian day, start from 1 -- Jan. 1 to 365 -- Dec. 31
c
c#######################################################################
c
      implicit none
 
      integer year, month, day, jday
      integer lpyear, lp

      integer mndys(12)     ! Day numbers for each month
      DATA mndys/0,31,59,90,120,151,181,212,243,273,304,334/
c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
      IF ( mod(year,400).eq.0 .or. 
     :     (mod(year,4).eq.0 .and. mod(year,100).ne.0 ) ) THEN
        lpyear = 1
      ELSE
        lpyear = 0
      ENDIF

      lp = 0
      IF ( month.gt.2 ) lp = lpyear

      jday = mndys(month) + day + lp
 
      RETURN
      END

c
c     ##################################################################
c     ##################################################################
c     ######                                                      ######
c     ######                SUBROUTINE CALDAY                     ######
c     ######                                                      ######
c     ######                     Developed by                     ######
c     ######     Center for Analysis and Prediction of Storms     ######
c     ######                University of Oklahoma                ######
c     ######                                                      ######
c     ##################################################################
c     ##################################################################
c
      SUBROUTINE CALDAY( jday, year, month, day )
c
c#######################################################################
c
c     PURPOSE:
c
c     Computes the month and day from Julian day. 
c
c     Start from Jan. 1 -- day 1 to Dec. 31 -- day 365, or 366 for leap.
c
c#######################################################################
c
c     AUTHOR: Yuhe Liu
c     07/29/93
c
c     MODIFICATIONS:
c
c     05/06/1998 (Yuhe Liu)
c     Corrected the leap year calculation.
c
c#######################################################################
c
c     INPUT:
c
c       year       Calendar year
c       jday       Julian day, start from 1 -- Jan. 1 to 365 -- Dec. 31
c
c    OUTPUT:
c
c       month      Monthe of the year
c       day        Day of the month
c
c#######################################################################
c
      implicit none
 
      integer jday
      integer year, month, day

      integer i, lp

      integer mndys(12)     ! Day numbers for each month
      DATA mndys/0,31,59,90,120,151,181,212,243,273,304,334/
c
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
C
C     Beginning of executable code...
C
C@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
c
      IF ( mod(year,400).eq.0 .or.
     :     (mod(year,4).eq.0 .and. mod(year,100).ne.0) ) THEN
        lp = 1
      ELSE
        lp = 0
      ENDIF

      DO 100 i = 1, 2
        IF ( jday .gt. mndys(i) ) THEN
          month = i
        ENDIF
100   CONTINUE

      DO 200 i = 3, 12
        IF ( jday .gt. mndys(i)+lp ) THEN
          month = i
        ENDIF
200   CONTINUE

      day = jday - mndys(month)
      IF ( month .gt. 2 ) day = day - lp
c
      RETURN
      END
