
###################################################################################
#
# SCRIPT : pdf.py
#
# PURPOSE: Calculates the probability density function, conditioned to the optical
#          air mass, of the clearness index, the diffuse fraction and the direct
#          fraction.
#
# INPUT PARAMETERS :
#         -i : ascii file with the data column-organized. The columns
#              order have to be: year, month, day, hour, minute and global
#              radiation
#         -o : text file with the fitting parameters of the Boltzmann
#              model given by the equation (32), in Section 7.1.2, for
#              each optical mass (1.0, 1.5, 2.0, 2.5, 3.0, 3.5)
#         -L : local standard meridian of the station site needed when the data are
#              provided using local time. See Iqbal, M., 1983, 'An introduction to
#              solar radiation', Canada: Academic Press, Section 1.4
#         -g : if passed, the histograms and curve fittings will be shown
#
# OUTPUTS:
#         Text file with the coefficients of the bimodal Boltzmann model and/or
#         the histograms and curve fittings
#
# AUTHORS:
#         Joaquin Tovar-Pescador and Jose A. Ruiz-Arias
#         University of Jaen, Spain
#         jtovar@ujaen.es
#
###################################################################################

import sys,datetime,warnings,time,operator

from pylab import *
from scipy import optimize
from scipy.stats import relfreq
from calendar import isleap
from math import radians,degrees
from optparse import OptionParser

######################################
# F U N C T I O N   U T I L I T I E S
######################################
def filter(data,value,test='<>',column=0):
    out=[]
    if test == '==':
        for i in xrange(len(data)):
            if data[i,column]==value: out.append(data[i,:])
    elif test == '<>' or test == '!=':
        for i in xrange(len(data)):
            if data[i,column]!=value: out.append(data[i,:])
    elif test == '<':
        for i in xrange(len(data)):
            if data[i,column]<value: out.append(data[i,:])
    elif test == '>':
        for i in xrange(len(data)):
            if data[i,column]>value: out.append(data[i,:])
    elif test == '<=':
        for i in xrange(len(data)):
            if data[i,column]<=value: out.append(data[i,:])
    elif test == '>=':
        for i in xrange(len(data)):
            if data[i,column]>=value: out.append(data[i,:])
    else:
        raise TypeError('Unknown test input')
    return array(out)   

def boltzmann_model_kt(p,kt):
    """
    Bimodal Boltzmann Model for clearness index
    Inputs :
        p : list with the six coefficients of the model (see equation 32)
        kt: clearness index
    Outputs:
        Bimodal Boltzmann model value for clearness index
    """
    A1,lambda1,kt01,lambda2,kt02=p
    return A1*((lambda1*exp((kt-kt01)*lambda1))/power(1+exp((kt-kt01)*lambda1),2))+(1.0-A1)*((lambda2*exp((kt-kt02)*lambda2))/power(1+exp((kt-kt02)*lambda2),2))

def guesskt(m):
    """
    Generates and returns a list with candidate coefficients to initialize
    the fitting process
    Inputs :
        m: optical air mass
    Outputs:
        Coefficients for the bimodal Boltzmann model for clearness index as
        function of the optical air mass for Armilla (Granada, Spain).
        See Tovar, J., Olmo, F.J., Alados-Arboledas, L., 1998, 'One-minute global
        irradiance probability density distributions conditioned to the optical
        air mass', Solar Energy, 62, 387-393
    """
    A1=0.699+0.1217*power(m,-2.1416)
    lambda1=91.375-40.092*m+6.489*m*m
    kt01=0.763-0.0152*m-0.012*m*m
    lambda2=6.737+1.248*m+0.4246*m*m
    kt02=0.469-0.0954*m+0.01*m*m
    return A1,lambda1,kt01,lambda2,kt02

def boltzmann_model_kd(p,kd):
    """
    Bimodal Boltzmann Model for diffuse fraction
    Inputs :
        p : list with the six coefficients of the model (see equation 45)
        kd: diffuse fraction
    Outputs:
        Boltzmann model value for diffuse fraction
    """
    A,lambdad,kd0,beta=p
    return A*((lambdad*exp((kd-kd0)*lambdad))/power(1+exp((kd-kd0)*(lambdad+beta)),2))

def guesskd(m):
    """
    Generates and returns a list with candidate coefficients to initialize
    the fitting process
    Inputs :
        m: optical air mass
    Outputs:
        Coefficients for the bimodal Boltzmann model for diffuse fraction as
        function of the optical air mass for Armilla (Granada, Spain).
        See Tovar, J., Olmo, F.J., Alados-Arboledas, L., 1998, 'One-minute kd and
        kb probability density distributions conditioned to the optical air mass',
        Solar Energy, 62, 387-393
    """
    A=0.07062-0.07609*m+0.02989*m*m
    lambdad=538.69-152.34*m
    kd0=0.0248+0.0222*m
    beta=-305.27+122.154*m-11.468*m*m
    return A,lambdad,kd0,beta

def boltzmann_model_kb(p,kb,m):
    """
    Bimodal Boltzmann Model for direct fraction
    Inputs :
        p : list with the six coefficients of the model (see equation 45)
        kb: direct fraction
        m : optical air mass
    Outputs:
        Bimodal Boltzmann model value for direct fraction
    """
    A,lambdab,kb0,beta=p
    if operator.isSequenceType(kb):
        out=[]
        for k in kb:
            if k<=0.02:
                out.append(17.34-36.75*exp(-0.975*m))
            else:
                out.append(A*((lambdab*exp((k-kb0)*lambdab))/power(1.+exp((k-kb0)*(lambdab+beta)),2)))
        return array(out)
    else:
        if kb<=0.02:
            return 17.34-36.75*exp(-0.975*m)
        else:
            return A*((lambdab*exp((kb-kb0)*lambdab))/power(1.+exp((kb-kb0)*(lambdab+beta)),2))

def guesskb(m):
    """
    Generates and returns a list with candidate coefficients to initialize
    the fitting process
    Inputs :
        m: optical air mass
    Outputs:
        Coefficients for the bimodal Boltzmann model for direct fraction as
        function of the optical air mass for Armilla (Granada, Spain).
        See Tovar, J., Olmo, F.J., Alados-Arboledas, L., 1998, 'One-minute kd and
        kb probability density distributions conditioned to the optical air mass',
        Solar Energy, 62, 387-393
    """
    A=0.9984-0.01686*m+0.00171*m*m
    lambdab=8.855-0.2666*m-0.3229*m*m
    kb0=0.7862-0.0546*m-0.00543*m*m
    beta=88.74-34.059*m-6.1846*m*m
    return A,lambdab,kb0,beta

##################################################################
#
# C L A S S   F O R   T H E   S U N - E A R T H   G E O M E T R Y
#
##################################################################
#Source:
#    Iqbal M., 1983, 'An introduction to solar radiation', Canada:
#    Academic Press, ISBN 0-12-373752-4
##################################################################
class SRad(object):
	
	_Isc=1367.0 # W/m2
	_p0=101325.0 # Pa
	
	def __init__(self,lat,year,month,day,hour=12,minute=0,second=0,stdlon=None,loclon=None):
		"""
		Class constructor
		"""
		self._lat=float(lat)
		self._day=int(day)
		self._month=int(month)
		self._year=int(year)
		self._hour=int(hour)
		self._minute=int(minute)
		self._second=int(second)
		self._slon=None
		if stdlon is not None: self._slon=float(stdlon)
		self._llon=None
		if loclon is not None: self._llon=float(loclon)

	def toLAT(self):
		"""
		Change from local standard time to local apparent time. See Iqbal, section 1.4
		"""
		if self._slon is None: raise IOError('Unknown standard meridian value')
		if self._llon is None: raise IOError('Unknown local meridian value')
		dangle=self.day_angle()
		et=(0.000075+0.001868*cos(dangle)-0.032077*sin(dangle)-0.014615*cos(2*dangle)-0.04089*sin(2*dangle))*(229.18)
		offset=4*(self._llon-self._slon)+et
		dt=datetime.datetime(self._year,self._month,self._day,self._hour,self._minute,self._second)
		newdt=dt+datetime.timedelta(minutes=offset)
		self._year=newdt.date().year
		self._month=newdt.date().month
		self._day=newdt.date().day
		self._hour=newdt.time().hour
		self._minute=newdt.time().minute
		self._second=newdt.time().second

	# getters and setters for the class attributes
	def get_Isc(self):
		return self._Isc
	
	def set_Isc(self, new_Isc):
		self._Isc=new_Isc
	
	Isc=property(fget=get_Isc,fset=set_Isc)
	
	def get_lat(self):
		return self._lat
	
	def set_lat(self,lat):
		self._lat=lat
	
	lat=property(fget=get_lat,fset=set_lat)
	
	def get_day(self):
		return self._day
	
	def set_day(self,day):
		self._day=day
	
	day=property(fget=get_day,fset=set_day)
	
	def get_month(self):
		return self._month
	
	def set_month(self,month):
		self._month=month
	
	month=property(fget=get_month,fset=set_month)
	
	def get_year(self):
		return self._year
	
	def set_year(self,year):
		self._year=year
	
	year=property(fget=get_year,fset=set_year)
	
	def get_hour(self):
		return self._hour
	
	def set_hour(self,hour):
		self._hour=hour
	
	hour=property(fget=get_hour,fset=set_hour)
	
	def get_minute(self):
		return self._minute
	
	def set_minute(self,minute):
		self._minute=minute
	
	minute=property(fget=get_minute,fset=set_minute)
	
	def get_second(self):
		return self._second
	
	def set_second(self,second):
		self._second=second
	
	second=property(fget=get_second,fset=set_second)
	
	def get_date(self):
		return datetime.date(self._year,self._month,self._day).strftime("%d %b %y")
	
	date=property(fget=get_date)
	
	def get_hms(self):
		return datetime.time(self._hour,self._minute,self._second).strftime("%H:%M:%S")
	
	hms=property(fget=get_hms)

	def get_stdlon(self):
		return self._slon

	def set_stdlon(self,stdlon):
		self._slon=float(stdlon)

	stdlon=property(fget=get_stdlon,fset=set_stdlon)

	def get_loclon(self):
		return self._llon

	def set_loclon(self,loclon):
		self._llon=float(loclon)

	loclon=property(fget=get_loclon,fset=set_loclon)

	def julian_day(self):
		"""
		Julian day from 1 to 366 for this object. Thus, accounting with the leap years
		"""
		ndays=[31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31]
		if isleap(self._year):
				ndays[1]=29
		julday=self._day
		for i in range(0,self._month-1):
			julday=julday+ndays[i]
		return julday
	
	def day_angle(self):
		"""
		Day angle for this object in radians. See Iqbal (1.2.2)
		"""
		N=365.0
		if isleap(self._year): N+=1
		return 2*pi*(self.julian_day()-1)/N;

	def hour_angle(self):
		"""
		Hour angle for the object in radians
		"""
		return -radians(15)*((float(self._hour)+float(self._minute)/60.0+float(self._second)/3600.0)-12.0)

	def eccentricity(self):
		"""
		Terrestrial orbit eccentricity for this object. See Iqbal (1.2.1)
		"""
		dangle=self.day_angle()
		return 1.00011+0.034221*cos(dangle)+0.00128*sin(dangle)+0.000719*cos(2*dangle)+0.000077*sin(2*dangle)

	def declination(self):
		"""
		Declination for this object in radians. See Iqbal (1.3.1)
		"""
		dangle = self.day_angle()
		return (0.006918 - 0.399912*cos(dangle) + 0.070257*sin(dangle) - 0.006758*cos(2*dangle)+0.000907*sin(2*dangle)-0.002697*cos(3*dangle)+0.00148*sin(3*dangle))

	def zenith(self,surf_slope=0.0,surf_azimuth=0.0):
		"""
		Zenith (solar) angle for this object in radians. See Iqbal (1.5.1)
		"""
		decrad=self.declination()
		latrad=radians(self._lat)
		hourang=self.hour_angle()
		sloperad=radians(surf_slope)
		azimuthrad=radians(surf_azimuth)
		teta = arccos((sin(latrad)*cos(sloperad)-cos(latrad)*sin(sloperad)*cos(azimuthrad))*sin(decrad)+(cos(latrad)*cos(sloperad)+sin(latrad)*sin(sloperad)*cos(azimuthrad))*cos(decrad)*cos(hourang)+cos(decrad)*sin(sloperad)*sin(azimuthrad)*sin(hourang))
		if teta > pi/2.0:
			return pi/2.0
		else:
			return teta
	
	def solar_height(self):
		"""
		Solar height angle for this object in radians. See Iqbal (1.5.1)
		"""
		dec=self.declination()
		latrad=radians(self._lat)
		omega=self.hour_angle()
		alt_solar=arcsin(sin(dec)*sin(latrad)+cos(dec)*cos(latrad)*cos(omega))
		if alt_solar<0.0: return 0.0
		else: return alt_solar
	
	def azimuth(self):
		"""
		Azimuth (solar) angle for this object in radians. See Iqbal (1.5.2a)
		"""
		dec=self.declination()
		alfa=self.solar_height()
		latrad=radians(self._lat)
		return arccos((sin(alfa)*sin(latrad)-sin(dec))/cos(alfa)*cos(latrad))
	
	def sunrise(self):
		"""
		Sunrise hour angle for this object in radians. See Iqbal (1.5.4)
		"""
		latrad=radians(self._lat)
		return arccos(-tan(latrad)*tan(self.declination()))
	
	def sunset(self):
		"""
		Sunset hour angle for this object in radians. It is supposse to be symmetrical
		with respect to the sunrise angle
		"""
		return -self.sunrise()
	
	def I0n(self):
		"""
		Normal extraterrestrial solar irradiance for this object in W/m2. See Iqbal (4.2.1)
		"""
		return self._Isc*self.eccentricity()

	def I0h(self):
		"""
		Horizontal extraterrestrial solar irradiance for this object in W/m2. See Iqbal (4.2.2)
		"""
		ioh=self._Isc*self.eccentricity()*cos(self.zenith())
		if ioh < 0:
			return 0.0
		else:
			return ioh
	
	def optical_mass(self):
		"""
		Relative optical air mass for this object
		Vollmer and Gedzelman, 'Colours of the Sun and Moon: the role of the optical air mass'
		European Journal of Physics, vol 27, pp 299-309, 2006
		"""
		ratio=6370000.0/8430.0 # Earth's radius/standard (homogeneous) atmosphere thickness
		cosenotetaz=cos(self.zenith())
		if self.I0h() == 0.0:
			return 0.0
		else:
			return sqrt(power(ratio*cosenotetaz,2)+2*ratio+1)-ratio*cosenotetaz

	def optical_mass_corr(self,pres):
		"""
		Optical air mass corrected for pressure in Pa. See Iqbal (5.7.3)
		"""
		return self.optical_mass()*(pres/self._p0)

	def clearness_index(self,rgblh,dataerror=None):
		"""
		Clearness index for this object and a given global radiation.
		"""
		if dataerror is not None and rgblh == dataerror: return dataerror
		ioh=self.I0h()
		if ioh < 1e-4: return 0.0
		kt = float(rgblh)/ioh
		if kt < 0.0:
			return 0.0
		if kt > 1.0:
			return 1.0
		return kt
	
	def diffuse_fraction(self,rgblh,scheme,dataerror=None):
		"""
		Diffuse fraction for this object and a given global radiation and diffuse regression
		equation (scheme). In this class are implemented some of these schemes
		"""
		if dataerror is not None and rgblh == dataerror: return dataerror
		if self.I0h() == 0.0: return 0.0
		try:
			return scheme(self.clearness_index(rgblh))
		except TypeError:
			return scheme(self.clearness_index(rgblh),self.optical_mass())

	def Rdifh(self,rgblh,scheme,dataerror=None):
		"""
		Horizontal diffuse radiation for a given global radiation and diffuse regression
		equation (scheme). In this class are implemented some of these schemes
		"""
		if dataerror is not None and rgblh == dataerror: return dataerror
		if self.I0h() == 0.0: return 0.0
		return rgblh*self.diffuse_fraction(rgblh,scheme)
	
	def Rdirh(self,rgblh,scheme,dataerror=None):
		"""
		Horizontal direct radiation for a given global radiation and diffuse regression
		equation (scheme). In this class are implemented some of these schemes
		"""
		if dataerror is not None and rgblh == dataerror: return dataerror
		if self.I0h() == 0.0: return 0.0
		return rgblh*(1.0-self.diffuse_fraction(rgblh,scheme))
	
	@staticmethod
	def orgill(kt):
		"""
		Diffuse fraction scheme of Orgill. See Orgill, J.E., Hollands, K.G.T., 1977, 'Correlation
		equation for hourly diffuse radiation on a horizontal surface', Solar Energy, 19, 357-359
		"""
		if kt < 0.35:
			return 1.0-0.249*kt
		elif kt >= 0.35 and kt <= 0.75:
			return 1.557-1.84*kt
		else:
			return 0.177 
	
	@staticmethod
	def reindl(kt):
		"""
		Diffuse fraction scheme of Orgill. See Reindl, D.T., Beckman, W.A., Duffie, J.A., 1990, 'Diffuse
		fraction correlations', Solar Energy, 45, 1-7
		"""
		if kt <= 0.3:
			return 1.020-0.284*kt
		elif kt > 0.3 and kt < 0.78:
			return 1.45-1.67*kt
		else:
			return 0.147

######################################
# M  A  I  N     P  R  O  G  R  A  M #
######################################
if __name__ == '__main__':

    # input parameters parser
    parser=OptionParser()
    parser.add_option("-i","--infile",dest="ifilename",help="input filename",metavar="FILE")
    parser.add_option("-o","--outfile",dest="ofilename",help="output filename for the fitting coefficients",metavar="FILE")
    parser.add_option("-L","--LST",dest="LST",help="data in local standard time as described in Iqbal, M., 1983, 'An introduction to solar radiation', Canada: Academic Press, ISBN 0-12-373752-4, Section 1.4 for this standart meridian")
    parser.add_option("-g","--graph",action="store_true",dest="dograph",default=False,help="Shows the probability density function for each optical mass")

    (options,args)=parser.parse_args()

    if options.ifilename is None:
        sys.stderr.write('FATAL ERROR: UNKNOWN INPUT FILE\n\n')
        parser.print_help()
        sys.exit(1)

    # longitude, latitude and elevation from the first line of the input data file
    ifile=open(options.ifilename,'r')
    sitelon,sitelat=ifile.readline().split()
    ifile.close()

    sitelon=float(sitelon)
    sitelat=float(sitelat)

    sys.stdout.write('\n')    
    if options.LST is not None:
        sys.stdout.write('Site longitude    : %6.2f\n' % sitelon)
        sys.stdout.write('Site latitude     : %6.2f\n' % sitelat)
        try:
            sys.stdout.write('Standard meridian : %6.2f\n' % float(options.LST))
        except ValueError:
            sys.stderr.write('FATAL ERROR: UNKNOWN STANDARD MERIDIAN\n\n')
            parser.print_help()
            sys.exit(1)
    else:
        sys.stdout.write('Site longitude : %6.2f\n' % sitelon)
        sys.stdout.write('Site latitude  : %6.2f\n' % sitelat)
    sys.stdout.write('\n')    

    N=50 # number of bins in the histograms

    # input data loading
    try:
        data=load(options.ifilename,skiprows=2)
    except ValueError:
        sys.stderr.write('FATAL ERROR: INVALID LITERAL FOUND IN INPUT FILE. CHECK INPUT FILE FORMAT')
        sys.exit(1)

    data=filter(data,0,test='>',column=5) # takes only diurnal records
    
    year  = [int(f)   for f in data[:,0]]
    month = [int(f)   for f in data[:,1]]
    day   = [int(f)   for f in data[:,2]]
    hour  = [int(f)   for f in data[:,3]]
    minute= [int(f)   for f in data[:,4]]
    rgblh = [float(f) for f in data[:,5]]
    if shape(data)[1]>6: rdifh = [float(f) for f in data[:,6]]
    if shape(data)[1]>7: rdirh = [float(f) for f in data[:,7]]

    sys.stdout.write('Found %s utilizable diurnal records.\n' % shape(data)[0] )
    sys.stdout.write('\n')    
    sys.stdout.write('Calculating the clearness index and the optical mass...\n')
    sys.stdout.write('\n\t---------------------------------\n')
    sys.stdout.write('\t|   20%   40%   60%   80%   100%|\n\t ')

    # assessment of the clearness index and the optical air mass
    binbar=len(year)/30
    progress=0
    kt,mr=[],[]
    sun=SRad(sitelat,2000,1,1,loclon=sitelon) # dummy initialization
    if options.LST is not None:
        sun.stdlon=float(options.LST)
        for y,M,d,h,m,r in zip(year,month,day,hour,minute,rgblh):
            if progress==0:
                progress=binbar
                sys.stdout.write('#') 
            sun.year=y                        # sets time instant in sun instance
            sun.month=M
            sun.day=d
            sun.hour=h
            sun.minute=m
            sun.second=0
            sun.toLAT()                       # Local Apparent Time correction
            kt.append(sun.clearness_index(r)) # clearness index calculation
            mr.append(sun.optical_mass())     # optical mass calculation
            progress=progress-1
    else:
         for y,M,d,h,m,r in zip(year,month,day,hour,minute,rgblh):
            if progress==0:
                progress=binbar
                sys.stdout.write('#') 
            sun.year=y                        # sets time instant in sun instance
            sun.month=M
            sun.day=d
            sun.hour=h
            sun.minute=m
            sun.second=0
            kt.append(sun.clearness_index(r)) # clearness index calculation
            mr.append(sun.optical_mass())     # optical mass calculation
            progress=progress-1
    sys.stdout.write('#\n\n')
    
    kt=array(kt)
    mr=array(mr)
    
    x=linspace(0.,1.,100)

    ##########################################################################
    # Probability density function and Boltzman curve for the clearness index
    # conditioned to the optical air masses 1.0, 1.5, 2.0, 2.5, 3.0, 3.5
    ##########################################################################
    datakt=transpose([kt,mr])
    datakt=filter(datakt,1,test='<',column=0) # clearness indices must to be less than one

    nfig=1
    figure(nfig)
    nfig=nfig+1

    coefskt=[]
    
    # m==1.0
    ax=subplot(321)
    data_m10=filter(datakt,1.1,test='<',column=1)
    data_m10=filter(data_m10,1.0,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m10[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kt(p,x)-y,guesskt(1.0),args=(bins,pdf))
        coefskt.append([p[0],p[1],p[2],1.0-p[0],p[3],p[4]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kt(p,x),'r-')
    except IndexError:
        coefskt.append(array([0.,0.,0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.1,0.8,'m=1.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Clearness index (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==1.5
    ax=subplot(322)
    data_m15=filter(datakt,1.55,test='<',column=1)
    data_m15=filter(data_m15,1.45,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m15[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kt(p,x)-y,guesskt(1.5),args=(bins,pdf))
        coefskt.append([p[0],p[1],p[2],1.0-p[0],p[3],p[4]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kt(p,x),'r-')
    except IndexError:
        coefskt.append(array([0.,0.,0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.1,0.8,'m=1.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Clearness index (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==2.0
    ax=subplot(323)
    data_m20=filter(datakt,2.05,test='<',column=1)
    data_m20=filter(data_m20,1.95,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m20[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kt(p,x)-y,guesskt(2.0),args=(bins,pdf))
        coefskt.append([p[0],p[1],p[2],1.0-p[0],p[3],p[4]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kt(p,x),'r-')
    except IndexError:
        coefskt.append(array([0.,0.,0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.1,0.8,'m=2.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Clearness index (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==2.5
    ax=subplot(324)
    data_m25=filter(datakt,2.55,test='<',column=1)
    data_m25=filter(data_m25,2.45,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m25[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kt(p,x)-y,guesskt(2.5),args=(bins,pdf))
        coefskt.append([p[0],p[1],p[2],1.0-p[0],p[3],p[4]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kt(p,x),'r-')
    except IndexError:
        coefskt.append(array([0.,0.,0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.1,0.8,'m=2.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Clearness index (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==3.0
    ax=subplot(325)
    data_m30=filter(datakt,3.05,test='<',column=1)
    data_m30=filter(data_m30,2.95,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m30[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kt(p,x)-y,guesskt(3.0),args=(bins,pdf))
        coefskt.append([p[0],p[1],p[2],1.0-p[0],p[3],p[4]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kt(p,x),'r-')
    except IndexError:
        coefskt.append(array([0.,0.,0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.1,0.8,'m=3.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Clearness index (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==3.5
    ax=subplot(326)
    data_m35=filter(datakt,3.55,test='<',column=1)
    data_m35=filter(data_m35,3.45,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m35[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kt(p,x)-y,guesskt(3.5),args=(bins,pdf))
        coefskt.append([p[0],p[1],p[2],1.0-p[0],p[3],p[4]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kt(p,x),'r-')
    except IndexError:
        coefskt.append(array([0.,0.,0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.1,0.8,'m=3.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Clearness index (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    ##########################################################################
    # Probability density function and Boltzman curve for the diffuse fraction
    # conditioned to the optical air masses 1.0, 1.5, 2.0, 2.5, 3.0, 3.5
    ##########################################################################
    rgblh=array(rgblh)
    rdifh=array(rdifh)
    rdirh=array(rdirh)
    kd=rdifh/rgblh # rgblh is never zero cause has been previously filtered
    kb=rdirh/rgblh
    ks=kb+kd # defined for filtering purposes

    dataks=transpose([ks,kd,kb,mr])
    dataks=filter(dataks,1,test='==',column=0) # takes only records where kd+kb=1

    datakd=transpose([dataks[:,1],dataks[:,3]])
    datakd=filter(datakd,1,test='<',column=0) # takes only records where kd<1

    figure(nfig)
    nfig=nfig+1

    coefskd=[]
    
    # m==1.0
    ax=subplot(321)
    data_m10=filter(datakd,1.2,test='<',column=1)
    data_m10=filter(data_m10,1.0,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m10[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kd(p,x)-y,guesskd(1.0),args=(bins,pdf))
        coefskd.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kd(p,x),'r-')
    except IndexError:
        coefskd.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=1.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Diffuse fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==1.5
    ax=subplot(322)
    data_m15=filter(datakd,1.6,test='<',column=1)
    data_m15=filter(data_m15,1.4,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m15[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kd(p,x)-y,guesskd(1.5),args=(bins,pdf))
        coefskd.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kd(p,x),'r-')
    except IndexError:
        coefskd.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=1.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Diffuse fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==2.0
    ax=subplot(323)
    data_m20=filter(datakd,2.1,test='<',column=1)
    data_m20=filter(data_m20,1.9,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m20[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kd(p,x)-y,guesskd(2.0),args=(bins,pdf))
        coefskd.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kd(p,x),'r-')
    except IndexError:
        coefskd.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=2.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Diffuse fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==2.5
    ax=subplot(324)
    data_m25=filter(datakd,2.6,test='<',column=1)
    data_m25=filter(data_m25,2.4,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m25[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kd(p,x)-y,guesskd(2.5),args=(bins,pdf))
        coefskd.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kd(p,x),'r-')
    except IndexError:
        coefskd.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=2.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Diffuse fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==3.0
    ax=subplot(325)
    data_m30=filter(datakd,3.1,test='<',column=1)
    data_m30=filter(data_m30,2.9,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m30[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kd(p,x)-y,guesskd(3.0),args=(bins,pdf))
        coefskd.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kd(p,x),'r-')
    except IndexError:
        coefskd.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=3.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Diffuse fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==3.5
    ax=subplot(326)
    data_m35=filter(datakd,3.6,test='<',column=1)
    data_m35=filter(data_m35,3.4,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m35[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y: boltzmann_model_kd(p,x)-y,guesskd(3.5),args=(bins,pdf))
        coefskd.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kd(p,x),'r-')
    except IndexError:
        coefskd.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=3.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Diffuse fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    ##########################################################################
    # Probability density function and Boltzman curve for the direct fraction
    # conditioned to the optical air masses 1.0, 1.5, 2.0, 2.5, 3.0, 3.5
    ##########################################################################
    datakb=transpose([dataks[:,2],dataks[:,3]])
    datakb=filter(datakb,1,test='<',column=0) # takes only records where kb<1

    figure(nfig)
    nfig=nfig+1

    coefskb=[]
    
    # m==1.0
    ax=subplot(321)
    data_m10=filter(datakb,1.2,test='<',column=1)
    data_m10=filter(data_m10,1.0,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m10[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y,z: boltzmann_model_kb(p,x,y)-z,guesskb(1.0),args=(bins[1:],1.0,pdf[1:]))
        coefskb.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kb(p,x,1.0),'r-')
    except IndexError:
        coefskb.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=1.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Direct fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==1.5
    ax=subplot(322)
    data_m15=filter(datakb,1.6,test='<',column=1)
    data_m15=filter(data_m15,1.4,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m15[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y,z: boltzmann_model_kb(p,x,y)-z,guesskb(1.5),args=(bins[1:],1.5,pdf[1:]))
        coefskb.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kb(p,x,1.5),'r-')
    except IndexError:
        coefskb.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=1.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Direct fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==2.0
    ax=subplot(323)
    data_m20=filter(datakb,2.1,test='<',column=1)
    data_m20=filter(data_m20,1.9,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m20[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y,z: boltzmann_model_kb(p,x,y)-z,guesskb(2.0),args=(bins[1:],2.0,pdf[1:]))
        coefskb.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kb(p,x,2.0),'r-')
    except IndexError:
        coefskb.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=2.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Direct fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==2.5
    ax=subplot(324)
    data_m25=filter(datakb,2.6,test='<',column=1)
    data_m25=filter(data_m25,2.4,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m25[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y,z: boltzmann_model_kb(p,x,y)-z,guesskb(2.5),args=(bins[1:],2.5,pdf[1:]))
        coefskb.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kb(p,x,2.5),'r-')
    except IndexError:
        coefskb.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=2.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Direct fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==3.0
    ax=subplot(325)
    data_m30=filter(datakb,3.1,test='<',column=1)
    data_m30=filter(data_m30,2.9,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m30[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y,z: boltzmann_model_kb(p,x,y)-z,guesskb(3.0),args=(bins[1:],3.0,pdf[1:]))
        coefskb.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kb(p,x,3.0),'r-')
    except IndexError:
        coefskb.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=3.0',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Direct fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # m==3.5
    ax=subplot(326)
    data_m35=filter(datakb,3.6,test='<',column=1)
    data_m35=filter(data_m35,3.4,test='>=',column=1)
    try:
        pdf,bin_min,bin_width,extra_points=relfreq(data_m35[:,0],N,(0.,1.))
        pdf=100*pdf
        bins=linspace(bin_min,(N-1)*bin_width,N)
        p,success=optimize.leastsq(lambda p,x,y,z: boltzmann_model_kb(p,x,y)-z,guesskb(3.5),args=(bins[1:],3.5,pdf[1:]))
        coefskb.append([p[0],p[1],p[2],p[3]])
        bar(bins,pdf,bin_width)
        plot(x,boltzmann_model_kb(p,x,3.5),'r-')
    except IndexError:
        coefskb.append(array([0.,0.,0.,0.]))
        plot() # empty plotting area
        text(0.4,0.8,'(no data available)',transform=ax.transAxes)
    text(0.8,0.8,'m=3.5',transform=ax.transAxes)
    xlim(0,1)
    ylim(0,axis()[3])
    xlabel('Direct fraction (0.00|%4.2f|1.00)' % float(bin_width))
    ylabel('Probability density')

    # dumping the results to the output file
    if options.ofilename is not None:
        sys.stdout.write('Dumping results to output file...\n')
        ofile=open(options.ofilename,'w')
        ofile.write('\n%s\n' % datetime.datetime.now().strftime("%d/%M/%y %H:%M:%S"))
        ofile.write('%s: found %d utilizable records\n' % (options.ifilename,len(year)))
        ofile.write('Bin size: %4.2f\n\n' %float(bin_width))
        
        coefskt=array(coefskt)
        ofile.write('Pdf for clearness index:\n')
        ofile.write('-'*69+'\n')
        ofile.write('%3s  %9s  %9s  %9s  %9s  %9s  %9s\n' % ('m','A1','lambda1','kt01','A2','lambda2','kt02'))
        ofile.write('-'*69+'\n')
        for r in range(shape(coefskt)[0]):
            for c in range(shape(coefskt)[1]):
                if c==0: ofile.write('%3.1f  %9.3f' % ((r+2)/2.,coefskt[r,c]))
                else: ofile.write('  %9.3f' % coefskt[r,c])
            ofile.write('\n')
        ofile.write('-'*69+'\n')
        
        coefskd=array(coefskd)
        ofile.write('\nPdf for diffuse fraction:\n')
        ofile.write('-'*47+'\n')
        ofile.write('%3s  %9s  %9s  %9s  %9s\n' % ('m','A','lambda','kd0','beta'))
        ofile.write('-'*47+'\n')
        for r in range(shape(coefskd)[0]):
            for c in range(shape(coefskd)[1]):
                if c==0: ofile.write('%3.1f  %9.3f' % ((r+2)/2.,coefskd[r,c]))
                else: ofile.write('  %9.3f' % coefskd[r,c])
            ofile.write('\n')
        ofile.write('-'*47+'\n')
        
        coefskb=array(coefskb)
        ofile.write('\nPdf for direct fraction:\n')
        ofile.write('-'*47+'\n')
        ofile.write('%3s  %9s  %9s  %9s  %9s\n' % ('m','A','lambda','kb0','beta'))
        ofile.write('-'*47+'\n')
        for r in range(shape(coefskb)[0]):
            for c in range(shape(coefskb)[1]):
                if c==0: ofile.write('%3.1f  %9.3f' % ((r+2)/2.,coefskb[r,c]))
                else: ofile.write('  %9.3f' % coefskb[r,c])
            ofile.write('\n')
        ofile.write('-'*47+'\n')
        
        ofile.close()

    # plot the histograms and Boltzman curves only if the flag -g is set    
    if options.dograph:
        sys.stdout.write('Close graph window to finish...\n')
        show()

    sys.stdout.write('\nAll work done!')
