function [day_angle_m,delta_m,omega_ss_m,S0_m,eccentricity_m,G0d_m,G0h_m,ier]=...
         monthly_averages(month_number,year_number,phi,lambda,gamma_riset)
%--------------------- HELIOSAT II ---------------------------------------------
%-------------> Geometry of Solar Beam
%
% Esta funcion calcula directamente el promedio mensual de varios parametros solares:
% el angulo diario (rad), la excentricidad, la declinacion (rad), el angulo horario de salida
% del sol (rad), la duracion del dia (horas decimales), irradiacion diaria extraterrestre (Wh/m2)
% y los 24 valores de irradiacion horaria extraterrestre (Wh/m2)
% El FLAG ier devuelve 0 si OK y 1 en caso contrario
%
% [day_angle_m,delta_m,omega_ss_m,S0_m,eccentricity_m,G0d_m,G0h_m,ier]=...
%         monthly_averages(month_number,year_number,phi,lambda,gamma_riset)
%
% INPUT: 
%       month_number        : numero del mes
%       year_number         : numero del ao (4 digitos)
%       phi                 : latitud del lugar (rad, positivo al norte)
%       lambda              : longitud del lugar (rqad, positivo al este)
%       gamma_riset         : elevacion solar cerca de la salida/puesta del sol
%                               -  0.0 para salida/puesta astronomica
%                               - -1.0 para salida/puesta corregida por refraccion
% OUTPUT:
%       day_angle_m         : promedio mensual del angulo de dia (rad)
%       delta_m             : promedio mensual de la declinacion solar (rad)
%       omega_ss_m          : promedio mensual del angulo horario de puesta (rad)
%       S0_m                : promedio mensual de la duracion del dia (horas decimales)
%       eccentricity_m      : promedio mensual de la excentricidad
%       G0d_m               : promedio mensual de la irradiacion extraterrestre diaria (Wh/m2)
%       G0h_m               : promedio mensual de los 24 valores de la irradiacion extraterrestre horaria (Wh/m2)
%       ier                 : FLAG de control
%-------------------------------------------------------------------------------

ier=1;
%inicializo
for i=1:24
    G0h_m(i,1)=0.0;
end

[number_days_month,ier]=nbdays_month(year_number,month_number);
if (ier ~= 0) break; end

for day_of_month=1:number_days_month
    [julian_day,ier]=make_julian_day(day_of_month,month_number,year_number);
    [day_angle(day_of_month,1),ier]=Day_Angle(julian_day);
    [delta(day_of_month,1),ier]=declination_sun(year_number,julian_day,lambda);
    [omega_sr(day_of_month,1),omega_ss(day_of_month,1),ier]=sunrise_hour_angle(phi,delta(day_of_month,1),gamma_riset);
    [t_sr(day_of_month,1),t_ss(day_of_month,1),S0(day_of_month,1),ier]...
        =timerise_daylenght(omega_sr(day_of_month,1),omega_ss(day_of_month,1));
    [eccentricity(day_of_month,1), ier]=corr_distance(day_angle(day_of_month,1));
    [G0d(day_of_month,1),ier]=G0_day(phi,eccentricity(day_of_month,1),delta(day_of_month,1));
    [G0h,ier]=G0_hours_profile(phi,eccentricity(day_of_month,1),delta(day_of_month,1));
    if (ier ~= 0) break; end
    for i=1:24
        G0h_m(i,1)=G0h_m(i,1)+G0h(i,1);
    end
end

day_angle_m=mean(day_angle);
delta_m=mean(delta);
omega_ss_m=mean(omega_ss);
S0_m=mean(S0);
eccentricity_m=mean(eccentricity);
G0d_m=mean(G0d);
for i=1:24
    G0h_m(i,1)=G0h_m(i,1)/number_days_month;
end