C     Last change:  A    28 Jun 2007    9:44 am
	PROGRAM MAIN
	IMPLICIT REAL*8(A-H,O-Z)
*--------------------------------------------------------------------
*  The program calculates spectral irradiances by the
*
*                     DELTA-EDDINGTON
*
*  method for the 290-325 nm band at Brewer spectrophotometer measurement
*  times for a 50 level atmosphere with cloud occuping the 2-3km layer.
*  There are two input files:
*  (1) DE.DAT selects model atmospheres and aerosol types.
*  (2) DAY.DAT contains Brewer radiation measurements and concurrent
*  meteorological data for Churchill, Manitoba for April 1993.
*  DE.RES contains measured and calculated spectral irradiances
*  at the Brewer measurement times and daily spectral sums.
*  Daily totals are calculated only for days with gaps between measurements
*  that are less than 2 hours.
*---------------------------------------------------------------------
	PARAMETER (nw = 36, nt = 70, nl =49)
	COMMON/O3DAT/wl(nw),solatlas(nw),sigatlas(nw)
	DIMENSION sol(nw),to3(nl,nw),taur(nl,nw),taua(nl,nw),wa(nl,nw),
     +  ga(nl,nw),taut(nl,nw),wt(nl,nw),gt(nl,nw)
	DIMENSION spm(nw,nt),gcs(nw,nt),gm1(nt),gc1(nt),gm2(nw),gc2(nw),
     +  xmusun(nt),time(nt),sza(nt),tco(nt),prs(nt),dbt(nt),rh(nt)
	CHARACTER id*7
	INTEGER dayn,year,day,mon
	DIMENSION total(6)
*     Total ozone content (atm-cm) for each model atmosphere
	DATA total/0.2773,0.3316,0.3768,0.3448,0.3757,0.3434/

	OPEN (1,file = 'DE.DAT', action = 'read')
	OPEN (2,file = 'DAY.DAT',action = 'read')
	OPEN (3,file = 'DE.RES',action = 'write')

	CALL SETUP(moda,modb,isc1,isc2a,isc2b,
     +  isc3a,isc3b,isc4,sol,height,iaer,ibtype,istype,cldtau,cssa,cgg)

	DO k = 1,1          ! Loop over days
	  CALL INPUT (id,year,day,mon,dayn,ozmeas,rat,nsang,time,
     +    sza,xmusun,gm1,tco,prs,dbt,rh,spm,albedo,snow,iflag,iftco)
          IF (iflag == 1) THEN
                EXIT
          END IF

*       If iftco = 1 there is missing tco data and the day is skipped

          IF (iftco == 1) THEN
                GOTO 99
          END IF

          WRITE (3,'(1x,a7,3i5,
     +    ''   Jd ='',i3,''  N ='',i3,''  Ozone = '',f5.1,
     +    ''  Albedo = '',f5.4,''   SD ='',f6.1)')id,day,mon,year,
     +    dayn,nsang,ozmeas*1000.d0,albedo,snow

*          Select Spring-summer or Fall-winter
	   IF (dayn > 90. .AND. dayn < 274.) THEN
		  model = moda
		  isc2  = isc2a
		  isc3  = isc3a
	   ELSE
		  model = modb
		  isc2  = isc2b
		  isc3  = isc3b
	   END IF

	   CALL METDTA(MODEL)

	   CALL FLIP (height,nlevel)

	   scale = ozmeas/total(model)

	   CALL O3UV2(to3,scale)

	   CALL RAY(taur)

	   CALL AEROSOL (iaer,isc1,isc2,isc3,isc4,ibtype,istype,taua,wa,
     +     ga)

	   CALL COMB (to3,taur,taua,wa,ga,taut,wt,gt)

	   CALL UVB (iaer,rat,sol,xmusun,nsang,taut,wt,gt,
     +    albedo,cldtau,cssa,cgg,tco,time,nlevel,spm,gcs,gm1,gc1,gm2,
     +    gc2,gm3,gc3)

           WRITE (3,'(2x,''GMT'',5x,70f11.2)') (time(i), i = 1,nsang)
           WRITE (3,'(2x,''SZA'',5x,70f11.2)') (sza(i), i = 1,nsang)
           WRITE (3,'(2x,''TCO'',5x,70f11.2)') (tco(i), i = 1,nsang)

	   DO iw = 1,nw
		  WRITE (3,'(i5,5x,70e11.4)') INT(wl(iw)),
     +            (spm(iw,i),i = 1,nsang),gm2(iw)
		  WRITE (3,'(10x,70e11.4)')
     +            (gcs(iw,i),i = 1,nsang),gc2(iw)
           END DO
           WRITE(3,*)
           WRITE (3,'(''   MEAS   ''70e11.4)') (gm1(i),i = 1,nsang), gm3
           WRITE (3,'(''   CALC   ''70e11.4)') (gc1(i),i = 1,nsang), gc3

99         CONTINUE
	ENDDO               ! End of day loop

	STOP
	END
*----------------------------------------------------------------------
	SUBROUTINE INPUT (id,yr,day,mon,jd,ozmeas,rat,nsol,time,sza,
     +  musun,gmm,tco,prs,dbt,rh,spm,albedo,snow,iflag,iftco)
	IMPLICIT REAL*8 (A-H,O-Z)
	PARAMETER (nw = 36,nt = 70)
	REAL*8 musun
	INTEGER yr,day
	CHARACTER id*7
	DIMENSION time(nt),sza(nt),gmm(nt),tco(nt),musun(nt),prs(nt),
     +  dbt(nt),rh(nt),spm(nw,nt)
	pi = 2.d0*dasin(1.d0)
        iflag = 0
        iftco = 0
	READ (2,'(a7,5i5,f10.4,10x,2f10.4)',END = 2000) id,yr,day,mon,
     +  jd,nsol,rat,ozmeas,snow
        ozmeas = ozmeas/1000.d0
	READ (2,'(4x,f7.2,70f11.2)') (time(i), i =1,nsol)
	READ (2,'(4x,70e11.4)') (sza(i), i =1,nsol)
	READ (2,'(4x,70e11.4)') (gmm(i), i = 1,nsol)  !skip sum at end
	READ (2,*)
	READ (2,*)
	READ (2,'(4x,70e11.4)') (tco(i), i = 1,nsol)
        DO l = 1,20
             READ(2,*)
        END DO
        READ(2,'(4x,70e11.4)') (prs(i), i = 1,nsol)
        READ(2,'(4x,70e11.4)') (dbt(i), i = 1,nsol)
        READ(2,'(4x,70e11.4)') (rh(i), i = 1,nsol)
        DO n =1,36
                READ (2,'(10x,70e11.4)') (spm(n,i), i = 1,nsol )
                DO i= 1,nsol
                      spm(n,i) = 1.06*spm(n,i)        !1.06 is the cosine error correction
                END DO
                IF (n == 36) THEN
                        EXIT
                END IF
                READ (2,*)
        END DO
	DO i = 1,nsol
	    musun(i) = cos(sza(i)*pi/180.d0)
	ENDDO

*     Snow albedo
	CALL SREF (snow,albedo)
*     Check for missing tco values. If tco<0, exit.
        DO i = 1,nsol
                IF(tco(i) < 0.d0) THEN
                        iftco = 1
                        RETURN
                ENDIF
        END DO

	RETURN
2000    iflag = 1

	END
*----------------------------------------------------------------------
	SUBROUTINE SREF (snow,albedo)
	REAL*8 snow,albedo,alfal,alfah,slope,sdu
	alfal = 0.05
	alfah = 0.75
	sdu = 30.0
	slope = (alfah-alfal)/sdu
	IF (snow .GT. 30.) THEN
	    albedo = alfah
	ELSE
	    albedo = alfal + slope*snow
	ENDIF
	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE SETUP(moda,modb,isc1,isc2a,isc2b,
     +  isc3a,isc3b,isc4,sol,height,iaer,ibtype,istype,cldtau,cssa,cgg)
	PARAMETER (nz=50,nw=36)
	IMPLICIT REAL*8 (A-H,O-Z)
	DIMENSION sol(nw)
	COMMON/O3DAT/wl(nw),solatlas(nw),sigatlas(nw)
	COMMON /ATM/ z(nz),p(nz),t(nz),h2o(nz),o3(nz)

*     Input data for this run
	READ(1,*) cldtau
        READ(1,*) cssa
        READ(1,*) cgg
	READ(1,*) height
	READ(1,*) iaer
	READ(1,*) ibtype
	READ(1,*) istype
	READ(1,*) isc1
	READ(1,*) isc4
	READ(1,*) moda
	READ(1,*) isc2a
	READ(1,*) isc3a
	READ(1,*) modb
	READ(1,*) isc2b
	READ(1,*) isc3b

	DO iw = 1,nw
	    sol(iw) = solatlas(iw)
	ENDDO

	RETURN
	END
*----------------------------------------------------------------------
	  SUBROUTINE FLIP(height,lb)
*       Inverts model profile and interpolates climatic variables for
*       the height fo the station. The station height becomes the
*       lowest profile height.

	  PARAMETER (nz=50)
	  IMPLICIT REAL*8 (A-H,O-Z)
	COMMON /ATM/ z(nz),p(nz),t(nz),h2o(nz),o3(nz)
	COMMON /ATM2/ z2(nz),p2(nz),t2(nz),h2o2(nz),o32(nz),nzz

*	Flip the profile
	DO i = 1,nz
		    ii=nz+1-i
		    z2(i) = z(ii)
		    p2(i) = p(ii)
		    t2(i) = t(ii)
		    h2o2(i) = h2o(ii)
		    o32(i) = o3(ii)
	  END DO

*       Interpolate climatic variables for the station height.
*       Find the profile layer which contains the station height.
*       lt = layer top
*       lb = layer bottom

	  lb = nz - INT(height)
	  lt = lb - 1

*       lb is the last index for each climatic variable array.
*       Interpolate values of pressure, temperature, water vapour and
*       ozone for this index. For pressure linearly interpolate in
*       p v ln(z2). Linearly interpolate for the other variables.


	  delhdh = (height - z2(lb))/(z2(lt) - z2(lb))
	  t2(lb) = t2(lb) + (t2(lt) - t2(lb))* delhdh
	  h2o2(lb) = h2o2(lb) + (h2o2(lt) - h2o2(lb))*delhdh
	  o32(lb) = o32(lb) + (o32(lt) - o32(lb))*delhdh
	  xx = dlog(p2(lb))+dlog(p2(lt)/p2(lb))*(height-z2(lb))/
     +  (z2(lt)-z2(lb))
	  p2(lb) = DEXP(xx)
	  z2(lb) = height

*       Number of items in each array is lb

	  nzz = lb

	  RETURN
	  END
c----------------------------------------------------------------------
	SUBROUTINE O3UV2(to3,scale)
*     Mean ozone spectral cross-sections for each nanometer between
*     290 and 325nm calculated from Bass's(1985) C0,C1 and C2 as
*     extracted from LOWTRAN7. The calculations are performed for
*     a selected model atmosphere (one of LOWTRAN's) and must be
*     redone for any other atmosphere since cross-sections vary with
*     temperature. The averages are Chandrasekhar means using the
*     extraterrestrial irradiance to weight.
*     Optical depths are calculated for 49 layers.
*----------------------------------------------------------------------
	IMPLICIT REAL*8(A-H,O-Z)
	PARAMETER (nz=50,nw=723,nnw=36,nl=49,f2=1.25467d21)
*     ATM2 contains climate data ordered from top to bottom and
*     nzz the number of levels, the last being at station height.
	COMMON /ATM2/ z(nz),p(nz),t(nz),h2o(nz),o3(nz),nzz
	COMMON/O3DAT/wll(nw),solatlas(nw),sigatlas(nw)
	COMMON /SUNDAT2/ atlas(5160)
	DIMENSION c(nw,3),xsec(nw),csig(nnw),xmsec(nz,nnw)
	DIMENSION xnmls(nz),sflx(nw),wl(nw),sig(nz,nw)
	DIMENSION nitem(nnw),istart(nnw),iend(nnw),to3(nl,nnw)
c     wll: the 36 1nm interval wavelengths in COMMON BLOCK O3DAT
c     wl : the 723 wavelengths for the Bass data
c     Extract Bass coefficients and interpolate extraterrestrial solar
c     radiation for the required wavelengths
	iwl = 28940
	DO iw = 1,nw
	    iwl = iwl + 5
	    wl(iw) = dble(iwl)/1.d5   ! now in um
	    v = 10000./wl(iw)
	    CALL O3HHT0(V,C0)
	    CALL O3HHT1(V,CT1)
	    CALL O3HHT2(V,CT2)
	    c(iw,1) = c0
	    c(iw,2) = ct1*c0
	    c(iw,3) = ct2*c0
	    sflx(iw) = atlas(2790 + iw - 1) ! verified
	ENDDO

*     Convert O3 from ppmv to g/m**3 by the equation of state and
*     compute ozone optical cross sections

	DO i = 1,nzz
	    rho = (p(i)/t(i))*1.d5/287.053d0
	    xnmls(i) = f2*scale*1.d-6*o3(i)*rho*47.9982/28.966
	    tc = t(i)-273.15
	    DO iw = 1,nw
		  sig(i,iw) = (c(iw,1)+c(iw,2)*tc+c(iw,3)*tc*tc)*1.d-20
	    ENDDO
	 ENDDO

*     Calculate mean optical cross sections for each nanometer

*     Find the number of wavelengths to be averaged for Chandrasekhar
*     means and the starting and ending wavelength indices, istart and iend.
*     Means are centred on nanometer wavelengths.

	CALL INDX(nitem,istart,iend,nnw)

	DO i = 1,nzz
	    DO iw = 1,nw
		xsec(iw) = sig(i,iw)
	    ENDDO
	    CALL CMEAN (istart,iend,sflx,xsec,csig,nnw)
	    DO niw = 1,nnw
		  xmsec(i,niw) = csig(niw)
	    ENDDO
	ENDDO

*     Layer optical depths

	DO niw = 1,nnw
	    DO i = 1,nzz - 1
		  dz = z(i) - z(i+1)
		  top = xmsec(i,niw)*xnmls(i)
		  bot = xmsec(i+1,niw)*xnmls(i+1)
		  to3(i,niw) = (top + bot)*dz/2.
	    ENDDO
	ENDDO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE INDX(nitem,istart,iend,nnw)
*     Loops over the wavelengths to be averaged, returning the
*     number of wavelengths and starting and ending indices for each
*     group.
	DIMENSION nitem(*),istart(*),iend(*)
	DO niw = 1,nnw   ! Loop over number of 1nm wavelengths
*         Find number of spectral values to be averaged
	    nitem(niw) = 23
	    IF (niw .EQ. 1) THEN
		  istart(1) = 1
	    ELSE
		  istart(niw) = iend(niw-1) - 2
	    ENDIF
	    iend(niw) = istart(niw) + nitem(niw) - 1
	ENDDO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE CMEAN (istart,iend,sflx,xsec,csig,nnw)
	IMPLICIT REAL*8(A-H,O-Z)
	DIMENSION sflx(*),csig(nnw),xsec(*),w(23)
	DIMENSION istart(*),iend(*)

*     w = weights for the triangular function.

	DATA w/0.,0.090909,0.181818,0.272727,0.363636,0.454545,0.545454,
     +0.636363,0.727272,0.818181,0.909090,1.0,0.909090,0.818181,
     +0.727272,0.636363,0.545454,0.454545,0.363636,0.272727,0.181818,
     +0.090909,0./

	DO niw = 1,nnw
	    sumw = 0.
	    sumsig = 0.
	    DO i = istart(niw), iend(niw)
		  ii = i - istart(niw) + 1
		  sumw = sumw + w(ii)*sflx(i)
		  sumsig = sumsig + w(ii)*sflx(i)*xsec(i)
	    ENDDO
	    csig(niw) = sumsig/sumw
	ENDDO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE RAY(taur)
	IMPLICIT REAL*8(A-H,O-Z)
	PARAMETER (nnw=36,nz=50,nl=49,xns=2.547d+19)
	COMMON /ATM2/ z(nz),p(nz),t(nz),h2o(nz),o3(nz),nzz
	COMMON/O3DAT/wl(nnw),solatlas(nnw),sigatlas(nnw)
	DIMENSION taur(nl,nnw)

*     xns = STP value for molecular number density (from 5S)
	con = 1.d-26*1.d+5*xns*273.15/1013.25

	DO niw = 1,nnw
	    DO i = 1,nzz-1
		  dz = z(i) - z(i+1)
		  taur(i,niw) = sigatlas(niw)*con*
     +        (p(i)/t(i) + p(i+1)/t(i+1))*dz/2.
	    ENDDO
	ENDDO

	RETURN
	END
c----------------------------------------------------------------------
	SUBROUTINE AEROSOL(iaer,isc1,isc2,isc3,isc4,ibtype,istype,taua,
     +  wa,ga)
	IMPLICIT REAL*8(A-H,O-Z)
	PARAMETER(nw=36,nz=50,nl=49)
	COMMON /ATM2/z2(nz),p2(nz),t2(nz),q2(nz),o32(nz),nzz
	COMMON /ATM/z(nz),p(nz),t(nz),q(nz),o3(nz)
	COMMON/O3DAT/wl(nw),solatlas(nw),sigatlas(nw)
*     BLOCK DATA PRFD provides aerosol attenuation coefficients at
*     0.55 microns.
	COMMON/PRFD/zht(34),hz2k(34,5),fawi50(34),fawi23(34),spsu50(34),
     1spsu23(34),bastfw(34),vumofw(34),hivufw(34),exvufw(34),bastss(34),
     2vumoss(34),hivuss(34),exvuss(34),upnatm(34),vutono(34),
     3vutoex(34),exupat(34)

*     BLOCK DATA EXTD provides scaled aeosol extinction and absorption
*     coefficients and asymmetry factors for 15 wavelengths between
*     0.2 and 4.5 microns. For altitude regions 0-2km and>2-10km
*     (RURAL,URBAN,OFCEANIC,TROPOSPHERIC) aerosol parameters also vary
*     with relative humidity (0,70,80 and 90%).
	COMMON /EXTD  /vx2(15),
     A rurext(15,4),rurabs(15,4),rurg(15,4),
     B urbext(15,4),urbabs(15,4),urbg(15,4),
     C ocnext(15,4),ocnabs(15,4),ocng(15,4),
     D troext(15,4),troabs(15,4),trog(15,4),
     E fg1ext(15),fg1abs(15),fg1g(15),
     F fg2ext(15),fg2abs(15),fg2g(15),
     G bstext(15),bstabs(15),bstg(15),
     H avoext(15),avoabs(15),avog(15),
     I fvoext(15),fvoabs(15),fvog(15),
     J dmeext(15),dmeabs(15),dmeg(15)

	DIMENSION  asf(34)
	DIMENSION  rh(nz), irh(nz), asf1(nz)
	DIMENSION  taua(nl,nw), wa(nl,nw), ga(nl,nw)
	DIMENSION  ubext(nz,nw),uom(nz,nw),ug(nz,nw)

*     Initialize aerosol parameters

	DO l = 1,nzz-1
	DO iw = 1,nw
	   taua(l,iw) = 0.0
	   wa(l,iw) = 0.0
	   ga(l,iw) = 0.0
	  END DO
	END DO

	IF (iaer  ==  0) RETURN

*     Build aerosol scaling factor (asf) profile from components
*     for the four aerosol zones (0-2, >2-10, >10-30, >30-100km).
*     Define scaling factor indices in the above PARAMETER statement.

	DO i = 1,3          ! 0-2km
!         isc1 = 1 = 50km visibility
!         isc1 = 2 = 23km
!         isc1 = 3 = 10km
!         isc1 = 4 = 5km
!         isc1 = 5 = 2km
!         isc1 = 6 = (50km + 23km)/2
	    IF(isc1  ==  6)THEN
		  asf(i) = (hz2k(i,1) + hz2k(i,2))/2.
	    ELSE
		  asf(i) = hz2k(i,isc1)
	    ENDIF
	END DO

	DO i = 4,11         ! >2-10km
	    SELECT CASE (isc2)
	    CASE (1)
		  asf(i) = fawi50(i)
	    CASE (2)
		  asf(i) = fawi23(i)
	    CASE (3)
		  asf(i) = spsu50(i)
	    CASE (4)
		  asf(i) = spsu23(i)
	    END SELECT
	END DO

	DO i = 12,27        ! >10-30km
	    SELECT CASE (isc3)
	    CASE (1)
		  asf(i) = bastfw(i)
	    CASE (2)
		  asf(i) = vumofw(i)
	    CASE (3)
		  asf(i) = hivufw(i)
	    CASE (4)
		  asf(i) = exvufw(i)
	    CASE (5)
		  asf(i) = bastss(i)
	    CASE (6)
		  asf(i) = vumoss(i)
	    CASE (7)
		  asf(i) = hivuss(i)
	    CASE (8)
		  asf(i) = exvuss(i)
	    END SELECT
	END DO

	DO i = 28,33        ! >30-100km
	    SELECT CASE (isc4)
	    CASE (1)
		  asf(i) = upnatm(i)
	    CASE (2)
		  asf(i) = vutono(i)
	    CASE (3)
		  asf(i) = vutoex(i)
	    CASE (4)
		  asf(i) = exupat(i)
	    END SELECT
	END DO

*     Interpolate from the asf v zht profile (33 heights) to get
*     asf1 v z (50 heights)

	DO i = 1,nzz
	    ii = nzz-i+1
	    IF(z2(i) .GE. 100.) THEN
		  asf1(i) = asf(33)
	    ELSE IF(z2(i) .LE. 25.) THEN
		  asf1(i) = asf(ii)
	    ELSE
		  CALL INTERP(zht,asf,z2(i),asf1(i))
	    ENDIF
	ENDDO

*     Compute relative humidity

	CALL HUMID (rh,irh)

	DO iw = 1,nw       ! START WAVELENGTH LOOP

	   wll = wl(iw)/1000.d0

*        Extract optical parameters

	 ! Tropospheric, rural, urban and maritime aerosol
	 ! models vary with relative humidity. Subroutine
	 ! INTERP1 interpolates extinction, absorption and
	 ! asymmetry factor for each wavelength.

	 DO i = 1, 22
	     CALL INTERP1 (asf1(i),vx2,wll,dmeext,dmeabs,dmeg,
     +     ubext(i,iw),uom(i,iw),ug(i,iw))
	 END DO

	 DO i = 23, 39
	    SELECT CASE (istype)
	    CASE (1)
		  CALL INTERP1 (asf1(i),vx2,wll,bstext,bstabs,bstg,
     +        ubext(i,iw),uom(i,iw),ug(i,iw))
	    CASE (2)
		  CALL INTERP1 (asf1(i),vx2,wll,avoext,avoabs,avog,
     +        ubext(i,iw),uom(i,iw),ug(i,iw))
	    CASE (3)
		  CALL INTERP1 (asf1(i),vx2,wll,fvoext,fvoabs,fvog,
     +        ubext(i,iw),uom(i,iw),ug(i,iw))
	    END SELECT
	 END DO

	 DO i = 40,47
	    CALL INTERP1 (asf1(i),vx2,wll,troext(1,irh(i)),
     +	  troabs(1,irh(i)),trog(1,irh(i)),ubext(i,iw),uom(i,iw),
     +    ug(i,iw))
	 END DO

	 DO i = 48,nzz
	    SELECT CASE (ibtype)
	    CASE (1)
		  CALL INTERP1 (asf1(i),vx2,wll,rurext(1,irh(i)),
     +        rurabs(1,irh(i)),rurg(1,irh(i)),ubext(i,iw),uom(i,iw),
     +        ug(i,iw))
	    CASE (2)
		  CALL INTERP1 (asf1(i),vx2,wll,urbext(1,irh(i)),
     +        urbabs(1,irh(i)),urbg(1,irh(i)),ubext(i,iw),uom(i,iw),
     +        ug(i,iw))
	    CASE (3)
		  CALL INTERP1 (asf1(i),vx2,wll,ocnext(1,irh(i)),
     +        ocnabs(1,irh(i)),ocng(1,irh(i)),ubext(i,iw),uom(i,iw),
     +        ug(i,iw))
	    CASE (4)
		  CALL INTERP1 (asf1(i),vx2,wll,troext(1,irh(i)),
     +        troabs(1,irh(i)),trog(1,irh(i)),ubext(i,iw),uom(i,iw),
     +        ug(i,iw))
	    CASE (5)
		  CALL INTERP1 (asf1(i),vx2,wll,fg1ext,fg1abs,fg1g,
     +        ubext(i,iw),uom(i,iw),ug(i,iw))
	    CASE (6)
		  CALL INTERP1 (asf1(i),vx2,wll,fg2ext,fg2abs,fg2g,
     +        ubext(i,iw),uom(i,iw),ug(i,iw))
	    END SELECT
	 END DO

c     Print values for one wavelength

c      IF(iw == 1) THEN
c          WRITE(3,'(''  I       Z              ASF1           UBEXT'',
c     +    ''          UOM            UG'')')
c          DO i = 1,nzz
c              WRITE(3,'(1x,i2,6e15.4)')i,z2(i),asf1(i),
c     +        ubext(i,1),uom(i,1),ug(i,1)
c          END DO
c      ENDIF

	END DO             ! END OF THE WAVEBAND LOOP

*     Loop over wavelength (1nm intervals)

	DO iw = 1,nw
	   DO i = 1,nzz-1
		dz = z2(i) - z2(i+1)
	    taua(i,iw) = ((ubext(i,iw) + ubext(i+1,iw))/2.)*dz
	    wa(i,iw)   = (uom(i,iw) + uom(i+1,iw))/2.
	    ga(i,iw)   = (ug(i,iw) + ug(i+1,iw))/2.
	 ENDDO
	ENDDO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE COMB (to3,taur,taua,wa,ga,tau,w,g)

*     Combine optical parameters for each layer

	IMPLICIT REAL*8 (A-H,O-Z)
	PARAMETER (nl=49,nz=50,nw=36)
	COMMON/O3DAT/wl(nw),solatlas(nw),sigatlas(nw)
	COMMON /ATM2/ z(nz),p(nz),t(nz),h2o(nz),o3(nz),nzz
	DIMENSION tau(nl,nw),w(nl,nw),g(nl,nw),to3(nl,nw),taur(nl,nw),
     +taua(nl,nw),wa(nl,nw),ga(nl,nw)

*     Combine taus, ssas and gs.

	DO l =1,nzz-1
	   DO iw = 1,nw
		tau(l,iw) = to3(l,iw) + taur(l,iw) + taua(l,iw)
		sumwt = taur(l,iw) + wa(l,iw)*taua(l,iw)
		w(l,iw) = sumwt/tau(l,iw)
		g(l,iw) = ga(l,iw)*wa(l,iw)*taua(l,iw)/sumwt
  	   END DO
	END DO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE UVB (iaer,rat,sol,xmusun,nsang,taut,wt,gt,
     +albedo,cldtau,cssa,cgg,tco,time,nlevel,spm,gcs,gm1,gc1,gm2,gc2,
     +gm3,gc3)

*     Uses the Delta-Eddington algorithm
*----------------------------------------------------------------------
*     DELTA-EDDINGTON VARIABLES
*     nz       = number of levels (level 1 is the TOA)
*     dtau(i)  = layer optical depth
*     om(i)    = layer single scattering albedo
*     g(i)     = layer asymmetry factor
*     nsol     = number of incident-beam zenith angles
*     musun(i) = cosine of incident beam angle
*     alb(i)   = surface albedo
*     mudept   = true, alb(i) corresponds to musun(i). false, alb(i)
*                is used for all values of musun(i)
*     nsol     = number of incident beam zenith angles
*     solflx   = incident beam flux (normal to beam) at the TOA
*     fdinc    = incident diffuse down-flux at the TOA
*     fuinc    = incident diffuse up-flux at level nz
*     nlev     = level dimension of arrays dtau, etc.
*     spm(iw,i)= measured spectral down-flux for time i
*     gcs(iw,i)= calculated spectral down-flux for time i
*     gmss(i)  = vector from spm for integration
*     gcss(i)  = vector from gcs for integration
*     gc1(i)   = integral of gcs(iw,i) over wavelength
*     gm1(i)   = integral of spm(iw,i) over wavelength
*     gc2(i)   = integral of gcs(iw,i) over time
*     gm2(i)   = integral of spm(iw,i) over time
*     gcsss(i) = vector from gcs(iw,i) for integration
*     gmsss(i) = vector from spm(iw,i) for integration
*     gc3      = daily total calculated radiation (time and wavelength
*                integral of gcs(iw,i))
*     gm3      = daily total measured radiation (time and wavelength
*                integral of spm(iw,i))
*----------------------------------------------------------------------
	IMPLICIT REAL*8(A-H,O-Z)
	PARAMETER (nw = 36, nt = 70, nl =49)
	REAL*8 musun
	LOGICAL mudept
	COMMON/O3DAT/wll(nw),solatlas(nw),sigatlas(nw)
	COMMON/IN/dtau(50),om(50),g(50),musun(20),alb(20),solflx,fdinc,
     1fuinc,mudept,nz,nzm1,nsol,nlev
	COMMON/out/dir(51,20),flxu(51,20),flxd(51,20)
	DIMENSION xmusun(nt),sol(nt),taut(nl,nw),wt(nl,nw),gt(nl,nw),
     +dfl1(nw),dfl2(nw),tco(nt),time(nt),
     +spm(nw,nt),gmss(nw),gm1(nt),gm2(nw),gmsss(nt),
     +gcs(nw,nt),gcss(nw),gc1(nt),gc2(nw),gcsss(nt)

*     Set fixed parameters
	nlev   = 50
	nz     = nlevel
	nzm1   = nz - 1
	mudept = .false.
	nsol = 1
	fdinc  = 0.d0
	fuinc  = 0.d0
	alb(1) = albedo

*       Initialize fluxes

        DO i = 1,nsang
                gc1(i) = -9.d0
                gm1(i) = -9.d0
        END DO

        DO iw = 1,nw
                gc2(iw) = -9.
                gm2(iw) = -9.
        END DO

        gc3 = -9.
        gm3 = -9.

	DO i = 1,nsang          ! Loop over time
	    IF (xmusun(i) <= 0.d0) THEN
		  CYCLE
	    ENDIF
	    musun(1) = xmusun(i)
	    ii = 0
	    DO iw = 1,nw        ! Loop over wavelength
		  ii = ii + 1
		  CALL OPPROP (iw,sol,rat,iaer,taut,wt,gt)
		  CALL DELTED
		  dfl1(ii) = dir(nz,1) + flxd(nz,1)
		  dtau(nz-3) = cldtau
                  om(nz-3) = cssa
                  g(nz-3) = cgg
		  CALL DELTED
		  dfl2(ii) = dir(nz,1) + flxd(nz,1)
		  gcs(iw,i) = (1.-tco(i))*dfl1(ii) + tco(i)*dfl2(ii)
*                 Extract 1D arrays from gcs and spm
                  gcss(iw) = gcs(iw,i)
                  gmss(iw) = spm(iw,i)
	    ENDDO

*         Integrate over wavelength (W/m**2/nm -> W/m**2)
             CALL XINT(wll,gcss,ii,1,ii,gc1(i),er)
             CALL XINT(wll,gmss,ii,1,ii,gm1(i),er)
	END DO           ! End time loop

*     Integrate over time

*     Check for time errors (time not increasing or gaps between measurements
*     that exceed 2 hours) and missing gc values. If criteria are
*     not satisfied, set fluxes to 0.
	itscan = 0
	DO i = 2,nsang
	    deltat = time(i) - time(i-1)
	    IF(deltat <= 0.d0 .OR. deltat > 2.d0 .OR. gc1(i) < 0.d0)
     +      itscan = 1
	ENDDO
	IF (itscan == 1 .OR. nsang < 4) THEN
	    gc3 = -9.
	    gm3 = -9.
	    DO iw = 1,nw
		  gc2(iw) = -9.
		  gm2(iw) = -9.
	    ENDDO
	ELSE
            CALL XINT(time,gc1,nsang,1,nsang,gc3,er)
            CALL XINT(time,gm1,nsang,1,nsang,gm3,er)
	    gc3 = 3.6*gc3          ! kJ/m**2/day
	    gm3 = 3.6*gm3
	    DO iw = 1,nw
                DO i =1,nsang
                        gcsss(i) = gcs(iw,i)
                        gmsss(i) = spm(iw,i)
                END DO
                  CALL XINT(time,gcsss,nsang,1,nsang,gc2(iw),er)
		  CALL XINT(time,gmsss,nsang,1,nsang,gm2(iw),er)
		  gc2(iw) = 3600*gc2(iw)  ! J/m**2/day/nm
                  gm2(iw) = 3600*gm2(iw)
	    ENDDO
	ENDIF

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE TRAP (x,y,n,ans)
*     Trapezoidal rule integration
	REAL*8 ans,x(*),y(*)

	ans = 0
	DO i = 1,n-1
	    ans = ans + (x(i+1) - x(i))*(y(i+1) + y(i))/2.
	ENDDO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE OPPROP (iw,sol,rat,iaer,taut,wt,gt)
	IMPLICIT REAL*8 (A-H,O-Z)
	PARAMETER (nw = 36,nl = 49)
	REAL*8 musun
	LOGICAL mudept
	COMMON/IN/dtau(50),om(50),g(50),musun(20),alb(20),solflx,fdinc,
     1fuinc,mudept,nz,nzm1,nsol,nlev
	DIMENSION sol(nw),taut(nl,nw),wt(nl,nw),gt(nl,nw)
	solflx  = sol(iw)/(1000.*rat**2)    ! W/m**2/nm
	do l = 1,nz-1
	    dtau(l) = taut(l,iw)
	    om(l)   = wt(l,iw)
	    CALL ASSYM (iaer,gt(l,iw),g(l))
	end do

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE ASSYM (iaer,gg,g)
	REAL*8 gg,g
	IF(iaer == 0) THEN
	    g = 0.d0
	ELSE
	    g = gg
	ENDIF
	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE INTERP(x,y,xp,yp)
	REAL*8 x(34),y(34), xd, xp, yp, dx
	DO n = 2,33
	    xd = xp - x(n)
	    IF(xd < 0.) THEN
		  EXIT
	    ELSE
		  CYCLE
	    ENDIF
	END DO
	dx = x(n) - x(n-1)
	yp = y(n) + (y(n) - y(n-1))*xd/dx
	RETURN
	END
*---------------------------------------------------------------------
	SUBROUTINE HUMID (rh,irh)
	IMPLICIT REAL*8(A-H,O-Z)
	PARAMETER (nz=50)

*     Computes RH for the first 10km where aerosol is humidity-
*     dependent.

	COMMON /ATM2/z(nz),p(nz),t(nz),q(nz),o3(nz),nzz
	DIMENSION rh(nz),irh(nz)

*     From Lowtran 5 (p.217-219)

	rv = 4.615e-3    ! gas constant for water vapour
	t0 = 273.15

*     Initialize arrays
	DO j = 1,nzz
	    rh(j) = -9.
	    irh(j) = -9
	ENDDO

	DO j = 1,nzz
	    IF(z(j) <= 10.)THEN
*             Convert humidity in ppmv to g/kg
		  rho = (p(j)/t(j))*1.d5/287.053d0
		  qq = (q(j)/1.d06)*(18.016/28.966)*rho
		  tt = t(j) - 273.15
		  a = t0/(t0 + tt)
		  qs = a*exp(18.9766 - 14.9595*a - 2.4388*a*a)
		  rstar = p(j)/(rv*t(j))
		  rh(j) = 100.*(qq/qs)*((rstar-qs)/(rstar-qq))
*             Assign RH index (i.e. IRH =1,2,3,4 for RH =0,70,80,90%)
		  IF (rh(j) .LT. 35.) THEN
			irh(j) = 1
		  ELSE IF (rh(j) .GE. 35. .AND. rh(j) .LT. 75.) THEN
			irh(j) = 2
		  ELSE IF (rh(j) .GE. 75. .AND. rh(j) .LT. 85.) THEN
			irh(j) = 3
		  ELSE
			irh(j) = 4
		  ENDIF
	    ENDIF
	END DO

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE INTERP1 (asf,vx,x,ext,ab,g,extt,w,gg)
	REAL*8 ext(15),ab(15),g(15),vx(15),xd,x,vxd,extt,asf,gg,w,abso
*     Imports 1-D arrays (ext,abs,g) with subscripts of wavelength
*     and returns interpolated values for wavelength x
	DO n = 2,15
	    xd= x-vx(n)
	    IF (xd < 0.) THEN
		  EXIT
	    ELSE
		  CYCLE
	    ENDIF
	END DO
	vxd = vx(n)-vx(n-1)
	extt = ext(n) + (ext(n)-ext(n-1))*xd/vxd
	abso = ab(n) + (ab(n)-ab(n-1))*xd/vxd
	gg = g(n) + (g(n)-g(n-1))*xd/vxd
	w = 1. - abso/extt
	extt = extt*asf

	RETURN
	END
*----------------------------------------------------------------------
	SUBROUTINE DELTED
      IMPLICIT REAL*8(A-H,O-Z)
	REAL*8 musun,lm,lmdtau
	LOGICAL mudept
	COMMON/IN/dtau(50),om(50),g(50),musun(20),alb(20),solflx,fdinc,
     1fuinc,mudept,nz,nzm1,nsol,nlev
	COMMON/out/dir(51,20),flxu(51,20),flxd(51,20)
	COMMON/INTERN/ss(98,5),cc(98,5),work(98,3),x(98),tau(50),lm(50),
     1pp(50),ex(50),exsun(50),alph(50),beta(50),tx(50),ty(50),tz(50),
     2isav(50),flxsun(20),nl2
c      COMMON/INTERN/ss(100,5),cc(100,5),work(100,3),x(100),tau(51),
c     1lm(51),pp(51),ex(51),exsun(51),alph(51),beta(51),tx(51),ty(51),
c     2tz(51),isav(51),flxsun(20),nl2
	DATA c1/0.66666666666667d0/
C************************* 10 X COMPUTER PRECISION *************************
	DATA prec/1.d-13/
C************************* CUT-OFF POINT FOR LM(I)*DTAUP ******************
	DATA cutpt/14.d0/
C 
C        Set incident flux at top of atmosphere
C 
	DO np=1,nsol
	    flxsun(np) = musun(np)*solflx
	ENDDO
	nl2 = 2 * nlev - 2
c
c        Scale optical depth, single-scattering albedo, and asymmetry factor
c        and calculate various funtions of these variables
c
	tau(1) = 0.d0
	do i = 1,nzm1
	    ff = g(i)**2
	    gp = g(i)/(1.d0+g(i))
	    scale = 1.d0-ff*om(i)
	    omp = (1.d0-ff)*om(i)/scale
	    if(om(i).eq.1.d0) omp = 1.d0-prec
	    t1 = 1.d0 - omp*gp
	    lm(i) =  dsqrt(3.d0*(1.d0-omp)*t1)
	    pp(i) = c1*lm(i)/t1
	    t1 = gp*(1.d0 - omp)
	    tx(i) = 0.75d0*solflx*omp*(1.d0+t1)
	    ty(i) = 0.5d0*solflx*omp
	    tz(i) = 3.d0*t1
	    isav(i) = i
	    dtaup = scale*dtau(i)
	    lmdtau = lm(i)*dtaup
c
c         test for a layer which is so highly absorbing that it would
c         cause ill-conditioning in the penta-diagonal matrix. if one
c         is found, subdivide it appropriately.
c
	    if (lmdtau.gt.cutpt) go to 15
	    ex(i) =  dexp(lmdtau)
	    tau(i+1) = tau(i) + dtaup
	ENDDO
	isav(nz) = nz
c
c        normal calculation
c
	if (nz.gt.2) call nlayde(nz)
	go to 500
c
c        sidestep potential ill-conditioning by subdividing offending
c        layer, and others like it.
c
  15  layers = nzm1
	ktr = i
c
  20  nsub = lmdtau/cutpt + 1.d0
	dtaup = dtaup/dble(nsub)
	ex(i) =  dexp(lm(i)*dtaup)
	tau(i+1) = tau(i) + dtaup
	ip1 = i+1
	iup = i + nsub
	iupm1 = iup - 1
	do ii = ip1,iupm1
	    lm(ii) = lm(i)
	    pp(ii) = pp(i)
	    tx(ii) = tx(i)
	    ty(ii) = ty(i)
	    tz(ii) = tz(i)
	    ex(ii) = ex(i)
	    tau(ii+1) = tau(ii) + dtaup
	ENDDO
	ktr = ktr + 1
	isav(ktr) = isav(ktr-1) + nsub
	layers = layers + nsub - 1
	if (layers.gt.nlev) stop
c
	if(iup.gt.layers) go to 100
	do i = iup,layers
	    ff = g(ktr)**2
	    gp = g(ktr)/(1.d0+g(ktr))
	    scale = 1.d0-ff*om(ktr)
	    omp = (1.d0-ff)*om(ktr)/scale
	    if (om(ktr).eq.1.d0) omp = 1.d0-prec
	    t1 = 1.d0 - omp*gp
	    lm(i) =  dsqrt(3.d0*(1.d0-omp)*t1)
	    pp(i) = c1*lm(i)/t1
	    t1 = gp*(1.d0-omp)
	    tx(i) = 0.75d0 * solflx * omp * (1.d0+t1)
	    ty(i) = 0.5d0 * solflx * omp
	    tz(i) = 3.d0*t1
	    dtaup = scale * dtau(ktr)
	    lmdtau = lm(i)*dtaup
c
c         test for a layer so highly absorbing that it would cause
c         ill-conditioning in the penta-diagonal matrix. if one is
c         found, subdivide it appropriately.
c
	    if (lmdtau.gt.cutpt) go to 20
	    ex(i) =  dexp(lmdtau)
	    tau(i+1) = tau(i) + dtaup
	    ktr = ktr + 1
	    isav(ktr) = isav(ktr - 1) + 1
	ENDDO
c
 100  CALL NLAYDE(LAYERS + 1)
c
 500  CONTINUE
	RETURN
	END
c----------------------------------------------------------------------
	SUBROUTINE NLAYDE(IB)
C 
C        MULTI-LAYER DELTA-EDDINGTON APPROXIMATION
C        IB = NUMBER OF LEVELS
C 
C        THE TOP AND BOTTOM BOUNDARY CONDITIONS PLUS FLUX CONTINUITY
C        CONDITIONS AT EACH INTERIOR LEVEL FORM A PENTA-DIAGONAL SYSTEM 
C        OF 2*IB - 2  EQUATIONS FOR THE UNKNOWN CONSTANTS (2 FOR EACH LAYER)
C        THE COLUMNS OF THE -SS- ARRAY CONTAIN THE DIAGONALS OF THE COEFFI- 
C        CIENT MATRIX. THE LOWERMOST DIAGONAL IN COLUMN 1, ETC. (THIS IS THE
C        SO-CALLED BAND STORAGE MODE REQUIRED BY IMSL ROUTINE LEQT2B).
C 
      IMPLICIT REAL*8(A-H,O-Z)
	REAL*8 MUSUN,LM
      LOGICAL MUDEPT
	COMMON/IN/dtau(50),om(50),g(50),musun(20),alb(20),solflx,fdinc,
     1fuinc,mudept,nz,nzm1,nsol,nlev
	COMMON/out/dir(51,20),flxu(51,20),flxd(51,20)
	COMMON/INTERN/ss(98,5),cc(98,5),work(98,3),x(98),tau(50),lm(50),
     1pp(50),ex(50),exsun(50),alph(50),beta(50),tx(50),ty(50),tz(50),
     2isav(50),flxsun(20),nl2
c      COMMON/INTERN/ss(100,5),cc(100,5),work(100,3),x(100),tau(51),
c     1lm(51),pp(51),ex(51),exsun(51),alph(51),beta(51),tx(51),ty(51),
c     2tz(51),isav(51),flxsun(20),nl2
C 
      IBM1 = IB - 1 
      LAST = 2*IB-2 
      SS(1,1) = 0.D0
      SS(1,2) = 0.D0
      SS(1,5) = 0.D0
      SS(1,3) = (1.D0-PP(1))/EX(1)
      SS(1,4) = (1.D0+PP(1))*EX(1)
      LASTM2 = LAST - 2 
      DO 100 J = 2,LASTM2,2 
      I = J/2 
      IP1 = I + 1 
      SS(J,1) = 0.D0 
      SS(J,2) = 1.D0 
      SS(J,3) = 1.D0 
      SS(J,4) = -1.D0/EX(IP1)
      SS(J,5) = -EX(IP1)
      SS(J+1,1) = -PP(I)
      SS(J+1,2) = PP(I) 
      SS(J+1,3) = PP(IP1)/EX(IP1) 
      SS(J+1,4) = -PP(IP1)*EX(IP1)
      SS(J+1,5) = 0.D0 
 100  CONTINUE
      SS(LAST,1) = 0.D0
      SS(LAST,4) = 0.D0
      SS(LAST,5) = 0.D0
      IF (MUDEPT) GO TO 150 
      SS(LAST,2) = 1.D0 + PP(IBM1) - ALB(1)*(1.D0-PP(IBM1)) 
      SS(LAST,3) = 1.D0 - PP(IBM1) - ALB(1)*(1.D0+PP(IBM1)) 
C 
C        CALCULATE THE L-U DECOMPOSITION OF PENTA-DIAGONAL MATRIX -SS-
C 
C         LEQ2B CALL FOR TESTING PURPOSES 
C        CALL LEQ2B(SS,LAST,2,2,NL2,X,1,NL2,1,WORK,NL2,WORK(1,8)) 
C 
C        LEQT1B DESTROYS THE INPUT COEFF MATRIX, SO SINCE WE MUST 
C        PRESERVE -SS-, WE MUST DESTROY -CC- INSTEAD. 
C 
      DO 120 IC = 1,5 
      DO 120 IR = 1,LAST
 120  CC(IR,IC) = SS(IR,IC) 
      CALL LEQT1B(CC,LAST,2,2,NL2,X,1,NL2,1,WORK) 
C 
C        FOR EACH SUN ANGLE, CALCULATE THE R.H.S. OF THE BANDED SYSTEM. 
C        SOLVE, AND USE THE SOLUTION TO CONSTRUCT THE FLUXES AT EACH LEVEL. 
C 
  150 DO 400 NP=1,NSOL
      RMU0 = 1.D0/MUSUN(NP) 
      T1 = RMU0**2 - LM(1)**2 
      IF (T1.EQ.0.D0) STOP 
      ALPH(1) = TX(1)/T1
      BETA(1) = TY(1)*(MUSUN(NP)*TZ(1)+RMU0)/T1 
      X(1) = ALPH(1) + BETA(1) + FDINC
C 
      DO 200 J = 2,LASTM2,2 
      I = J/2 
      IP1 = I + 1 
      T1 = RMU0**2 - LM(IP1)**2 
      ALPH(IP1) = TX(IP1)/T1
      BETA(IP1) = TY(IP1)*(MUSUN(NP)*TZ(IP1)+RMU0)/T1 
      EXSUN(IP1) = DEXP(-RMU0*TAU(IP1))
      X(J) = (ALPH(I)-ALPH(IP1)) * EXSUN(IP1) 
      X(J+1) = (BETA(I)-BETA(IP1))*EXSUN(IP1) 
 200  CONTINUE
      EXSUN(IB) = DEXP(-RMU0*TAU(IB))
C 
      IF (MUDEPT) GO TO 250 
      ALBDO = ALB(1)
      IJOB = 2
      GO TO 275 
 250  ALBDO = ALB(NP) 
      IJOB = 0
      SS(LAST,2) = (1.D0+PP(IBM1)) - ALB(NP)*(1.D0-PP(IBM1))
      SS(LAST,3) = 1.D0-PP(IBM1) - ALB(NP)*(1.D0+PP(IBM1))
      DO 260 IC = 1,5 
      DO 260 IR = 1,LAST
 260  CC(IR,IC) = SS(IR,IC) 
C 
 275  X(LAST) = (ALPH(IBM1)-BETA(IBM1) + ALBDO*(FLXSUN(NP)-ALPH(IBM1)-
     1BETA(IBM1)))*EXSUN(IB) + FUINC
C 
C        SOLVE PENTA-DIAGONAL SYSTEM WITH R.H.S. -X-. SOLN GOES INTO -X-
C 
C        CALL LEQT2B(SS,LAST,2,2,NL2,X,1,NL2,IJOB,WORK(1,8))
C 
      CALL LEQT1B(CC,LAST,2,2,NL2,X,1,NL2,IJOB,WORK)
C 
      DIR(1,NP) = FLXSUN(NP)
      FLXD(1,NP)  = FDINC 
      FLXU(1,NP) = (1.D0+PP(1))/EX(1)*X(1)+(1.D0-PP(1))*EX(1)*X(2)-
     1ALPH(1)+BETA(1) 
       KTR = 2
      DO 350 I = 1,IBM1 
      IF (I+1.NE.ISAV(KTR)) GO TO 350 
      DIR(KTR,NP) = FLXSUN(NP)*EXSUN(I+1) 
      FLXD(KTR,NP) = (1.D0-PP(I))*X(2*I-1)+(1.D0+PP(I))*X(2*I)-(ALPH(I)+
     1BETA(I))*EXSUN(I+1) 
      FLXU(KTR,NP) = (1.D0+PP(I))*X(2*I-1)+(1.D0-PP(I))*X(2*I)-(ALPH(I)-
     1BETA(I))*EXSUN(I+1) 
      KTR = KTR+1 
 350  CONTINUE
 400  CONTINUE
C 
      RETURN
      END 
c----------------------------------------------------------------------
      SUBROUTINE LEQT1B (A,N,NLC,NUC,IA,B,M,IB,IJOB,XL) 
C 
C   IMSL ROUTINE NAME   - LEQT1B(JANUARY 1,1979)
C 
C   PURPOSE             - LINEAR EQUATION SOLUTION - BAND STORAGE 
C                           MODE - SPACE ECONOMIZER SOLUTION
C 
C   ARGUMENTS    A      - INPUT/OUTPUT MATRIX OF DIMENSION N BY 
C                           (NUC+NLC+1). SEE PARAMETER IJOB.
C                N      - ORDER OF MATRIX A AND THE NUMBER OF ROWS IN 
C                           B. (INPUT)
C                NLC    - NUMBER OF LOWER CODIAGONALS IN MATRIX A.
C                           (INPUT) 
C                NUC    - NUMBER OF UPPER CODIAGONALS IN MATRIX A.
C                           (INPUT) 
C                IA     - ROW DIMENSION OF MATRIX A EXACTLY AS
C                           SPECIFIED IN THE DIMENSION STATEMENT IN THE 
C                           CALLING PROGRAM. (INPUT)
C                B      - INPUT/OUTPUT MATRIX OF DIMENSION N BY M.
C                           ON INPUT, B CONTAINS THE M RIGHT-HAND SIDES 
C                           OF THE EQUATION AX = B. ON OUTPUT, THE
C                           SOLUTION MATRIX X REPLACES B. IF IJOB = 1,
C                           B IS NOT USED.
C                M      - NUMBER OF RIGHT HAND SIDES (COLUMNS IN B).
C                           (INPUT) 
C                IB     - ROW DIMENSION OF MATRIX B EXACTLY AS
C                           SPECIFIED IN THE DIMENSION STATEMENT IN THE 
C                           CALLING PROGRAM. (INPUT)
C                IJOB   - INPUT OPTION PARAMETER. IJOB = I IMPLIES WHEN 
C                           I = 0, FACTOR THE MATRIX A AND SOLVE THE
C                             EQUATION AX = B. ON INPUT, A CONTAINS THE 
C                             COEFFICIENT MATRIX OF THE EQUATION AX = B,
C                             WHERE A IS ASSUMED TO BE AN N BY N BAND 
C                             MATRIX. A IS STORED IN BAND STORAGE MODE
C                             AND THEREFORE HAS DIMENSION N BY
C                             (NLC+NUC+1). ON OUTPUT, A IS REPLACED 
C                             BY THE U MATRIX OF THE L-U DECOMPOSITION
C                             OF A ROWWISE PERMUTATION OF MATRIX A. U 
C                             IS STORED IN BAND STORAGE MODE. 
C                           I = 1, FACTOR THE MATRIX A. A CONTAINS THE
C                             SAME INPUT/OUTPUT INFORMATION AS IF 
C                             IJOB = 0. 
C                           I = 2, SOLVE THE EQUATION AX = B. THIS
C                             OPTION IMPLIES THAT LEQT1B HAS ALREADY
C                             BEEN CALLED USING IJOB = 0 OR 1 SO THAT 
C                             THE MATRIX A HAS ALREADY BEEN FACTORED. 
C                             IN THIS CASE, OUTPUT MATRICES A AND XL
C                             MUST HAVE BEEN SAVED FOR REUSE IN THE 
C                             CALL TO LEQT1B. 
C                XL     - WORK AREA OF DIMENSION N*(NLC+1). THE FIRST 
C                           NLC*N LOCATIONS OF XL CONTAIN COMPONENTS OF 
C                           THE L MATRIX OF THE L-U DECOMPOSITION OF A
C                           ROWWISE PERMUTATION OF A. THE LAST N
C                           LOCATIONS CONTAIN THE PIVOT INDICES.
C                IER    - ERROR PARAMETER. (OUTPUT) 
C                         TERMINAL ERROR
C                           IER = 129 INDICATES THAT MATRIX A IS
C                             ALGORITHMICALLY SINGULAR. 
C 
c----------------------------------------------------------------------
      IMPLICIT REAL*8(A-H,O-Z)
      DIMENSION          A(IA,5),XL(N,3),B(IB,2)
      DATA               ZERO/0.D0/,ONE/1.D0/
      JBEG = NLC+1
      NLC1 = JBEG 
      IF (IJOB .EQ. 2) GO TO 80 
      RN = N
C                                  RESTRUCTURE THE MATRIX 
C                                  FIND RECIPROCAL OF THE LARGEST 
C                                  ABSOLUTE VALUE IN ROW I
      I = 1 
      NC = JBEG+NUC 
      NN = NC 
      JEND = NC 
      IF (N .EQ. 1 .OR. NLC .EQ. 0) GO TO 25
    5 K = 1 
      P = ZERO
      DO 10 J = JBEG,JEND 
         A(I,K) = A(I,J)
         Q =  DABS(A(I,K)) 
         IF (Q .GT. P) P = Q
         K = K+1
   10 CONTINUE
      IF (P .EQ. ZERO) GO TO 135
      XL(I,NLC1) = ONE/P
      IF (K .GT. NC) GO TO 20 
      DO 15 J = K,NC
         A(I,J) = ZERO
   15 CONTINUE
   20 I = I+1 
      JBEG = JBEG-1 
      IF (JEND-JBEG .EQ. N) JEND = JEND-1 
      IF (I .LE. NLC) GO TO 5 
      JBEG = I
      NN = JEND 
   25 JEND = N-NUC
      DO 40 I = JBEG,N
         P = ZERO 
         DO 30 J = 1,NN 
            Q =  DABS(A(I,J))
            IF (Q .GT. P) P = Q 
   30    CONTINUE 
         IF (P .EQ. ZERO) GO TO 135 
         XL(I,NLC1) = ONE/P 
         IF (I .EQ. JEND) GO TO 37
         IF (I .LT. JEND) GO TO 40
         K = NN+1 
         DO 35 J = K,NC 
            A(I,J) = ZERO 
   35    CONTINUE 
   37    NN = NN-1
   40 CONTINUE
      L = NLC 
C                                  L-U DECOMPOSITION
      DO 75 K = 1,N 
         P =  DABS(A(K,1))*XL(K,NLC1)
         I = K
         IF (L .LT. N) L = L+1
         K1 = K+1 
         IF (K1 .GT. L) GO TO 50
         DO 45 J = K1,L 
            Q = DABS(A(J,1))*XL(J,NLC1)
            IF (Q .LE. P) GO TO 45
            P = Q 
            I = J 
   45    CONTINUE 
   50    XL(I,NLC1) = XL(K,NLC1)
         XL(K,NLC1) = I 
C                                  SINGULARITY FOUND
         Q = RN+P 
         IF (Q .EQ. RN) GO TO 135 
C                                  INTERCHANGE ROWS I AND K 
         IF (K .EQ. I) GO TO 60 
         DO 55 J = 1,NC 
            P = A(K,J)
            A(K,J) = A(I,J) 
            A(I,J) = P
   55    CONTINUE 
   60    IF (K1 .GT. L) GO TO 75
         DO 70 I = K1,L 
            P = A(I,1)/A(K,1) 
            IK = I-K
            XL(K1,IK) = P 
            DO 65 J = 2,NC
               A(I,J-1) = A(I,J)-P*A(K,J) 
   65    CONTINUE 
         A(I,NC) = ZERO 
   70    CONTINUE 
   75 CONTINUE
      IF (IJOB .EQ. 1) GO TO 9005 
C                                  FORWARD SUBSTITUTION 
   80 L = NLC 
      DO 105 K = 1,N
         I = XL(K,NLC1) 
         IF (I .EQ. K) GO TO 90 
         DO 85 J = 1,M
            P = B(K,J)
            B(K,J) = B(I,J) 
            B(I,J) = P
   85    CONTINUE 
   90    IF (L .LT. N) L = L+1
         K1 = K+1 
         IF (K1 .GT. L) GO TO 105 
         DO 100 I = K1,L
            IK = I-K
            P = XL(K1,IK) 
            DO 95 J = 1,M 
               B(I,J) = B(I,J)-P*B(K,J) 
   95       CONTINUE
  100    CONTINUE 
  105 CONTINUE
C                                  BACKWARD SUBSTITUTION
      JBEG = NUC+NLC
      DO 125 J = 1,M
         L = 1
         K1 = N+1 
         DO 120 I = 1,N 
            K = K1-I
            P = B(K,J)
            IF (L .EQ. 1) GO TO 115 
            DO 110 KK = 2,L 
               IK = KK+K
		   P = P-A(K,KK)*B(IK-1,J)
  110       CONTINUE
  115       B(K,J) = P/A(K,1) 
            IF (L .LE. JBEG) L = L+1
  120    CONTINUE 
  125 CONTINUE
 9005 RETURN
  135 CONTINUE 
      WRITE(*,888)
  888 FORMAT(1X,'MATRIX ALGORITHMICALLY SINGULAR')
      STOP
      END
c----------------------------------------------------------------------
	subroutine xint(x,f,n,ia,ib,int,er)
*     An algorithm for the integration of unequally spaced data
*     P.E.Gill and G.F.Miller (1972), J.Comp.
*     ia      =   array index for the integral's lower limit (usually 1)
*     ib      =   array index for the integral's upper limit (usually n)
	integer n,ia,ib,i,j,k
	real*8 x(*),f(*),er,int,e,h1,h2,h3,h4,r1,r2,r3,r4,d1,d2,d3,c,s

	if((n .lt. 3).or.(ia .gt. ib).or.(ia .lt. 1).or.(ib .gt. n))then
	    print '('' Error'')'
	endif
	if (ia .eq. ib) then
	    int = 0.
	    er = 0.
	    return
	endif
	data e,s,c,r4 /4*0.0/
c      j = 3
c      k = n-1
	if (ia .eq. n-1 .and. n .gt. 4) then
	    j =  n-2
	elseif(ia .gt. 3) then
	    j = ia
	else
	    j = 3   ! When the integral starts with the first array value
	endif
	if (ib .eq. 1 .and. n .gt. 4) then
	    k =3
	elseif(n  .gt. ib+2) then
	    k = ib+1
	else
	    k = n-1 ! When the integral starts with the last array value
	endif

	do i = j,k
	    if (i .eq. j) then
		  h2 = x(j-1)-x(j-2)
		  d3 = (f(j-1)-f(j-2))/h2
		  h3 = x(j)-x(j-1)
		  d1 = (f(j)-f(j-1))/h3
		  h1 = h2+h3
		  d2 = (d1-d3)/h1
		  h4 = x(j+1)-x(j)
		  r1 = (f(j+1)-f(j))/h4
		  r2 = (r1-d1)/(h4+h3)
		  h1 = h1+h4
		  r3 = (r2-d2)/h1
		  if (ia .eq. 1) then
		    int = h2*(f(1)+h2*(d3/2.-h2*(d2/6.-(h2+2.*h3)*r3/12.
     +                  )))
			s = -h2**3*(h2*(3.*h2+5.*h4)+10.*h3*h1)/60.
		  endif
	    else
		  h4 = x(i+1)-x(i)
		  r1 = (f(i+1)-f(i))/h4
		  r4 = h4+h3
		  r2 = (r1-d1)/r4
		  r4 = r4+h2
		  r3 = (r2-d2)/r4
		  r4 = r4+h1
		  r4 = (r3-d3)/r4
	    endif
	    if (i .le. ib .and. i .gt. ia) then
		  int =int+h3*((f(i)+f(i-1))/2.-h3*h3*(d2+r2+(h2-h4)*r3)
     +              /12.)
		  c = h3**3*(2.*h3*h3+5.*(h3*(h4+h2)+2.*h4*h2))/120.
		  e = e+(c+s)*r4
		  s = c
		  if (i .eq. j) s = 2.*c+s
	    else
		  e = e+r4*s
	    endif
	    if(i .eq. k) then
		  if(ib .eq. n) then
		      int = int+h4*(f(n)-h4*(r1/2.+h4*(r2/6.+(2.*h3+h4)*
     +            r3/12.)))
		     e = e-h4**3*r4*(h4*(3.*h4+5.*h2)+10.*h3*(h2+h3+h4))
     +            /60.
		  elseif (ib .ge. n-1) then
			e = e+s*r4
		  endif
	     else
		  h1 = h2
		  h2 = h3
		  h3 = h4
		  d1 = r1
		  d2 = r2
		  d3 = r3
	     endif
	enddo
	er = e
	return
	end
c----------------------------------------------------------------------
	SUBROUTINE METDTA(MODEL)
c     Select a standard atmospheric model
	IF (model .EQ. 1) CALL L7TROPIC
	IF (model .EQ. 2) CALL L7MIDSUM
	IF (model .EQ. 3) CALL L7MIDWIN
	IF (model .EQ. 4) CALL L7SUBSUM
	IF (model .EQ. 5) CALL L7SUBWIN
	IF (model .EQ. 6) CALL L7US76
	RETURN
	END
c----------------------------------------------------------------------
	BLOCK DATA UVNANO
	IMPLICIT REAL*8(A-H,O-Z)
	PARAMETER(nw=36)
	COMMON/O3DAT/wl(nw),solatlas(nw),sigatlas(nw)
c     SOLLOW and RAYSIG values revised July 20, 1993 using NFLUX.FOR
c     Values were interpolated for
c     every 0.05nm - the interval for the Bass coefficients. All averaging
c     was done using D01GAF.
c     Wavelengths (nm)
	DATA wl/
     +290.,291.,292.,293.,294.,295.,296.,297.,298.,299.,
     +300.,301.,302.,303.,304.,305.,306.,307.,308.,309.,
     +310.,311.,312.,313.,314.,315.,316.,317.,318.,319.,
     +320.,321.,322.,323.,324.,325./
*     Triangularly averaged solar spectrum values from van Hoosier.
        DATA solatlas/
     +592.32,605.73,568.57,536.78,501.15,506.52,607.94,417.03,596.11,
     +544.51,450.32,487.27,397.48,668.10,588.83,660.80,530.88,625.43,
     +663.84,604.18,483.83,824.53,657.76,715.97,805.91,726.37,531.16,
     +784.77,684.16,767.22,817.19,768.81,805.39,639.57,761.06,790.42/
*     Triangularly averaged-sun weightd Rayleigh scattering optical
*     cross sections using van Hoosier's solar spectrum (multiply by
*     E-26 before use)
        DATA sigatlas/
     +6.5691,6.4722,6.3772,6.2840,6.1924,6.1026,6.0144,5.9278,5.8429,
     +5.7594,5.6775,5.5970,5.5180,5.4403,5.3641,5.2892,5.2157,5.1434,
     +5.0724,5.0026,4.9341,4.8667,4.8005,4.7355,4.6715,4.6087,4.5469,
     +4.4861,4.4264,4.3677,4.3100,4.2532,4.1974,4.1425,4.0885,4.0354/
	END
c----------------------------------------------------------------------
	BLOCK DATA PRFDTA
	IMPLICIT REAL*8(A-H,O-Z)
C        AEROSOL PROFILE DATA
C        AEROSOL ATTENUATION COEFFICIENTS AT 0.55 MICRONS (/KM)
C        CALLED "AEROSOL SCALING FACTORS"
C        NO CHANGES MADE TO THE ORIGINAL LOWTRAN 7 CODE
C
CCC         0-2KM
CCC           HZ2K=5 VIS PROFILES- 50KM,23KM,10KM,5KM,2KM
CCC         >2-10KM
CCC           FAWI50=FALL/WINTER   50KM VIS
CCC           FAWI23=FALL/WINTER    23KM VIS
CCC           SPSU50=SPRING/SUMMER  50KM VIS
CCC           SPSU23=SPRING/SUMMER  23KM VIS
CCC         >10-30KM
CCC           BASTFW=BACKGROUND STRATOSPHERIC   FALL/WINTER
CCC           VUMOFW=MODERATE VOLCANIC          FALL/WINTER
CCC           HIVUFW=HIGH VOLCANIC              FALL/WINTER
CCC           EXVUFW=EXTREME VOLCANIC           FALL/WINTER
CCC           BASTSS,VUMOSS,HIVUSS,EXVUSS=      SPRING/SUMMER
CCC         >30-100KM
CCC           UPNATM=NORMAL UPPER ATMOSPHERIC
CCC           VUTONO=TRANSITION FROM VOLCANIC TO NORMAL
CCC           VUTOEX=TRANSITION FROM VOLCANIC TO EXTREME
CCC           EXUPAT=EXTREME UPPER ATMOSPHERIC
      COMMON/PRFD  /ZHT(34),HZ2K(34,5),FAWI50(34),FAWI23(34),SPSU50(34),
     1SPSU23(34),BASTFW(34),VUMOFW(34),HIVUFW(34),EXVUFW(34),BASTSS(34),
     2VUMOSS(34),HIVUSS(34),EXVUSS(34),UPNATM(34),VUTONO(34),
     3VUTOEX(34),EXUPAT(34)
	DATA ZHT/
     *    0.,    1.,    2.,    3.,    4.,    5.,    6.,    7.,    8.,
     *    9.,   10.,   11.,   12.,   13.,   14.,   15.,   16.,   17.,
     *   18.,   19.,   20.,   21.,   22.,   23.,   24.,   25.,   30.,
     *   35.,   40.,   45.,   50.,   70.,  100.,99999./
	 DATA HZ2K(1,1),HZ2K(1,2),HZ2K(1,3),HZ2K(1,4),HZ2K(1,5)/
     1 6.62E-02, 1.58E-01, 3.79E-01, 7.70E-01, 1.94E+00/
	 DATA HZ2K(2,1),HZ2K(2,2),HZ2K(2,3),HZ2K(2,4),HZ2K(2,5)/
     1 4.15E-02, 9.91E-02, 3.79E-01, 7.70E-01, 1.94E+00/
	 DATA HZ2K(3,1),HZ2K(3,2),HZ2K(3,3),HZ2K(3,4),HZ2K(3,5)/
     1 2.60E-02, 6.21E-02, 6.21E-02, 6.21E-02, 6.21E-02/
	DATA FAWI50  /3*0.,
     1 1.14E-02, 6.43E-03, 4.85E-03, 3.54E-03, 2.31E-03, 1.41E-03,
     2 9.80E-04,7.87E-04,23*0./
	DATA FAWI23              /3*0.,
     1 2.72E-02, 1.20E-02, 4.85E-03, 3.54E-03, 2.31E-03, 1.41E-03,
     2 9.80E-04,7.87E-04,23*0./
	DATA  SPSU50              / 3*0.,
     1 1.46E-02, 1.02E-02, 9.31E-03, 7.71E-03, 6.23E-03, 3.37E-03,
     2 1.82E-03  ,1.14E-03,23*0./
	DATA  SPSU23              / 3*0.,
     1 3.46E-02, 1.85E-02, 9.31E-03, 7.71E-03, 6.23E-03, 3.37E-03,
     2 1.82E-03  ,1.14E-03,23*0./
	DATA BASTFW       /11*0.,
     1           7.14E-04, 6.64E-04, 6.23E-04, 6.45E-04, 6.43E-04,
     2 6.41E-04, 6.00E-04, 5.62E-04, 4.91E-04, 4.23E-04, 3.52E-04,
     3 2.95E-04, 2.42E-04, 1.90E-04, 1.50E-04, 3.32E-05 ,7*0./
	DATA    VUMOFW       /11*0.,
     1           1.79E-03, 2.21E-03, 2.75E-03, 2.89E-03, 2.92E-03,
     2 2.73E-03, 2.46E-03, 2.10E-03, 1.71E-03, 1.35E-03, 1.09E-03,
     3 8.60E-04, 6.60E-04, 5.15E-04, 4.09E-04, 7.60E-05 ,7*0./
	DATA    HIVUFW       /11*0.,
     1           2.31E-03, 3.25E-03, 4.52E-03, 6.40E-03, 7.81E-03,
     2 9.42E-03, 1.07E-02, 1.10E-02, 8.60E-03, 5.10E-03, 2.70E-03,
     3 1.46E-03, 8.90E-04, 5.80E-04, 4.09E-04, 7.60E-05 ,7*0./
	DATA    EXVUFW       /11*0.,
     1           2.31E-03, 3.25E-03, 4.52E-03, 6.40E-03, 1.01E-02,
     2 2.35E-02, 6.10E-02, 1.00E-01, 4.00E-02, 9.15E-03, 3.13E-03,
     3 1.46E-03, 8.90E-04, 5.80E-04, 4.09E-04, 7.60E-05 ,7*0./
	DATA    BASTSS       /11*0.,
     1           7.99E-04, 6.41E-04, 5.17E-04, 4.42E-04, 3.95E-04,
     2 3.82E-04, 4.25E-04, 5.20E-04, 5.81E-04, 5.89E-04, 5.02E-04,
     3 4.20E-04, 3.00E-04, 1.98E-04, 1.31E-04, 3.32E-05 ,7*0./
	DATA    VUMOSS       /11*0.,
     1           2.12E-03, 2.45E-03, 2.80E-03, 2.89E-03, 2.92E-03,
     2 2.73E-03, 2.46E-03, 2.10E-03, 1.71E-03, 1.35E-03, 1.09E-03,
     3 8.60E-04, 6.60E-04, 5.15E-04, 4.09E-04, 7.60E-05 ,7*0./
	DATA    HIVUSS       /11*0.,
     1           2.12E-03, 2.45E-03, 2.80E-03, 3.60E-03, 5.23E-03,
     2 8.11E-03, 1.20E-02, 1.52E-02, 1.53E-02, 1.17E-02, 7.09E-03,
     3 4.50E-03, 2.40E-03, 1.28E-03, 7.76E-04, 7.60E-05 ,7*0./
	DATA    EXVUSS       /11*0.,
     1           2.12E-03, 2.45E-03, 2.80E-03, 3.60E-03, 5.23E-03,
     2 8.11E-03, 1.27E-02, 2.32E-02, 4.85E-02, 1.00E-01, 5.50E-02,
     3 6.10E-03, 2.40E-03, 1.28E-03, 7.76E-04, 7.60E-05 ,7*0./
	DATA UPNATM       /26*0.,
     1 3.32E-05, 1.64E-05, 7.99E-06, 4.01E-06, 2.10E-06, 1.60E-07,
     2 9.31E-10, 0.      /
	DATA VUTONO       /26*0.,
     1 7.60E-05, 2.45E-05, 7.99E-06, 4.01E-06, 2.10E-06, 1.60E-07,
     2 9.31E-10, 0.      /
	DATA VUTOEX       /26*0.,
     1 7.60E-05, 7.20E-05, 6.95E-05, 6.60E-05, 5.04E-05, 1.03E-05,
     2 4.50E-07, 0.      /
	DATA EXUPAT       /26*0.,
     1 3.32E-05, 4.25E-05, 5.59E-05, 6.60E-05, 5.04E-05, 1.03E-05,
     2 4.50E-07, 0.      /
	END
C----------------------------------------------------------------------
	BLOCK DATA EXTDTA
	IMPLICIT REAL*8(A-H,O-Z)
C     CHANGES FROM LOWTRAN 7 - JUNE 8, 1992 (JAD)
C        -REMOVED CLOUD DATA
C        -DOUBLE DIMENSIONED RH-DEDPENDENT VARIABLES
C        -ONLY SOLAR WAVELENGTHS: 0.2 - 4.5 MICRONS
CCC
CCC   ALTITUDE REGIONS FOR AEROSOL EXTINCTION COEFFICIENTS
CCC
CCC
CCC         0-2KM
CCC           RUREXT=RURAL EXTINCTION   RURABS=RURAL ABSORPTION
CCC           RURG=RURAL ASYMMETRY FACTORS
CCC           URBEXT=URBAN EXTINCTION   URBABS=URBAN ABSORPTION
CCC           URBG=URBAN ASYMMETRY FACTORS
CCC           OCNEXT=MARITIME EXTINCTION  OCNABS=MARITIME ABSORPTION
CCC           OCNG=MARITIME ASYMMETRY FACTORS
CCC           TROEXT=TROPSPHER EXTINCTION  TROABS=TROPOSPHER ABSORPTION
CCC           TROG=TROPSPHERIC ASYMMETRY FACTORS
CCC           FG1EXT=FOG1 .2KM VIS EXTINCTION  FG1ABS=FOG1 ABSORPTION
CCC           FG1G=FOG1 ASYMMETRY FACTORS
CCC           FG2EXT=FOG2 .5KM VIS EXTINCTION  FG2ABS=FOG2 ABSORPTION
CCC           FG2G=FOG2 ASYMMETRY FACTORS
CCC         >2-10KM
CCC           TROEXT=TROPOSPHER EXTINCTION  TROABS=TROPOSPHER ABSORPTION
CCC           TROG=TROPOSPHERIC ASYMMETRY FACTORS
CCC         >10-30KM
CCC           BSTEXT=BACKGROUND STRATOSPHERIC EXTINCTION
CCC           BSTABS=BACKGROUND STRATOSPHERIC ABSORPTION
CCC           BSTG=BACKGROUND STRATOSPHERIC ASYMMETRY FACTORS
CCC           AVOEXT=AGED VOLCANIC EXTINCTION
CCC           AVOABS=AGED VOLCANIC ABSORPTION
CCC           AVOG=AGED VOLCANIC ASYMMETRY FACTORS
CCC           FVOEXT=FRESH VOLCANIC EXTINCTION
CCC           FVOABS=FRESH VOLCANIC ABSORPTION
CCC           FVOG=FRESH VOLCANIC ASYMMETRY FACTORS
CCC         >30-100KM
CCC           DMEEXT=METEORIC DUST EXTINCTION
CCC           DMEABS=METEORIC DUST ABSORPTION
CCC           DMEG=METEORIC DUST ASYMMETRY FACTORS
C
C     AEROSOL EXTINCTION AND ABSORPTION COEFFICIENTS
C     AND ASYMMETRY FACTOR
C
C     15 WAVELENGTHS
C     4 RELATIVE HUMIDITIES (0,70,80,90%)
C
	COMMON /EXTD  /VX2(15),
     A RUREXT(15,4),RURABS(15,4),RURG(15,4),
     B URBEXT(15,4),URBABS(15,4),URBG(15,4),
     C OCNEXT(15,4),OCNABS(15,4),OCNG(15,4),
     D TROEXT(15,4),TROABS(15,4),TROG(15,4),
     E FG1EXT(15),FG1ABS(15),FG1G(15),
     F FG2EXT(15),FG2ABS(15),FG2G(15),
     G BSTEXT(15),BSTABS(15),BSTG(15),
     H AVOEXT(15),AVOABS(15),AVOG(15),
     I FVOEXT(15),FVOABS(15),FVOG(15),
     J DMEEXT(15),DMEABS(15),DMEG(15)
C
C     15 WAVELENGTHS IN MICOMETERS
	DATA VX2 /
     *   .2000,   .3000,   .3371,   .5500,   .6943,  1.0600,  1.5360,
     *  2.0000,  2.2500,  2.5000,  2.7000,  3.0000,  3.3923,  3.7500,
     *  4.5000/
	DATA (RUREXT(I,1),I=1,15) /
     1 2.09291, 1.74582, 1.60500, 1.00000,  .75203,  .41943,  .24070,
     2  .14709,  .13304,  .12234,  .13247,  .11196,  .10437,  .09956,
     3  .09190/
	DATA (RUREXT(I,2),I=1,15) /
     1 2.09544, 1.74165, 1.59981, 1.00000,  .75316,  .42171,  .24323,
     2  .15108,  .13608,  .12430,  .13222,  .13823,  .11076,  .10323,
     3  .09475/
	DATA (RUREXT(I,3),I=1,15) /
     1 2.07082, 1.71456, 1.57962, 1.00000,  .76095,  .43228,  .25348,
     2  .16456,  .14677,  .13234,  .13405,  .20316,  .12873,  .11506,
     3  .10481/
	DATA (RUREXT(I,4),I=1,15) /
     1 1.66076, 1.47886, 1.40139, 1.00000,  .80652,  .50595,  .32259,
     2  .23468,  .20772,  .18532,  .17348,  .35114,  .20006,  .17386,
     3  .16139/
	DATA (RURABS(I,1),I=1,15) /
     1  .67196,  .11937,  .08506,  .05930,  .05152,  .05816,  .05006,
     2  .01968,  .02070,  .02101,  .05652,  .02785,  .01316,  .00867,
     3  .01462/
	DATA (RURABS(I,2),I=1,15) /
     1  .62968,  .10816,  .07671,  .05380,  .04684,  .05335,  .04614,
     2  .01829,  .01899,  .01962,  .05525,  .06816,  .01652,  .00867,
     3  .01544/
	DATA (RURABS(I,3),I=1,15) /
     1  .51899,  .08278,  .05816,  .04082,  .03570,  .04158,  .03620,
     2  .01513,  .01481,  .01633,  .05278,  .13690,  .02494,  .00886,
     3  .01804/
	DATA (RURABS(I,4),I=1,15) /
     1  .21943,  .02848,  .01943,  .01342,  .01171,  .01437,  .01323,
     2  .01152,  .00696,  .01329,  .06108,  .24690,  .05323,  .01430,
     3  .03361/
	DATA (RURG(I,1),I=1,15) /
     1  .7581,   .6785,   .6712,   .6479,   .6342,   .6176,   .6334,
     2  .7063,   .7271,   .7463,   .7788,   .7707,   .7424,   .7312,
     3  .7442/
	DATA (RURG(I,2),I=1,15) /
     1  .7632,   .6928,   .6865,   .6638,   .6498,   .6314,   .6440,
     2  .7098,   .7303,   .7522,   .7903,   .7804,   .7380,   .7319,
     3  .7508/
	DATA (RURG(I,3),I=1,15) /
     1  .7725,   .7240,   .7197,   .6997,   .6858,   .6650,   .6702,
     2  .7181,   .7378,   .7653,   .8168,   .7661,   .7286,   .7336,
     3  .7654/
	DATA (RURG(I,4),I=1,15) /
     1  .7778,   .7793,   .7786,   .7717,   .7628,   .7444,   .7365,
     2  .7491,   .7609,   .7921,   .8688,   .7537,   .7294,   .7413,
     3  .7928/
	DATA (URBEXT(I,1),I=1,15) /
     1 1.88816, 1.63316, 1.51867, 1.00000,  .77785,  .47095,  .30006,
     2  .21392,  .19405,  .17886,  .18127,  .16133,  .14785,  .14000,
     3  .12715/
	DATA (URBEXT(I,2),I=1,15) /
     1 1.95582, 1.64994, 1.53070, 1.00000,  .77614,  .46639,  .29487,
     2  .21051,  .18943,  .17285,  .17209,  .21418,  .15354,  .14051,
     3  .12728/
	DATA (URBEXT(I,3),I=1,15) /
     1 1.96430, 1.64032, 1.52392, 1.00000,  .77709,  .46253,  .28690,
     2  .20310,  .17981,  .16101,  .15614,  .26475,  .15456,  .13563,
     3  .12215/
	DATA (URBEXT(I,4),I=1,15) /
     1 1.41266, 1.33816, 1.29114, 1.00000,  .83646,  .55025,  .35342,
     2  .25285,  .21576,  .18310,  .16215,  .37854,  .20494,  .16665,
     3  .14778/
	DATA (URBABS(I,1),I=1,15) /
     1  .78437,  .58975,  .54285,  .36184,  .29222,  .20886,  .15658,
     2  .12329,  .11462,  .10747,  .11797,  .10025,  .08759,  .08184,
     3  .07506/
	DATA (URBABS(I,2),I=1,15) /
     1  .69032,  .49367,  .45165,  .29741,  .24070,  .17399,  .13146,
     2  .10354,  .09589,  .09025,  .10411,  .15101,  .07880,  .06949,
     3  .06570/
	DATA (URBABS(I,3),I=1,15) /
     1  .54848,  .37101,  .33734,  .21949,  .17785,  .12968,  .09854,
     2  .07804,  .07165,  .06791,  .08563,  .19639,  .06722,  .05316,
     3  .05316/
	DATA (URBABS(I,4),I=1,15) /
     1  .15975,  .10000,  .09013,  .05785,  .04671,  .03424,  .02633,
     2  .02525,  .01975,  .02354,  .06241,  .26690,  .05810,  .02285,
     3  .03810/
	DATA (URBG(I,1),I=1,15) /
     1  .7785,   .7182,   .7067,   .6617,   .6413,   .6166,   .6287,
     2  .6883,   .7070,   .7243,   .7370,   .7446,   .7391,   .7371,
     3  .7414/
	DATA (URBG(I,2),I=1,15) /
     1  .7906,   .7476,   .7385,   .6998,   .6803,   .6536,   .6590,
     2  .7066,   .7258,   .7484,   .7769,   .7405,   .7351,   .7459,
     3  .7625/
	DATA (URBG(I,3),I=1,15) /
     1  .7949,   .7713,   .7650,   .7342,   .7162,   .6873,   .6820,
     2  .7131,   .7312,   .7583,   .8030,   .7171,   .7185,   .7400,
     3  .7698/
	DATA (URBG(I,4),I=1,15) /
     1  .7814,   .7993,   .7995,   .7948,   .7870,   .7682,   .7751,
     2  .7501,   .7565,   .7809,   .8516,   .7137,   .7039,   .7241,
     3  .7728/
	DATA (OCNEXT(I,1),I=1,15) /
     1 1.47576, 1.32614, 1.26171, 1.00000,  .88133,  .70297,  .56487,
     2  .46006,  .42044,  .38310,  .35076,  .42266,  .32278,  .28810,
     3  .24905/
	DATA (OCNEXT(I,2),I=1,15) /
     1 1.36924, 1.25443, 1.20835, 1.00000,  .91367,  .77089,  .64987,
     2  .54886,  .50247,  .45038,  .38209,  .50589,  .43766,  .38076,
     3  .31658/
	DATA (OCNEXT(I,3),I=1,15) /
     1 1.22259, 1.14627, 1.11842, 1.00000,  .94766,  .87538,  .80418,
     2  .72930,  .68582,  .62165,  .49962,  .67949,  .66468,  .59253,
     3  .49551/
	DATA (OCNEXT(I,4),I=1,15) /
     1 1.09133, 1.06601, 1.05620, 1.00000,  .97506,  .94791,  .94203,
     2  .93671,  .92867,  .90411,  .80253,  .89222,  .94462,  .92146,
     3  .85797/
	DATA (OCNABS(I,1),I=1,15) /
     1  .30987,  .04354,  .02880,  .01797,  .01468,  .01766,  .01582,
     2  .00816,  .01146,  .01677,  .03310,  .03380,  .00715,  .00443,
     3  .00500/
	DATA (OCNABS(I,2),I=1,15) /
     1  .23367,  .03127,  .02070,  .01297,  .01063,  .01285,  .01190,
     2  .00937,  .00911,  .01576,  .05576,  .23487,  .03949,  .00905,
     3  .02057/
	DATA (OCNABS(I,3),I=1,15) /
     1  .13025,  .01557,  .01013,  .00646,  .00532,  .00665,  .00722,
     2  .01335,  .00728,  .01810,  .09835,  .37329,  .09703,  .01968,
     3  .05114/
	DATA (OCNABS(I,4),I=1,15) /
     1  .03506,  .00323,  .00215,  .00139,  .00114,  .00171,  .00532,
     2  .03082,  .01101,  .03741,  .20101,  .47608,  .21165,  .05234,
     3  .12886/
	DATA (OCNG(I,1),I=1,15) /
     1  .7516,   .6960,   .6920,   .6756,   .6767,   .6844,   .6936,
     2  .7055,   .7110,   .7177,   .7367,   .6287,   .6779,   .6784,
     3  .6599/
	DATA (OCNG(I,2),I=1,15) /
     1  .7708,   .7288,   .7243,   .7214,   .7211,   .7330,   .7445,
     2  .7579,   .7649,   .7790,   .8182,   .7673,   .7171,   .7205,
     3  .7235/
	DATA (OCNG(I,3),I=1,15) /
     1  .7954,   .7782,   .7752,   .7717,   .7721,   .7777,   .7872,
     2  .8013,   .8089,   .8301,   .8844,   .8332,   .7557,   .7597,
     3  .7823/
	DATA (OCNG(I,4),I=1,15) /
     1  .8208,   .8270,   .8260,   .8196,   .8176,   .8096,   .8096,
     2  .8202,   .8255,   .8520,   .9228,   .8950,   .7965,   .7847,
     3  .8242/
	DATA (TROEXT(I,1),I=1,15) /
     1 2.21222, 1.82753, 1.67032, 1.00000,  .72424,  .35272,  .15234,
     2  .05165,  .03861,  .02994,  .04671,  .02462,  .01538,  .01146,
     3  .01032/
	DATA (TROEXT(I,2),I=1,15) /
     1 2.21519, 1.82266, 1.66557, 1.00000,  .72525,  .35481,  .15449,
     2  .05475,  .04044,  .03082,  .04620,  .05272,  .01867,  .01266,
     3  .01127/
	DATA (TROEXT(I,3),I=1,15) /
     1 2.19082, 1.79462, 1.64456, 1.00000,  .73297,  .36443,  .16278,
     2  .06468,  .04658,  .03399,  .04538,  .11892,  .02835,  .01646,
     3  .01386/
	DATA (TROEXT(I,4),I=1,15) /
     1 1.75696, 1.54829, 1.45962, 1.00000,  .77816,  .43139,  .21778,
     2  .11329,  .08101,  .05506,  .04943,  .25291,  .06816,  .03703,
     3  .02601/
	DATA (TROABS(I,1),I=1,15) /
     1  .69671,  .09905,  .06563,  .04101,  .03354,  .03627,  .02810,
     2  .00873,  .00918,  .00930,  .03215,  .01285,  .00513,  .00316,
     3  .00557/
	DATA (TROABS(I,2),I=1,15) /
     1  .65000,  .08791,  .05816,  .03652,  .02994,  .03278,  .02557,
     2  .00810,  .00842,  .00867,  .03139,  .03949,  .00646,  .00316,
     3  .00595/
	DATA (TROABS(I,3),I=1,15) /
     1  .52804,  .06367,  .04158,  .02633,  .02184,  .02443,  .01937,
     2  .00658,  .00646,  .00709,  .02949,  .10013,  .00968,  .00310,
     3  .00677/
	DATA (TROABS(I,4),I=1,15) /
     1  .19829,  .01842,  .01215,  .00791,  .00665,  .00778,  .00652,
     2  .00361,  .00253,  .00399,  .02570,  .20690,  .01715,  .00316,
     3  .00873/
	DATA (TROG(I,1),I=1,15) /
     1  .7518,   .6710,   .6638,   .6345,   .6152,   .5736,   .5280,
     2  .4949,   .4700,   .4467,   .4204,   .4028,   .3777,   .3563,
     3  .3150/
	DATA (TROG(I,2),I=1,15) /
     1  .7571,   .6858,   .6790,   .6510,   .6315,   .5887,   .5418,
     2  .5075,   .4829,   .4598,   .4338,   .4043,   .3890,   .3680,
     3  .3259/
	DATA (TROG(I,3),I=1,15) /
     1  .7667,   .7176,   .7128,   .6879,   .6690,   .6255,   .5769,
     2  .5403,   .5167,   .4947,   .4703,   .4143,   .4190,   .3993,
     3  .3563/
	DATA (TROG(I,4),I=1,15) /
     1  .7696,   .7719,   .7710,   .7606,   .7478,   .7142,   .6727,
     2  .6381,   .6201,   .6050,   .5912,   .4849,   .5137,   .5019,
     3  .4625/
	DATA FG1EXT /
     1  .98519,  .99158,  .99089, 1.00000, 1.00576, 1.01747, 1.03177,
     2 1.04146, 1.04696, 1.05323, 1.05886, 1.04899, 1.06823, 1.07804,
     3 1.09272/
	DATA FG1ABS /
     1  .00013, 0.00000, 0.00000, 0.00000, 0.00000,  .00095,  .01513,
     2  .10861,  .03892,  .13272,  .47133,  .49696,  .45785,  .17918,
     3  .37373/
	DATA FG1G /
     1  .8578,   .8726,   .8722,   .8717,   .8703,   .8652,   .8618,
     2  .8798,   .8689,   .8918,   .9641,   .9502,   .9297,   .8544,
     3  .9007/
	DATA FG2EXT /
     1  .94791,  .96215,  .97063, 1.00000, 1.00937, 1.05177, 1.12519,
     2 1.29570, 1.39203, 1.41120, 1.04715, 1.10816, 1.43285, 1.45272,
     3 1.18709/
	DATA FG2ABS /
     1 0.00000, 0.00000, 0.00000, 0.00000, 0.00000,  .00013,  .00247,
     2  .01987,  .00620,  .02323,  .17209,  .57930,  .19810,  .03475,
     3  .09639/
	DATA FG2G /
     1  .8388,   .8459,   .8419,   .8286,   .8224,   .7883,   .7763,
     2  .8133,   .8393,   .8767,   .9258,   .8982,   .7887,   .8082,
     3  .8319/
	DATA BSTEXT /
     E 2.0752E0,  1.8656E0,  1.7246E0,  1.0000E0,  7.0156E-1, 3.0170E-1,
     E 1.1440E-1, 5.1225E-2, 3.4285E-2, 2.3475E-2, 1.6878E-2, 6.6506E-2,
     E 1.0943E-1, 8.9653E-2, 6.7609E-2/
	DATA BSTABS /
     A 2.4347E-7, 1.4949E-7, 1.3020E-7, 6.8716E-8, 1.0053E-7, 4.2384E-6,
     A 2.3075E-4, 1.4889E-3, 1.8377E-3, 3.3645E-3, 4.6858E-3, 5.9424E-2,
     A 1.0129E-1, 8.2396E-2, 6.3621E-2/
	DATA BSTG /
     G 0.6749, 0.6943, 0.6991, 0.6846, 0.6572, 0.5861, 0.4965,
     G 0.4164, 0.3772, 0.3385, 0.3069, 0.2599, 0.2234, 0.2028,
     G 0.1554/
	DATA AVOEXT /
     1 1.14880, 1.19171, 1.18013, 1.00000,  .84873,  .53019,  .27968,
     2  .14551,  .11070,  .08633,  .07184,  .06076,  .04506,  .03399,
     3  .02095/
	DATA AVOABS /
     1  .44816,  .11259,  .08500,  .05272,  .04082,  .02449,  .01487,
     2  .01019,  .00867,  .00842,  .00842,  .00949,  .00741,  .00487,
     3  .00316/
	DATA AVOG /
     1  .8272,   .7148,   .7076,   .6978,   .6886,   .6559,   .6062,
     2  .5561,   .5255,   .4958,   .4729,   .4401,   .4015,   .3699,
     3  .3125/
	DATA FVOEXT /
     1  .88715,  .92532,  .94013, 1.00000, 1.03013, 1.05975, 1.01171,
     2  .88677,  .82538,  .76361,  .71563,  .67424,  .60589,  .55057,
     3  .45222/
	DATA FVOABS /
     1  .41582,  .22892,  .19108,  .14468,  .12475,  .09158,  .06601,
     2  .04943,  .04367,  .04342,  .04399,  .05076,  .04133,  .02829,
     3  .01924/
	DATA FVOG /
     1  .9295,   .8115,   .7897,   .7473,   .7314,   .7132,   .7113,
     2  .7238,   .7199,   .7165,   .7134,   .6989,   .6840,   .6687,
     3  .6409/
	DATA DMEEXT /
     1 1.05019, 1.05880, 1.05259, 1.00000,  .94949,  .81456,  .66051,
     2  .54380,  .49133,  .44677,  .41671,  .38063,  .34778,  .32804,
     3  .29722/
	DATA DMEABS /
     1  .00063,  .00152,  .00184,  .00506,  .00791,  .01829,  .03728,
     2  .06158,  .07538,  .08943,  .10051,  .11614,  .13310,  .14348,
     3  .14633/
	DATA DMEG /
     1  .7173,   .7039,   .7020,   .6908,   .6872,   .6848,   .6891,
     2  .6989,   .7046,   .7099,   .7133,   .7159,   .7134,   .7058,
     3  .6827/
	END
*----------------------------------------------------------------------
	BLOCK DATA VANH
	IMPLICIT REAL*8(A-H,O-Z)

*     SUSIM ATLAS-3 solar spectrum data for 150-407.95nm with a grid
*     spacing of 0.05nm. Data were downloaded from the SUSIM website
*     on November 4, 1998 (ftp susim.nrl.navy.mil). This spectrum was
*     recommended by Dianne Prinz.

*      COMMON /SUNDAT2/ ATLAS(5160)

	COMMON /SUNDAT2/ ATLASB01(108),ATLASB02(108),ATLASB03(108),
     +ATLASB04(108),ATLASB05(108),ATLASB06(108),ATLASB07(108),
     +ATLASB08(108),ATLASB09(108),ATLASB10(108),ATLASB11(108),
     +ATLASB12(108),ATLASB13(108),ATLASB14(108),ATLASB15(108),
     +ATLASB16(108),ATLASB17(108),ATLASB18(108),ATLASB19(108),
     +ATLASB20(108),ATLASB21(108),ATLASB22(108),ATLASB23(108),
     +ATLASB24(108),ATLASB25(108),ATLASB26(108),ATLASB27(108),
     +ATLASB28(108),ATLASB29(108),ATLASB30(108),ATLASB31(108),
     +ATLASB32(108),ATLASB33(108),ATLASB34(108),ATLASB35(108),
     +ATLASB36(108),ATLASB37(108),ATLASB38(108),ATLASB39(108),
     +ATLASB40(108),ATLASB41(108),ATLASB42(108),ATLASB43(108),
     +ATLASB44(108),ATLASB45(108),ATLASB46(108),ATLASB47(108),
     +ATLASB48(84)

*                    SOLAR SPECTRUM FROM 150.000 TO 155.350 nm
      DATA ATLASB01 /
     +1.0240E-01,9.8110E-02,9.5490E-02,9.7540E-02,9.5250E-02,9.7430E-02,
     +1.0090E-01,9.4290E-02,9.3810E-02,9.5210E-02,9.7880E-02,9.4550E-02,
     +9.3350E-02,9.6590E-02,9.7190E-02,9.7210E-02,9.3960E-02,9.5200E-02,
     +9.7590E-02,1.0120E-01,1.1300E-01,1.2000E-01,1.2380E-01,1.1390E-01,
     +1.0380E-01,1.0260E-01,9.9240E-02,9.9240E-02,9.8390E-02,1.0010E-01,
     +9.9240E-02,9.9360E-02,1.0450E-01,1.0130E-01,1.0050E-01,1.0500E-01,
     +1.0200E-01,1.0380E-01,1.0150E-01,1.0430E-01,1.0570E-01,1.0520E-01,
     +1.0750E-01,1.0850E-01,1.1000E-01,1.1200E-01,1.1500E-01,1.1590E-01,
     +1.1390E-01,1.1510E-01,1.1900E-01,1.2030E-01,1.4920E-01,1.9990E-01,
     +1.9020E-01,1.4960E-01,1.2190E-01,1.1350E-01,1.1950E-01,1.1780E-01,
     +1.1600E-01,1.1830E-01,1.2420E-01,1.2140E-01,1.1850E-01,1.3770E-01,
     +1.9080E-01,2.1760E-01,1.8860E-01,1.5660E-01,1.3130E-01,1.2240E-01,
     +1.2520E-01,1.3100E-01,1.4400E-01,1.4900E-01,1.3470E-01,1.1950E-01,
     +1.1460E-01,1.2020E-01,1.4140E-01,1.5410E-01,1.6040E-01,1.5910E-01,
     +1.4450E-01,1.3120E-01,1.2190E-01,1.2000E-01,1.2920E-01,1.3700E-01,
     +1.4740E-01,1.4550E-01,1.2830E-01,1.3050E-01,1.4700E-01,2.9550E-01,
     +6.7860E-01,6.8480E-01,4.3190E-01,2.0820E-01,2.3020E-01,4.1090E-01,
     +4.9380E-01,3.8880E-01,2.1960E-01,1.5390E-01,1.3250E-01,1.2700E-01/
*                    SOLAR SPECTRUM FROM 155.400 TO 160.750 nm 
      DATA ATLASB02 /
     +1.2200E-01,1.2620E-01,1.3460E-01,1.4050E-01,1.4410E-01,1.4280E-01,
     +1.3450E-01,1.3560E-01,1.5950E-01,2.1130E-01,2.5080E-01,2.4880E-01,
     +2.6480E-01,3.4470E-01,3.7310E-01,3.4990E-01,2.6760E-01,1.7750E-01,
     +1.5550E-01,1.8110E-01,1.8520E-01,1.6710E-01,1.4520E-01,1.3930E-01,
     +1.4440E-01,1.5910E-01,1.7150E-01,1.7860E-01,1.7380E-01,1.6400E-01,
     +1.6980E-01,1.9660E-01,2.1790E-01,2.1110E-01,1.8820E-01,1.7240E-01,
     +1.5890E-01,1.5870E-01,1.6610E-01,2.0040E-01,2.3660E-01,2.5580E-01,
     +2.4940E-01,2.0060E-01,1.6930E-01,1.6210E-01,1.7150E-01,1.6510E-01,
     +1.4880E-01,1.4720E-01,1.4320E-01,1.5010E-01,1.7400E-01,2.0090E-01,
     +2.0090E-01,1.6800E-01,1.5050E-01,1.4240E-01,1.5200E-01,1.6030E-01,
     +1.7990E-01,1.9410E-01,1.9670E-01,1.8270E-01,1.7380E-01,1.6970E-01,
     +1.7310E-01,1.9390E-01,2.0890E-01,2.0560E-01,1.8440E-01,1.8140E-01,
     +1.8130E-01,1.7850E-01,1.8260E-01,1.7950E-01,1.8250E-01,1.7990E-01,
     +1.7590E-01,1.8740E-01,1.9300E-01,2.0770E-01,2.0460E-01,1.8990E-01,
     +1.8110E-01,1.6920E-01,1.6750E-01,1.7110E-01,1.7730E-01,1.7060E-01,
     +1.6910E-01,1.7680E-01,1.8940E-01,1.9110E-01,2.0180E-01,2.1810E-01,
     +2.2600E-01,2.1920E-01,2.0620E-01,1.9490E-01,1.8580E-01,1.9020E-01,
     +1.8940E-01,1.9310E-01,1.9090E-01,1.9140E-01,1.9550E-01,2.1090E-01/
*                    SOLAR SPECTRUM FROM 160.800 TO 166.150 nm 
      DATA ATLASB03 /
     +2.2940E-01,2.3750E-01,2.0850E-01,1.9440E-01,2.1060E-01,2.4220E-01,
     +2.6840E-01,2.4830E-01,2.4030E-01,2.5910E-01,2.9080E-01,2.8910E-01,
     +2.5290E-01,2.4470E-01,2.2050E-01,2.1040E-01,2.1360E-01,2.2380E-01,
     +2.2910E-01,2.2400E-01,2.3720E-01,2.5580E-01,2.4940E-01,2.3340E-01,
     +2.3220E-01,2.3480E-01,2.5040E-01,2.6950E-01,2.7570E-01,2.8420E-01,
     +2.9320E-01,2.8820E-01,2.5850E-01,2.5870E-01,2.7930E-01,3.2660E-01,
     +3.2440E-01,2.8070E-01,2.7010E-01,2.7680E-01,2.6820E-01,2.7250E-01,
     +2.9470E-01,3.0700E-01,2.8960E-01,2.6560E-01,2.5410E-01,2.5690E-01,
     +2.7080E-01,3.0600E-01,3.1380E-01,3.1960E-01,3.3190E-01,3.1280E-01,
     +2.7720E-01,2.6720E-01,2.6610E-01,2.8340E-01,3.2180E-01,3.4120E-01,
     +3.3190E-01,2.9390E-01,2.8890E-01,3.4810E-01,4.4770E-01,5.0430E-01,
     +4.1690E-01,3.2300E-01,2.9570E-01,2.9860E-01,3.3040E-01,3.6820E-01,
     +3.6680E-01,3.3640E-01,3.1760E-01,3.0720E-01,2.9780E-01,2.9570E-01,
     +3.0130E-01,3.0540E-01,3.0020E-01,2.9890E-01,3.1370E-01,3.2710E-01,
     +3.1160E-01,2.9390E-01,2.8870E-01,2.9480E-01,2.9310E-01,2.9490E-01,
     +2.9200E-01,3.0700E-01,3.5630E-01,3.7630E-01,3.6220E-01,3.8250E-01,
     +6.2750E-01,8.5660E-01,9.6670E-01,1.0240E+00,9.4060E-01,7.1880E-01,
     +5.8990E-01,5.1670E-01,4.2570E-01,3.6510E-01,3.6660E-01,3.6370E-01/
*                    SOLAR SPECTRUM FROM 166.200 TO 171.550 nm 
      DATA ATLASB04 /
     +3.6020E-01,3.7350E-01,3.9990E-01,4.1080E-01,3.8580E-01,3.5030E-01,
     +3.4360E-01,3.6060E-01,3.8630E-01,4.0140E-01,3.7730E-01,3.6050E-01,
     +3.5080E-01,3.4500E-01,3.5450E-01,3.7130E-01,4.0800E-01,5.4710E-01,
     +5.7820E-01,4.7710E-01,3.5590E-01,3.3750E-01,3.6900E-01,4.1110E-01,
     +4.3440E-01,4.5260E-01,4.2130E-01,3.8810E-01,3.7960E-01,4.0780E-01,
     +4.2540E-01,4.0900E-01,3.9510E-01,3.9220E-01,4.1360E-01,4.3050E-01,
     +4.2370E-01,4.2380E-01,4.5170E-01,4.4860E-01,4.1060E-01,4.0450E-01,
     +4.1520E-01,4.3000E-01,4.3290E-01,4.3080E-01,4.3600E-01,4.8750E-01,
     +5.6370E-01,6.0140E-01,5.7360E-01,5.0730E-01,5.0520E-01,5.3390E-01,
     +5.3050E-01,5.1030E-01,5.3570E-01,5.6310E-01,5.9090E-01,5.9710E-01,
     +5.7390E-01,5.6230E-01,5.5710E-01,5.7540E-01,6.1210E-01,6.3310E-01,
     +6.3390E-01,6.1580E-01,6.0710E-01,6.8130E-01,7.2810E-01,7.1450E-01,
     +6.3350E-01,6.3750E-01,6.7330E-01,6.8600E-01,6.6290E-01,6.4910E-01,
     +6.8990E-01,7.6280E-01,8.2260E-01,7.8130E-01,7.0320E-01,6.8700E-01,
     +6.7770E-01,6.8230E-01,6.7400E-01,6.5680E-01,6.6360E-01,6.6930E-01,
     +6.6740E-01,6.6090E-01,6.8860E-01,7.4800E-01,7.7690E-01,7.7340E-01,
     +7.5930E-01,7.3880E-01,7.1920E-01,7.0060E-01,6.5620E-01,6.4900E-01,
     +7.1530E-01,7.0400E-01,6.4840E-01,6.1500E-01,6.3600E-01,6.7220E-01/
*                    SOLAR SPECTRUM FROM 171.600 TO 176.950 nm 
      DATA ATLASB05 /
     +7.0800E-01,7.5020E-01,7.9610E-01,7.9920E-01,7.6910E-01,7.5700E-01,
     +7.5100E-01,7.4040E-01,7.7410E-01,8.1450E-01,8.4030E-01,7.7720E-01,
     +7.3130E-01,7.0700E-01,7.0280E-01,6.9180E-01,7.2500E-01,7.9050E-01,
     +8.3470E-01,8.2640E-01,8.0710E-01,8.0790E-01,7.9990E-01,7.9340E-01,
     +8.0150E-01,7.9260E-01,7.5310E-01,7.0290E-01,6.6080E-01,6.6870E-01,
     +6.8990E-01,7.2330E-01,7.4160E-01,7.4130E-01,7.5640E-01,7.7180E-01,
     +7.6660E-01,7.7280E-01,7.7850E-01,7.9170E-01,8.0880E-01,8.0570E-01,
     +8.0270E-01,7.9250E-01,7.9760E-01,8.2200E-01,8.4570E-01,8.4690E-01,
     +8.4630E-01,8.5480E-01,8.9700E-01,9.1990E-01,9.4180E-01,9.4700E-01,
     +9.5260E-01,9.5670E-01,9.6040E-01,9.9250E-01,9.8480E-01,9.5780E-01,
     +9.6420E-01,9.6950E-01,9.0800E-01,8.5650E-01,8.6860E-01,8.9450E-01,
     +9.3220E-01,9.4110E-01,9.5180E-01,9.9480E-01,1.0550E+00,1.0960E+00,
     +1.0870E+00,1.0760E+00,1.0750E+00,1.0910E+00,1.1130E+00,1.1230E+00,
     +1.1760E+00,1.1730E+00,1.1260E+00,1.1330E+00,1.1540E+00,1.1810E+00,
     +1.1750E+00,1.1700E+00,1.2020E+00,1.2040E+00,1.1900E+00,1.1910E+00,
     +1.2180E+00,1.2680E+00,1.2780E+00,1.2500E+00,1.1610E+00,1.1170E+00,
     +1.1530E+00,1.1650E+00,1.1690E+00,1.0760E+00,1.0370E+00,1.1500E+00,
     +1.2850E+00,1.4310E+00,1.3980E+00,1.3140E+00,1.2520E+00,1.2540E+00/
*                    SOLAR SPECTRUM FROM 177.000 TO 182.350 nm 
      DATA ATLASB06 /
     +1.3350E+00,1.4030E+00,1.4560E+00,1.5160E+00,1.5680E+00,1.6460E+00,
     +1.7240E+00,1.6960E+00,1.6050E+00,1.4160E+00,1.2730E+00,1.1900E+00,
     +1.2240E+00,1.2930E+00,1.3100E+00,1.3270E+00,1.3570E+00,1.4560E+00,
     +1.5590E+00,1.5960E+00,1.5980E+00,1.6520E+00,1.6590E+00,1.5850E+00,
     +1.5100E+00,1.4680E+00,1.5160E+00,1.5680E+00,1.5900E+00,1.5800E+00,
     +1.5810E+00,1.5990E+00,1.6390E+00,1.6390E+00,1.5990E+00,1.5930E+00,
     +1.5970E+00,1.6050E+00,1.7090E+00,1.7750E+00,1.7560E+00,1.7430E+00,
     +1.7710E+00,1.7840E+00,1.7230E+00,1.6330E+00,1.4970E+00,1.4210E+00,
     +1.4010E+00,1.4310E+00,1.4830E+00,1.5640E+00,1.6090E+00,1.6020E+00,
     +1.5670E+00,1.5930E+00,1.6520E+00,1.6690E+00,1.6540E+00,1.6940E+00,
     +1.7450E+00,1.7690E+00,1.7980E+00,1.8100E+00,1.8050E+00,1.8060E+00,
     +1.8160E+00,1.8080E+00,1.8110E+00,1.8590E+00,1.9180E+00,1.9070E+00,
     +1.8700E+00,1.8030E+00,1.7910E+00,2.1520E+00,2.7620E+00,2.5410E+00,
     +2.1580E+00,1.8860E+00,1.8900E+00,1.9690E+00,1.9290E+00,1.8930E+00,
     +1.8610E+00,1.8900E+00,1.8660E+00,1.8190E+00,1.7550E+00,1.7710E+00,
     +1.8340E+00,1.8870E+00,2.2310E+00,3.5420E+00,4.2950E+00,3.9620E+00,
     +3.2120E+00,2.3860E+00,2.2200E+00,2.1800E+00,2.1060E+00,1.9880E+00,
     +2.0450E+00,2.1920E+00,2.2660E+00,2.2780E+00,2.3820E+00,2.4760E+00/
*                    SOLAR SPECTRUM FROM 182.400 TO 187.750 nm 
      DATA ATLASB07 /
     +2.5350E+00,2.5400E+00,2.4390E+00,2.3200E+00,2.2270E+00,2.1800E+00,
     +2.2230E+00,2.1490E+00,2.0840E+00,2.0820E+00,2.2390E+00,2.2880E+00,
     +2.2480E+00,2.1930E+00,2.2030E+00,2.3210E+00,2.4020E+00,2.3930E+00,
     +2.3630E+00,2.3700E+00,2.4390E+00,2.4990E+00,2.5650E+00,2.5070E+00,
     +2.3810E+00,2.3250E+00,2.3630E+00,2.4850E+00,2.4610E+00,2.4700E+00,
     +2.6070E+00,2.5680E+00,2.3290E+00,2.2360E+00,2.1140E+00,1.9970E+00,
     +2.0610E+00,2.1890E+00,2.2800E+00,2.2950E+00,2.2600E+00,2.2910E+00,
     +2.2580E+00,2.1070E+00,2.0380E+00,2.0870E+00,2.0080E+00,1.8600E+00,
     +1.7560E+00,1.9470E+00,2.2030E+00,2.3640E+00,2.2780E+00,1.9930E+00,
     +1.9150E+00,1.9660E+00,2.0230E+00,2.0190E+00,2.0950E+00,2.3010E+00,
     +2.5720E+00,2.8070E+00,2.9000E+00,2.8130E+00,2.7910E+00,2.8050E+00,
     +2.6990E+00,2.5120E+00,2.4320E+00,2.4950E+00,2.4530E+00,2.3990E+00,
     +2.4290E+00,2.5240E+00,2.5730E+00,2.5430E+00,2.5740E+00,2.6820E+00,
     +2.8210E+00,2.7100E+00,2.6950E+00,2.5730E+00,2.5480E+00,2.5450E+00,
     +2.6330E+00,2.8030E+00,3.0280E+00,3.1850E+00,3.2130E+00,3.2200E+00,
     +3.2400E+00,3.2640E+00,3.2910E+00,3.2440E+00,3.3190E+00,3.3390E+00,
     +3.2190E+00,3.1530E+00,3.1260E+00,3.2280E+00,3.3360E+00,3.3240E+00,
     +3.2110E+00,3.1550E+00,3.1500E+00,3.1760E+00,3.1240E+00,3.1210E+00/
*                    SOLAR SPECTRUM FROM 187.800 TO 193.150 nm 
      DATA ATLASB08 /
     +3.0990E+00,3.0710E+00,3.0570E+00,3.0350E+00,3.0010E+00,3.0280E+00,
     +3.0250E+00,3.0170E+00,3.1610E+00,3.3910E+00,3.4760E+00,3.5390E+00,
     +3.5360E+00,3.5140E+00,3.5330E+00,3.4950E+00,3.4770E+00,3.5230E+00,
     +3.5160E+00,3.5030E+00,3.4280E+00,3.4930E+00,3.5910E+00,3.6920E+00,
     +3.7580E+00,3.7290E+00,3.8120E+00,3.9330E+00,4.1610E+00,4.0700E+00,
     +3.7270E+00,3.6010E+00,3.7660E+00,3.9780E+00,3.9530E+00,3.8220E+00,
     +3.7410E+00,3.8120E+00,3.7520E+00,3.6110E+00,3.4500E+00,3.3070E+00,
     +3.3610E+00,3.5640E+00,3.6850E+00,3.7140E+00,3.5190E+00,3.4990E+00,
     +3.5950E+00,3.7970E+00,3.9530E+00,4.0780E+00,4.1480E+00,4.1300E+00,
     +4.0300E+00,3.9270E+00,3.8700E+00,3.8720E+00,4.0200E+00,4.1290E+00,
     +4.1940E+00,4.3200E+00,4.5160E+00,4.3900E+00,4.0760E+00,3.8930E+00,
     +3.8840E+00,4.0410E+00,4.1840E+00,4.1990E+00,4.1860E+00,4.0960E+00,
     +4.1910E+00,4.3300E+00,4.4660E+00,4.6620E+00,4.6420E+00,4.5100E+00,
     +4.4180E+00,4.3480E+00,4.2250E+00,4.2060E+00,4.2660E+00,4.4220E+00,
     +4.5920E+00,4.7350E+00,4.7830E+00,4.8820E+00,4.8530E+00,4.7000E+00,
     +4.6650E+00,4.5870E+00,4.5670E+00,4.4740E+00,4.4340E+00,4.4960E+00,
     +4.5820E+00,4.4540E+00,4.3770E+00,4.4350E+00,4.6110E+00,4.6500E+00,
     +4.5500E+00,4.3740E+00,4.0580E+00,3.6090E+00,3.2940E+00,3.1940E+00/
*                    SOLAR SPECTRUM FROM 193.200 TO 198.550 nm 
      DATA ATLASB09 /
     +3.1540E+00,3.2420E+00,3.4060E+00,3.5210E+00,3.5800E+00,3.5350E+00,
     +3.4030E+00,3.2780E+00,3.1110E+00,3.0060E+00,3.1790E+00,3.4720E+00,
     +3.7680E+00,4.3240E+00,4.7610E+00,4.9480E+00,5.0040E+00,4.9820E+00,
     +5.1370E+00,5.5110E+00,5.8270E+00,5.9960E+00,6.0750E+00,6.0410E+00,
     +5.9330E+00,5.7270E+00,5.5610E+00,5.6240E+00,5.5150E+00,5.4130E+00,
     +5.4020E+00,5.9160E+00,6.3060E+00,6.3370E+00,6.2880E+00,6.0440E+00,
     +5.7280E+00,5.5770E+00,5.5020E+00,5.2880E+00,5.0090E+00,4.9290E+00,
     +5.0060E+00,5.5930E+00,6.0390E+00,6.0380E+00,5.8300E+00,5.5110E+00,
     +5.6180E+00,5.8940E+00,5.9540E+00,5.7540E+00,5.4980E+00,5.6090E+00,
     +5.8620E+00,6.1120E+00,6.0370E+00,5.7850E+00,5.6050E+00,5.4880E+00,
     +5.4710E+00,5.5300E+00,5.6080E+00,5.6730E+00,5.8690E+00,6.1480E+00,
     +6.4260E+00,6.5150E+00,6.6980E+00,6.8110E+00,6.8820E+00,6.8940E+00,
     +6.9050E+00,6.9620E+00,7.0110E+00,6.9880E+00,6.8900E+00,6.5370E+00,
     +6.3660E+00,6.3980E+00,6.3580E+00,6.4120E+00,6.5190E+00,6.6110E+00,
     +6.4480E+00,6.2930E+00,6.3150E+00,6.3600E+00,6.4240E+00,6.3570E+00,
     +6.1150E+00,6.0520E+00,6.2390E+00,6.3450E+00,6.1930E+00,6.2570E+00,
     +6.2750E+00,6.2930E+00,6.4880E+00,6.9380E+00,7.1410E+00,7.0390E+00,
     +6.6690E+00,6.4360E+00,6.4590E+00,6.5860E+00,6.8210E+00,6.6600E+00/
*                    SOLAR SPECTRUM FROM 198.600 TO 203.950 nm 
      DATA ATLASB10 /
     +6.3330E+00,5.9750E+00,5.8900E+00,6.1300E+00,6.2740E+00,6.0800E+00,
     +5.9760E+00,6.1780E+00,6.4080E+00,6.4930E+00,6.4560E+00,6.4570E+00,
     +6.6440E+00,6.8800E+00,7.3490E+00,7.6690E+00,7.2920E+00,7.0590E+00,
     +6.9200E+00,6.9900E+00,7.0300E+00,6.9830E+00,6.7260E+00,6.6910E+00,
     +7.0960E+00,7.4140E+00,7.2800E+00,7.1020E+00,7.1470E+00,7.1510E+00,
     +7.0390E+00,7.1370E+00,7.5610E+00,7.6420E+00,7.6290E+00,7.7020E+00,
     +7.6180E+00,7.3230E+00,7.3730E+00,7.5890E+00,8.0400E+00,7.9500E+00,
     +7.7860E+00,7.5230E+00,7.5090E+00,7.6470E+00,7.6880E+00,7.8050E+00,
     +8.0640E+00,8.1910E+00,7.7820E+00,7.3320E+00,7.3760E+00,8.0990E+00,
     +8.6250E+00,8.7910E+00,8.5760E+00,8.2510E+00,8.0680E+00,8.0360E+00,
     +7.9760E+00,8.0730E+00,8.2970E+00,8.7010E+00,8.9680E+00,9.2070E+00,
     +9.0100E+00,8.7050E+00,8.7080E+00,8.7640E+00,8.6020E+00,8.3000E+00,
     +7.9320E+00,7.7760E+00,8.1290E+00,8.7810E+00,9.0820E+00,9.0040E+00,
     +8.4030E+00,7.3400E+00,6.6800E+00,6.2360E+00,6.8320E+00,8.1680E+00,
     +8.9520E+00,9.2450E+00,9.2690E+00,9.0500E+00,8.9590E+00,9.1700E+00,
     +9.6520E+00,1.0190E+01,1.0280E+01,1.0030E+01,9.5700E+00,8.8220E+00,
     +8.2220E+00,8.1360E+00,8.3380E+00,8.9790E+00,9.7530E+00,1.0160E+01,
     +1.0250E+01,1.0200E+01,1.0220E+01,1.0130E+01,9.6530E+00,9.0530E+00/
*                    SOLAR SPECTRUM FROM 204.000 TO 209.350 nm 
      DATA ATLASB11 /
     +8.8200E+00,8.9650E+00,9.5540E+00,1.0150E+01,1.0700E+01,1.1150E+01,
     +1.1300E+01,1.1140E+01,1.0460E+01,9.9990E+00,1.0020E+01,1.0290E+01,
     +1.1000E+01,1.1460E+01,1.1230E+01,1.0680E+01,1.0090E+01,1.0270E+01,
     +1.0600E+01,1.0710E+01,1.0620E+01,1.0890E+01,1.1100E+01,1.1050E+01,
     +1.1050E+01,1.1160E+01,1.1300E+01,1.1490E+01,1.1530E+01,1.1450E+01,
     +1.1100E+01,1.0770E+01,1.0020E+01,1.0080E+01,1.0980E+01,1.1380E+01,
     +1.1220E+01,1.0240E+01,9.8800E+00,1.0440E+01,1.1230E+01,1.1560E+01,
     +1.1620E+01,1.1460E+01,1.0860E+01,1.0100E+01,1.0200E+01,1.1020E+01,
     +1.1450E+01,1.1440E+01,1.1650E+01,1.1800E+01,1.1480E+01,1.1190E+01,
     +1.1530E+01,1.2150E+01,1.2340E+01,1.2310E+01,1.2050E+01,1.1840E+01,
     +1.1810E+01,1.1920E+01,1.2320E+01,1.2770E+01,1.2680E+01,1.2490E+01,
     +1.2410E+01,1.2790E+01,1.3310E+01,1.3420E+01,1.3410E+01,1.3560E+01,
     +1.3890E+01,1.4110E+01,1.4390E+01,1.4370E+01,1.4140E+01,1.3720E+01,
     +1.3480E+01,1.3590E+01,1.4140E+01,1.4550E+01,1.4230E+01,1.3790E+01,
     +1.3480E+01,1.3160E+01,1.3130E+01,1.3550E+01,1.4120E+01,1.3750E+01,
     +1.3290E+01,1.3390E+01,1.5190E+01,1.6900E+01,1.7320E+01,1.7260E+01,
     +1.7070E+01,1.9070E+01,2.0130E+01,1.8990E+01,1.6670E+01,1.6280E+01,
     +1.6820E+01,1.7730E+01,1.8600E+01,1.9200E+01,2.2580E+01,2.4630E+01/
*                    SOLAR SPECTRUM FROM 209.400 TO 214.750 nm 
      DATA ATLASB12 /
     +2.4350E+01,1.9730E+01,1.8190E+01,2.0310E+01,2.2740E+01,2.4800E+01,
     +2.3860E+01,2.3290E+01,2.4320E+01,2.7080E+01,2.5630E+01,2.3510E+01,
     +2.1470E+01,1.9880E+01,2.1220E+01,2.6530E+01,3.1560E+01,3.5020E+01,
     +3.0680E+01,2.4000E+01,1.8710E+01,1.9890E+01,2.8850E+01,3.6050E+01,
     +3.9840E+01,3.8680E+01,3.5600E+01,3.3910E+01,3.3290E+01,2.9500E+01,
     +2.4490E+01,2.0510E+01,2.0040E+01,2.0870E+01,2.3120E+01,3.1120E+01,
     +3.5460E+01,3.9750E+01,4.0270E+01,3.3240E+01,2.9420E+01,2.8770E+01,
     +2.7410E+01,2.5210E+01,2.9870E+01,3.7060E+01,4.2250E+01,4.6070E+01,
     +4.4180E+01,3.9750E+01,3.5670E+01,3.7290E+01,4.5700E+01,5.1060E+01,
     +5.2530E+01,4.5720E+01,3.5890E+01,3.1720E+01,2.9340E+01,2.6250E+01,
     +1.8950E+01,1.4500E+01,1.3450E+01,1.5580E+01,2.4050E+01,3.1260E+01,
     +3.5350E+01,3.7350E+01,3.7320E+01,3.7390E+01,3.7620E+01,3.8290E+01,
     +3.5000E+01,2.8620E+01,2.5300E+01,2.6030E+01,2.9840E+01,3.0380E+01,
     +2.9610E+01,2.9970E+01,3.2690E+01,3.1570E+01,2.7650E+01,2.5500E+01,
     +2.6680E+01,3.1190E+01,3.5700E+01,4.1770E+01,4.3680E+01,3.6090E+01,
     +2.7370E+01,2.3040E+01,2.3880E+01,3.1400E+01,4.4470E+01,5.2050E+01,
     +5.3210E+01,5.0620E+01,4.7930E+01,4.8890E+01,5.0150E+01,4.9530E+01,
     +4.6040E+01,4.1550E+01,3.6800E+01,3.4910E+01,3.4080E+01,3.4330E+01/
*                    SOLAR SPECTRUM FROM 214.800 TO 220.150 nm 
      DATA ATLASB13 /
     +3.5470E+01,3.7090E+01,4.1010E+01,4.4670E+01,4.6840E+01,4.8290E+01,
     +3.7170E+01,3.0950E+01,3.2270E+01,3.5150E+01,3.8140E+01,4.4590E+01,
     +4.9550E+01,5.0430E+01,4.5360E+01,4.0860E+01,3.8420E+01,3.5790E+01,
     +3.3950E+01,2.7220E+01,2.3450E+01,2.2190E+01,2.1720E+01,2.3960E+01,
     +2.8510E+01,3.4630E+01,4.1180E+01,4.5450E+01,4.2630E+01,3.9830E+01,
     +3.9510E+01,4.1200E+01,3.9190E+01,3.7400E+01,3.7470E+01,3.7060E+01,
     +3.0720E+01,2.3680E+01,1.9620E+01,1.9310E+01,2.5420E+01,3.2140E+01,
     +3.2640E+01,3.0780E+01,2.8080E+01,3.1970E+01,3.6590E+01,3.7560E+01,
     +3.7160E+01,3.9280E+01,4.1090E+01,4.1660E+01,3.6110E+01,3.0040E+01,
     +2.6100E+01,2.5680E+01,2.7900E+01,3.4020E+01,3.6450E+01,3.4030E+01,
     +3.0230E+01,2.6190E+01,3.0740E+01,3.7920E+01,4.7380E+01,5.4130E+01,
     +5.5820E+01,5.6540E+01,5.7370E+01,5.8870E+01,6.0960E+01,5.8920E+01,
     +5.2040E+01,4.3330E+01,4.1520E+01,4.8220E+01,5.3170E+01,5.3230E+01,
     +3.8330E+01,2.6580E+01,2.7110E+01,3.6760E+01,4.3670E+01,3.9860E+01,
     +3.7470E+01,4.1810E+01,4.7280E+01,4.5060E+01,4.0800E+01,3.8020E+01,
     +4.6510E+01,6.1540E+01,6.6880E+01,6.5200E+01,5.6810E+01,4.5970E+01,
     +3.7990E+01,3.4140E+01,3.5440E+01,4.3040E+01,5.2400E+01,5.7860E+01,
     +6.1470E+01,6.0800E+01,5.6060E+01,4.9350E+01,3.8700E+01,3.1780E+01/
*                    SOLAR SPECTRUM FROM 220.200 TO 225.550 nm 
      DATA ATLASB14 /
     +3.8300E+01,4.7710E+01,5.6730E+01,6.4930E+01,6.9980E+01,6.7240E+01,
     +6.2750E+01,5.8880E+01,6.0680E+01,5.6730E+01,5.1910E+01,4.6380E+01,
     +4.2580E+01,3.6720E+01,3.3960E+01,3.6340E+01,3.9870E+01,3.2760E+01,
     +2.6190E+01,2.2120E+01,2.1270E+01,2.7860E+01,3.7830E+01,4.6610E+01,
     +5.1360E+01,4.6380E+01,4.5120E+01,4.7860E+01,4.7690E+01,3.9190E+01,
     +2.8260E+01,2.4180E+01,2.4490E+01,2.9060E+01,3.6060E+01,4.3340E+01,
     +5.1410E+01,5.4100E+01,5.4010E+01,5.6450E+01,5.9290E+01,5.8250E+01,
     +5.2210E+01,4.6290E+01,4.6700E+01,5.3020E+01,5.3070E+01,5.0780E+01,
     +5.0180E+01,5.0260E+01,4.6590E+01,4.3660E+01,4.3230E+01,4.6050E+01,
     +5.4430E+01,5.7330E+01,5.5790E+01,5.2160E+01,4.7340E+01,4.4050E+01,
     +4.4460E+01,5.1290E+01,6.7340E+01,8.1590E+01,8.5120E+01,8.3850E+01,
     +7.7180E+01,7.7110E+01,8.0130E+01,8.0970E+01,7.5830E+01,6.9370E+01,
     +6.4270E+01,6.0470E+01,5.4170E+01,6.1020E+01,7.0500E+01,7.2500E+01,
     +6.9730E+01,5.8530E+01,5.3900E+01,5.3380E+01,5.3810E+01,5.2920E+01,
     +5.0320E+01,4.7650E+01,4.6710E+01,5.4030E+01,6.3030E+01,6.5880E+01,
     +6.6290E+01,6.8810E+01,7.4690E+01,7.5560E+01,7.1210E+01,5.6830E+01,
     +5.3070E+01,5.5450E+01,5.5750E+01,5.2680E+01,5.1640E+01,5.8400E+01,
     +6.3640E+01,6.0080E+01,4.7830E+01,4.6480E+01,5.3580E+01,5.7300E+01/
*                    SOLAR SPECTRUM FROM 225.600 TO 230.950 nm 
      DATA ATLASB15 /
     +5.2790E+01,4.6800E+01,4.6640E+01,5.0040E+01,5.3990E+01,5.3690E+01,
     +5.4490E+01,5.7930E+01,5.3710E+01,4.2080E+01,3.7960E+01,4.1190E+01,
     +4.6780E+01,5.1910E+01,5.1220E+01,4.8860E+01,4.3910E+01,4.0710E+01,
     +3.6520E+01,3.4700E+01,3.3140E+01,3.3970E+01,3.5420E+01,3.5150E+01,
     +3.3910E+01,3.3310E+01,3.2970E+01,3.1940E+01,3.1170E+01,3.1500E+01,
     +3.1500E+01,3.2670E+01,3.3820E+01,3.4960E+01,3.6900E+01,3.8230E+01,
     +3.7680E+01,3.5430E+01,3.6890E+01,4.0010E+01,3.8510E+01,3.6940E+01,
     +3.9070E+01,4.3670E+01,4.6080E+01,4.6710E+01,4.7480E+01,4.8570E+01,
     +4.8090E+01,4.6400E+01,4.8770E+01,5.7290E+01,6.2800E+01,6.5670E+01,
     +6.6570E+01,6.0690E+01,5.2100E+01,5.1390E+01,5.4190E+01,6.4260E+01,
     +6.8580E+01,6.3710E+01,5.3170E+01,4.2040E+01,3.8610E+01,4.4930E+01,
     +5.2310E+01,5.5330E+01,5.0300E+01,4.4800E+01,4.2800E+01,4.3780E+01,
     +4.7170E+01,4.9300E+01,5.0470E+01,5.2110E+01,5.4100E+01,5.3480E+01,
     +5.2790E+01,5.4700E+01,6.1310E+01,6.3950E+01,5.8690E+01,4.7710E+01,
     +3.5570E+01,3.2030E+01,3.2920E+01,3.6580E+01,4.0480E+01,4.1210E+01,
     +4.0270E+01,4.0760E+01,4.7070E+01,5.2130E+01,5.1920E+01,4.7380E+01,
     +4.3540E+01,4.8370E+01,5.5990E+01,6.2990E+01,7.2330E+01,8.0620E+01,
     +7.9240E+01,7.7290E+01,7.0820E+01,6.5780E+01,5.7150E+01,5.5800E+01/
*                    SOLAR SPECTRUM FROM 231.000 TO 236.350 nm 
      DATA ATLASB16 /
     +5.1660E+01,4.5510E+01,4.1380E+01,3.6760E+01,3.4740E+01,3.4960E+01,
     +3.5630E+01,3.5320E+01,3.5140E+01,4.1960E+01,5.1520E+01,6.4750E+01,
     +6.9480E+01,6.5830E+01,5.8140E+01,5.5600E+01,5.5370E+01,5.8070E+01,
     +6.1550E+01,5.6190E+01,4.8320E+01,3.9470E+01,3.6120E+01,3.9410E+01,
     +4.1160E+01,4.7180E+01,6.1020E+01,7.7170E+01,8.5790E+01,8.6300E+01,
     +7.6880E+01,6.0250E+01,4.6050E+01,3.8820E+01,3.7380E+01,3.9530E+01,
     +4.1440E+01,4.6510E+01,5.3780E+01,6.0300E+01,6.1930E+01,5.8910E+01,
     +5.4950E+01,4.8410E+01,4.0810E+01,3.7860E+01,3.6190E+01,3.4120E+01,
     +3.8380E+01,4.4960E+01,4.8780E+01,5.3310E+01,6.0170E+01,6.3850E+01,
     +5.9720E+01,4.8100E+01,3.5550E+01,2.8960E+01,2.7800E+01,2.9920E+01,
     +3.5400E+01,4.0280E+01,4.4060E+01,4.6750E+01,5.1220E+01,5.3040E+01,
     +4.8450E+01,4.1300E+01,2.9900E+01,2.4520E+01,2.5470E+01,2.9010E+01,
     +3.0800E+01,3.2200E+01,3.5470E+01,3.7500E+01,3.6040E+01,3.0250E+01,
     +2.9800E+01,3.2470E+01,4.3410E+01,5.2040E+01,5.4530E+01,5.4400E+01,
     +5.5580E+01,6.0160E+01,6.5250E+01,6.6990E+01,6.6190E+01,5.8860E+01,
     +5.0630E+01,4.6110E+01,4.8110E+01,6.0130E+01,6.7110E+01,6.7880E+01,
     +6.6920E+01,6.5480E+01,5.7240E+01,4.4170E+01,3.6340E+01,3.1330E+01,
     +3.2530E+01,4.0210E+01,4.4890E+01,4.7500E+01,5.4420E+01,6.2710E+01/
*                    SOLAR SPECTRUM FROM 236.400 TO 241.750 nm 
      DATA ATLASB17 /
     +6.2830E+01,5.4200E+01,4.3980E+01,3.9320E+01,4.1020E+01,4.4710E+01,
     +4.3500E+01,4.0720E+01,4.2020E+01,4.3960E+01,4.5980E+01,4.9820E+01,
     +5.4970E+01,5.7760E+01,5.7750E+01,6.1870E+01,6.8210E+01,7.1490E+01,
     +6.7210E+01,5.3100E+01,3.8760E+01,3.5280E+01,3.8120E+01,4.1280E+01,
     +4.5610E+01,5.2160E+01,5.8100E+01,6.6920E+01,6.6880E+01,6.1770E+01,
     +5.2460E+01,4.3840E+01,3.9960E+01,4.0150E+01,3.8950E+01,3.4620E+01,
     +2.7760E+01,2.4220E+01,2.2770E+01,2.3780E+01,2.6930E+01,3.2400E+01,
     +3.7570E+01,4.2530E+01,5.0030E+01,5.5460E+01,5.6520E+01,5.5650E+01,
     +5.0460E+01,3.9710E+01,3.1750E+01,2.9700E+01,3.5820E+01,5.0110E+01,
     +6.0150E+01,6.4550E+01,6.1310E+01,5.7730E+01,5.8130E+01,5.8700E+01,
     +5.6220E+01,4.9620E+01,4.1700E+01,3.4230E+01,2.6690E+01,2.5100E+01,
     +3.1070E+01,3.7710E+01,4.3960E+01,4.7760E+01,4.3890E+01,3.7540E+01,
     +3.3700E+01,3.6480E+01,4.6720E+01,5.2350E+01,5.3370E+01,5.0820E+01,
     +4.9490E+01,4.9520E+01,4.9450E+01,4.4040E+01,3.2200E+01,2.6860E+01,
     +2.9750E+01,3.3870E+01,3.3560E+01,3.3220E+01,3.6850E+01,4.3750E+01,
     +4.9870E+01,4.9640E+01,4.9100E+01,4.3760E+01,3.4690E+01,2.8100E+01,
     +3.1800E+01,3.6490E+01,4.1230E+01,4.0860E+01,3.8730E+01,4.1060E+01,
     +4.7500E+01,5.3320E+01,5.6160E+01,5.7370E+01,6.0780E+01,6.2610E+01/
*                    SOLAR SPECTRUM FROM 241.800 TO 247.150 nm 
      DATA ATLASB18 /
     +6.1890E+01,6.2950E+01,6.8010E+01,6.9940E+01,6.9230E+01,7.1420E+01,
     +7.4100E+01,7.6490E+01,8.0360E+01,8.3860E+01,8.4060E+01,7.5180E+01,
     +6.5610E+01,6.2700E+01,6.1800E+01,6.3070E+01,6.7110E+01,7.3980E+01,
     +8.1020E+01,8.8270E+01,8.5430E+01,7.6410E+01,7.1020E+01,6.6170E+01,
     +6.7630E+01,7.3980E+01,7.7150E+01,7.8780E+01,7.6660E+01,6.9090E+01,
     +6.3240E+01,6.1070E+01,6.4280E+01,6.8380E+01,6.6110E+01,5.6030E+01,
     +5.3200E+01,5.7130E+01,6.0240E+01,6.3400E+01,7.0350E+01,7.3890E+01,
     +6.8320E+01,6.4040E+01,5.7480E+01,5.8120E+01,6.4990E+01,7.1380E+01,
     +7.5940E+01,7.8420E+01,7.8380E+01,7.4610E+01,6.7980E+01,6.2860E+01,
     +6.3320E+01,6.4730E+01,6.5420E+01,6.3280E+01,6.0570E+01,5.6150E+01,
     +5.3620E+01,5.0470E+01,5.0290E+01,5.3910E+01,5.5700E+01,5.3290E+01,
     +5.0610E+01,4.8570E+01,4.7600E+01,4.5900E+01,4.6290E+01,4.7240E+01,
     +4.6730E+01,4.6610E+01,4.8580E+01,5.2340E+01,5.2890E+01,5.0780E+01,
     +4.8520E+01,4.9920E+01,5.2780E+01,5.6280E+01,5.5820E+01,5.1850E+01,
     +5.0630E+01,5.0870E+01,5.0110E+01,4.7140E+01,4.5290E+01,4.2640E+01,
     +4.0100E+01,4.0160E+01,4.4570E+01,4.8190E+01,5.1430E+01,5.2760E+01,
     +5.1920E+01,5.1580E+01,5.0910E+01,5.1840E+01,5.5180E+01,5.7080E+01,
     +5.7500E+01,5.7510E+01,5.6580E+01,5.5440E+01,5.5360E+01,5.8100E+01/
*                    SOLAR SPECTRUM FROM 247.200 TO 252.550 nm 
      DATA ATLASB19 /
     +6.1570E+01,5.6120E+01,5.1270E+01,4.4020E+01,4.8100E+01,6.0630E+01,
     +6.5460E+01,6.5070E+01,6.4510E+01,6.3320E+01,6.1490E+01,6.0080E+01,
     +5.9710E+01,5.6740E+01,5.3000E+01,5.0130E+01,4.4680E+01,4.5130E+01,
     +5.1530E+01,5.9610E+01,6.0210E+01,5.4040E+01,4.2730E+01,3.6400E+01,
     +3.1100E+01,3.1960E+01,4.2510E+01,4.7040E+01,5.3810E+01,5.6040E+01,
     +5.2780E+01,4.9400E+01,4.4340E+01,3.7460E+01,3.5320E+01,3.6490E+01,
     +3.6800E+01,3.4900E+01,3.1150E+01,3.0210E+01,3.2750E+01,4.0900E+01,
     +4.8490E+01,5.4460E+01,5.6200E+01,5.9500E+01,6.6220E+01,7.2850E+01,
     +7.4800E+01,7.0070E+01,6.5300E+01,6.6870E+01,6.9970E+01,7.0130E+01,
     +6.8190E+01,6.9910E+01,7.3650E+01,7.6770E+01,7.1880E+01,5.5220E+01,
     +4.7220E+01,4.9920E+01,5.8200E+01,6.4120E+01,6.7180E+01,6.9370E+01,
     +7.1550E+01,6.6170E+01,5.9380E+01,5.1290E+01,4.4400E+01,3.8790E+01,
     +4.1370E+01,4.8910E+01,5.9280E+01,6.5940E+01,6.9020E+01,6.8920E+01,
     +6.5610E+01,5.6010E+01,5.2590E+01,5.3920E+01,5.4870E+01,5.3710E+01,
     +4.9440E+01,4.5200E+01,4.0270E+01,3.8970E+01,3.9080E+01,3.6530E+01,
     +3.4600E+01,3.6430E+01,3.9690E+01,4.0430E+01,4.0000E+01,3.9450E+01,
     +3.9970E+01,4.5410E+01,5.0530E+01,5.2850E+01,5.0650E+01,4.6000E+01,
     +4.1200E+01,3.7530E+01,3.5660E+01,3.5650E+01,3.6510E+01,3.9630E+01/
*                    SOLAR SPECTRUM FROM 252.600 TO 257.950 nm 
      DATA ATLASB20 /
     +4.4770E+01,4.8450E+01,4.9840E+01,4.8700E+01,4.3490E+01,3.7640E+01,
     +3.4760E+01,3.3020E+01,3.3880E+01,4.2130E+01,4.9170E+01,5.7350E+01,
     +6.1960E+01,6.1160E+01,5.8830E+01,5.7530E+01,5.7080E+01,5.6390E+01,
     +5.7080E+01,5.6320E+01,5.3750E+01,5.1100E+01,5.1400E+01,5.2910E+01,
     +5.5320E+01,5.8100E+01,5.5520E+01,5.3960E+01,5.6740E+01,6.2320E+01,
     +5.9790E+01,5.5030E+01,5.3850E+01,5.7020E+01,6.0470E+01,6.6160E+01,
     +6.9430E+01,6.8640E+01,6.5900E+01,6.3330E+01,6.1560E+01,5.7950E+01,
     +5.5030E+01,5.7990E+01,6.1200E+01,6.1880E+01,5.7040E+01,5.1260E+01,
     +4.8590E+01,5.1480E+01,6.0480E+01,7.3710E+01,8.3780E+01,9.2260E+01,
     +9.2730E+01,8.2420E+01,8.4110E+01,9.1500E+01,9.6630E+01,9.1080E+01,
     +8.3650E+01,8.3200E+01,8.5500E+01,8.2350E+01,7.9770E+01,8.3260E+01,
     +9.2070E+01,1.0300E+02,1.0570E+02,1.0040E+02,9.7290E+01,9.5960E+01,
     +8.9770E+01,8.0570E+01,7.2510E+01,6.6260E+01,6.8110E+01,7.5670E+01,
     +8.4470E+01,9.4090E+01,1.0980E+02,1.2540E+02,1.2830E+02,1.2360E+02,
     +1.1930E+02,1.1980E+02,1.1960E+02,1.2200E+02,1.3270E+02,1.4110E+02,
     +1.4160E+02,1.3470E+02,1.3220E+02,1.3580E+02,1.3650E+02,1.4150E+02,
     +1.5250E+02,1.5120E+02,1.3700E+02,1.1390E+02,8.9950E+01,7.2090E+01,
     +7.2770E+01,8.6130E+01,1.0010E+02,1.1550E+02,1.3130E+02,1.3810E+02/
*                    SOLAR SPECTRUM FROM 258.000 TO 263.350 nm 
      DATA ATLASB21 /
     +1.4680E+02,1.5730E+02,1.6700E+02,1.7550E+02,1.9170E+02,1.8470E+02,
     +1.3960E+02,1.1670E+02,1.2230E+02,1.3050E+02,1.1290E+02,9.1730E+01,
     +7.8510E+01,6.4280E+01,6.9080E+01,8.9370E+01,1.0760E+02,1.1660E+02,
     +1.3080E+02,1.6090E+02,1.8260E+02,1.9300E+02,1.6800E+02,1.4120E+02,
     +1.1950E+02,1.1580E+02,1.2360E+02,1.1340E+02,9.9130E+01,8.0700E+01,
     +8.7870E+01,1.1090E+02,1.2360E+02,1.2250E+02,1.1570E+02,1.0530E+02,
     +9.2540E+01,7.5640E+01,5.6620E+01,4.3530E+01,4.1120E+01,4.6020E+01,
     +6.0350E+01,8.5180E+01,1.0760E+02,1.1980E+02,1.2610E+02,1.2690E+02,
     +1.2320E+02,1.1690E+02,1.1280E+02,1.0260E+02,8.4870E+01,7.3740E+01,
     +6.5990E+01,5.8660E+01,6.1740E+01,7.4200E+01,8.6280E+01,9.7510E+01,
     +1.0450E+02,1.0760E+02,1.0690E+02,9.1240E+01,6.9780E+01,5.7440E+01,
     +5.7270E+01,6.4930E+01,6.8500E+01,6.5830E+01,6.8360E+01,8.6810E+01,
     +1.1130E+02,1.2470E+02,1.2480E+02,1.1720E+02,9.5280E+01,8.0400E+01,
     +8.1680E+01,9.6950E+01,1.2280E+02,1.3290E+02,1.2790E+02,1.1770E+02,
     +1.1130E+02,1.1180E+02,1.2690E+02,1.3680E+02,1.3610E+02,1.3810E+02,
     +1.4070E+02,1.3160E+02,1.0860E+02,8.9150E+01,9.1670E+01,1.0320E+02,
     +1.0800E+02,9.5420E+01,8.4820E+01,8.6440E+01,9.5290E+01,9.7330E+01,
     +8.4810E+01,7.1030E+01,6.2250E+01,6.6300E+01,8.2710E+01,1.0200E+02/
*                    SOLAR SPECTRUM FROM 263.400 TO 268.750 nm 
      DATA ATLASB22 /
     +1.2330E+02,1.4550E+02,1.7080E+02,1.7800E+02,1.7640E+02,1.7940E+02,
     +2.0360E+02,2.1950E+02,2.3490E+02,2.6550E+02,2.9330E+02,2.9870E+02,
     +2.8790E+02,2.6270E+02,2.4400E+02,2.3630E+02,2.3340E+02,2.4710E+02,
     +3.0190E+02,3.4570E+02,3.5290E+02,3.3910E+02,3.0080E+02,2.7350E+02,
     +2.6450E+02,2.4890E+02,2.3890E+02,2.3680E+02,2.3220E+02,2.1460E+02,
     +1.9220E+02,1.8220E+02,1.9690E+02,2.1250E+02,2.1980E+02,2.1950E+02,
     +2.2110E+02,2.2730E+02,2.3740E+02,2.5310E+02,2.5690E+02,2.5720E+02,
     +2.8520E+02,3.2250E+02,3.2440E+02,2.9770E+02,2.7880E+02,2.9070E+02,
     +3.0000E+02,2.8970E+02,2.7100E+02,2.6220E+02,2.8390E+02,2.9870E+02,
     +2.9700E+02,2.8480E+02,2.5220E+02,2.3350E+02,2.4130E+02,2.5330E+02,
     +2.2960E+02,2.0990E+02,2.1740E+02,2.5920E+02,2.8420E+02,2.6290E+02,
     +2.2090E+02,2.1170E+02,2.5440E+02,2.8490E+02,2.6890E+02,2.5350E+02,
     +2.2850E+02,2.3270E+02,2.6920E+02,2.9790E+02,2.9490E+02,2.2070E+02,
     +1.8490E+02,1.9710E+02,2.1130E+02,2.8400E+02,3.1650E+02,3.2660E+02,
     +3.2280E+02,3.1130E+02,2.9930E+02,2.7960E+02,2.5230E+02,2.3850E+02,
     +2.1680E+02,2.0790E+02,2.1170E+02,2.2900E+02,2.4700E+02,2.6070E+02,
     +2.7000E+02,2.7240E+02,2.6770E+02,2.6100E+02,2.5290E+02,2.4710E+02,
     +2.3950E+02,2.4500E+02,2.5750E+02,2.7290E+02,2.7980E+02,2.7940E+02/
*                    SOLAR SPECTRUM FROM 268.800 TO 274.150 nm 
      DATA ATLASB23 /
     +2.7460E+02,2.5780E+02,2.4270E+02,2.2970E+02,2.1590E+02,2.0590E+02,
     +2.1210E+02,2.2720E+02,2.4350E+02,2.5580E+02,2.5980E+02,2.6090E+02,
     +2.6710E+02,2.7890E+02,2.6660E+02,2.1970E+02,1.8550E+02,1.8400E+02,
     +2.0600E+02,2.3900E+02,2.5100E+02,2.3820E+02,2.2770E+02,2.3100E+02,
     +2.8570E+02,3.2900E+02,3.4390E+02,3.1290E+02,2.6770E+02,2.5140E+02,
     +2.7080E+02,2.9500E+02,3.0060E+02,2.9640E+02,3.0830E+02,3.4100E+02,
     +3.2590E+02,2.7850E+02,2.5280E+02,2.4670E+02,2.6140E+02,2.6520E+02,
     +2.5670E+02,2.4840E+02,2.6110E+02,2.7200E+02,2.7020E+02,2.6370E+02,
     +2.5070E+02,2.6810E+02,2.8670E+02,3.1840E+02,3.0890E+02,2.4070E+02,
     +1.9720E+02,1.8690E+02,2.0530E+02,2.4390E+02,2.5680E+02,2.6360E+02,
     +2.5050E+02,2.1290E+02,1.6520E+02,1.3210E+02,1.1550E+02,1.2300E+02,
     +1.2740E+02,1.3680E+02,1.5150E+02,1.9120E+02,2.1070E+02,2.1460E+02,
     +2.0040E+02,1.8220E+02,1.8400E+02,1.9280E+02,2.0550E+02,2.1610E+02,
     +2.1280E+02,2.0420E+02,1.8980E+02,1.9610E+02,2.4560E+02,2.7440E+02,
     +3.2100E+02,3.3100E+02,2.8960E+02,2.5520E+02,2.5480E+02,2.6430E+02,
     +2.6060E+02,2.3560E+02,1.9650E+02,1.8250E+02,1.9280E+02,2.0670E+02,
     +2.1040E+02,2.0810E+02,1.8790E+02,1.5170E+02,1.2820E+02,1.4340E+02,
     +1.7180E+02,1.6280E+02,1.3760E+02,1.2090E+02,1.3600E+02,1.5990E+02/
*                    SOLAR SPECTRUM FROM 274.200 TO 279.550 nm 
      DATA ATLASB24 /
     +1.6260E+02,1.5070E+02,1.2320E+02,1.0260E+02,9.8660E+01,1.0320E+02,
     +1.2430E+02,1.5710E+02,1.8060E+02,1.7320E+02,1.2840E+02,1.0800E+02,
     +1.1260E+02,1.3650E+02,1.5070E+02,1.3040E+02,9.9890E+01,8.8030E+01,
     +1.0210E+02,1.3860E+02,1.8510E+02,2.1850E+02,2.3750E+02,2.3770E+02,
     +2.1690E+02,1.9610E+02,1.7800E+02,1.5830E+02,1.2810E+02,1.0750E+02,
     +1.0750E+02,1.3910E+02,1.8720E+02,2.3960E+02,2.8300E+02,3.1440E+02,
     +2.9300E+02,2.6960E+02,2.6520E+02,2.6790E+02,2.4280E+02,2.0800E+02,
     +1.8380E+02,1.9660E+02,2.3630E+02,2.6380E+02,2.8020E+02,3.0520E+02,
     +3.1490E+02,2.9800E+02,2.6880E+02,2.5240E+02,2.4880E+02,2.4410E+02,
     +2.3290E+02,2.2220E+02,2.2160E+02,2.4610E+02,2.7390E+02,2.8890E+02,
     +2.8940E+02,2.6760E+02,2.4500E+02,2.5040E+02,2.8150E+02,3.0050E+02,
     +2.9500E+02,2.8130E+02,2.6600E+02,2.4990E+02,2.3030E+02,2.2130E+02,
     +2.2490E+02,2.1300E+02,1.8500E+02,1.7500E+02,1.7260E+02,1.7060E+02,
     +1.6920E+02,1.7050E+02,1.7220E+02,1.7870E+02,1.8320E+02,1.7560E+02,
     +1.6880E+02,1.6790E+02,1.7730E+02,1.8380E+02,1.8570E+02,1.8550E+02,
     +1.8320E+02,1.8140E+02,1.7290E+02,1.5330E+02,1.4080E+02,1.3580E+02,
     +1.3410E+02,1.2600E+02,1.1770E+02,1.0910E+02,1.0270E+02,1.0100E+02,
     +9.2240E+01,8.6760E+01,7.6690E+01,7.0880E+01,6.5180E+01,7.1500E+01/
*                    SOLAR SPECTRUM FROM 279.600 TO 284.950 nm 
      DATA ATLASB25 /
     +9.2440E+01,1.0010E+02,8.6900E+01,6.6000E+01,6.1620E+01,6.4240E+01,
     +6.8990E+01,7.7980E+01,8.5190E+01,8.8630E+01,8.6290E+01,7.8220E+01,
     +7.0920E+01,7.4790E+01,8.1070E+01,8.7900E+01,8.1310E+01,7.4880E+01,
     +7.4360E+01,8.5540E+01,1.0040E+02,1.0980E+02,1.1430E+02,1.2290E+02,
     +1.3340E+02,1.4620E+02,1.5460E+02,1.6470E+02,1.7550E+02,1.8150E+02,
     +1.8120E+02,1.7870E+02,1.7820E+02,1.8000E+02,1.8110E+02,1.8160E+02,
     +1.9940E+02,2.3510E+02,2.4870E+02,2.6920E+02,2.6830E+02,2.6130E+02,
     +2.5320E+02,2.4830E+02,2.5050E+02,2.6220E+02,2.7320E+02,2.8090E+02,
     +2.9500E+02,3.0780E+02,3.1010E+02,3.0790E+02,2.9960E+02,2.9510E+02,
     +2.9150E+02,3.0160E+02,3.2320E+02,3.4750E+02,3.5790E+02,3.2600E+02,
     +2.8930E+02,2.8490E+02,2.9800E+02,3.1190E+02,3.1730E+02,3.1860E+02,
     +3.2500E+02,3.3360E+02,3.5100E+02,3.6710E+02,3.6290E+02,3.6080E+02,
     +3.5740E+02,3.4800E+02,3.3220E+02,3.3280E+02,3.5170E+02,3.4740E+02,
     +3.2320E+02,2.8810E+02,2.5100E+02,2.4640E+02,3.0760E+02,3.4990E+02,
     +3.6870E+02,3.6900E+02,3.6000E+02,3.5010E+02,3.4680E+02,3.2800E+02,
     +2.9540E+02,2.8530E+02,3.0570E+02,3.3430E+02,3.2640E+02,2.7440E+02,
     +2.2970E+02,2.2260E+02,2.6750E+02,3.0360E+02,2.8610E+02,2.6010E+02,
     +2.3340E+02,2.1890E+02,2.0150E+02,1.8930E+02,1.6390E+02,1.4420E+02/
*                    SOLAR SPECTRUM FROM 285.000 TO 290.350 nm 
      DATA ATLASB26 /
     +1.2930E+02,1.2000E+02,1.0450E+02,8.9740E+01,7.3140E+01,6.0610E+01,
     +5.1290E+01,5.6210E+01,7.3560E+01,8.9660E+01,1.0690E+02,1.2700E+02,
     +1.4140E+02,1.5290E+02,1.6670E+02,1.9170E+02,2.1700E+02,2.2400E+02,
     +2.2640E+02,2.5200E+02,2.9980E+02,3.2700E+02,3.3790E+02,3.3210E+02,
     +3.3550E+02,3.4030E+02,3.3760E+02,3.3330E+02,3.5240E+02,3.6880E+02,
     +3.8250E+02,3.7460E+02,3.6470E+02,3.7540E+02,3.8030E+02,3.5970E+02,
     +3.2410E+02,3.1490E+02,3.1270E+02,3.0340E+02,3.0240E+02,3.3860E+02,
     +3.8620E+02,4.2400E+02,4.5660E+02,4.6240E+02,4.4260E+02,4.2620E+02,
     +3.8950E+02,3.6180E+02,3.6450E+02,3.8210E+02,3.7800E+02,3.4750E+02,
     +3.3110E+02,3.4110E+02,3.6900E+02,3.6970E+02,3.4480E+02,3.0870E+02,
     +2.7070E+02,2.4560E+02,2.1580E+02,1.7020E+02,1.4140E+02,1.3070E+02,
     +1.6860E+02,2.1290E+02,2.5470E+02,3.1010E+02,3.8570E+02,4.2300E+02,
     +4.5050E+02,4.4460E+02,4.2970E+02,4.1900E+02,4.1280E+02,4.0740E+02,
     +4.1570E+02,4.2310E+02,4.1520E+02,4.0690E+02,4.1490E+02,4.1540E+02,
     +4.1430E+02,4.2170E+02,4.2600E+02,4.2650E+02,4.2510E+02,4.2590E+02,
     +4.3890E+02,4.6430E+02,5.0270E+02,5.3790E+02,5.5180E+02,5.4670E+02,
     +5.4580E+02,5.3910E+02,5.3720E+02,5.5460E+02,6.0050E+02,6.4560E+02,
     +6.8530E+02,6.7950E+02,6.3430E+02,5.8360E+02,5.7210E+02,6.2960E+02/
*                    SOLAR SPECTRUM FROM 290.400 TO 295.750 nm 
      DATA ATLASB27 /
     +6.7570E+02,6.9230E+02,6.9340E+02,6.8430E+02,6.6720E+02,6.4160E+02,
     +6.1450E+02,6.1130E+02,6.0080E+02,5.9180E+02,5.7710E+02,5.7600E+02,
     +5.8180E+02,5.9010E+02,5.8570E+02,5.8080E+02,5.9080E+02,6.0090E+02,
     +6.3020E+02,6.9450E+02,7.2440E+02,6.7720E+02,6.1310E+02,5.7270E+02,
     +5.3880E+02,5.4770E+02,5.6370E+02,5.8360E+02,5.6370E+02,5.3280E+02,
     +5.2350E+02,5.6000E+02,5.8740E+02,5.9420E+02,5.8470E+02,5.8270E+02,
     +5.9870E+02,6.0140E+02,5.9470E+02,5.7180E+02,5.1270E+02,4.8570E+02,
     +4.9340E+02,5.0540E+02,4.8520E+02,4.4850E+02,4.4160E+02,4.5560E+02,
     +4.9610E+02,4.9530E+02,4.3890E+02,3.9960E+02,4.3020E+02,5.7380E+02,
     +6.4860E+02,6.6280E+02,6.5520E+02,6.7630E+02,6.7230E+02,6.4650E+02,
     +6.0670E+02,6.0260E+02,6.4530E+02,6.8180E+02,6.9620E+02,6.1920E+02,
     +4.6410E+02,3.6050E+02,3.4990E+02,4.0850E+02,4.6520E+02,4.7950E+02,
     +4.7880E+02,5.1980E+02,5.5200E+02,5.2780E+02,4.8200E+02,4.8030E+02,
     +5.8040E+02,6.3480E+02,6.2390E+02,5.6880E+02,5.2170E+02,5.1750E+02,
     +5.6040E+02,6.4340E+02,6.7860E+02,5.9890E+02,4.8360E+02,3.6890E+02,
     +3.1480E+02,3.3990E+02,3.8550E+02,4.4270E+02,5.2410E+02,5.9390E+02,
     +6.3980E+02,6.7270E+02,6.9130E+02,6.7640E+02,5.7530E+02,4.6490E+02,
     +4.5390E+02,5.6740E+02,6.3390E+02,6.4970E+02,6.0860E+02,5.1970E+02/
*                    SOLAR SPECTRUM FROM 295.800 TO 301.150 nm 
      DATA ATLASB28 /
     +4.9240E+02,5.1790E+02,6.3170E+02,6.8190E+02,6.6010E+02,6.0740E+02,
     +5.6420E+02,5.6280E+02,5.8250E+02,6.2930E+02,6.7660E+02,7.2740E+02,
     +7.2340E+02,6.8490E+02,6.1450E+02,4.7210E+02,3.8090E+02,3.3940E+02,
     +3.1510E+02,2.8750E+02,3.4700E+02,4.2970E+02,4.9030E+02,5.0130E+02,
     +4.1660E+02,3.4680E+02,3.2120E+02,3.9410E+02,5.1230E+02,5.5790E+02,
     +5.2560E+02,4.3620E+02,3.5260E+02,4.1760E+02,5.1510E+02,6.2340E+02,
     +6.5370E+02,6.2240E+02,6.0760E+02,6.6960E+02,7.8560E+02,8.2920E+02,
     +8.0000E+02,6.9840E+02,6.0550E+02,5.7040E+02,5.5870E+02,5.2040E+02,
     +4.3150E+02,3.7660E+02,3.9780E+02,4.2450E+02,3.7430E+02,3.0560E+02,
     +2.8940E+02,3.0330E+02,3.1480E+02,3.4700E+02,4.0240E+02,4.3200E+02,
     +4.2980E+02,4.3780E+02,4.7850E+02,5.2930E+02,5.8610E+02,6.6280E+02,
     +7.1660E+02,7.1420E+02,6.8410E+02,6.1920E+02,5.8190E+02,5.7580E+02,
     +5.6840E+02,4.6250E+02,3.6030E+02,3.1660E+02,3.6160E+02,5.0030E+02,
     +6.0020E+02,6.4170E+02,6.7440E+02,6.9630E+02,6.3640E+02,5.4420E+02,
     +4.3880E+02,3.4630E+02,2.9600E+02,2.8450E+02,2.9120E+02,3.1920E+02,
     +3.1370E+02,2.9490E+02,2.9160E+02,3.9250E+02,5.0700E+02,5.7540E+02,
     +5.9630E+02,5.9460E+02,5.1200E+02,4.5950E+02,4.0740E+02,3.7210E+02,
     +3.5680E+02,3.7950E+02,4.1920E+02,4.6340E+02,5.5730E+02,6.2630E+02/
*                    SOLAR SPECTRUM FROM 301.200 TO 306.550 nm 
      DATA ATLASB29 /
     +5.9840E+02,5.3360E+02,5.0730E+02,5.6170E+02,5.9180E+02,5.8950E+02,
     +5.8140E+02,5.7500E+02,5.7630E+02,5.7040E+02,5.6190E+02,5.1450E+02,
     +4.7220E+02,4.2080E+02,3.9160E+02,3.6510E+02,3.4350E+02,2.9920E+02,
     +2.3700E+02,1.8520E+02,2.7190E+02,3.1270E+02,4.7770E+02,5.8560E+02,
     +6.3310E+02,6.2130E+02,5.7190E+02,5.2230E+02,4.6280E+02,4.0720E+02,
     +4.4580E+02,5.6470E+02,7.1980E+02,8.0070E+02,8.2900E+02,7.6100E+02,
     +6.9190E+02,6.4510E+02,6.2970E+02,6.0970E+02,5.9410E+02,6.5600E+02,
     +7.0740E+02,7.8110E+02,8.0600E+02,7.7090E+02,7.5070E+02,7.6640E+02,
     +7.8870E+02,7.9750E+02,7.6940E+02,6.1030E+02,4.5000E+02,4.2120E+02,
     +4.3480E+02,5.5000E+02,6.5200E+02,6.5100E+02,6.2920E+02,6.0910E+02,
     +5.3780E+02,5.0150E+02,5.4210E+02,5.3710E+02,6.3970E+02,7.2420E+02,
     +7.4920E+02,7.4440E+02,7.5540E+02,7.4890E+02,7.5280E+02,6.2980E+02,
     +5.1450E+02,4.4430E+02,4.6420E+02,5.6320E+02,6.8010E+02,7.2380E+02,
     +6.7910E+02,6.4920E+02,7.0790E+02,8.4410E+02,8.3930E+02,8.5750E+02,
     +8.1250E+02,6.3460E+02,5.7010E+02,6.1070E+02,6.3350E+02,6.6170E+02,
     +6.6600E+02,5.6490E+02,4.2570E+02,3.5780E+02,3.3660E+02,3.4610E+02,
     +3.7560E+02,4.3110E+02,5.7900E+02,6.6530E+02,7.0740E+02,7.4330E+02,
     +7.4320E+02,7.2040E+02,6.9980E+02,6.0820E+02,5.4490E+02,5.6670E+02/
*                    SOLAR SPECTRUM FROM 306.600 TO 311.950 nm 
      DATA ATLASB30 /
     +6.0580E+02,6.1440E+02,5.2030E+02,4.6280E+02,4.2450E+02,4.5270E+02,
     +5.3430E+02,6.2520E+02,6.9790E+02,7.3910E+02,7.4500E+02,7.3000E+02,
     +7.2700E+02,6.5880E+02,6.4060E+02,6.6470E+02,6.8690E+02,7.0300E+02,
     +7.1020E+02,6.2880E+02,5.6910E+02,5.3330E+02,6.0650E+02,7.0840E+02,
     +7.3280E+02,6.9830E+02,6.4310E+02,6.5420E+02,7.3150E+02,7.6910E+02,
     +7.5020E+02,7.1050E+02,6.1270E+02,5.4940E+02,5.3000E+02,5.4110E+02,
     +5.8120E+02,5.8530E+02,6.9970E+02,8.0570E+02,8.4080E+02,8.2120E+02,
     +8.0620E+02,7.1900E+02,6.2810E+02,5.8830E+02,5.8290E+02,6.1810E+02,
     +6.5150E+02,6.3350E+02,6.0360E+02,5.6230E+02,5.0430E+02,4.7060E+02,
     +4.5550E+02,4.0660E+02,3.8520E+02,3.9020E+02,4.8730E+02,6.2010E+02,
     +6.6930E+02,6.0640E+02,5.6220E+02,4.8030E+02,4.3430E+02,5.0330E+02,
     +5.7760E+02,6.0780E+02,5.5030E+02,4.2610E+02,3.2870E+02,2.9440E+02,
     +3.1030E+02,3.4630E+02,4.2990E+02,5.2210E+02,6.4710E+02,7.5270E+02,
     +7.8490E+02,7.7440E+02,7.5440E+02,7.3530E+02,7.5950E+02,7.7480E+02,
     +8.1400E+02,8.5390E+02,9.0050E+02,9.1220E+02,9.0310E+02,8.3270E+02,
     +7.7350E+02,7.3410E+02,7.5600E+02,8.0270E+02,8.4700E+02,8.7740E+02,
     +8.8820E+02,8.2810E+02,7.5840E+02,7.5050E+02,7.9550E+02,8.2340E+02,
     +7.9260E+02,6.9540E+02,6.5180E+02,6.5110E+02,6.4230E+02,6.3940E+02/
*                    SOLAR SPECTRUM FROM 312.000 TO 317.350 nm 
      DATA ATLASB31 /
     +6.3060E+02,6.1320E+02,6.0440E+02,6.0640E+02,6.0870E+02,6.0020E+02,
     +6.1530E+02,6.8110E+02,7.2470E+02,7.8340E+02,8.0360E+02,7.5400E+02,
     +6.7940E+02,6.2550E+02,6.3190E+02,6.8550E+02,7.2750E+02,7.4790E+02,
     +7.3800E+02,7.2840E+02,7.3250E+02,7.5200E+02,7.4160E+02,7.1310E+02,
     +7.0160E+02,7.0570E+02,7.0320E+02,7.0170E+02,6.9520E+02,6.7560E+02,
     +6.4380E+02,6.6970E+02,7.2040E+02,7.4800E+02,7.3430E+02,7.4240E+02,
     +7.7470E+02,8.1320E+02,8.7190E+02,9.0200E+02,8.7880E+02,8.4520E+02,
     +8.1930E+02,8.2810E+02,8.6590E+02,8.6890E+02,8.6440E+02,6.9950E+02,
     +5.5100E+02,4.5350E+02,3.8560E+02,3.7020E+02,5.0130E+02,5.8870E+02,
     +6.2890E+02,6.5630E+02,6.4910E+02,6.6330E+02,6.8090E+02,7.8690E+02,
     +8.3380E+02,8.1470E+02,7.9030E+02,7.5570E+02,7.6550E+02,7.6240E+02,
     +7.5940E+02,7.7330E+02,7.8250E+02,7.8680E+02,7.5550E+02,7.2700E+02,
     +7.2490E+02,7.2700E+02,6.9320E+02,6.6910E+02,5.4200E+02,4.6000E+02,
     +4.3810E+02,4.2970E+02,4.3730E+02,4.6850E+02,4.8810E+02,4.9030E+02,
     +4.8720E+02,5.1130E+02,5.6090E+02,6.3020E+02,6.8360E+02,7.1010E+02,
     +6.9920E+02,6.6140E+02,5.5580E+02,6.5020E+02,6.9370E+02,6.9860E+02,
     +7.3740E+02,7.6660E+02,7.2560E+02,6.8680E+02,6.9210E+02,7.5020E+02,
     +8.3960E+02,8.9680E+02,9.4170E+02,9.6340E+02,9.5070E+02,9.5430E+02/
*                    SOLAR SPECTRUM FROM 317.400 TO 322.750 nm 
      DATA ATLASB32 /
     +9.4200E+02,9.1450E+02,8.7000E+02,8.4420E+02,8.3570E+02,8.7190E+02,
     +9.3190E+02,9.4040E+02,9.4650E+02,8.7820E+02,7.5000E+02,6.6370E+02,
     +5.9380E+02,5.3440E+02,5.2710E+02,5.4870E+02,5.4840E+02,5.3870E+02,
     +5.5200E+02,5.8750E+02,6.0210E+02,6.2320E+02,6.4730E+02,6.6340E+02,
     +6.8620E+02,7.1090E+02,7.4400E+02,7.4050E+02,7.3080E+02,7.4690E+02,
     +7.7330E+02,8.4270E+02,9.4310E+02,9.7340E+02,8.9020E+02,7.8400E+02,
     +7.0100E+02,6.5120E+02,5.5790E+02,4.8920E+02,5.2750E+02,5.6930E+02,
     +6.4380E+02,7.0290E+02,7.1740E+02,7.2110E+02,7.2030E+02,6.9650E+02,
     +6.9470E+02,7.7310E+02,8.1780E+02,8.9130E+02,9.4760E+02,8.9620E+02,
     +7.7540E+02,7.6020E+02,8.2450E+02,8.8590E+02,8.7970E+02,8.5170E+02,
     +8.6950E+02,8.9460E+02,9.5610E+02,9.8160E+02,9.3730E+02,9.0990E+02,
     +9.2980E+02,9.5290E+02,9.6080E+02,9.2560E+02,8.5500E+02,7.7990E+02,
     +7.4650E+02,7.0630E+02,7.0520E+02,6.5690E+02,6.1200E+02,5.9870E+02,
     +5.9350E+02,6.0050E+02,6.1150E+02,5.7330E+02,5.6070E+02,6.0270E+02,
     +6.7680E+02,7.4170E+02,7.5950E+02,7.3710E+02,7.0270E+02,7.3950E+02,
     +8.2930E+02,9.0720E+02,9.0410E+02,8.7650E+02,8.6240E+02,9.0090E+02,
     +8.7440E+02,7.8180E+02,6.9570E+02,6.6190E+02,6.8420E+02,7.9240E+02,
     +8.4240E+02,8.3460E+02,7.6440E+02,6.6190E+02,6.2150E+02,6.3320E+02/
*                    SOLAR SPECTRUM FROM 322.800 TO 328.150 nm 
      DATA ATLASB33 /
     +6.4810E+02,6.1990E+02,5.8110E+02,5.5920E+02,5.4650E+02,5.4540E+02,
     +5.9480E+02,6.1600E+02,6.8910E+02,7.7210E+02,7.9790E+02,8.0150E+02,
     +7.6680E+02,6.9930E+02,6.1190E+02,5.9970E+02,6.2810E+02,6.8230E+02,
     +6.6240E+02,6.2570E+02,6.4930E+02,7.3640E+02,7.8750E+02,8.0540E+02,
     +7.3260E+02,7.0150E+02,8.1500E+02,8.8420E+02,9.0180E+02,8.5540E+02,
     +8.2310E+02,8.1070E+02,7.9180E+02,7.8980E+02,7.7940E+02,8.2470E+02,
     +8.6410E+02,8.9500E+02,9.2280E+02,8.8110E+02,8.4890E+02,7.6210E+02,
     +7.0960E+02,7.1500E+02,7.3910E+02,7.6470E+02,7.7250E+02,7.6640E+02,
     +7.7190E+02,8.0170E+02,8.2950E+02,8.4100E+02,8.3710E+02,8.2830E+02,
     +8.0970E+02,9.1360E+02,9.9180E+02,1.0660E+03,1.0600E+03,1.0340E+03,
     +1.0010E+03,9.8890E+02,9.9710E+02,1.0630E+03,1.1040E+03,1.1110E+03,
     +1.1050E+03,1.0630E+03,1.0510E+03,1.0150E+03,1.0100E+03,1.0470E+03,
     +1.0670E+03,1.0280E+03,9.4240E+02,8.8090E+02,8.7260E+02,9.1500E+02,
     +9.7520E+02,1.0660E+03,1.0920E+03,1.0770E+03,1.0780E+03,1.0750E+03,
     +1.0660E+03,1.0450E+03,1.0210E+03,9.7820E+02,8.9970E+02,8.8560E+02,
     +9.4240E+02,9.8820E+02,1.0430E+03,1.0220E+03,1.0140E+03,1.0310E+03,
     +1.0680E+03,1.0590E+03,1.0210E+03,9.5380E+02,9.4630E+02,9.5690E+02,
     +9.8860E+02,9.7910E+02,9.4570E+02,9.2220E+02,9.2650E+02,9.6090E+02/
*                    SOLAR SPECTRUM FROM 328.200 TO 333.550 nm 
      DATA ATLASB34 /
     +9.9680E+02,9.7860E+02,9.5740E+02,9.1560E+02,9.4080E+02,9.7780E+02,
     +1.0050E+03,1.0190E+03,9.5300E+02,9.1610E+02,8.5280E+02,8.0500E+02,
     +7.4870E+02,7.8340E+02,8.4540E+02,9.0890E+02,9.9250E+02,1.0860E+03,
     +1.1480E+03,1.1430E+03,1.1130E+03,1.0850E+03,1.0680E+03,1.0820E+03,
     +1.1090E+03,1.1720E+03,1.1940E+03,1.1910E+03,1.1420E+03,1.1110E+03,
     +1.1250E+03,1.1410E+03,1.1600E+03,1.1560E+03,1.1320E+03,1.1060E+03,
     +1.1110E+03,1.1760E+03,1.2810E+03,1.3350E+03,1.2930E+03,1.2030E+03,
     +1.0850E+03,9.8520E+02,9.6860E+02,1.0250E+03,1.0770E+03,1.0230E+03,
     +1.0050E+03,9.1540E+02,8.1370E+02,7.7970E+02,8.3330E+02,9.0910E+02,
     +1.0030E+03,1.0530E+03,1.0710E+03,1.0820E+03,1.0650E+03,1.0510E+03,
     +1.0610E+03,1.0640E+03,1.0420E+03,9.9980E+02,9.8780E+02,9.6250E+02,
     +9.3380E+02,8.8650E+02,8.7600E+02,9.0820E+02,9.8460E+02,1.0470E+03,
     +1.0640E+03,1.0720E+03,1.0890E+03,1.0900E+03,1.0920E+03,1.0720E+03,
     +1.0390E+03,1.0260E+03,1.0090E+03,1.0290E+03,9.4390E+02,8.5340E+02,
     +8.0100E+02,7.9370E+02,8.7500E+02,9.5310E+02,1.0510E+03,1.0960E+03,
     +1.1240E+03,1.0690E+03,1.0100E+03,9.8230E+02,9.9290E+02,9.6100E+02,
     +9.1450E+02,8.7110E+02,8.7050E+02,9.6910E+02,1.0430E+03,1.0760E+03,
     +1.0380E+03,1.0250E+03,1.0520E+03,1.0840E+03,1.0320E+03,9.4960E+02/
*                    SOLAR SPECTRUM FROM 333.600 TO 338.950 nm 
      DATA ATLASB35 /
     +8.9490E+02,8.1930E+02,7.7100E+02,7.4460E+02,7.8730E+02,8.2820E+02,
     +9.1040E+02,9.7860E+02,1.0010E+03,9.6340E+02,9.7260E+02,9.7510E+02,
     +9.9340E+02,9.7030E+02,9.5110E+02,9.1310E+02,9.9500E+02,1.0590E+03,
     +1.0960E+03,1.0990E+03,1.0950E+03,1.1000E+03,1.1130E+03,1.0960E+03,
     +1.0790E+03,1.0630E+03,1.0460E+03,9.2130E+02,7.7780E+02,6.9900E+02,
     +7.4740E+02,8.4990E+02,1.0290E+03,1.0950E+03,1.1090E+03,1.1060E+03,
     +1.0920E+03,1.0460E+03,1.0210E+03,1.0370E+03,1.0960E+03,1.1620E+03,
     +1.1570E+03,1.1490E+03,1.1260E+03,1.0890E+03,1.0290E+03,8.8290E+02,
     +7.2420E+02,5.9550E+02,5.2200E+02,4.9070E+02,4.6870E+02,5.0370E+02,
     +6.2910E+02,6.9180E+02,8.5880E+02,9.6520E+02,1.0140E+03,1.0390E+03,
     +1.0920E+03,1.0760E+03,1.0110E+03,9.4750E+02,9.0500E+02,8.2580E+02,
     +7.8090E+02,7.6730E+02,7.1760E+02,6.7880E+02,6.9820E+02,7.3530E+02,
     +7.5560E+02,7.3260E+02,7.0060E+02,6.9080E+02,7.1080E+02,7.9480E+02,
     +8.7210E+02,8.9200E+02,9.7500E+02,1.0740E+03,1.1400E+03,1.1540E+03,
     +1.1470E+03,1.1370E+03,1.0620E+03,9.4820E+02,8.8630E+02,8.0060E+02,
     +6.7750E+02,6.4290E+02,6.7330E+02,8.5230E+02,9.8800E+02,1.0100E+03,
     +9.8470E+02,9.2840E+02,9.2610E+02,9.5870E+02,1.0210E+03,1.0500E+03,
     +1.0890E+03,1.1430E+03,1.1200E+03,1.0790E+03,9.2740E+02,8.9880E+02/
*                    SOLAR SPECTRUM FROM 339.000 TO 344.350 nm 
      DATA ATLASB36 /
     +1.0120E+03,1.1020E+03,1.1590E+03,1.1090E+03,1.0050E+03,9.5660E+02,
     +8.3860E+02,7.6640E+02,7.1900E+02,7.2730E+02,8.0680E+02,9.0260E+02,
     +9.5480E+02,9.9410E+02,1.0170E+03,1.0440E+03,1.0810E+03,1.1060E+03,
     +1.1190E+03,1.1350E+03,1.1420E+03,1.1290E+03,1.2040E+03,1.2770E+03,
     +1.2560E+03,1.1110E+03,1.0820E+03,1.0650E+03,1.0700E+03,1.0790E+03,
     +1.0390E+03,1.0120E+03,1.0070E+03,1.0070E+03,1.0390E+03,1.0290E+03,
     +9.3360E+02,8.9000E+02,9.1450E+02,9.4670E+02,9.4990E+02,9.3690E+02,
     +9.4700E+02,9.9850E+02,1.0740E+03,1.1120E+03,1.0650E+03,1.0560E+03,
     +8.7640E+02,8.3230E+02,7.6790E+02,7.0700E+02,7.2270E+02,8.1100E+02,
     +8.6780E+02,9.8060E+02,1.0060E+03,9.8200E+02,9.8070E+02,1.0140E+03,
     +1.1050E+03,1.1720E+03,1.2010E+03,1.1940E+03,1.1450E+03,1.1080E+03,
     +1.0980E+03,1.0110E+03,9.1800E+02,8.7170E+02,8.6130E+02,9.0110E+02,
     +1.0210E+03,1.0840E+03,1.0890E+03,1.0480E+03,9.3700E+02,9.2720E+02,
     +9.8320E+02,1.0020E+03,1.0970E+03,1.2050E+03,1.2640E+03,1.2490E+03,
     +1.2190E+03,1.1280E+03,1.0550E+03,9.8710E+02,9.3290E+02,8.4020E+02,
     +8.7730E+02,9.8680E+02,1.1360E+03,1.1830E+03,1.1430E+03,1.0800E+03,
     +1.0080E+03,9.3740E+02,9.2820E+02,9.5510E+02,9.7250E+02,8.6810E+02,
     +7.9450E+02,6.4490E+02,4.9970E+02,5.1490E+02,5.7310E+02,6.4360E+02/
*                    SOLAR SPECTRUM FROM 344.400 TO 349.750 nm 
      DATA ATLASB37 /
     +6.6800E+02,6.4730E+02,6.6320E+02,7.2720E+02,7.7210E+02,8.3280E+02,
     +8.1040E+02,7.8450E+02,8.0710E+02,8.9280E+02,9.6190E+02,9.8190E+02,
     +9.9380E+02,1.0070E+03,1.0750E+03,1.1530E+03,1.1980E+03,1.1180E+03,
     +1.0240E+03,9.0370E+02,8.4560E+02,8.6200E+02,1.0020E+03,1.1410E+03,
     +1.1910E+03,1.1550E+03,1.0690E+03,1.0380E+03,1.0150E+03,1.0050E+03,
     +9.2290E+02,8.4780E+02,8.3210E+02,8.4170E+02,8.2110E+02,7.9250E+02,
     +7.7340E+02,7.5090E+02,7.7810E+02,8.6190E+02,9.8630E+02,1.0350E+03,
     +1.1040E+03,1.1360E+03,1.0880E+03,1.0190E+03,8.6010E+02,8.5210E+02,
     +9.1790E+02,1.0300E+03,1.1020E+03,1.1200E+03,1.0580E+03,1.0400E+03,
     +1.1120E+03,1.1830E+03,1.2010E+03,1.1690E+03,1.0040E+03,9.2930E+02,
     +9.2950E+02,9.2900E+02,8.9510E+02,8.3310E+02,7.4670E+02,6.9640E+02,
     +6.6970E+02,6.7940E+02,7.3740E+02,8.4510E+02,9.3640E+02,1.0210E+03,
     +1.0600E+03,1.0530E+03,1.0470E+03,1.0320E+03,1.0240E+03,1.0350E+03,
     +1.0140E+03,9.6030E+02,8.8350E+02,8.5410E+02,8.6230E+02,8.9350E+02,
     +8.9570E+02,9.0040E+02,9.3720E+02,1.0680E+03,1.1760E+03,1.2100E+03,
     +1.1250E+03,1.0260E+03,8.6080E+02,8.0510E+02,7.6410E+02,7.2200E+02,
     +7.8920E+02,9.0960E+02,9.1310E+02,8.3480E+02,7.8870E+02,9.0090E+02,
     +1.0010E+03,1.0410E+03,9.5840E+02,8.8660E+02,8.3400E+02,7.7100E+02/
*                    SOLAR SPECTRUM FROM 349.800 TO 355.150 nm 
      DATA ATLASB38 /
     +7.2160E+02,7.4410E+02,8.3570E+02,9.6720E+02,1.1560E+03,1.2120E+03,
     +1.1980E+03,1.1720E+03,1.1500E+03,1.1470E+03,1.1350E+03,1.1620E+03,
     +1.2560E+03,1.3140E+03,1.3000E+03,1.1530E+03,1.0770E+03,1.0360E+03,
     +1.0270E+03,1.0260E+03,1.0710E+03,1.1350E+03,1.1820E+03,1.1840E+03,
     +1.1430E+03,1.0360E+03,8.9940E+02,9.0750E+02,9.7440E+02,1.0200E+03,
     +1.0110E+03,9.8370E+02,9.0030E+02,8.3610E+02,8.0070E+02,7.8010E+02,
     +8.1360E+02,9.4900E+02,1.0620E+03,1.1330E+03,1.1890E+03,1.2620E+03,
     +1.2580E+03,1.2290E+03,1.1910E+03,1.1180E+03,1.0500E+03,9.8540E+02,
     +9.3580E+02,9.2170E+02,9.6390E+02,9.7150E+02,9.4130E+02,8.2610E+02,
     +7.4340E+02,7.2180E+02,7.2090E+02,7.0590E+02,6.6000E+02,6.7420E+02,
     +7.9970E+02,9.6210E+02,1.0720E+03,1.1530E+03,1.1520E+03,1.1180E+03,
     +1.0950E+03,1.1230E+03,1.1180E+03,1.0830E+03,9.6250E+02,8.8950E+02,
     +9.2810E+02,1.0180E+03,1.1550E+03,1.2130E+03,1.1920E+03,1.1550E+03,
     +1.1110E+03,1.0840E+03,1.0670E+03,1.0560E+03,1.0850E+03,1.2060E+03,
     +1.2920E+03,1.3160E+03,1.2280E+03,1.0990E+03,1.0180E+03,1.0050E+03,
     +1.0350E+03,1.0880E+03,1.1440E+03,1.1970E+03,1.2230E+03,1.2200E+03,
     +1.1860E+03,1.1660E+03,1.1830E+03,1.1980E+03,1.1520E+03,1.1300E+03,
     +1.0850E+03,1.1290E+03,1.2240E+03,1.2410E+03,1.2360E+03,1.2030E+03/
*                    SOLAR SPECTRUM FROM 355.200 TO 360.550 nm 
      DATA ATLASB39 /
     +1.1680E+03,1.1520E+03,1.1290E+03,1.0810E+03,1.0230E+03,9.8170E+02,
     +9.3240E+02,9.0530E+02,9.6990E+02,1.1140E+03,1.1380E+03,1.1180E+03,
     +1.1050E+03,1.0860E+03,1.0260E+03,9.7330E+02,1.0020E+03,1.0320E+03,
     +1.0560E+03,1.0670E+03,1.0630E+03,1.0960E+03,1.1580E+03,1.2340E+03,
     +1.2830E+03,1.2810E+03,1.1890E+03,9.8840E+02,7.3520E+02,5.7240E+02,
     +5.8270E+02,6.8750E+02,8.4580E+02,9.8210E+02,9.9870E+02,9.1650E+02,
     +7.3640E+02,5.7170E+02,5.3310E+02,6.0350E+02,7.1870E+02,7.9990E+02,
     +8.4290E+02,8.8680E+02,9.4180E+02,9.3930E+02,9.3910E+02,9.3020E+02,
     +9.4710E+02,9.7350E+02,1.0030E+03,1.0400E+03,1.0620E+03,1.0110E+03,
     +8.9500E+02,8.3380E+02,8.1450E+02,8.1470E+02,7.8000E+02,6.4880E+02,
     +4.7270E+02,4.4340E+02,4.9300E+02,5.9110E+02,7.2690E+02,7.9630E+02,
     +7.9890E+02,7.3450E+02,5.9920E+02,5.0790E+02,5.3230E+02,5.5710E+02,
     +5.8780E+02,6.5220E+02,7.2330E+02,7.7950E+02,7.8380E+02,7.8820E+02,
     +8.2900E+02,9.0760E+02,1.0630E+03,1.1740E+03,1.1740E+03,1.0750E+03,
     +1.0010E+03,9.5340E+02,9.9330E+02,1.0400E+03,1.0880E+03,1.1620E+03,
     +1.2160E+03,1.2570E+03,1.2600E+03,1.2100E+03,1.1830E+03,1.1550E+03,
     +1.1630E+03,1.1990E+03,1.2940E+03,1.3400E+03,1.3500E+03,1.2170E+03,
     +1.0600E+03,9.8060E+02,9.4090E+02,9.4220E+02,9.4830E+02,9.4580E+02/
*                    SOLAR SPECTRUM FROM 360.600 TO 365.950 nm 
      DATA ATLASB40 /
     +9.0780E+02,8.6920E+02,9.1020E+02,9.7860E+02,1.0320E+03,1.0270E+03,
     +9.1680E+02,7.4050E+02,6.3960E+02,6.5040E+02,7.2480E+02,7.9410E+02,
     +9.0100E+02,1.0160E+03,1.0700E+03,1.0320E+03,9.8330E+02,9.1140E+02,
     +9.2230E+02,1.0020E+03,1.0690E+03,1.1440E+03,1.1690E+03,1.1320E+03,
     +1.0790E+03,9.2110E+02,7.1480E+02,5.4160E+02,4.9150E+02,5.7910E+02,
     +7.3370E+02,8.6750E+02,8.9300E+02,9.3110E+02,1.0460E+03,1.1400E+03,
     +1.1240E+03,1.0910E+03,1.0160E+03,1.0220E+03,1.1450E+03,1.1910E+03,
     +1.3130E+03,1.3610E+03,1.3850E+03,1.3500E+03,1.3150E+03,1.3070E+03,
     +1.3120E+03,1.2740E+03,1.1450E+03,8.5350E+02,7.1010E+02,5.8640E+02,
     +6.5150E+02,8.1230E+02,9.3910E+02,9.7260E+02,9.5390E+02,8.9840E+02,
     +9.4830E+02,1.0060E+03,1.0510E+03,1.0590E+03,1.0510E+03,1.0700E+03,
     +1.1030E+03,1.1730E+03,1.2390E+03,1.2660E+03,1.2790E+03,1.2700E+03,
     +1.2660E+03,1.2200E+03,1.1910E+03,1.1230E+03,1.0990E+03,1.0730E+03,
     +1.0400E+03,9.6710E+02,9.4450E+02,9.4640E+02,1.0160E+03,1.0490E+03,
     +9.1920E+02,8.6320E+02,7.6920E+02,8.5080E+02,9.4090E+02,9.1330E+02,
     +9.1790E+02,1.0190E+03,1.0910E+03,1.1060E+03,1.1470E+03,1.2710E+03,
     +1.3440E+03,1.2990E+03,1.2880E+03,1.2800E+03,1.2660E+03,1.2410E+03,
     +1.2620E+03,1.3260E+03,1.3500E+03,1.3310E+03,1.3310E+03,1.3900E+03/
*                    SOLAR SPECTRUM FROM 366.000 TO 371.350 nm 
      DATA ATLASB41 /
     +1.4300E+03,1.4200E+03,1.3810E+03,1.3670E+03,1.3890E+03,1.3960E+03,
     +1.3870E+03,1.3270E+03,1.2640E+03,1.2080E+03,1.1820E+03,1.2110E+03,
     +1.2980E+03,1.3440E+03,1.3020E+03,1.2730E+03,1.2570E+03,1.2800E+03,
     +1.3080E+03,1.2540E+03,1.1470E+03,1.0820E+03,1.0700E+03,1.1360E+03,
     +1.2240E+03,1.3610E+03,1.4270E+03,1.4340E+03,1.3730E+03,1.2920E+03,
     +1.2740E+03,1.2840E+03,1.3370E+03,1.3770E+03,1.3790E+03,1.2560E+03,
     +1.1880E+03,1.0720E+03,1.0690E+03,1.1250E+03,1.1280E+03,1.0730E+03,
     +9.9760E+02,1.0040E+03,1.0530E+03,1.1510E+03,1.1840E+03,1.1830E+03,
     +1.2020E+03,1.2710E+03,1.2900E+03,1.2600E+03,1.1870E+03,1.1040E+03,
     +1.0810E+03,1.0620E+03,9.9540E+02,9.2090E+02,9.5560E+02,1.0170E+03,
     +1.0690E+03,1.1630E+03,1.2470E+03,1.3400E+03,1.4300E+03,1.4880E+03,
     +1.4970E+03,1.4370E+03,1.3220E+03,1.2210E+03,1.1800E+03,1.1950E+03,
     +1.2600E+03,1.2870E+03,1.2710E+03,1.2650E+03,1.2630E+03,1.2850E+03,
     +1.3150E+03,1.3560E+03,1.4240E+03,1.4910E+03,1.5080E+03,1.4660E+03,
     +1.4230E+03,1.3830E+03,1.3800E+03,1.3670E+03,1.3440E+03,1.2360E+03,
     +1.0900E+03,1.0100E+03,9.6820E+02,8.7680E+02,8.6060E+02,9.3620E+02,
     +9.9060E+02,9.9290E+02,9.7960E+02,9.7170E+02,9.3620E+02,9.1640E+02,
     +1.0480E+03,1.1830E+03,1.2650E+03,1.2950E+03,1.3310E+03,1.3780E+03/
*                    SOLAR SPECTRUM FROM 371.400 TO 376.750 nm 
      DATA ATLASB42 /
     +1.4640E+03,1.5070E+03,1.5760E+03,1.5950E+03,1.5340E+03,1.4570E+03,
     +1.3750E+03,1.3870E+03,1.4220E+03,1.4270E+03,1.3410E+03,1.1860E+03,
     +9.8350E+02,8.3240E+02,6.5890E+02,6.4490E+02,6.9270E+02,7.1070E+02,
     +7.1040E+02,7.7930E+02,8.7690E+02,1.0630E+03,1.1430E+03,1.2450E+03,
     +1.2950E+03,1.3930E+03,1.4180E+03,1.2610E+03,1.0800E+03,9.3020E+02,
     +9.6420E+02,1.1430E+03,1.3070E+03,1.3870E+03,1.3320E+03,1.2260E+03,
     +1.1490E+03,1.1410E+03,1.1230E+03,1.0150E+03,8.7980E+02,7.6240E+02,
     +6.9790E+02,5.8410E+02,5.1670E+02,5.6870E+02,6.1390E+02,6.2510E+02,
     +6.3340E+02,6.7170E+02,7.5560E+02,9.5570E+02,1.1030E+03,1.1800E+03,
     +1.2180E+03,1.3320E+03,1.3510E+03,1.3440E+03,1.3090E+03,1.1490E+03,
     +9.8760E+02,9.2620E+02,9.2910E+02,9.4150E+02,8.2010E+02,7.2390E+02,
     +6.9880E+02,7.6430E+02,8.8910E+02,8.4210E+02,7.6650E+02,6.3550E+02,
     +5.2880E+02,5.4040E+02,6.0170E+02,7.9760E+02,1.0110E+03,1.1580E+03,
     +1.2360E+03,1.2530E+03,1.2530E+03,1.2820E+03,1.3570E+03,1.4310E+03,
     +1.5200E+03,1.5550E+03,1.5300E+03,1.4610E+03,1.2590E+03,9.9650E+02,
     +8.0110E+02,7.4150E+02,8.0400E+02,9.7090E+02,1.1100E+03,1.1700E+03,
     +1.1900E+03,1.1830E+03,1.2130E+03,1.2020E+03,1.0910E+03,9.8300E+02,
     +9.9470E+02,1.0970E+03,1.2030E+03,1.2480E+03,1.1770E+03,1.0350E+03/
*                    SOLAR SPECTRUM FROM 376.800 TO 382.150 nm 
      DATA ATLASB43 /
     +9.5620E+02,9.5090E+02,1.1180E+03,1.1610E+03,1.2320E+03,1.2430E+03,
     +1.1120E+03,1.0640E+03,1.0790E+03,1.1620E+03,1.2960E+03,1.3710E+03,
     +1.4140E+03,1.4290E+03,1.4100E+03,1.3880E+03,1.3720E+03,1.3540E+03,
     +1.3560E+03,1.3930E+03,1.4250E+03,1.4180E+03,1.3770E+03,1.3690E+03,
     +1.3860E+03,1.4610E+03,1.5480E+03,1.5830E+03,1.5620E+03,1.5340E+03,
     +1.5070E+03,1.5030E+03,1.5220E+03,1.5500E+03,1.6170E+03,1.6640E+03,
     +1.5890E+03,1.3980E+03,1.1930E+03,1.1270E+03,1.1450E+03,1.1650E+03,
     +1.1630E+03,1.1520E+03,1.1040E+03,1.0730E+03,1.0660E+03,1.1290E+03,
     +1.2020E+03,1.2960E+03,1.3000E+03,1.2490E+03,1.1930E+03,1.1290E+03,
     +1.0640E+03,1.0070E+03,9.8350E+02,1.0010E+03,1.0800E+03,1.1050E+03,
     +1.0550E+03,9.2320E+02,8.3670E+02,8.1270E+02,8.3290E+02,9.2280E+02,
     +1.0660E+03,1.2110E+03,1.2590E+03,1.2430E+03,1.2650E+03,1.3260E+03,
     +1.3910E+03,1.4480E+03,1.4550E+03,1.4280E+03,1.4040E+03,1.3880E+03,
     +1.3090E+03,1.1960E+03,1.1130E+03,1.0920E+03,1.1980E+03,1.3130E+03,
     +1.3700E+03,1.4040E+03,1.4520E+03,1.4710E+03,1.4570E+03,1.4320E+03,
     +1.3800E+03,1.2630E+03,1.1270E+03,1.0650E+03,1.0380E+03,1.0120E+03,
     +9.4300E+02,8.3960E+02,7.8120E+02,8.7550E+02,1.0170E+03,1.1420E+03,
     +1.2140E+03,1.2070E+03,1.1230E+03,9.5400E+02,7.6110E+02,6.2670E+02/
*                    SOLAR SPECTRUM FROM 382.200 TO 387.550 nm 
      DATA ATLASB44 /
     +6.5390E+02,7.5900E+02,9.3420E+02,1.1190E+03,1.1260E+03,1.0930E+03,
     +9.4920E+02,7.7740E+02,7.0010E+02,6.4050E+02,6.1180E+02,6.4840E+02,
     +7.0080E+02,7.1160E+02,7.0370E+02,7.1970E+02,7.3510E+02,7.4190E+02,
     +7.9940E+02,8.8290E+02,8.7300E+02,7.9760E+02,7.1500E+02,6.0800E+02,
     +6.0360E+02,6.1460E+02,6.1280E+02,6.4020E+02,7.0530E+02,8.0040E+02,
     +8.3970E+02,8.5600E+02,8.3580E+02,7.2270E+02,6.4500E+02,5.9090E+02,
     +6.2930E+02,6.7020E+02,6.8460E+02,6.4050E+02,6.6540E+02,8.3950E+02,
     +9.7650E+02,1.1230E+03,1.1510E+03,1.1410E+03,1.1660E+03,1.1970E+03,
     +1.1940E+03,1.1700E+03,1.1460E+03,1.1360E+03,1.1850E+03,1.2540E+03,
     +1.3060E+03,1.3040E+03,1.1270E+03,1.0270E+03,9.5090E+02,9.2340E+02,
     +1.0200E+03,1.1960E+03,1.2910E+03,1.2850E+03,1.2200E+03,1.1430E+03,
     +1.0980E+03,1.0750E+03,1.0720E+03,1.0260E+03,9.4310E+02,9.2390E+02,
     +1.0110E+03,1.0590E+03,1.0080E+03,8.9090E+02,7.4080E+02,6.3460E+02,
     +6.0210E+02,6.2350E+02,7.5780E+02,9.3870E+02,1.0900E+03,1.2640E+03,
     +1.3390E+03,1.3310E+03,1.2930E+03,1.2750E+03,1.2120E+03,1.1360E+03,
     +1.1490E+03,1.1700E+03,1.1810E+03,1.1570E+03,1.1380E+03,1.1350E+03,
     +1.1190E+03,1.0580E+03,9.5540E+02,8.9670E+02,8.1020E+02,7.6410E+02,
     +7.6490E+02,7.8050E+02,8.4790E+02,9.8940E+02,1.1260E+03,1.2480E+03/
*                    SOLAR SPECTRUM FROM 387.600 TO 392.950 nm 
      DATA ATLASB45 /
     +1.3390E+03,1.3670E+03,1.3290E+03,1.3170E+03,1.2260E+03,1.0630E+03,
     +8.6240E+02,8.2990E+02,8.6360E+02,1.0200E+03,1.1130E+03,1.1280E+03,
     +1.0850E+03,9.9390E+02,8.7460E+02,8.2920E+02,8.7610E+02,1.0570E+03,
     +1.2520E+03,1.2930E+03,1.2140E+03,1.0480E+03,8.5760E+02,7.7880E+02,
     +8.3520E+02,9.5840E+02,9.4810E+02,9.0600E+02,8.7050E+02,9.0990E+02,
     +1.0230E+03,1.1460E+03,1.2190E+03,1.3170E+03,1.3660E+03,1.3540E+03,
     +1.3300E+03,1.3100E+03,1.3200E+03,1.2900E+03,1.2550E+03,1.1890E+03,
     +1.2180E+03,1.3790E+03,1.4690E+03,1.4620E+03,1.3800E+03,1.3440E+03,
     +1.3040E+03,1.2410E+03,1.2210E+03,1.2910E+03,1.3840E+03,1.4390E+03,
     +1.3650E+03,1.2010E+03,1.1340E+03,1.1470E+03,1.2130E+03,1.2050E+03,
     +1.1590E+03,1.0340E+03,9.8400E+02,1.1270E+03,1.1980E+03,1.3230E+03,
     +1.4030E+03,1.4440E+03,1.4670E+03,1.4440E+03,1.3860E+03,1.3730E+03,
     +1.3950E+03,1.4600E+03,1.5460E+03,1.5410E+03,1.5360E+03,1.4680E+03,
     +1.4160E+03,1.4470E+03,1.4710E+03,1.4600E+03,1.3930E+03,1.3590E+03,
     +1.3730E+03,1.3990E+03,1.3940E+03,1.3200E+03,1.2920E+03,1.2640E+03,
     +1.1840E+03,1.0510E+03,1.0080E+03,1.0410E+03,1.1070E+03,1.1200E+03,
     +1.1200E+03,1.1550E+03,1.2530E+03,1.2720E+03,1.2280E+03,1.1670E+03,
     +1.0810E+03,1.0420E+03,9.8880E+02,9.4220E+02,8.0000E+02,7.0770E+02/
*                    SOLAR SPECTRUM FROM 393.000 TO 398.350 nm 
      DATA ATLASB46 /
     +6.7370E+02,6.4530E+02,5.9900E+02,5.3710E+02,5.0090E+02,4.7430E+02,
     +4.3430E+02,3.6930E+02,3.1350E+02,2.7110E+02,2.5390E+02,2.8530E+02,
     +3.3220E+02,3.8020E+02,4.3070E+02,5.1430E+02,5.9180E+02,6.2680E+02,
     +6.7340E+02,7.7090E+02,8.6440E+02,9.2880E+02,9.8940E+02,1.0110E+03,
     +1.0140E+03,1.0120E+03,1.0460E+03,1.0730E+03,1.0350E+03,9.5990E+02,
     +9.0020E+02,8.9650E+02,1.0270E+03,1.1540E+03,1.3050E+03,1.3970E+03,
     +1.3670E+03,1.2980E+03,1.2450E+03,1.2270E+03,1.2800E+03,1.3760E+03,
     +1.4390E+03,1.4980E+03,1.5270E+03,1.4920E+03,1.4290E+03,1.2740E+03,
     +1.2380E+03,1.3140E+03,1.4400E+03,1.5350E+03,1.5180E+03,1.4430E+03,
     +1.3470E+03,1.2590E+03,1.2830E+03,1.3360E+03,1.3850E+03,1.4380E+03,
     +1.5030E+03,1.5100E+03,1.4680E+03,1.2870E+03,1.0340E+03,8.9190E+02,
     +8.2770E+02,8.4260E+02,8.8890E+02,9.1100E+02,9.1160E+02,8.7100E+02,
     +8.1600E+02,7.3460E+02,6.5170E+02,5.6310E+02,4.6760E+02,3.8990E+02,
     +3.4760E+02,3.0550E+02,2.9120E+02,3.2940E+02,4.1280E+02,4.6920E+02,
     +5.8510E+02,6.6750E+02,7.3080E+02,7.8880E+02,8.4730E+02,8.9860E+02,
     +9.3830E+02,9.8700E+02,1.1070E+03,1.1860E+03,1.2060E+03,1.2250E+03,
     +1.2620E+03,1.3370E+03,1.3890E+03,1.4370E+03,1.5030E+03,1.5690E+03,
     +1.6090E+03,1.6350E+03,1.6270E+03,1.5580E+03,1.5250E+03,1.5130E+03/
*                    SOLAR SPECTRUM FROM 398.400 TO 403.750 nm 
      DATA ATLASB47 /
     +1.5090E+03,1.4790E+03,1.4630E+03,1.4990E+03,1.5560E+03,1.5800E+03,
     +1.5550E+03,1.4960E+03,1.4580E+03,1.6170E+03,1.7230E+03,1.7580E+03,
     +1.7110E+03,1.6440E+03,1.6120E+03,1.6220E+03,1.6800E+03,1.7240E+03,
     +1.7390E+03,1.7560E+03,1.7780E+03,1.7460E+03,1.7060E+03,1.6790E+03,
     +1.6620E+03,1.6590E+03,1.6500E+03,1.6200E+03,1.5640E+03,1.4910E+03,
     +1.5010E+03,1.6590E+03,1.7140E+03,1.8250E+03,1.8580E+03,1.8010E+03,
     +1.7490E+03,1.7560E+03,1.7850E+03,1.8590E+03,1.9130E+03,1.9030E+03,
     +1.8290E+03,1.6740E+03,1.4240E+03,1.3330E+03,1.3870E+03,1.4670E+03,
     +1.5870E+03,1.6930E+03,1.8110E+03,1.8270E+03,1.7940E+03,1.7630E+03,
     +1.7680E+03,1.7830E+03,1.8100E+03,1.8170E+03,1.8160E+03,1.8350E+03,
     +1.8480E+03,1.8160E+03,1.7740E+03,1.7820E+03,1.8290E+03,1.8760E+03,
     +1.8500E+03,1.7980E+03,1.7010E+03,1.6340E+03,1.6890E+03,1.7940E+03,
     +1.8860E+03,1.9110E+03,1.8420E+03,1.8230E+03,1.7990E+03,1.8010E+03,
     +1.8080E+03,1.8430E+03,1.8580E+03,1.8440E+03,1.7870E+03,1.7200E+03,
     +1.7070E+03,1.7570E+03,1.8290E+03,1.8830E+03,1.9020E+03,1.9280E+03,
     +1.9470E+03,1.9570E+03,1.9490E+03,1.8100E+03,1.6220E+03,1.4710E+03,
     +1.4550E+03,1.5300E+03,1.5520E+03,1.4990E+03,1.4840E+03,1.5080E+03,
     +1.5630E+03,1.5950E+03,1.6390E+03,1.6950E+03,1.7940E+03,1.9040E+03/
*                    SOLAR SPECTRUM FROM 403.800 TO   0.114 nm 
      DATA ATLASB48 /
     +1.9760E+03,2.0160E+03,2.0290E+03,2.0160E+03,1.9880E+03,1.9560E+03,
     +1.8580E+03,1.7760E+03,1.7690E+03,1.8030E+03,1.9430E+03,2.0420E+03,
     +2.0120E+03,1.8900E+03,1.6790E+03,1.4340E+03,1.2080E+03,1.0780E+03,
     +1.0460E+03,1.3390E+03,1.6260E+03,1.7880E+03,1.8900E+03,1.8730E+03,
     +1.8540E+03,1.8750E+03,1.9470E+03,2.0060E+03,1.9950E+03,1.9400E+03,
     +1.8290E+03,1.7610E+03,1.7730E+03,1.7900E+03,1.7580E+03,1.6900E+03,
     +1.6630E+03,1.6850E+03,1.7450E+03,1.7430E+03,1.6410E+03,1.5560E+03,
     +1.5240E+03,1.5540E+03,1.6370E+03,1.7120E+03,1.8300E+03,1.9080E+03,
     +1.9080E+03,1.8700E+03,1.7530E+03,1.5380E+03,1.3000E+03,1.1830E+03,
     +1.2270E+03,1.4720E+03,1.6670E+03,1.7450E+03,1.7070E+03,1.6140E+03,
     +1.5620E+03,1.5790E+03,1.7130E+03,1.8560E+03,1.9460E+03,1.9670E+03,
     +1.9020E+03,1.6980E+03,1.4220E+03,1.3020E+03,1.3090E+03,1.5660E+03,
     +1.7880E+03,1.8660E+03,1.8560E+03,1.8020E+03,1.7490E+03,1.7160E+03,
     +1.6150E+03,1.4730E+03,1.3850E+03,1.3740E+03,1.4360E+03,1.6600E+03/

      END
c----------------------------------------------------------------------
	SUBROUTINE L7TROPIC
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /ATM/Z(50),P(50),T(50),H2O(50),O3(50)
	DIMENSION Z2(50),P2(50),T2(50),WH2(50),WO3(50)

C     MODEL: TROPICAL MCCLATCHEY

	DATA(Z2(I),I=1, 50) /
     C       0.0,       1.0,       2.0,       3.0,       4.0,
     C       5.0,       6.0,       7.0,       8.0,       9.0,
     C      10.0,      11.0,      12.0,      13.0,      14.0,
     C      15.0,      16.0,      17.0,      18.0,      19.0,
     C      20.0,      21.0,      22.0,      23.0,      24.0,
     C      25.0,      27.5,      30.0,      32.5,      35.0,
     C      37.5,      40.0,      42.5,      45.0,      47.5,
     C      50.0,      55.0,      60.0,      65.0,      70.0,
     C      75.0,      80.0,      85.0,      90.0,      95.0,
     C     100.0,     105.0,     110.0,     115.0,     120.0/
	DATA(P2(I),I=1,50) /
     C 1.013E+03, 9.040E+02, 8.050E+02, 7.150E+02, 6.330E+02,
     C 5.590E+02, 4.920E+02, 4.320E+02, 3.780E+02, 3.290E+02,
     C 2.860E+02, 2.470E+02, 2.130E+02, 1.820E+02, 1.560E+02,
     C 1.320E+02, 1.110E+02, 9.370E+01, 7.890E+01, 6.660E+01,
     C 5.650E+01, 4.800E+01, 4.090E+01, 3.500E+01, 3.000E+01,
     C 2.570E+01, 1.763E+01, 1.220E+01, 8.520E+00, 6.000E+00,
     C 4.260E+00, 3.050E+00, 2.200E+00, 1.590E+00, 1.160E+00,
     C 8.540E-01, 4.560E-01, 2.390E-01, 1.210E-01, 5.800E-02,
     C 2.600E-02, 1.100E-02, 4.400E-03, 1.720E-03, 6.880E-04,
     C 2.890E-04, 1.300E-04, 6.470E-05, 3.600E-05, 2.250E-05/
	DATA(T2(I),I=1,50) /
     C    299.70,    293.70,    287.70,    283.70,    277.00,
     C    270.30,    263.60,    257.00,    250.30,    243.60,
     C    237.00,    230.10,    223.60,    217.00,    210.30,
     C    203.70,    197.00,    194.80,    198.80,    202.70,
     C    206.70,    210.70,    214.60,    217.00,    219.20,
     C    221.40,    227.00,    232.30,    237.70,    243.10,
     C    248.50,    254.00,    259.40,    264.80,    269.60,
     C    270.20,    263.40,    253.10,    236.00,    218.90,
     C    201.80,    184.80,    177.10,    177.00,    184.30,
     C    190.70,    212.00,    241.60,    299.70,    380.00/
	DATA(WH2(I),I=1,50) /
     C 2.593E+04, 1.949E+04, 1.534E+04, 8.600E+03, 4.441E+03,
     C 3.346E+03, 2.101E+03, 1.289E+03, 7.637E+02, 4.098E+02,
     C 1.912E+02, 7.306E+01, 2.905E+01, 9.900E+00, 6.220E+00,
     C 4.000E+00, 3.000E+00, 2.900E+00, 2.750E+00, 2.600E+00,
     C 2.600E+00, 2.650E+00, 2.800E+00, 2.900E+00, 3.200E+00,
     C 3.250E+00, 3.600E+00, 4.000E+00, 4.300E+00, 4.600E+00,
     C 4.900E+00, 5.200E+00, 5.500E+00, 5.700E+00, 5.900E+00,
     C 6.000E+00, 6.000E+00, 6.000E+00, 5.400E+00, 4.500E+00,
     C 3.300E+00, 2.100E+00, 1.300E+00, 8.500E-01, 5.400E-01,
     C 4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01/
	DATA (WO3(I),I=1,50) /
     C 2.869E-02, 3.150E-02, 3.342E-02, 3.504E-02, 3.561E-02,
     C 3.767E-02, 3.989E-02, 4.223E-02, 4.471E-02, 5.000E-02,
     C 5.595E-02, 6.613E-02, 7.815E-02, 9.289E-02, 1.050E-01,
     C 1.256E-01, 1.444E-01, 2.500E-01, 5.000E-01, 9.500E-01,
     C 1.400E+00, 1.800E+00, 2.400E+00, 3.400E+00, 4.300E+00,
     C 5.400E+00, 7.800E+00, 9.300E+00, 9.850E+00, 9.700E+00,
     C 8.800E+00, 7.500E+00, 5.900E+00, 4.500E+00, 3.450E+00,
     C 2.800E+00, 1.800E+00, 1.100E+00, 6.500E-01, 3.000E-01,
     C 1.800E-01, 3.300E-01, 5.000E-01, 5.200E-01, 5.000E-01,
     C 4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04/
	DO I=1,50
	   Z(I)=Z2(I)
	   P(I)=P2(I)
	   T(I)=T2(I)
	   H2O(I)=WH2(I)
	   O3(I)=WO3(I)
	END DO
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE L7MIDSUM
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /ATM/Z(50),P(50),T(50),H2O(50),O3(50)
	DIMENSION Z2(50),P2(50),T2(50),WH2(50),WO3(50)

C     MODEL: MIDLATITUDE SUMMER MCCLATCHEY

	DATA(Z2(I),I=1,50) /
     C       0.0,       1.0,       2.0,       3.0,       4.0,
     C       5.0,       6.0,       7.0,       8.0,       9.0,
     C      10.0,      11.0,      12.0,      13.0,      14.0,
     C      15.0,      16.0,      17.0,      18.0,      19.0,
     C      20.0,      21.0,      22.0,      23.0,      24.0,
     C      25.0,      27.5,      30.0,      32.5,      35.0,
     C      37.5,      40.0,      42.5,      45.0,      47.5,
     C      50.0,      55.0,      60.0,      65.0,      70.0,
     C      75.0,      80.0,      85.0,      90.0,      95.0,
     C     100.0,     105.0,     110.0,     115.0,     120.0/
	DATA(P2(I),I=1,50) /
     C 1.013E+03, 9.020E+02, 8.020E+02, 7.100E+02, 6.280E+02,
     C 5.540E+02, 4.870E+02, 4.260E+02, 3.720E+02, 3.240E+02,
     C 2.810E+02, 2.430E+02, 2.090E+02, 1.790E+02, 1.530E+02,
     C 1.300E+02, 1.110E+02, 9.500E+01, 8.120E+01, 6.950E+01,
     C 5.950E+01, 5.100E+01, 4.370E+01, 3.760E+01, 3.220E+01,
     C 2.770E+01, 1.907E+01, 1.320E+01, 9.300E+00, 6.520E+00,
     C 4.640E+00, 3.330E+00, 2.410E+00, 1.760E+00, 1.290E+00,
     C 9.510E-01, 5.150E-01, 2.720E-01, 1.390E-01, 6.700E-02,
     C 3.000E-02, 1.200E-02, 4.480E-03, 1.640E-03, 6.250E-04,
     C 2.580E-04, 1.170E-04, 6.110E-05, 3.560E-05, 2.270E-05/
	DATA(T2(I),I=1,50)/
     C    294.20,    289.70,    285.20,    279.20,    273.20,
     C    267.20,    261.20,    254.70,    248.20,    241.70,
     C    235.30,    228.80,    222.30,    215.80,    215.70,
     C    215.70,    215.70,    215.70,    216.80,    217.90,
     C    219.20,    220.40,    221.60,    222.80,    223.90,
     C    225.10,    228.45,    233.70,    239.00,    245.20,
     C    251.30,    257.50,    263.70,    269.90,    275.20,
     C    275.70,    269.30,    257.10,    240.10,    218.10,
     C    196.10,    174.10,    165.10,    165.00,    178.30,
     C    190.50,    222.20,    262.40,    316.80,    380.00/
	DATA (WH2(I),I=1,50)/
     C 1.876E+04, 1.378E+04, 9.680E+03, 5.984E+03, 3.813E+03,
     C 2.225E+03, 1.510E+03, 1.020E+03, 6.464E+02, 4.129E+02,
     C 2.472E+02, 9.556E+01, 2.944E+01, 8.000E+00, 5.000E+00,
     C 3.400E+00, 3.300E+00, 3.200E+00, 3.150E+00, 3.200E+00,
     C 3.300E+00, 3.450E+00, 3.600E+00, 3.850E+00, 4.000E+00,
     C 4.200E+00, 4.450E+00, 4.700E+00, 4.850E+00, 4.950E+00,
     C 5.000E+00, 5.100E+00, 5.300E+00, 5.450E+00, 5.500E+00,
     C 5.500E+00, 5.350E+00, 5.000E+00, 4.400E+00, 3.700E+00,
     C 2.950E+00, 2.100E+00, 1.330E+00, 8.500E-01, 5.400E-01,
     C 4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01/
	DATA(WO3(I),I=1,50) /
     C 3.017E-02, 3.337E-02, 3.694E-02, 4.222E-02, 4.821E-02,
     C 5.512E-02, 6.408E-02, 7.764E-02, 9.126E-02, 1.111E-01,
     C 1.304E-01, 1.793E-01, 2.230E-01, 3.000E-01, 4.400E-01,
     C 5.000E-01, 6.000E-01, 7.000E-01, 1.000E+00, 1.500E+00,
     C 2.000E+00, 2.400E+00, 2.900E+00, 3.400E+00, 4.000E+00,
     C 4.800E+00, 6.000E+00, 7.000E+00, 8.100E+00, 8.900E+00,
     C 8.700E+00, 7.550E+00, 5.900E+00, 4.500E+00, 3.500E+00,
     C 2.800E+00, 1.800E+00, 1.300E+00, 8.000E-01, 4.000E-01,
     C 1.900E-01, 2.000E-01, 5.700E-01, 7.500E-01, 7.000E-01,
     C 4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04/
	DO I=1,50
	   Z(I)=Z2(I)
	   P(I)=P2(I)
	   T(I)=T2(I)
	   H2O(I)=WH2(I)
	   O3(I)=WO3(I)
	END DO
	RETURN
	END
c----------------------------------------------------------------------
	SUBROUTINE L7MIDWIN
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /ATM/Z(50),P(50),T(50),H2O(50),O3(50)
	DIMENSION Z2(50),P2(50),T2(50),WH2(50),WO3(50)

C     MODEL: MIDLATITUDE WINTER MCCLATCHEY

	DATA(Z2(I),I=1, 50) /
     C       0.0,       1.0,       2.0,       3.0,       4.0,
     C       5.0,       6.0,       7.0,       8.0,       9.0,
     C      10.0,      11.0,      12.0,      13.0,      14.0,
     C      15.0,      16.0,      17.0,      18.0,      19.0,
     C      20.0,      21.0,      22.0,      23.0,      24.0,
     C      25.0,      27.5,      30.0,      32.5,      35.0,
     C      37.5,      40.0,      42.5,      45.0,      47.5,
     C      50.0,      55.0,      60.0,      65.0,      70.0,
     C      75.0,      80.0,      85.0,      90.0,      95.0,
     C     100.0,     105.0,     110.0,     115.0,     120.0/
	DATA(P2(I),I=1, 50) /
     C 1.018E+03, 8.973E+02, 7.897E+02, 6.938E+02, 6.081E+02,
     C 5.313E+02, 4.627E+02, 4.016E+02, 3.473E+02, 2.993E+02,
     C 2.568E+02, 2.199E+02, 1.882E+02, 1.611E+02, 1.378E+02,
     C 1.178E+02, 1.007E+02, 8.610E+01, 7.360E+01, 6.280E+01,
     C 5.370E+01, 4.580E+01, 3.910E+01, 3.340E+01, 2.860E+01,
     C 2.440E+01, 1.646E+01, 1.110E+01, 7.560E+00, 5.180E+00,
     C 3.600E+00, 2.530E+00, 1.800E+00, 1.290E+00, 9.400E-01,
     C 6.830E-01, 3.620E-01, 1.880E-01, 9.500E-02, 4.700E-02,
     C 2.220E-02, 1.030E-02, 4.560E-03, 1.980E-03, 8.770E-04,
     C 4.074E-04, 2.000E-04, 1.057E-04, 5.980E-05, 3.600E-05/
	DATA(T2(I),I=1,50)/
     C    272.20,    268.70,    265.20,    261.70,    255.70,
     C    249.70,    243.70,    237.70,    231.70,    225.70,
     C    219.70,    219.20,    218.70,    218.20,    217.70,
     C    217.20,    216.70,    216.20,    215.70,    215.20,
     C    215.20,    215.20,    215.20,    215.20,    215.20,
     C    215.20,    215.50,    217.40,    220.40,    227.90,
     C    235.50,    243.20,    250.80,    258.50,    265.10,
     C    265.70,    260.60,    250.80,    240.90,    230.70,
     C    220.40,    210.10,    199.80,    199.50,    208.30,
     C    218.60,    237.10,    259.50,    293.00,    333.00/
	DATA(WH2(I),I=1,50)/
     C 4.316E+03, 3.454E+03, 2.788E+03, 2.088E+03, 1.280E+03,
     C 8.241E+02, 5.103E+02, 2.321E+02, 1.077E+02, 5.566E+01,
     C 2.960E+01, 1.000E+01, 6.000E+00, 5.000E+00, 4.800E+00,
     C 4.700E+00, 4.600E+00, 4.500E+00, 4.500E+00, 4.500E+00,
     C 4.500E+00, 4.500E+00, 4.530E+00, 4.550E+00, 4.600E+00,
     C 4.650E+00, 4.700E+00, 4.750E+00, 4.800E+00, 4.850E+00,
     C 4.900E+00, 4.950E+00, 5.000E+00, 5.000E+00, 5.000E+00,
     C 4.950E+00, 4.850E+00, 4.500E+00, 4.000E+00, 3.300E+00,
     C 2.700E+00, 2.000E+00, 1.330E+00, 8.500E-01, 5.400E-01,
     C 4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01/
	DATA (WO3(I),I=1,50)/
     C 2.778E-02, 2.800E-02, 2.849E-02, 3.200E-02, 3.567E-02,
     C 4.720E-02, 5.837E-02, 7.891E-02, 1.039E-01, 1.567E-01,
     C 2.370E-01, 3.624E-01, 5.232E-01, 7.036E-01, 8.000E-01,
     C 9.000E-01, 1.100E+00, 1.400E+00, 1.800E+00, 2.300E+00,
     C 2.900E+00, 3.500E+00, 3.900E+00, 4.300E+00, 4.700E+00,
     C 5.100E+00, 5.600E+00, 6.100E+00, 6.800E+00, 7.100E+00,
     C 7.200E+00, 6.900E+00, 5.900E+00, 4.600E+00, 3.700E+00,
     C 2.750E+00, 1.700E+00, 1.000E-00, 5.500E-01, 3.200E-01,
     C 2.500E-01, 2.300E-01, 5.500E-01, 8.000E-01, 8.000E-01,
     C 4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04/
	DO I=1,50
	   Z(I)=Z2(I)
	   P(I)=P2(I)
	   T(I)=T2(I)
	   H2O(I)=WH2(I)
	   O3(I)=WO3(I)
	END DO
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE L7SUBSUM
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /ATM/Z(50),P(50),T(50),H2O(50),O3(50)
	DIMENSION Z2(50),P2(50),T2(50),WH2(50),WO3(50)

C     MODEL: SUBARCTIC SUMMER MCCLATCHEY

	DATA(Z2(I),I=1, 50) /
     C       0.0,       1.0,       2.0,       3.0,       4.0,
     C       5.0,       6.0,       7.0,       8.0,       9.0,
     C      10.0,      11.0,      12.0,      13.0,      14.0,
     C      15.0,      16.0,      17.0,      18.0,      19.0,
     C      20.0,      21.0,      22.0,      23.0,      24.0,
     C      25.0,      27.5,      30.0,      32.5,      35.0,
     C      37.5,      40.0,      42.5,      45.0,      47.5,
     C      50.0,      55.0,      60.0,      65.0,      70.0,
     C      75.0,      80.0,      85.0,      90.0,      95.0,
     C     100.0,     105.0,     110.0,     115.0,     120.0/
	DATA(P2(I),I=1,50) /
     C 1.010E+03, 8.960E+02, 7.929E+02, 7.000E+02, 6.160E+02,
     C 5.410E+02, 4.740E+02, 4.130E+02, 3.590E+02, 3.108E+02,
     C 2.677E+02, 2.300E+02, 1.977E+02, 1.700E+02, 1.460E+02,
     C 1.260E+02, 1.080E+02, 9.280E+01, 7.980E+01, 6.860E+01,
     C 5.900E+01, 5.070E+01, 4.360E+01, 3.750E+01, 3.228E+01,
     C 2.780E+01, 1.923E+01, 1.340E+01, 9.400E+00, 6.610E+00,
     C 4.720E+00, 3.400E+00, 2.480E+00, 1.820E+00, 1.340E+00,
     C 9.870E-01, 5.370E-01, 2.880E-01, 1.470E-01, 7.100E-02,
     C 3.200E-02, 1.250E-02, 4.510E-03, 1.610E-03, 6.060E-04,
     C 2.480E-04, 1.130E-04, 6.000E-05, 3.540E-05, 2.260E-05/
	DATA(T2(I),I=1,50) /
     C    287.20,    281.70,    276.30,    270.90,    265.50,
     C    260.10,    253.10,    246.10,    239.20,    232.20,
     C    225.20,    225.20,    225.20,    225.20,    225.20,
     C    225.20,    225.20,    225.20,    225.20,    225.20,
     C    225.20,    225.20,    225.20,    225.20,    226.60,
     C    228.10,    231.00,    235.10,    240.00,    247.20,
     C    254.60,    262.10,    269.50,    273.60,    276.20,
     C    277.20,    274.00,    262.70,    239.70,    216.60,
     C    193.60,    170.60,    161.70,    161.60,    176.80,
     C    190.40,    226.00,    270.10,    322.70,    380.00/
	DATA(WH2(I),I=1,50)/
     C 1.194E+04, 8.701E+03, 6.750E+03, 4.820E+03, 3.380E+03,
     C 2.218E+03, 1.330E+03, 7.971E+02, 3.996E+02, 1.300E+02,
     C 4.240E+01, 1.330E+01, 6.000E+00, 4.450E+00, 4.000E+00,
     C 4.000E+00, 4.000E+00, 4.050E+00, 4.300E+00, 4.500E+00,
     C 4.600E+00, 4.700E+00, 4.800E+00, 4.830E+00, 4.850E+00,
     C 4.900E+00, 4.950E+00, 5.000E+00, 5.000E+00, 5.000E+00,
     C 5.000E+00, 5.000E+00, 5.000E+00, 5.000E+00, 5.000E+00,
     C 4.950E+00, 4.850E+00, 4.500E+00, 4.000E+00, 3.300E+00,
     C 2.700E+00, 2.000E+00, 1.330E+00, 8.500E-01, 5.400E-01,
     C 4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01/
	DATA(WO3(I),I=1,50) /
     C 2.412E-02, 2.940E-02, 3.379E-02, 3.887E-02, 4.478E-02,
     C 5.328E-02, 6.564E-02, 7.738E-02, 9.114E-02, 1.420E-01,
     C 1.890E-01, 3.050E-01, 4.100E-01, 5.000E-01, 6.000E-01,
     C 7.000E-01, 8.500E-01, 1.000E+00, 1.300E+00, 1.700E+00,
     C 2.100E+00, 2.700E+00, 3.300E+00, 3.700E+00, 4.200E+00,
     C 4.500E+00, 5.300E+00, 5.700E+00, 6.900E+00, 7.700E+00,
     C 7.800E+00, 7.000E+00, 5.400E+00, 4.200E+00, 3.200E+00,
     C 2.500E+00, 1.700E+00, 1.200E+00, 8.000E-01, 4.000E-01,
     C 2.000E-01, 1.800E-01, 6.500E-01, 9.000E-01, 8.000E-01,
     C 4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04/
	DO I=1,50
	   Z(I)=Z2(I)
	   P(I)=P2(I)
	   T(I)=T2(I)
	   H2O(I)=WH2(I)
	   O3(I)=WO3(I)
	END DO
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE L7SUBWIN
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /ATM/Z(50),P(50),T(50),H2O(50),O3(50)
	DIMENSION Z2(50),P2(50),T2(50),WH2(50),WO3(50)

C     MODEL: SUBARCTIC WINTER MCCLATCHEY

	DATA(Z2(I),I=1, 50) /
     C       0.0,       1.0,       2.0,       3.0,       4.0,
     C       5.0,       6.0,       7.0,       8.0,       9.0,
     C      10.0,      11.0,      12.0,      13.0,      14.0,
     C      15.0,      16.0,      17.0,      18.0,      19.0,
     C      20.0,      21.0,      22.0,      23.0,      24.0,
     C      25.0,      27.5,      30.0,      32.5,      35.0,
     C      37.5,      40.0,      42.5,      45.0,      47.5,
     C      50.0,      55.0,      60.0,      65.0,      70.0,
     C      75.0,      80.0,      85.0,      90.0,      95.0,
     C     100.0,     105.0,     110.0,     115.0,     120.0/
	DATA(P2(I),I=1,50) /
     C 1.013E+03, 8.878E+02, 7.775E+02, 6.798E+02, 5.932E+02,
     C 5.158E+02, 4.467E+02, 3.853E+02, 3.308E+02, 2.829E+02,
     C 2.418E+02, 2.067E+02, 1.766E+02, 1.510E+02, 1.291E+02,
     C 1.103E+02, 9.431E+01, 8.058E+01, 6.882E+01, 5.875E+01,
     C 5.014E+01, 4.277E+01, 3.647E+01, 3.109E+01, 2.649E+01,
     C 2.256E+01, 1.513E+01, 1.020E+01, 6.910E+00, 4.701E+00,
     C 3.230E+00, 2.243E+00, 1.570E+00, 1.113E+00, 7.900E-01,
     C 5.719E-01, 2.990E-01, 1.550E-01, 7.900E-02, 4.000E-02,
     C 2.000E-02, 9.660E-03, 4.500E-03, 2.022E-03, 9.070E-04,
     C 4.230E-04, 2.070E-04, 1.080E-04, 6.000E-05, 3.590E-05/
	DATA(T2(I),I=1,50) /
     C    257.20,    259.10,    255.90,    252.70,    247.70,
     C    240.90,    234.10,    227.30,    220.60,    217.20,
     C    217.20,    217.20,    217.20,    217.20,    217.20,
     C    217.20,    216.60,    216.00,    215.40,    214.80,
     C    214.20,    213.60,    213.00,    212.40,    211.80,
     C    211.20,    213.60,    216.00,    218.50,    222.30,
     C    228.50,    234.70,    240.80,    247.00,    253.20,
     C    259.30,    259.10,    250.90,    248.40,    245.40,
     C    234.70,    223.90,    213.10,    202.30,    211.00,
     C    218.50,    234.00,    252.60,    288.50,    333.00/
	DATA(WH2(I),I=1,50)/
     C 1.405E+03, 1.615E+03, 1.427E+03, 1.166E+03, 7.898E+02,
     C 4.309E+02, 2.369E+02, 1.470E+02, 3.384E+01, 2.976E+01,
     C 2.000E+01, 1.000E+01, 6.000E+00, 4.450E+00, 4.500E+00,
     C 4.550E+00, 4.600E+00, 4.650E+00, 4.700E+00, 4.750E+00,
     C 4.800E+00, 4.850E+00, 4.900E+00, 4.950E+00, 5.000E+00,
     C 5.000E+00, 5.000E+00, 5.000E+00, 5.000E+00, 5.000E+00,
     C 5.000E+00, 5.000E+00, 5.000E+00, 5.000E+00, 5.000E+00,
     C 4.950E+00, 4.850E+00, 4.500E+00, 4.000E+00, 3.300E+00,
     C 2.700E+00, 2.000E+00, 1.330E+00, 8.500E-01, 5.400E-01,
     C 4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01/
	DATA(WO3(I),I=1,50) /
     C 1.802E-02, 2.072E-02, 2.336E-02, 2.767E-02, 3.253E-02,
     C 3.801E-02, 4.446E-02, 7.252E-02, 1.040E-01, 2.100E-01,
     C 3.000E-01, 3.500E-01, 4.000E-01, 6.500E-01, 9.000E-01,
     C 1.200E+00, 1.500E+00, 1.900E+00, 2.450E+00, 3.100E+00,
     C 3.700E+00, 4.000E+00, 4.200E+00, 4.500E+00, 4.600E+00,
     C 4.700E+00, 4.900E+00, 5.400E+00, 5.900E+00, 6.200E+00,
     C 6.250E+00, 5.900E+00, 5.100E+00, 4.100E+00, 3.000E+00,
     C 2.600E+00, 1.600E+00, 9.500E-01, 6.500E-01, 5.000E-01,
     C 3.300E-01, 1.300E-01, 7.500E-01, 8.000E-01, 8.000E-01,
     C 4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04/
	DO I=1,50
	   Z(I)=Z2(I)
	   P(I)=P2(I)
	   T(I)=T2(I)
	   H2O(I)=WH2(I)
	   O3(I)=WO3(I)
	END DO
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE L7US76
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /ATM/Z(50),P(50),T(50),H2O(50),O3(50)
	DIMENSION Z2(50),P2(50),T2(50),WH2(50),WO3(50)

C     MODEL: 1976 US STANDARD ATMOSPHERE MCCLATCHEY

	DATA(Z2(I),I=1, 50) /
     C       0.0,       1.0,       2.0,       3.0,       4.0,
     C       5.0,       6.0,       7.0,       8.0,       9.0,
     C      10.0,      11.0,      12.0,      13.0,      14.0,
     C      15.0,      16.0,      17.0,      18.0,      19.0,
     C      20.0,      21.0,      22.0,      23.0,      24.0,
     C      25.0,      27.5,      30.0,      32.5,      35.0,
     C      37.5,      40.0,      42.5,      45.0,      47.5,
     C      50.0,      55.0,      60.0,      65.0,      70.0,
     C      75.0,      80.0,      85.0,      90.0,      95.0,
     C     100.0,     105.0,     110.0,     115.0,     120.0/
	DATA(P2(I),I=1,50) /
     C 1.013E+03, 8.988E+02, 7.950E+02, 7.012E+02, 6.166E+02,
     C 5.405E+02, 4.722E+02, 4.111E+02, 3.565E+02, 3.080E+02,
     C 2.650E+02, 2.270E+02, 1.940E+02, 1.658E+02, 1.417E+02,
     C 1.211E+02, 1.035E+02, 8.850E+01, 7.565E+01, 6.467E+01,
     C 5.529E+01, 4.729E+01, 4.047E+01, 3.467E+01, 2.972E+01,
     C 2.549E+01, 1.743E+01, 1.197E+01, 8.010E+00, 5.746E+00,
     C 4.150E+00, 2.871E+00, 2.060E+00, 1.491E+00, 1.090E+00,
     C 7.978E-01, 4.250E-01, 2.190E-01, 1.090E-01, 5.220E-02,
     C 2.400E-02, 1.050E-02, 4.460E-03, 1.840E-03, 7.600E-04,
     C 3.200E-04, 1.450E-04, 7.100E-05, 4.010E-05, 2.540E-05/
	DATA(T2(I),I=1,50) /
     C    288.20,    281.70,    275.20,    268.70,    262.20,
     C    255.70,    249.20,    242.70,    236.20,    229.70,
     C    223.30,    216.80,    216.70,    216.70,    216.70,
     C    216.70,    216.70,    216.70,    216.70,    216.70,
     C    216.70,    217.60,    218.60,    219.60,    220.60,
     C    221.60,    224.00,    226.50,    230.00,    236.50,
     C    242.90,    250.40,    257.30,    264.20,    270.60,
     C    270.70,    260.80,    247.00,    233.30,    219.60,
     C    208.40,    198.60,    188.90,    186.90,    188.40,
     C    195.10,    208.80,    240.00,    300.00,    360.00/
	DATA(WH2(I),I=1,50)/
     C 7.745E+03, 6.071E+03, 4.631E+03, 3.182E+03, 2.158E+03,
     C 1.397E+03, 9.254E+02, 5.720E+02, 3.667E+02, 1.583E+02,
     C 6.996E+01, 3.613E+01, 1.906E+01, 1.085E+01, 5.927E+00,
     C 5.000E+00, 3.950E+00, 3.850E+00, 3.825E+00, 3.850E+00,
     C 3.900E+00, 3.975E+00, 4.065E+00, 4.200E+00, 4.300E+00,
     C 4.425E+00, 4.575E+00, 4.725E+00, 4.825E+00, 4.900E+00,
     C 4.950E+00, 5.025E+00, 5.150E+00, 5.225E+00, 5.250E+00,
     C 5.225E+00, 5.100E+00, 4.750E+00, 4.200E+00, 3.500E+00,
     C 2.825E+00, 2.050E+00, 1.330E+00, 8.500E-01, 5.400E-01,
     C 4.000E-01, 3.400E-01, 2.800E-01, 2.400E-01, 2.000E-01/
	DATA(WO3(I),I=1,50) /
     C 2.660E-02, 2.931E-02, 3.237E-02, 3.318E-02, 3.387E-02,
     C 3.768E-02, 4.112E-02, 5.009E-02, 5.966E-02, 9.168E-02,
     C 1.313E-01, 2.149E-01, 3.095E-01, 3.846E-01, 5.030E-01,
     C 6.505E-01, 8.701E-01, 1.187E+00, 1.587E+00, 2.030E+00,
     C 2.579E+00, 3.028E+00, 3.647E+00, 4.168E+00, 4.627E+00,
     C 5.118E+00, 5.803E+00, 6.553E+00, 7.373E+00, 7.837E+00,
     C 7.800E+00, 7.300E+00, 6.200E+00, 5.250E+00, 4.100E+00,
     C 3.100E+00, 1.800E+00, 1.100E+00, 7.000E-01, 3.000E-01,
     C 2.500E-01, 3.000E-01, 5.000E-01, 7.000E-01, 7.000E-01,
     C 4.000E-01, 2.000E-01, 5.000E-02, 5.000E-03, 5.000E-04/
	DO I=1,50
	   Z(I)=Z2(I)
	   P(I)=P2(I)
	   T(I)=T2(I)
	   H2O(I)=WH2(I)
	   O3(I)=WO3(I)
	END DO
	RETURN
	END
c---------------------------------------------------------------------
	SUBROUTINE O3HHT0(V,C)
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /O3HH0/ V1S,V2S,DVS,S(2280),NPTS
	CALL O3INT(V ,V1S,DVS,NPTS,S,C)
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE O3HHT1(V,C)
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /O3HH1/ V1S,V2S,DVS,S(2280),NPTS
	CALL O3INT(V ,V1S,DVS,NPTS,S,C)
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE O3HHT2(V,C)
	IMPLICIT REAL*8(A-H,O-Z)
	COMMON /O3HH2/ V1S,V2S,DVS,S(2280),NPTS
	CALL O3INT(V ,V1S,DVS,NPTS,S,C)
	RETURN
	END
C----------------------------------------------------------------------
	SUBROUTINE O3INT(V1C,V1,DV,NPT,CONTI,CONTO)
	IMPLICIT REAL*8(A-H,O-Z)
C     INTERPOLATION FOR O3 CONTINUUM
	DIMENSION CONTI(2280)
	CONTO=0.
	I=(V1C-V1)/DV+1.00001
	IF(I.LT.1  )GO TO 10
	IF(I.GT.NPT)GO TO 10
	CONTO=CONTI(I)
10    CONTINUE
	RETURN
	END
C----------------------------------------------------------------------
	BLOCK DATA BO3HH0
	IMPLICIT REAL*8(A-H,O-Z)
C
C     O3HH0 CONTAINS O3 HARTLEY HUGGINS CROSS SECTIONS FOR 273K
C               UNITS OF (CM**2/MOL)*1.E-20
C     DATA FROM BASS(1985)

	COMMON /O3HH0/  V1C,V2C,DVC,
     X           C00001(80),C00081(80),C00161(80),C00241(80),C00321(80),
     X           C00401(80),C00481(80),C00561(80),C00641(80),C00721(80),
     X           C00801(80),C00881(80),C00961(80),C01041(80),C01121(80),
     X           C01201(80),C01281(80),C01361(80),C01441(80),C01521(80),
     X           C01601(80),C01681(80),C01761(80),C01841(80),C01921(80),
     X           C02001(80),C02081(80),C02161(80),C02241(40),
     +           NC
C
	DATA V1C /29405./, V2C /40800./ ,DVC /5./, NC /2280/
C         DATA FROM BASS, JUNE 1985
C
	DATA C00001 /
     X 1.81094E-01, 1.57760E-01, 1.37336E-01, 1.19475E-01, 1.17191E-01,
     X 1.14331E-01, 1.15984E-01, 1.10412E-01, 1.12660E-01, 1.16014E-01,
     X 1.15060E-01, 1.12041E-01, 1.11611E-01, 1.00378E-01, 9.54850E-02,
     X 9.87528E-02, 9.46153E-02, 9.53093E-02, 9.72653E-02, 9.66468E-02,

     X 9.39750E-02, 1.03552E-01, 1.01361E-01, 1.04315E-01, 1.12842E-01,
     X 1.02800E-01, 1.09576E-01, 1.05577E-01, 1.17334E-01, 1.25763E-01,
     X 1.27597E-01, 1.34267E-01, 1.44799E-01, 1.57366E-01, 1.67369E-01,
     X 1.81778E-01, 1.89207E-01, 2.01376E-01, 2.10310E-01, 2.21721E-01,
     X 2.43162E-01, 2.55542E-01, 2.75312E-01, 2.88576E-01, 3.02505E-01,
     X 3.15141E-01, 3.28908E-01, 3.49000E-01, 3.56620E-01, 3.59852E-01,
     X 3.57517E-01, 3.12924E-01, 2.63610E-01, 2.50854E-01, 2.25642E-01,
     X 2.15954E-01, 2.12099E-01, 2.13039E-01, 2.12286E-01, 2.17214E-01,
     X 2.28784E-01, 2.28276E-01, 2.34677E-01, 2.30730E-01, 2.16107E-01,
     X 1.99471E-01, 1.85629E-01, 1.72730E-01, 1.56229E-01, 1.38156E-01,
     X 1.37641E-01, 1.33169E-01, 1.32759E-01, 1.30102E-01, 1.35396E-01,
     X 1.37976E-01, 1.41571E-01, 1.46448E-01, 1.44508E-01, 1.47612E-01/
	DATA C00081 /
     X 1.47424E-01, 1.48173E-01, 1.52936E-01, 1.58908E-01, 1.58808E-01,
     X 1.59860E-01, 1.73936E-01, 1.84109E-01, 1.95143E-01, 2.08267E-01,
     X 2.19256E-01, 2.31653E-01, 2.46400E-01, 2.60437E-01, 2.70792E-01,
     X 2.79749E-01, 2.91068E-01, 2.98080E-01, 3.10421E-01, 3.24540E-01,
     X 3.39003E-01, 3.58322E-01, 3.81520E-01, 4.02798E-01, 4.35972E-01,
     X 4.56220E-01, 4.79037E-01, 5.02597E-01, 5.24648E-01, 5.33964E-01,
     X 5.39211E-01, 5.43613E-01, 5.28793E-01, 4.94103E-01, 4.34481E-01,
     X 3.76792E-01, 3.37161E-01, 3.15750E-01, 3.11042E-01, 3.08745E-01,
     X 3.09195E-01, 3.05859E-01, 3.01443E-01, 2.88111E-01, 2.81303E-01,
     X 2.75329E-01, 2.60812E-01, 2.59337E-01, 2.45576E-01, 2.40470E-01,
     X 2.39705E-01, 2.45389E-01, 2.49801E-01, 2.53235E-01, 2.54387E-01,
     X 2.64311E-01, 2.74146E-01, 2.89737E-01, 2.96673E-01, 3.07337E-01,
     X 3.24380E-01, 3.42266E-01, 3.59522E-01, 3.78005E-01, 3.97178E-01,
     X 4.23351E-01, 4.45925E-01, 4.63029E-01, 4.94843E-01, 5.19418E-01,
     X 5.49928E-01, 5.69115E-01, 6.02396E-01, 6.43471E-01, 6.76401E-01,
     X 7.14024E-01, 7.42425E-01, 7.60916E-01, 7.83319E-01, 7.98299E-01/
	DATA C00161 /
     X 7.76672E-01, 7.22769E-01, 6.45967E-01, 5.80850E-01, 5.76514E-01,
     X 5.79380E-01, 5.90359E-01, 6.21721E-01, 6.37540E-01, 6.52572E-01,
     X 6.63442E-01, 6.69026E-01, 6.69038E-01, 6.53319E-01, 6.21950E-01,
     X 5.47619E-01, 4.58994E-01, 4.14888E-01, 3.97736E-01, 3.88775E-01,
     X 3.87424E-01, 3.93567E-01, 4.03442E-01, 4.05217E-01, 4.12848E-01,
     X 4.12246E-01, 4.16620E-01, 4.13195E-01, 4.08467E-01, 4.13104E-01,
     X 4.24498E-01, 4.32002E-01, 4.46361E-01, 4.61131E-01, 4.77228E-01,
     X 4.96519E-01, 5.16764E-01, 5.38966E-01, 5.54187E-01, 5.73748E-01,
     X 6.07260E-01, 6.34358E-01, 6.60286E-01, 6.95533E-01, 7.37090E-01,
     X 7.83894E-01, 8.19557E-01, 8.49244E-01, 8.91832E-01, 9.44885E-01,
     X 9.86271E-01, 1.02262E+00, 1.07242E+00, 1.12162E+00, 1.18287E+00,
     X 1.22402E+00, 1.24978E+00, 1.24392E+00, 1.19668E+00, 1.11562E+00,
     X 1.03983E+00, 9.31884E-01, 8.35307E-01, 7.92620E-01, 7.81980E-01,
     X 7.89623E-01, 8.05987E-01, 8.27344E-01, 8.57514E-01, 8.66302E-01,
     X 8.72092E-01, 8.66840E-01, 8.40536E-01, 7.87360E-01, 7.35743E-01,
     X 6.92039E-01, 6.64032E-01, 6.48360E-01, 6.46288E-01, 6.49505E-01/
	DATA C00241 /
     X 6.69937E-01, 6.81006E-01, 7.00969E-01, 7.19834E-01, 7.26964E-01,
     X 7.50591E-01, 7.73600E-01, 8.00673E-01, 8.20347E-01, 8.37855E-01,
     X 8.66780E-01, 9.04297E-01, 9.46300E-01, 9.69134E-01, 9.97928E-01,
     X 1.06388E+00, 1.11032E+00, 1.15221E+00, 1.21324E+00, 1.24462E+00,
     X 1.31978E+00, 1.35617E+00, 1.38792E+00, 1.39196E+00, 1.35161E+00,
     X 1.29381E+00, 1.30295E+00, 1.32965E+00, 1.37024E+00, 1.44064E+00,
     X 1.50484E+00, 1.57200E+00, 1.62097E+00, 1.67874E+00, 1.72676E+00,
     X 1.73383E+00, 1.66091E+00, 1.54936E+00, 1.35454E+00, 1.20070E+00,
     X 1.14609E+00, 1.13642E+00, 1.13784E+00, 1.14609E+00, 1.14531E+00,
     X 1.16024E+00, 1.16891E+00, 1.16111E+00, 1.14192E+00, 1.09903E+00,
     X 1.05745E+00, 1.02341E+00, 1.00121E+00, 1.00036E+00, 1.00576E+00,
     X 1.02405E+00, 1.04379E+00, 1.07623E+00, 1.11347E+00, 1.17305E+00,
     X 1.20016E+00, 1.22697E+00, 1.27479E+00, 1.32572E+00, 1.38690E+00,
     X 1.43768E+00, 1.48379E+00, 1.55317E+00, 1.64020E+00, 1.71268E+00,
     X 1.77183E+00, 1.85824E+00, 1.95131E+00, 2.04609E+00, 2.13151E+00,
     X 2.17777E+00, 2.22832E+00, 2.26886E+00, 2.19775E+00, 2.05087E+00/
	DATA C00321 /
     X 1.96103E+00, 1.95554E+00, 1.98037E+00, 2.05440E+00, 2.11629E+00,
     X 2.17893E+00, 2.24384E+00, 2.30464E+00, 2.32525E+00, 2.29945E+00,
     X 2.21712E+00, 2.03430E+00, 1.82139E+00, 1.70354E+00, 1.64631E+00,
     X 1.62164E+00, 1.61356E+00, 1.63900E+00, 1.66313E+00, 1.67409E+00,
     X 1.69143E+00, 1.70181E+00, 1.69165E+00, 1.67699E+00, 1.67879E+00,
     X 1.67312E+00, 1.68133E+00, 1.70002E+00, 1.72500E+00, 1.76308E+00,
     X 1.80634E+00, 1.87548E+00, 1.94924E+00, 1.99812E+00, 2.05333E+00,
     X 2.14035E+00, 2.21847E+00, 2.27412E+00, 2.29752E+00, 2.30750E+00,
     X 2.36165E+00, 2.44394E+00, 2.52782E+00, 2.61343E+00, 2.71640E+00,
     X 2.81613E+00, 2.93679E+00, 3.01577E+00, 3.15995E+00, 3.15931E+00,
     X 2.96658E+00, 2.73295E+00, 2.67480E+00, 2.66652E+00, 2.69393E+00,
     X 2.75102E+00, 2.86503E+00, 2.99163E+00, 2.99576E+00, 3.02603E+00,
     X 2.98415E+00, 2.79309E+00, 2.65337E+00, 2.50962E+00, 2.43207E+00,
     X 2.34812E+00, 2.34872E+00, 2.35186E+00, 2.39477E+00, 2.42629E+00,
     X 2.48068E+00, 2.55087E+00, 2.55952E+00, 2.56497E+00, 2.64323E+00,
     X 2.67961E+00, 2.66263E+00, 2.70243E+00, 2.74911E+00, 2.81786E+00/
	DATA C00401 /
     X 2.88684E+00, 2.97790E+00, 3.04305E+00, 3.13053E+00, 3.23857E+00,
     X 3.35582E+00, 3.40654E+00, 3.38117E+00, 3.36296E+00, 3.39480E+00,
     X 3.49066E+00, 3.60832E+00, 3.71817E+00, 3.83924E+00, 3.96355E+00,
     X 4.03656E+00, 4.00518E+00, 3.90389E+00, 3.74790E+00, 3.61385E+00,
     X 3.57066E+00, 3.59438E+00, 3.66182E+00, 3.71176E+00, 3.75255E+00,
     X 3.79101E+00, 3.85278E+00, 3.85027E+00, 3.81112E+00, 3.72553E+00,
     X 3.61017E+00, 3.54384E+00, 3.52406E+00, 3.54097E+00, 3.59375E+00,
     X 3.66312E+00, 3.72632E+00, 3.76825E+00, 3.86798E+00, 3.92916E+00,
     X 3.95610E+00, 4.00120E+00, 4.05865E+00, 4.11981E+00, 4.14634E+00,
     X 4.19109E+00, 4.20317E+00, 4.25754E+00, 4.35131E+00, 4.48573E+00,
     X 4.58716E+00, 4.67462E+00, 4.78228E+00, 4.91196E+00, 5.01871E+00,
     X 5.10663E+00, 5.17780E+00, 5.21393E+00, 5.18144E+00, 5.04379E+00,
     X 4.86504E+00, 4.78569E+00, 4.72717E+00, 4.69132E+00, 4.65797E+00,
     X 4.60305E+00, 4.59798E+00, 4.65300E+00, 4.69707E+00, 4.74790E+00,
     X 4.82581E+00, 4.80953E+00, 4.80517E+00, 4.82685E+00, 4.82321E+00,
     X 4.84806E+00, 4.88591E+00, 4.91759E+00, 4.98074E+00, 5.07071E+00/
	DATA C00481 /
     X 5.18733E+00, 5.30567E+00, 5.38670E+00, 5.43942E+00, 5.51797E+00,
     X 5.62652E+00, 5.71228E+00, 5.82347E+00, 5.91434E+00, 6.00171E+00,
     X 6.06977E+00, 6.13040E+00, 6.21990E+00, 6.29980E+00, 6.37206E+00,
     X 6.48233E+00, 6.53068E+00, 6.53275E+00, 6.56858E+00, 6.54577E+00,
     X 6.50472E+00, 6.41504E+00, 6.33853E+00, 6.31184E+00, 6.21253E+00,
     X 6.22034E+00, 6.26918E+00, 6.28982E+00, 6.29461E+00, 6.35418E+00,
     X 6.40956E+00, 6.38020E+00, 6.39784E+00, 6.45383E+00, 6.50134E+00,
     X 6.56808E+00, 6.58850E+00, 6.58882E+00, 6.65097E+00, 6.75259E+00,
     X 6.83256E+00, 6.92593E+00, 6.98083E+00, 7.03632E+00, 7.11147E+00,
     X 7.15622E+00, 7.21106E+00, 7.27319E+00, 7.33382E+00, 7.38601E+00,
     X 7.48971E+00, 7.61459E+00, 7.70134E+00, 7.76194E+00, 7.85534E+00,
     X 7.99519E+00, 8.12227E+00, 8.25461E+00, 8.34670E+00, 8.42733E+00,
     X 8.51806E+00, 8.57638E+00, 8.56481E+00, 8.55461E+00, 8.55593E+00,
     X 8.58756E+00, 8.50070E+00, 8.54400E+00, 8.57575E+00, 8.62083E+00,
     X 8.60684E+00, 8.67824E+00, 8.72069E+00, 8.79127E+00, 8.85479E+00,
     X 8.86770E+00, 8.90574E+00, 8.91531E+00, 8.94800E+00, 9.00167E+00/
	DATA C00561 /
     X 9.14051E+00, 9.25421E+00, 9.39694E+00, 9.50896E+00, 9.53190E+00,
     X 9.55977E+00, 9.53482E+00, 9.49662E+00, 9.53359E+00, 9.54007E+00,
     X 9.49809E+00, 9.49373E+00, 9.53282E+00, 9.63757E+00, 9.67855E+00,
     X 9.67633E+00, 9.67045E+00, 9.79481E+00, 9.93420E+00, 1.00234E+01,
     X 1.01372E+01, 1.02577E+01, 1.05056E+01, 1.07873E+01, 1.09967E+01,
     X 1.10873E+01, 1.11624E+01, 1.13006E+01, 1.14875E+01, 1.16106E+01,
     X 1.16744E+01, 1.17582E+01, 1.17709E+01, 1.18537E+01, 1.19623E+01,
     X 1.19763E+01, 1.19879E+01, 1.20384E+01, 1.20763E+01, 1.20826E+01,
     X 1.20449E+01, 1.19747E+01, 1.20227E+01, 1.21805E+01, 1.23134E+01,
     X 1.24042E+01, 1.25614E+01, 1.26828E+01, 1.26645E+01, 1.26963E+01,
     X 1.28226E+01, 1.28720E+01, 1.28981E+01, 1.29462E+01, 1.29363E+01,
     X 1.29199E+01, 1.29797E+01, 1.28860E+01, 1.29126E+01, 1.30205E+01,
     X 1.31327E+01, 1.31722E+01, 1.31901E+01, 1.33189E+01, 1.34833E+01,
     X 1.36228E+01, 1.37474E+01, 1.38548E+01, 1.39450E+01, 1.40926E+01,
     X 1.43099E+01, 1.44836E+01, 1.46257E+01, 1.47755E+01, 1.49163E+01,
     X 1.51038E+01, 1.53308E+01, 1.54194E+01, 1.54852E+01, 1.55968E+01/
	DATA C00641 /
     X 1.57025E+01, 1.58667E+01, 1.60365E+01, 1.61427E+01, 1.62967E+01,
     X 1.64735E+01, 1.66123E+01, 1.67268E+01, 1.67673E+01, 1.67825E+01,
     X 1.68898E+01, 1.68178E+01, 1.68216E+01, 1.68574E+01, 1.68799E+01,
     X 1.70317E+01, 1.70767E+01, 1.71508E+01, 1.72965E+01, 1.73421E+01,
     X 1.73937E+01, 1.74420E+01, 1.74535E+01, 1.75110E+01, 1.75497E+01,
     X 1.75149E+01, 1.75955E+01, 1.78260E+01, 1.78271E+01, 1.79750E+01,
     X 1.80600E+01, 1.81597E+01, 1.83454E+01, 1.85243E+01, 1.87382E+01,
     X 1.88904E+01, 1.90395E+01, 1.92759E+01, 1.95398E+01, 1.97712E+01,
     X 1.98487E+01, 1.99522E+01, 2.02363E+01, 2.03271E+01, 2.07090E+01,
     X 2.09195E+01, 2.10974E+01, 2.11702E+01, 2.12964E+01, 2.14339E+01,
     X 2.15764E+01, 2.17351E+01, 2.18486E+01, 2.19700E+01, 2.21663E+01,
     X 2.24244E+01, 2.24813E+01, 2.25248E+01, 2.26357E+01, 2.26457E+01,
     X 2.27249E+01, 2.27172E+01, 2.27123E+01, 2.26859E+01, 2.27216E+01,
     X 2.29306E+01, 2.30711E+01, 2.31374E+01, 2.31815E+01, 2.33423E+01,
     X 2.33810E+01, 2.36430E+01, 2.36807E+01, 2.36676E+01, 2.38607E+01,
     X 2.41559E+01, 2.43413E+01, 2.44401E+01, 2.45968E+01, 2.47927E+01/
	DATA C00721 /
     X 2.50743E+01, 2.53667E+01, 2.55749E+01, 2.57357E+01, 2.58927E+01,
     X 2.61523E+01, 2.64110E+01, 2.66650E+01, 2.68829E+01, 2.70635E+01,
     X 2.72797E+01, 2.75064E+01, 2.77229E+01, 2.80341E+01, 2.82003E+01,
     X 2.83346E+01, 2.83909E+01, 2.86212E+01, 2.88006E+01, 2.89577E+01,
     X 2.90965E+01, 2.91834E+01, 2.93224E+01, 2.94094E+01, 2.94848E+01,
     X 2.96584E+01, 2.96749E+01, 2.97760E+01, 2.99163E+01, 3.00238E+01,
     X 3.01290E+01, 3.02307E+01, 3.03663E+01, 3.05897E+01, 3.07937E+01,
     X 3.10403E+01, 3.11778E+01, 3.13271E+01, 3.15799E+01, 3.18435E+01,
     X 3.21614E+01, 3.25097E+01, 3.27701E+01, 3.29600E+01, 3.32583E+01,
     X 3.36348E+01, 3.40282E+01, 3.41751E+01, 3.44128E+01, 3.46199E+01,
     X 3.49363E+01, 3.52087E+01, 3.54056E+01, 3.55596E+01, 3.56694E+01,
     X 3.58104E+01, 3.60276E+01, 3.62818E+01, 3.63505E+01, 3.66069E+01,
     X 3.67544E+01, 3.70664E+01, 3.72525E+01, 3.73491E+01, 3.76006E+01,
     X 3.77102E+01, 3.78970E+01, 3.81254E+01, 3.82728E+01, 3.81720E+01,
     X 3.82781E+01, 3.84982E+01, 3.87202E+01, 3.89958E+01, 3.94148E+01,
     X 3.98434E+01, 3.98952E+01, 4.01573E+01, 4.06014E+01, 4.09651E+01/
	DATA C00801 /
     X 4.12821E+01, 4.16849E+01, 4.19899E+01, 4.22719E+01, 4.27736E+01,
     X 4.32254E+01, 4.33883E+01, 4.39831E+01, 4.39414E+01, 4.42613E+01,
     X 4.46503E+01, 4.49027E+01, 4.50384E+01, 4.52929E+01, 4.57269E+01,
     X 4.56433E+01, 4.57350E+01, 4.60128E+01, 4.60487E+01, 4.61183E+01,
     X 4.64397E+01, 4.68211E+01, 4.70706E+01, 4.72821E+01, 4.74972E+01,
     X 4.78253E+01, 4.81615E+01, 4.84480E+01, 4.85703E+01, 4.87397E+01,
     X 4.90015E+01, 4.93673E+01, 4.97291E+01, 4.99836E+01, 5.02975E+01,
     X 5.05572E+01, 5.08226E+01, 5.13433E+01, 5.17112E+01, 5.19703E+01,
     X 5.23128E+01, 5.27305E+01, 5.30599E+01, 5.34555E+01, 5.39625E+01,
     X 5.43627E+01, 5.45446E+01, 5.49263E+01, 5.53511E+01, 5.57270E+01,
     X 5.60904E+01, 5.63875E+01, 5.68475E+01, 5.73172E+01, 5.81134E+01,
     X 5.86399E+01, 5.90384E+01, 5.91417E+01, 5.90883E+01, 5.93610E+01,
     X 5.95794E+01, 5.99600E+01, 5.98493E+01, 5.99441E+01, 6.02748E+01,
     X 6.04778E+01, 6.05233E+01, 6.07194E+01, 6.11589E+01, 6.13324E+01,
     X 6.17685E+01, 6.23166E+01, 6.31055E+01, 6.38211E+01, 6.42320E+01,
     X 6.45195E+01, 6.51125E+01, 6.56765E+01, 6.59286E+01, 6.62716E+01/
	DATA C00881 /
     X 6.65693E+01, 6.68906E+01, 6.72246E+01, 6.75177E+01, 6.78476E+01,
     X 6.82599E+01, 6.84400E+01, 6.89072E+01, 6.95720E+01, 7.01410E+01,
     X 7.05519E+01, 7.09367E+01, 7.13975E+01, 7.22128E+01, 7.28222E+01,
     X 7.33808E+01, 7.38828E+01, 7.44496E+01, 7.49983E+01, 7.54178E+01,
     X 7.60554E+01, 7.62484E+01, 7.67892E+01, 7.71262E+01, 7.76235E+01,
     X 7.81413E+01, 7.85694E+01, 7.91248E+01, 7.94715E+01, 7.96200E+01,
     X 8.00270E+01, 8.03783E+01, 8.07100E+01, 8.11929E+01, 8.17375E+01,
     X 8.18410E+01, 8.23341E+01, 8.26754E+01, 8.30893E+01, 8.34232E+01,
     X 8.35533E+01, 8.36017E+01, 8.38589E+01, 8.43366E+01, 8.47593E+01,
     X 8.51614E+01, 8.55271E+01, 8.58979E+01, 8.64892E+01, 8.74367E+01,
     X 8.82440E+01, 8.89105E+01, 8.90980E+01, 8.97266E+01, 9.04886E+01,
     X 9.12709E+01, 9.21243E+01, 9.26673E+01, 9.31331E+01, 9.38190E+01,
     X 9.44877E+01, 9.50636E+01, 9.57445E+01, 9.65211E+01, 9.68623E+01,
     X 9.75356E+01, 9.81991E+01, 9.88881E+01, 9.94554E+01, 9.99292E+01,
     X 1.00357E+02, 1.00670E+02, 1.01227E+02, 1.01529E+02, 1.01889E+02,
     X 1.02033E+02, 1.02254E+02, 1.02731E+02, 1.02914E+02, 1.03120E+02/
	DATA C00961 /
     X 1.03674E+02, 1.03768E+02, 1.04146E+02, 1.04850E+02, 1.05525E+02,
     X 1.06263E+02, 1.06653E+02, 1.07084E+02, 1.07461E+02, 1.08052E+02,
     X 1.08793E+02, 1.09395E+02, 1.09811E+02, 1.10079E+02, 1.10656E+02,
     X 1.11575E+02, 1.12544E+02, 1.13453E+02, 1.14440E+02, 1.15292E+02,
     X 1.15869E+02, 1.16925E+02, 1.17854E+02, 1.18723E+02, 1.19574E+02,
     X 1.19940E+02, 1.21108E+02, 1.21807E+02, 1.22490E+02, 1.23278E+02,
     X 1.24094E+02, 1.24816E+02, 1.25469E+02, 1.26217E+02, 1.26878E+02,
     X 1.27536E+02, 1.28168E+02, 1.28682E+02, 1.29076E+02, 1.30171E+02,
     X 1.30667E+02, 1.31242E+02, 1.31665E+02, 1.31961E+02, 1.32347E+02,
     X 1.32805E+02, 1.33152E+02, 1.33869E+02, 1.34261E+02, 1.34498E+02,
     X 1.35028E+02, 1.36049E+02, 1.36577E+02, 1.37491E+02, 1.38078E+02,
     X 1.38389E+02, 1.38819E+02, 1.39653E+02, 1.39770E+02, 1.40812E+02,
     X 1.40926E+02, 1.41267E+02, 1.41872E+02, 1.42233E+02, 1.43447E+02,
     X 1.44641E+02, 1.45500E+02, 1.45996E+02, 1.47040E+02, 1.48767E+02,
     X 1.48785E+02, 1.49525E+02, 1.50266E+02, 1.50814E+02, 1.51443E+02,
     X 1.52272E+02, 1.52846E+02, 1.54000E+02, 1.54629E+02, 1.54907E+02/
	DATA C01041 /
     X 1.55527E+02, 1.56642E+02, 1.57436E+02, 1.59036E+02, 1.59336E+02,
     X 1.59661E+02, 1.60287E+02, 1.61202E+02, 1.62410E+02, 1.63040E+02,
     X 1.62872E+02, 1.63248E+02, 1.63776E+02, 1.64313E+02, 1.65782E+02,
     X 1.65692E+02, 1.66049E+02, 1.66701E+02, 1.67786E+02, 1.69150E+02,
     X 1.69996E+02, 1.71634E+02, 1.71137E+02, 1.71372E+02, 1.72525E+02,
     X 1.73816E+02, 1.75219E+02, 1.76091E+02, 1.78260E+02, 1.79299E+02,
     X 1.79904E+02, 1.81718E+02, 1.83807E+02, 1.85488E+02, 1.85929E+02,
     X 1.86787E+02, 1.88282E+02, 1.89546E+02, 1.91489E+02, 1.92646E+02,
     X 1.93399E+02, 1.93838E+02, 1.94406E+02, 1.95829E+02, 1.96745E+02,
     X 1.96978E+02, 1.97243E+02, 1.97636E+02, 1.98025E+02, 1.98227E+02,
     X 1.99552E+02, 2.00304E+02, 2.01031E+02, 2.01788E+02, 2.02432E+02,
     X 2.03817E+02, 2.04866E+02, 2.05561E+02, 2.06180E+02, 2.07024E+02,
     X 2.08303E+02, 2.09426E+02, 2.10575E+02, 2.11637E+02, 2.12559E+02,
     X 2.13361E+02, 2.14191E+02, 2.15264E+02, 2.16366E+02, 2.17316E+02,
     X 2.17717E+02, 2.17154E+02, 2.19172E+02, 2.20346E+02, 2.20849E+02,
     X 2.21539E+02, 2.22810E+02, 2.22740E+02, 2.22824E+02, 2.23285E+02/
	DATA C01121 /
     X 2.23696E+02, 2.23864E+02, 2.23968E+02, 2.23544E+02, 2.24804E+02,
     X 2.25953E+02, 2.26753E+02, 2.27732E+02, 2.29505E+02, 2.30108E+02,
     X 2.31232E+02, 2.32552E+02, 2.33979E+02, 2.36677E+02, 2.38481E+02,
     X 2.41797E+02, 2.44025E+02, 2.45113E+02, 2.47373E+02, 2.47258E+02,
     X 2.48617E+02, 2.49790E+02, 2.50562E+02, 2.51198E+02, 2.51289E+02,
     X 2.52509E+02, 2.54136E+02, 2.55335E+02, 2.55808E+02, 2.56567E+02,
     X 2.57977E+02, 2.58987E+02, 2.59622E+02, 2.60170E+02, 2.61127E+02,
     X 2.60655E+02, 2.62129E+02, 2.64020E+02, 2.65659E+02, 2.67086E+02,
     X 2.67615E+02, 2.69800E+02, 2.71452E+02, 2.73314E+02, 2.76972E+02,
     X 2.78005E+02, 2.79815E+02, 2.81709E+02, 2.84043E+02, 2.87070E+02,
     X 2.88842E+02, 2.90555E+02, 2.92401E+02, 2.94314E+02, 2.96074E+02,
     X 2.97103E+02, 2.98037E+02, 2.98113E+02, 2.97705E+02, 2.97350E+02,
     X 2.97329E+02, 2.97016E+02, 2.96752E+02, 2.96599E+02, 2.96637E+02,
     X 2.97057E+02, 2.97585E+02, 2.98179E+02, 2.98997E+02, 3.00012E+02,
     X 3.00806E+02, 3.00908E+02, 3.02369E+02, 3.04063E+02, 3.05325E+02,
     X 3.06737E+02, 3.08066E+02, 3.09694E+02, 3.11530E+02, 3.13132E+02/
	DATA C01201 /
     X 3.13296E+02, 3.15513E+02, 3.16887E+02, 3.17682E+02, 3.18296E+02,
     X 3.18654E+02, 3.18912E+02, 3.19236E+02, 3.19626E+02, 3.20020E+02,
     X 3.20186E+02, 3.20709E+02, 3.21628E+02, 3.22625E+02, 3.23504E+02,
     X 3.25479E+02, 3.26825E+02, 3.28146E+02, 3.29404E+02, 3.30512E+02,
     X 3.32634E+02, 3.34422E+02, 3.35602E+02, 3.36833E+02, 3.39372E+02,
     X 3.43446E+02, 3.46374E+02, 3.48719E+02, 3.50881E+02, 3.53160E+02,
     X 3.54890E+02, 3.57162E+02, 3.59284E+02, 3.60876E+02, 3.62295E+02,
     X 3.63987E+02, 3.64835E+02, 3.65257E+02, 3.65738E+02, 3.65904E+02,
     X 3.65976E+02, 3.66460E+02, 3.67087E+02, 3.67377E+02, 3.69079E+02,
     X 3.70694E+02, 3.70940E+02, 3.70557E+02, 3.72693E+02, 3.73852E+02,
     X 3.75679E+02, 3.77863E+02, 3.79964E+02, 3.81368E+02, 3.82716E+02,
     X 3.85556E+02, 3.89072E+02, 3.91796E+02, 3.92766E+02, 3.96551E+02,
     X 3.97833E+02, 3.97285E+02, 4.01929E+02, 4.02158E+02, 4.04553E+02,
     X 4.06451E+02, 4.06236E+02, 4.08135E+02, 4.07797E+02, 4.08415E+02,
     X 4.10111E+02, 4.11781E+02, 4.12735E+02, 4.11547E+02, 4.11606E+02,
     X 4.13548E+02, 4.12557E+02, 4.12923E+02, 4.12866E+02, 4.13009E+02/
	DATA C01281 /
     X 4.14447E+02, 4.16032E+02, 4.17032E+02, 4.19064E+02, 4.22458E+02,
     X 4.26021E+02, 4.25192E+02, 4.25684E+02, 4.27536E+02, 4.29972E+02,
     X 4.31994E+02, 4.36037E+02, 4.39132E+02, 4.40363E+02, 4.40716E+02,
     X 4.40342E+02, 4.42063E+02, 4.44408E+02, 4.45454E+02, 4.47835E+02,
     X 4.48256E+02, 4.48831E+02, 4.50257E+02, 4.51427E+02, 4.52373E+02,
     X 4.53899E+02, 4.55496E+02, 4.56311E+02, 4.57314E+02, 4.59922E+02,
     X 4.61048E+02, 4.59840E+02, 4.62144E+02, 4.63152E+02, 4.64565E+02,
     X 4.66715E+02, 4.69380E+02, 4.70751E+02, 4.72012E+02, 4.73482E+02,
     X 4.75524E+02, 4.79307E+02, 4.82035E+02, 4.84423E+02, 4.86712E+02,
     X 4.88754E+02, 4.90102E+02, 4.92047E+02, 4.94150E+02, 4.95375E+02,
     X 4.95828E+02, 4.97555E+02, 4.98559E+02, 4.97618E+02, 4.99265E+02,
     X 4.99979E+02, 5.00681E+02, 5.01386E+02, 5.00868E+02, 5.01935E+02,
     X 5.03151E+02, 5.04329E+02, 5.05546E+02, 5.08259E+02, 5.09222E+02,
     X 5.09818E+02, 5.11397E+02, 5.12391E+02, 5.13326E+02, 5.14329E+02,
     X 5.15443E+02, 5.16533E+02, 5.21417E+02, 5.25071E+02, 5.26581E+02,
     X 5.27762E+02, 5.29274E+02, 5.31704E+02, 5.34310E+02, 5.35727E+02/
	DATA C01361 /
     X 5.36838E+02, 5.37082E+02, 5.36733E+02, 5.36170E+02, 5.36063E+02,
     X 5.36451E+02, 5.37870E+02, 5.40475E+02, 5.42268E+02, 5.41972E+02,
     X 5.42532E+02, 5.44764E+02, 5.46844E+02, 5.47525E+02, 5.49150E+02,
     X 5.52049E+02, 5.55423E+02, 5.56259E+02, 5.57424E+02, 5.59189E+02,
     X 5.61167E+02, 5.64512E+02, 5.66753E+02, 5.68183E+02, 5.69628E+02,
     X 5.73474E+02, 5.76192E+02, 5.78058E+02, 5.79588E+02, 5.81619E+02,
     X 5.83530E+02, 5.84852E+02, 5.85326E+02, 5.88130E+02, 5.90570E+02,
     X 5.91785E+02, 5.91371E+02, 5.90931E+02, 5.90942E+02, 5.91168E+02,
     X 5.91291E+02, 5.89791E+02, 5.91146E+02, 5.90804E+02, 5.87847E+02,
     X 5.89067E+02, 5.91027E+02, 5.90951E+02, 5.89227E+02, 5.93389E+02,
     X 5.92921E+02, 5.92739E+02, 5.94544E+02, 5.98941E+02, 6.02302E+02,
     X 6.03908E+02, 6.04265E+02, 6.06737E+02, 6.08560E+02, 6.11272E+02,
     X 6.14992E+02, 6.18595E+02, 6.20930E+02, 6.22107E+02, 6.22957E+02,
     X 6.26710E+02, 6.28657E+02, 6.30132E+02, 6.31543E+02, 6.33043E+02,
     X 6.36932E+02, 6.38248E+02, 6.37126E+02, 6.41648E+02, 6.48274E+02,
     X 6.52638E+02, 6.53922E+02, 6.56647E+02, 6.59351E+02, 6.60525E+02/
	DATA C01441 /
     X 6.60130E+02, 6.61375E+02, 6.62660E+02, 6.63976E+02, 6.65181E+02,
     X 6.64820E+02, 6.64458E+02, 6.64927E+02, 6.66555E+02, 6.66759E+02,
     X 6.68218E+02, 6.70323E+02, 6.72703E+02, 6.76085E+02, 6.79180E+02,
     X 6.80850E+02, 6.80017E+02, 6.79928E+02, 6.80886E+02, 6.82038E+02,
     X 6.82271E+02, 6.84057E+02, 6.85309E+02, 6.86816E+02, 6.90180E+02,
     X 6.93205E+02, 6.95870E+02, 6.98794E+02, 7.03776E+02, 7.04010E+02,
     X 7.05041E+02, 7.07254E+02, 7.07432E+02, 7.10736E+02, 7.13791E+02,
     X 7.15542E+02, 7.16468E+02, 7.17412E+02, 7.17783E+02, 7.17340E+02,
     X 7.18184E+02, 7.18716E+02, 7.18809E+02, 7.18282E+02, 7.20317E+02,
     X 7.18568E+02, 7.16274E+02, 7.19119E+02, 7.20852E+02, 7.21727E+02,
     X 7.22607E+02, 7.26369E+02, 7.26412E+02, 7.27101E+02, 7.29404E+02,
     X 7.30786E+02, 7.30910E+02, 7.30656E+02, 7.30566E+02, 7.33408E+02,
     X 7.37064E+02, 7.39178E+02, 7.36713E+02, 7.37365E+02, 7.40861E+02,
     X 7.45281E+02, 7.46178E+02, 7.46991E+02, 7.48035E+02, 7.49777E+02,
     X 7.54665E+02, 7.56585E+02, 7.57408E+02, 7.58131E+02, 7.58155E+02,
     X 7.60838E+02, 7.64792E+02, 7.68161E+02, 7.69263E+02, 7.73166E+02/
	DATA C01521 /
     X 7.79006E+02, 7.82037E+02, 7.83109E+02, 7.84674E+02, 7.87444E+02,
     X 7.89510E+02, 7.90130E+02, 7.91364E+02, 7.95225E+02, 8.03599E+02,
     X 8.06340E+02, 8.05105E+02, 8.05120E+02, 8.08515E+02, 8.10907E+02,
     X 8.11388E+02, 8.13432E+02, 8.12579E+02, 8.10564E+02, 8.08719E+02,
     X 8.07682E+02, 8.05009E+02, 8.01754E+02, 8.01013E+02, 7.99926E+02,
     X 7.99067E+02, 7.98369E+02, 7.94090E+02, 7.92883E+02, 7.94244E+02,
     X 7.98220E+02, 7.98201E+02, 7.98332E+02, 7.99289E+02, 8.02355E+02,
     X 8.03621E+02, 8.05302E+02, 8.08368E+02, 8.09983E+02, 8.11529E+02,
     X 8.13068E+02, 8.14717E+02, 8.16441E+02, 8.19241E+02, 8.22944E+02,
     X 8.23768E+02, 8.25030E+02, 8.26103E+02, 8.26374E+02, 8.28331E+02,
     X 8.32620E+02, 8.38618E+02, 8.43666E+02, 8.45212E+02, 8.46324E+02,
     X 8.48536E+02, 8.50192E+02, 8.53083E+02, 8.56653E+02, 8.59614E+02,
     X 8.62000E+02, 8.64593E+02, 8.67678E+02, 8.70908E+02, 8.73408E+02,
     X 8.74779E+02, 8.74005E+02, 8.76718E+02, 8.80445E+02, 8.84365E+02,
     X 8.83806E+02, 8.84292E+02, 8.85539E+02, 8.87474E+02, 8.84905E+02,
     X 8.84039E+02, 8.85105E+02, 8.83733E+02, 8.82224E+02, 8.79865E+02/
	DATA C01601 /
     X 8.75663E+02, 8.75575E+02, 8.73144E+02, 8.68602E+02, 8.70278E+02,
     X 8.69659E+02, 8.68701E+02, 8.69250E+02, 8.71057E+02, 8.72860E+02,
     X 8.74361E+02, 8.74458E+02, 8.77576E+02, 8.81613E+02, 8.84358E+02,
     X 8.87440E+02, 8.91549E+02, 8.96568E+02, 8.99836E+02, 9.02880E+02,
     X 9.05428E+02, 9.06891E+02, 9.07349E+02, 9.10151E+02, 9.15917E+02,
     X 9.16197E+02, 9.18571E+02, 9.21219E+02, 9.20292E+02, 9.21949E+02,
     X 9.24509E+02, 9.27454E+02, 9.29474E+02, 9.31348E+02, 9.32818E+02,
     X 9.32658E+02, 9.36280E+02, 9.39512E+02, 9.39667E+02, 9.44078E+02,
     X 9.47196E+02, 9.48291E+02, 9.46150E+02, 9.46918E+02, 9.49093E+02,
     X 9.51372E+02, 9.53109E+02, 9.56308E+02, 9.61335E+02, 9.58214E+02,
     X 9.56188E+02, 9.55660E+02, 9.58633E+02, 9.57541E+02, 9.54879E+02,
     X 9.51663E+02, 9.52839E+02, 9.52055E+02, 9.49253E+02, 9.50187E+02,
     X 9.50323E+02, 9.50937E+02, 9.54362E+02, 9.55855E+02, 9.56350E+02,
     X 9.55908E+02, 9.57963E+02, 9.61866E+02, 9.66948E+02, 9.69786E+02,
     X 9.74302E+02, 9.79061E+02, 9.82465E+02, 9.86019E+02, 9.89930E+02,
     X 9.94294E+02, 9.97011E+02, 9.98207E+02, 9.98607E+02, 1.00175E+03/
	DATA C01681 /
     X 1.00275E+03, 1.00284E+03, 1.00294E+03, 1.00485E+03, 1.00593E+03,
     X 1.00524E+03, 1.00415E+03, 1.00335E+03, 1.00278E+03, 1.00185E+03,
     X 9.99982E+02, 9.98177E+02, 9.97959E+02, 9.99161E+02, 9.98810E+02,
     X 9.95415E+02, 9.94342E+02, 9.92998E+02, 9.91340E+02, 9.90900E+02,
     X 9.90407E+02, 9.89232E+02, 9.85447E+02, 9.86312E+02, 9.87461E+02,
     X 9.86090E+02, 9.86670E+02, 9.85534E+02, 9.81877E+02, 9.84946E+02,
     X 9.86392E+02, 9.86709E+02, 9.88086E+02, 9.90269E+02, 9.92566E+02,
     X 9.94029E+02, 9.95795E+02, 9.97788E+02, 1.00005E+03, 1.00287E+03,
     X 1.00566E+03, 1.00833E+03, 1.00982E+03, 1.01348E+03, 1.01862E+03,
     X 1.02322E+03, 1.02786E+03, 1.03179E+03, 1.03339E+03, 1.03833E+03,
     X 1.04317E+03, 1.04598E+03, 1.04753E+03, 1.04981E+03, 1.05321E+03,
     X 1.05492E+03, 1.05721E+03, 1.05978E+03, 1.06033E+03, 1.06107E+03,
     X 1.06155E+03, 1.06035E+03, 1.05838E+03, 1.05649E+03, 1.05553E+03,
     X 1.05498E+03, 1.05387E+03, 1.05171E+03, 1.04877E+03, 1.04725E+03,
     X 1.04748E+03, 1.04733E+03, 1.04704E+03, 1.04643E+03, 1.04411E+03,
     X 1.04435E+03, 1.04520E+03, 1.04233E+03, 1.04047E+03, 1.03992E+03/
	DATA C01761 /
     X 1.04192E+03, 1.04171E+03, 1.04140E+03, 1.04197E+03, 1.04415E+03,
     X 1.04548E+03, 1.04533E+03, 1.04616E+03, 1.04705E+03, 1.04800E+03,
     X 1.05025E+03, 1.05219E+03, 1.05412E+03, 1.05808E+03, 1.06062E+03,
     X 1.06292E+03, 1.06780E+03, 1.07219E+03, 1.07610E+03, 1.07913E+03,
     X 1.08405E+03, 1.08798E+03, 1.08835E+03, 1.09140E+03, 1.09447E+03,
     X 1.09676E+03, 1.10015E+03, 1.10272E+03, 1.10410E+03, 1.10749E+03,
     X 1.10991E+03, 1.11121E+03, 1.10981E+03, 1.10981E+03, 1.11063E+03,
     X 1.10714E+03, 1.10500E+03, 1.10357E+03, 1.10093E+03, 1.09898E+03,
     X 1.09679E+03, 1.09188E+03, 1.09088E+03, 1.09040E+03, 1.08586E+03,
     X 1.08178E+03, 1.07752E+03, 1.07243E+03, 1.07178E+03, 1.07084E+03,
     X 1.06693E+03, 1.06527E+03, 1.06405E+03, 1.06285E+03, 1.06287E+03,
     X 1.06276E+03, 1.06221E+03, 1.06464E+03, 1.06579E+03, 1.06498E+03,
     X 1.06596E+03, 1.06812E+03, 1.07159E+03, 1.07361E+03, 1.07556E+03,
     X 1.07751E+03, 1.08128E+03, 1.08523E+03, 1.08927E+03, 1.09193E+03,
     X 1.09612E+03, 1.10133E+03, 1.10435E+03, 1.10781E+03, 1.11168E+03,
     X 1.11641E+03, 1.12217E+03, 1.12839E+03, 1.13298E+03, 1.13575E+03/
	DATA C01841 /
     X 1.13742E+03, 1.13929E+03, 1.14132E+03, 1.14340E+03, 1.14518E+03,
     X 1.14742E+03, 1.14943E+03, 1.14935E+03, 1.14975E+03, 1.15086E+03,
     X 1.15420E+03, 1.15267E+03, 1.15007E+03, 1.15155E+03, 1.14982E+03,
     X 1.14663E+03, 1.14301E+03, 1.13986E+03, 1.13676E+03, 1.13307E+03,
     X 1.12898E+03, 1.12516E+03, 1.12284E+03, 1.12068E+03, 1.11855E+03,
     X 1.11632E+03, 1.11464E+03, 1.11318E+03, 1.11180E+03, 1.11163E+03,
     X 1.11160E+03, 1.11035E+03, 1.11178E+03, 1.11395E+03, 1.11447E+03,
     X 1.11439E+03, 1.11440E+03, 1.11582E+03, 1.11560E+03, 1.11478E+03,
     X 1.11448E+03, 1.11454E+03, 1.11494E+03, 1.11607E+03, 1.11736E+03,
     X 1.11854E+03, 1.11875E+03, 1.11989E+03, 1.12165E+03, 1.12427E+03,
     X 1.12620E+03, 1.12758E+03, 1.12774E+03, 1.12870E+03, 1.13001E+03,
     X 1.13006E+03, 1.13078E+03, 1.13172E+03, 1.12971E+03, 1.12857E+03,
     X 1.12810E+03, 1.12740E+03, 1.12659E+03, 1.12564E+03, 1.12338E+03,
     X 1.12117E+03, 1.11902E+03, 1.11878E+03, 1.11855E+03, 1.11828E+03,
     X 1.11791E+03, 1.11784E+03, 1.11815E+03, 1.11957E+03, 1.12046E+03,
     X 1.12042E+03, 1.11929E+03, 1.12074E+03, 1.12708E+03, 1.12600E+03/
	DATA C01921 /
     X 1.12538E+03, 1.12871E+03, 1.13167E+03, 1.13388E+03, 1.13444E+03,
     X 1.13595E+03, 1.13801E+03, 1.14096E+03, 1.14230E+03, 1.14304E+03,
     X 1.14421E+03, 1.14580E+03, 1.14767E+03, 1.15000E+03, 1.15126E+03,
     X 1.15181E+03, 1.15197E+03, 1.15364E+03, 1.15626E+03, 1.15538E+03,
     X 1.15636E+03, 1.15908E+03, 1.16024E+03, 1.16188E+03, 1.16411E+03,
     X 1.16310E+03, 1.16430E+03, 1.16927E+03, 1.17035E+03, 1.17052E+03,
     X 1.17013E+03, 1.16968E+03, 1.16969E+03, 1.17106E+03, 1.17123E+03,
     X 1.17006E+03, 1.16536E+03, 1.16087E+03, 1.15691E+03, 1.15608E+03,
     X 1.15388E+03, 1.15077E+03, 1.14967E+03, 1.14793E+03, 1.14554E+03,
     X 1.14212E+03, 1.13908E+03, 1.13654E+03, 1.13499E+03, 1.13308E+03,
     X 1.13033E+03, 1.13051E+03, 1.13073E+03, 1.12898E+03, 1.12941E+03,
     X 1.13051E+03, 1.13086E+03, 1.13189E+03, 1.13304E+03, 1.13192E+03,
     X 1.13131E+03, 1.13110E+03, 1.13499E+03, 1.13914E+03, 1.14359E+03,
     X 1.14383E+03, 1.14390E+03, 1.14435E+03, 1.14540E+03, 1.14646E+03,
     X 1.14716E+03, 1.14880E+03, 1.15062E+03, 1.15170E+03, 1.15093E+03,
     X 1.14926E+03, 1.15133E+03, 1.15167E+03, 1.15043E+03, 1.15134E+03/
	DATA C02001 /
     X 1.15135E+03, 1.15000E+03, 1.15087E+03, 1.15118E+03, 1.14935E+03,
     X 1.14780E+03, 1.14647E+03, 1.14560E+03, 1.14404E+03, 1.14238E+03,
     X 1.14406E+03, 1.14245E+03, 1.13781E+03, 1.13664E+03, 1.13653E+03,
     X 1.13778E+03, 1.13813E+03, 1.13794E+03, 1.13681E+03, 1.13515E+03,
     X 1.13328E+03, 1.13132E+03, 1.13080E+03, 1.13130E+03, 1.13400E+03,
     X 1.13526E+03, 1.13494E+03, 1.13193E+03, 1.12898E+03, 1.12654E+03,
     X 1.12739E+03, 1.12849E+03, 1.12774E+03, 1.12733E+03, 1.12733E+03,
     X 1.12943E+03, 1.13014E+03, 1.12967E+03, 1.12731E+03, 1.12671E+03,
     X 1.12885E+03, 1.13050E+03, 1.13201E+03, 1.13345E+03, 1.13488E+03,
     X 1.13605E+03, 1.13530E+03, 1.13737E+03, 1.14186E+03, 1.14250E+03,
     X 1.14305E+03, 1.14383E+03, 1.14510E+03, 1.14659E+03, 1.14848E+03,
     X 1.14949E+03, 1.14995E+03, 1.14934E+03, 1.15058E+03, 1.15368E+03,
     X 1.15435E+03, 1.15422E+03, 1.15296E+03, 1.15228E+03, 1.15189E+03,
     X 1.15198E+03, 1.15081E+03, 1.14881E+03, 1.14562E+03, 1.14276E+03,
     X 1.14030E+03, 1.13637E+03, 1.13254E+03, 1.12942E+03, 1.12653E+03,
     X 1.12362E+03, 1.11987E+03, 1.11712E+03, 1.11522E+03, 1.11403E+03/
	DATA C02081 /
     X 1.11226E+03, 1.10947E+03, 1.10956E+03, 1.10976E+03, 1.10748E+03,
     X 1.10673E+03, 1.10688E+03, 1.10675E+03, 1.10533E+03, 1.10230E+03,
     X 1.10384E+03, 1.10496E+03, 1.10274E+03, 1.10197E+03, 1.10196E+03,
     X 1.10278E+03, 1.10257E+03, 1.10147E+03, 1.10205E+03, 1.10308E+03,
     X 1.10478E+03, 1.10358E+03, 1.10197E+03, 1.10305E+03, 1.10390E+03,
     X 1.10456E+03, 1.10526E+03, 1.10588E+03, 1.10640E+03, 1.10747E+03,
     X 1.10904E+03, 1.11214E+03, 1.11350E+03, 1.11359E+03, 1.11604E+03,
     X 1.11706E+03, 1.11594E+03, 1.11600E+03, 1.11616E+03, 1.11561E+03,
     X 1.11556E+03, 1.11547E+03, 1.11370E+03, 1.11289E+03, 1.11276E+03,
     X 1.11338E+03, 1.11437E+03, 1.11595E+03, 1.11309E+03, 1.10958E+03,
     X 1.10887E+03, 1.10573E+03, 1.10068E+03, 1.10194E+03, 1.10165E+03,
     X 1.09813E+03, 1.09973E+03, 1.10233E+03, 1.10121E+03, 1.10097E+03,
     X 1.10149E+03, 1.10162E+03, 1.10222E+03, 1.10389E+03, 1.10315E+03,
     X 1.10158E+03, 1.10193E+03, 1.10186E+03, 1.10135E+03, 1.10336E+03,
     X 1.10500E+03, 1.10459E+03, 1.10592E+03, 1.10784E+03, 1.10076E+03,
     X 1.09615E+03, 1.09496E+03, 1.09422E+03, 1.09350E+03, 1.09244E+03/
	DATA C02161 /
     X 1.08955E+03, 1.08535E+03, 1.08379E+03, 1.08184E+03, 1.07889E+03,
     X 1.07563E+03, 1.07238E+03, 1.07042E+03, 1.06882E+03, 1.06761E+03,
     X 1.06816E+03, 1.06772E+03, 1.06327E+03, 1.06313E+03, 1.06563E+03,
     X 1.06254E+03, 1.06072E+03, 1.06095E+03, 1.06173E+03, 1.06269E+03,
     X 1.06361E+03, 1.06438E+03, 1.06501E+03, 1.06465E+03, 1.06481E+03,
     X 1.06685E+03, 1.06642E+03, 1.06447E+03, 1.06701E+03, 1.06791E+03,
     X 1.06612E+03, 1.06471E+03, 1.06403E+03, 1.06774E+03, 1.06823E+03,
     X 1.06524E+03, 1.06479E+03, 1.06453E+03, 1.06346E+03, 1.06175E+03,
     X 1.05958E+03, 1.05941E+03, 1.05936E+03, 1.05938E+03, 1.05736E+03,
     X 1.05449E+03, 1.05307E+03, 1.05180E+03, 1.05074E+03, 1.04810E+03,
     X 1.04536E+03, 1.04477E+03, 1.04389E+03, 1.04272E+03, 1.04006E+03,
     X 1.03739E+03, 1.03533E+03, 1.03476E+03, 1.03516E+03, 1.03275E+03,
     X 1.03093E+03, 1.03062E+03, 1.02997E+03, 1.02919E+03, 1.02993E+03,
     X 1.02983E+03, 1.02837E+03, 1.02611E+03, 1.02386E+03, 1.02426E+03,
     X 1.02542E+03, 1.02750E+03, 1.02638E+03, 1.02496E+03, 1.02608E+03,
     X 1.02568E+03, 1.02388E+03, 1.02522E+03, 1.02692E+03, 1.02834E+03/
	DATA C02241 /
     X 1.02828E+03, 1.02716E+03, 1.02667E+03, 1.02607E+03, 1.02503E+03,
     X 1.02723E+03, 1.03143E+03, 1.02881E+03, 1.02646E+03, 1.02500E+03,
     X 1.02569E+03, 1.02743E+03, 1.02608E+03, 1.02548E+03, 1.02620E+03,
     X 1.02733E+03, 1.02839E+03, 1.02575E+03, 1.02432E+03, 1.02471E+03,
     X 1.02392E+03, 1.02267E+03, 1.02077E+03, 1.01964E+03, 1.01957E+03,
     X 1.01848E+03, 1.01704E+03, 1.01524E+03, 1.01352E+03, 1.01191E+03,
     X 1.01066E+03, 1.00952E+03, 1.00849E+03, 1.00660E+03, 1.00368E+03,
     X 9.99713E+02, 9.95921E+02, 9.94845E+02, 9.93286E+02, 9.91204E+02/
C
	END
C----------------------------------------------------------------------
	BLOCK DATA BO3HH1
	IMPLICIT REAL*8(A-H,O-Z)
C
C     RATIO (C1/C0)
C     DATA FROM BASS 1985
C
	COMMON /O3HH1/  V1C,V2C,DVC,
     X           C10001(80),C10081(80),C10161(80),C10241(80),C10321(80),
     X           C10401(80),C10481(80),C10561(80),C10641(80),C10721(80),
     X           C10801(80),C10881(80),C10961(80),C11041(80),C11121(80),
     X           C11201(80),C11281(80),C11361(80),C11441(80),C11521(80),
     X           C11601(80),C11681(80),C11761(80),C11841(80),C11921(80),
     X           C12001(80),C12081(80),C12161(80),C12241(40),
     +           NC
C
	DATA V1C /29405./, V2C /40800./ ,DVC /5./, NC /2280/
C
	DATA C10001 /
     X 6.45621E-03, 7.11308E-03, 1.06130E-02, 1.36338E-02, 1.27746E-02,
     X 1.42152E-02, 1.41144E-02, 1.64830E-02, 1.67110E-02, 1.57368E-02,
     X 1.54644E-02, 1.45248E-02, 1.43206E-02, 1.56946E-02, 1.54268E-02,
     X 1.37500E-02, 1.50224E-02, 1.60919E-02, 1.49099E-02, 1.53960E-02,
     X 1.61871E-02, 1.46539E-02, 1.38258E-02, 1.32571E-02, 1.21580E-02,
     X 1.39596E-02, 1.16029E-02, 1.47042E-02, 1.07441E-02, 1.08999E-02,
     X 1.05562E-02, 1.00589E-02, 9.60711E-03, 9.36950E-03, 7.65303E-03,
     X 6.86216E-03, 7.05344E-03, 6.90728E-03, 6.78627E-03, 6.97435E-03,
     X 5.75456E-03, 5.81685E-03, 5.00915E-03, 4.90259E-03, 4.42545E-03,
     X 4.14633E-03, 3.61657E-03, 3.08178E-03, 2.91680E-03, 2.94554E-03,
     X 3.35794E-03, 5.49025E-03, 7.09867E-03, 6.82592E-03, 8.84835E-03,
     X 9.15718E-03, 9.17935E-03, 8.31848E-03, 7.79481E-03, 7.75125E-03,
     X 6.95844E-03, 7.34506E-03, 7.53823E-03, 7.03272E-03, 7.57051E-03,
     X 9.20239E-03, 1.10864E-02, 1.16188E-02, 1.30029E-02, 1.44364E-02,
     X 1.29292E-02, 1.36031E-02, 1.35967E-02, 1.30412E-02, 1.29874E-02,
     X 1.14829E-02, 1.18009E-02, 1.20829E-02, 1.17831E-02, 1.21489E-02/
	DATA C10081 /
     X 1.27019E-02, 1.25557E-02, 1.23812E-02, 1.20158E-02, 1.26749E-02,
     X 1.17139E-02, 1.14552E-02, 1.11268E-02, 9.79143E-03, 8.79741E-03,
     X 8.85709E-03, 8.57653E-03, 8.93908E-03, 8.46205E-03, 8.56506E-03,
     X 8.14319E-03, 8.14415E-03, 7.74205E-03, 7.80727E-03, 7.49886E-03,
     X 7.71114E-03, 6.55963E-03, 6.87550E-03, 6.39162E-03, 5.55359E-03,
     X 5.43275E-03, 4.90649E-03, 4.41165E-03, 4.21875E-03, 3.62592E-03,
     X 3.40700E-03, 2.40267E-03, 2.61479E-03, 2.75677E-03, 4.10842E-03,
     X 5.79601E-03, 7.10708E-03, 8.07826E-03, 8.16166E-03, 8.72620E-03,
     X 8.85878E-03, 8.72755E-03, 8.25811E-03, 8.12100E-03, 7.78534E-03,
     X 7.39762E-03, 8.43880E-03, 8.53789E-03, 9.90072E-03, 1.01668E-02,
     X 1.00827E-02, 9.73556E-03, 9.57462E-03, 1.01289E-02, 1.10670E-02,
     X 1.03508E-02, 1.00929E-02, 9.10236E-03, 9.39459E-03, 8.79601E-03,
     X 8.67936E-03, 8.53862E-03, 7.95459E-03, 8.04037E-03, 7.95361E-03,
     X 7.87432E-03, 6.99165E-03, 7.37107E-03, 6.09187E-03, 6.21030E-03,
     X 5.33277E-03, 5.04633E-03, 4.45811E-03, 4.34153E-03, 3.98596E-03,
     X 3.84225E-03, 3.41943E-03, 3.60535E-03, 2.81691E-03, 2.49771E-03/
	DATA C10161 /
     X 2.35046E-03, 2.50947E-03, 3.75462E-03, 4.92349E-03, 5.09294E-03,
     X 4.98312E-03, 5.19325E-03, 4.41827E-03, 4.25192E-03, 4.46745E-03,
     X 4.08731E-03, 3.84776E-03, 3.67507E-03, 3.76845E-03, 3.69210E-03,
     X 4.59864E-03, 6.42677E-03, 7.83255E-03, 7.89247E-03, 8.10883E-03,
     X 8.00825E-03, 8.40322E-03, 7.97108E-03, 8.24714E-03, 8.39006E-03,
     X 8.68787E-03, 8.61108E-03, 8.81552E-03, 9.36996E-03, 9.08243E-03,
     X 9.69116E-03, 9.66185E-03, 9.22856E-03, 9.65086E-03, 9.35398E-03,
     X 9.06358E-03, 8.76851E-03, 8.43072E-03, 7.85659E-03, 7.93936E-03,
     X 7.49712E-03, 7.20199E-03, 6.94581E-03, 6.64086E-03, 6.12627E-03,
     X 6.13967E-03, 5.67310E-03, 5.09928E-03, 4.59112E-03, 3.95257E-03,
     X 3.67652E-03, 3.28781E-03, 2.77471E-03, 2.74494E-03, 2.15529E-03,
     X 1.95283E-03, 1.75043E-03, 1.60419E-03, 1.82688E-03, 2.34667E-03,
     X 2.92502E-03, 3.88322E-03, 4.39984E-03, 4.67814E-03, 4.80395E-03,
     X 4.69130E-03, 4.54564E-03, 4.46773E-03, 4.59178E-03, 4.37498E-03,
     X 4.12706E-03, 4.18299E-03, 4.57267E-03, 5.60127E-03, 6.51936E-03,
     X 7.10498E-03, 7.49870E-03, 7.89554E-03, 7.97428E-03, 8.21044E-03/
	DATA C10241 /
     X 8.06324E-03, 7.76648E-03, 7.62238E-03, 7.77675E-03, 7.46905E-03,
     X 7.61115E-03, 7.42715E-03, 7.28461E-03, 7.51514E-03, 7.38782E-03,
     X 6.97206E-03, 6.52738E-03, 6.10147E-03, 5.87553E-03, 5.49218E-03,
     X 4.94873E-03, 4.47920E-03, 4.25005E-03, 3.98094E-03, 3.92084E-03,
     X 3.41707E-03, 3.30501E-03, 3.09208E-03, 3.19686E-03, 3.55283E-03,
     X 4.20775E-03, 4.11155E-03, 3.72193E-03, 3.52000E-03, 3.13572E-03,
     X 2.87629E-03, 2.64251E-03, 2.33451E-03, 2.22426E-03, 2.05800E-03,
     X 1.75214E-03, 2.32530E-03, 2.68651E-03, 3.66315E-03, 4.93904E-03,
     X 5.32850E-03, 5.43978E-03, 5.32656E-03, 5.15649E-03, 5.42096E-03,
     X 5.37193E-03, 5.23454E-03, 5.34557E-03, 5.50533E-03, 6.13216E-03,
     X 6.65129E-03, 7.09357E-03, 7.46042E-03, 7.68605E-03, 7.91866E-03,
     X 7.52953E-03, 7.48272E-03, 7.17800E-03, 6.80060E-03, 6.60427E-03,
     X 6.43049E-03, 6.45975E-03, 6.20534E-03, 5.93094E-03, 5.67360E-03,
     X 5.38584E-03, 5.19364E-03, 4.92599E-03, 4.60655E-03, 4.24669E-03,
     X 3.94253E-03, 3.55894E-03, 3.24256E-03, 2.92974E-03, 2.62760E-03,
     X 2.52238E-03, 2.24714E-03, 2.26350E-03, 2.44380E-03, 3.03798E-03/
	DATA C10321 /
     X 3.50000E-03, 3.55416E-03, 3.43661E-03, 3.19814E-03, 3.02155E-03,
     X 2.73890E-03, 2.50078E-03, 2.34595E-03, 2.18282E-03, 2.19285E-03,
     X 2.49482E-03, 3.13434E-03, 4.18947E-03, 4.72069E-03, 5.29712E-03,
     X 5.39004E-03, 5.44846E-03, 5.37952E-03, 5.09935E-03, 5.08741E-03,
     X 5.05257E-03, 5.10339E-03, 5.17968E-03, 5.31841E-03, 5.58106E-03,
     X 5.65031E-03, 5.65680E-03, 5.76184E-03, 5.71213E-03, 5.48515E-03,
     X 5.32168E-03, 5.18505E-03, 4.99640E-03, 4.78746E-03, 4.57244E-03,
     X 4.32728E-03, 4.14464E-03, 3.97659E-03, 4.01874E-03, 4.10588E-03,
     X 3.99644E-03, 3.84584E-03, 3.64222E-03, 3.39590E-03, 3.00386E-03,
     X 2.73790E-03, 2.45095E-03, 2.29068E-03, 1.64530E-03, 1.68602E-03,
     X 2.32934E-03, 3.14851E-03, 3.65706E-03, 3.70878E-03, 3.75103E-03,
     X 3.79183E-03, 3.32032E-03, 2.42604E-03, 2.48775E-03, 2.34603E-03,
     X 2.36349E-03, 3.33744E-03, 3.44617E-03, 4.27280E-03, 4.61076E-03,
     X 5.20165E-03, 5.14851E-03, 5.22909E-03, 5.08278E-03, 5.16125E-03,
     X 5.01572E-03, 4.51685E-03, 4.67541E-03, 4.83421E-03, 4.57546E-03,
     X 4.55111E-03, 5.03093E-03, 4.67838E-03, 4.44282E-03, 4.40774E-03/
	DATA C10401 /
     X 4.48123E-03, 4.24410E-03, 4.03559E-03, 3.73969E-03, 3.45458E-03,
     X 3.18217E-03, 3.16115E-03, 3.36877E-03, 3.62026E-03, 3.69898E-03,
     X 3.49845E-03, 3.13839E-03, 2.77731E-03, 2.40106E-03, 2.03935E-03,
     X 1.84377E-03, 2.07757E-03, 2.39550E-03, 2.86272E-03, 3.27900E-03,
     X 3.42304E-03, 3.50211E-03, 3.29197E-03, 3.24784E-03, 3.20864E-03,
     X 3.28063E-03, 3.01328E-03, 3.00379E-03, 3.19562E-03, 3.45113E-03,
     X 3.75149E-03, 3.98520E-03, 4.19181E-03, 4.15773E-03, 4.02490E-03,
     X 3.95936E-03, 3.79001E-03, 3.77647E-03, 3.48528E-03, 3.55768E-03,
     X 3.62812E-03, 3.48650E-03, 3.35434E-03, 3.20088E-03, 3.25316E-03,
     X 3.04467E-03, 3.12633E-03, 3.23602E-03, 3.07723E-03, 2.80070E-03,
     X 2.72498E-03, 2.74752E-03, 2.58943E-03, 2.32482E-03, 2.20218E-03,
     X 2.10846E-03, 2.05991E-03, 2.01844E-03, 2.16224E-03, 2.48456E-03,
     X 2.88022E-03, 2.93939E-03, 3.01176E-03, 2.98886E-03, 2.96947E-03,
     X 3.38082E-03, 3.61657E-03, 3.42654E-03, 3.41274E-03, 3.22475E-03,
     X 2.97658E-03, 3.21944E-03, 3.32032E-03, 3.33273E-03, 3.58854E-03,
     X 3.67023E-03, 3.64069E-03, 3.74557E-03, 3.77703E-03, 3.64042E-03/
	DATA C10481 /
     X 3.39468E-03, 3.22657E-03, 3.16466E-03, 3.24224E-03, 3.24801E-03,
     X 3.19487E-03, 3.40155E-03, 3.16940E-03, 2.92293E-03, 3.00998E-03,
     X 2.82851E-03, 2.60381E-03, 2.59242E-03, 2.48530E-03, 2.76677E-03,
     X 2.45506E-03, 2.21845E-03, 2.30407E-03, 2.28136E-03, 2.37278E-03,
     X 2.25313E-03, 2.47836E-03, 2.77858E-03, 2.89803E-03, 2.86131E-03,
     X 3.14118E-03, 3.14119E-03, 2.88881E-03, 3.19502E-03, 2.99538E-03,
     X 2.91212E-03, 3.22739E-03, 3.05960E-03, 3.18901E-03, 3.05805E-03,
     X 3.12205E-03, 2.95636E-03, 3.24111E-03, 3.29433E-03, 3.09206E-03,
     X 3.06696E-03, 2.97735E-03, 2.90897E-03, 2.88979E-03, 2.75105E-03,
     X 2.92156E-03, 3.03445E-03, 2.91664E-03, 2.85559E-03, 2.98405E-03,
     X 2.95376E-03, 2.80234E-03, 2.78349E-03, 2.73421E-03, 2.70035E-03,
     X 2.60074E-03, 2.34840E-03, 2.37626E-03, 2.32927E-03, 2.20842E-03,
     X 2.31080E-03, 2.42771E-03, 2.43339E-03, 2.53280E-03, 2.37093E-03,
     X 2.37377E-03, 2.73453E-03, 2.60836E-03, 2.55568E-03, 2.44062E-03,
     X 2.71093E-03, 2.64421E-03, 2.66969E-03, 2.55560E-03, 2.71800E-03,
     X 2.79534E-03, 2.59070E-03, 2.55373E-03, 2.45272E-03, 2.55571E-03/
	DATA C10561 /
     X 2.54606E-03, 2.57349E-03, 2.46807E-03, 2.35634E-03, 2.44470E-03,
     X 2.47050E-03, 2.57131E-03, 2.71649E-03, 2.58800E-03, 2.54524E-03,
     X 2.69505E-03, 2.89122E-03, 2.77399E-03, 2.63306E-03, 2.82269E-03,
     X 2.95684E-03, 3.07415E-03, 2.70594E-03, 2.65650E-03, 2.90613E-03,
     X 2.96666E-03, 2.94767E-03, 2.81765E-03, 2.64829E-03, 2.43062E-03,
     X 2.33816E-03, 2.38210E-03, 2.45701E-03, 2.38508E-03, 2.40746E-03,
     X 2.49779E-03, 2.28209E-03, 2.26185E-03, 2.26604E-03, 2.19232E-03,
     X 2.19160E-03, 2.32246E-03, 2.11108E-03, 2.26220E-03, 2.26849E-03,
     X 2.34787E-03, 2.49323E-03, 2.46872E-03, 2.52974E-03, 2.35858E-03,
     X 2.36865E-03, 2.33533E-03, 2.21338E-03, 2.24610E-03, 2.24776E-03,
     X 2.24423E-03, 2.29276E-03, 2.18487E-03, 2.27621E-03, 2.31141E-03,
     X 2.44095E-03, 2.45198E-03, 2.56919E-03, 2.56823E-03, 2.41982E-03,
     X 2.39968E-03, 2.62447E-03, 2.55339E-03, 2.51556E-03, 2.47477E-03,
     X 2.50276E-03, 2.48381E-03, 2.48484E-03, 2.48316E-03, 2.38541E-03,
     X 2.41183E-03, 2.55888E-03, 2.42810E-03, 2.43356E-03, 2.25996E-03,
     X 2.34736E-03, 2.10305E-03, 2.13870E-03, 2.17472E-03, 2.05354E-03/
	DATA C10641 /
     X 2.11572E-03, 2.19557E-03, 2.09545E-03, 2.07831E-03, 1.94425E-03,
     X 1.89333E-03, 1.98025E-03, 1.98328E-03, 2.01702E-03, 1.98333E-03,
     X 2.01150E-03, 2.02484E-03, 2.10759E-03, 2.11892E-03, 2.10175E-03,
     X 2.05314E-03, 2.13338E-03, 2.25764E-03, 2.19055E-03, 2.10818E-03,
     X 2.05100E-03, 2.05685E-03, 2.10843E-03, 2.10228E-03, 2.10646E-03,
     X 2.22640E-03, 2.31253E-03, 2.31230E-03, 2.21885E-03, 2.19568E-03,
     X 2.23583E-03, 2.34754E-03, 2.28622E-03, 2.21876E-03, 2.26679E-03,
     X 2.30828E-03, 2.24944E-03, 2.13851E-03, 2.02938E-03, 1.96770E-03,
     X 2.05953E-03, 2.13814E-03, 2.03158E-03, 2.24655E-03, 1.95119E-03,
     X 2.12979E-03, 2.08581E-03, 2.02434E-03, 1.98926E-03, 1.98792E-03,
     X 1.97237E-03, 1.93397E-03, 1.92360E-03, 1.90805E-03, 1.89300E-03,
     X 1.83548E-03, 1.87215E-03, 1.85589E-03, 1.85718E-03, 1.79361E-03,
     X 1.77984E-03, 1.91506E-03, 2.04256E-03, 2.04095E-03, 1.94031E-03,
     X 1.90447E-03, 2.02049E-03, 1.98360E-03, 2.04364E-03, 2.02519E-03,
     X 2.20802E-03, 1.96964E-03, 1.94559E-03, 2.09922E-03, 2.11184E-03,
     X 2.05706E-03, 2.02257E-03, 2.01781E-03, 2.01055E-03, 1.86538E-03/
	DATA C10721 /
     X 1.86899E-03, 1.76798E-03, 1.85871E-03, 1.95363E-03, 1.96404E-03,
     X 1.84169E-03, 1.82851E-03, 1.84582E-03, 1.81997E-03, 1.76461E-03,
     X 1.68384E-03, 1.65530E-03, 1.73550E-03, 1.62463E-03, 1.68793E-03,
     X 1.60472E-03, 1.67560E-03, 1.67431E-03, 1.61779E-03, 1.66446E-03,
     X 1.66403E-03, 1.55724E-03, 1.62351E-03, 1.71545E-03, 1.69645E-03,
     X 1.59540E-03, 1.62948E-03, 1.66784E-03, 1.66416E-03, 1.66131E-03,
     X 1.71502E-03, 1.76555E-03, 1.75182E-03, 1.72327E-03, 1.72338E-03,
     X 1.69993E-03, 1.78819E-03, 1.73517E-03, 1.74802E-03, 1.81751E-03,
     X 1.70973E-03, 1.65075E-03, 1.70784E-03, 1.73655E-03, 1.71670E-03,
     X 1.67367E-03, 1.69338E-03, 1.61772E-03, 1.54914E-03, 1.56009E-03,
     X 1.59467E-03, 1.60761E-03, 1.57117E-03, 1.54045E-03, 1.53102E-03,
     X 1.44516E-03, 1.49898E-03, 1.56048E-03, 1.60087E-03, 1.62636E-03,
     X 1.62472E-03, 1.53931E-03, 1.55536E-03, 1.61649E-03, 1.66493E-03,
     X 1.86915E-03, 1.59984E-03, 1.60483E-03, 1.66549E-03, 1.73449E-03,
     X 1.73673E-03, 1.68393E-03, 1.67434E-03, 1.77880E-03, 1.76154E-03,
     X 1.43028E-03, 1.69651E-03, 1.60934E-03, 1.69413E-03, 1.70514E-03/
	DATA C10801 /
     X 1.62471E-03, 1.74854E-03, 1.76480E-03, 1.63495E-03, 1.59364E-03,
     X 1.39603E-03, 1.47897E-03, 1.49509E-03, 1.70002E-03, 1.63048E-03,
     X 1.44807E-03, 1.45071E-03, 1.53998E-03, 1.45276E-03, 1.29129E-03,
     X 1.52900E-03, 1.64444E-03, 1.37450E-03, 1.42574E-03, 1.47355E-03,
     X 1.51202E-03, 1.54376E-03, 1.51421E-03, 1.43989E-03, 1.45732E-03,
     X 1.42912E-03, 1.59906E-03, 1.56748E-03, 1.52383E-03, 1.47665E-03,
     X 1.51465E-03, 1.55582E-03, 1.54521E-03, 1.55189E-03, 1.56772E-03,
     X 1.45401E-03, 1.55775E-03, 1.43120E-03, 1.39659E-03, 1.41451E-03,
     X 1.45157E-03, 1.48303E-03, 1.42540E-03, 1.26387E-03, 1.37479E-03,
     X 1.46381E-03, 1.38134E-03, 1.32733E-03, 1.38030E-03, 1.44619E-03,
     X 1.41344E-03, 1.31982E-03, 1.24944E-03, 1.20096E-03, 1.21107E-03,
     X 1.27999E-03, 1.22523E-03, 1.22193E-03, 1.35957E-03, 1.41427E-03,
     X 1.35679E-03, 1.15438E-03, 1.41184E-03, 1.49093E-03, 1.32193E-03,
     X 1.25009E-03, 1.37625E-03, 1.49022E-03, 1.44180E-03, 1.27628E-03,
     X 1.29670E-03, 1.31636E-03, 1.28874E-03, 1.31177E-03, 1.35732E-03,
     X 1.33854E-03, 1.30253E-03, 1.31374E-03, 1.27379E-03, 1.18339E-03/
	DATA C10881 /
     X 1.22016E-03, 1.26551E-03, 1.26371E-03, 1.28180E-03, 1.36024E-03,
     X 1.45759E-03, 1.29413E-03, 1.35858E-03, 1.26528E-03, 1.18623E-03,
     X 1.21812E-03, 1.28799E-03, 1.37028E-03, 1.29268E-03, 1.27639E-03,
     X 1.19487E-03, 1.23542E-03, 1.25010E-03, 1.17418E-03, 1.13914E-03,
     X 1.21951E-03, 1.13780E-03, 1.16443E-03, 1.17883E-03, 1.11982E-03,
     X 1.05708E-03, 1.04865E-03, 1.05884E-03, 1.06599E-03, 1.13828E-03,
     X 1.10373E-03, 1.07739E-03, 1.04632E-03, 1.06118E-03, 1.15445E-03,
     X 1.17300E-03, 1.00675E-03, 1.04235E-03, 1.08398E-03, 1.06587E-03,
     X 1.05536E-03, 1.08614E-03, 1.09026E-03, 1.09141E-03, 1.13051E-03,
     X 1.08667E-03, 1.04016E-03, 1.04897E-03, 1.08894E-03, 1.09682E-03,
     X 1.09638E-03, 9.79254E-04, 1.00668E-03, 1.02569E-03, 1.00581E-03,
     X 9.74433E-04, 9.66321E-04, 9.78440E-04, 9.01587E-04, 1.02149E-03,
     X 9.87464E-04, 9.41872E-04, 9.05021E-04, 8.59547E-04, 9.03963E-04,
     X 8.66415E-04, 8.84726E-04, 8.77087E-04, 8.70584E-04, 8.81338E-04,
     X 8.97658E-04, 8.97586E-04, 9.19028E-04, 8.82438E-04, 9.00710E-04,
     X 9.54329E-04, 9.54490E-04, 9.10940E-04, 9.95472E-04, 9.50134E-04/
	DATA C10961 /
     X 9.17127E-04, 9.70916E-04, 9.87575E-04, 9.65026E-04, 9.71779E-04,
     X 1.00967E-03, 1.00053E-03, 9.26063E-04, 9.34721E-04, 9.76354E-04,
     X 9.78436E-04, 9.36012E-04, 9.64448E-04, 9.95903E-04, 9.89960E-04,
     X 9.41143E-04, 9.04393E-04, 8.84719E-04, 8.41396E-04, 8.67234E-04,
     X 8.55864E-04, 8.63314E-04, 8.72317E-04, 8.40899E-04, 7.79593E-04,
     X 7.88481E-04, 8.21075E-04, 7.38342E-04, 7.56537E-04, 7.57278E-04,
     X 7.35854E-04, 7.32765E-04, 6.67398E-04, 7.45338E-04, 7.33094E-04,
     X 7.01840E-04, 6.85595E-04, 6.95740E-04, 7.24015E-04, 7.00907E-04,
     X 7.28498E-04, 6.89410E-04, 6.91728E-04, 7.40601E-04, 7.62775E-04,
     X 7.40912E-04, 7.35021E-04, 7.07799E-04, 7.54113E-04, 8.44845E-04,
     X 8.53956E-04, 6.42186E-04, 7.40557E-04, 7.54340E-04, 7.55544E-04,
     X 7.88986E-04, 7.97902E-04, 6.98460E-04, 7.74873E-04, 6.81178E-04,
     X 7.15567E-04, 7.56723E-04, 7.98438E-04, 8.83150E-04, 8.45671E-04,
     X 7.40924E-04, 7.35498E-04, 7.77829E-04, 6.93566E-04, 5.10188E-04,
     X 7.52717E-04, 6.94185E-04, 6.71928E-04, 6.73286E-04, 6.89415E-04,
     X 7.22917E-04, 7.89448E-04, 8.53812E-04, 7.45132E-04, 7.68732E-04/
	DATA C11041 /
     X 8.10104E-04, 7.55615E-04, 7.09145E-04, 6.80676E-04, 7.54594E-04,
     X 7.89416E-04, 7.88579E-04, 7.49805E-04, 6.13534E-04, 7.22491E-04,
     X 7.95410E-04, 7.80604E-04, 7.74283E-04, 7.93224E-04, 6.86522E-04,
     X 8.06038E-04, 8.30285E-04, 8.37763E-04, 8.03863E-04, 7.33526E-04,
     X 7.42588E-04, 6.31046E-04, 8.16153E-04, 8.95391E-04, 8.61330E-04,
     X 8.38726E-04, 8.16761E-04, 8.16118E-04, 6.37058E-04, 6.30868E-04,
     X 7.26410E-04, 7.03464E-04, 5.93454E-04, 6.01985E-04, 6.51157E-04,
     X 6.68569E-04, 6.56297E-04, 6.58732E-04, 5.99721E-04, 5.34301E-04,
     X 5.33271E-04, 5.57992E-04, 5.70096E-04, 5.59932E-04, 5.32110E-04,
     X 5.64713E-04, 6.25026E-04, 6.38973E-04, 6.05323E-04, 7.17460E-04,
     X 6.19407E-04, 5.90228E-04, 5.43682E-04, 5.38446E-04, 6.56146E-04,
     X 6.09081E-04, 6.04737E-04, 6.45526E-04, 6.46978E-04, 5.89738E-04,
     X 5.63852E-04, 6.18018E-04, 5.71768E-04, 5.75433E-04, 6.05766E-04,
     X 5.93065E-04, 5.31708E-04, 5.41187E-04, 5.76985E-04, 5.78176E-04,
     X 5.75339E-04, 6.85426E-04, 5.51038E-04, 6.02049E-04, 6.20406E-04,
     X 5.80169E-04, 5.36399E-04, 5.59608E-04, 5.46575E-04, 5.66979E-04/
	DATA C11121 /
     X 5.94982E-04, 6.18469E-04, 6.56281E-04, 8.22124E-04, 7.81716E-04,
     X 7.29616E-04, 7.14460E-04, 7.08969E-04, 6.53794E-04, 7.33138E-04,
     X 8.29513E-04, 8.99395E-04, 9.05526E-04, 7.98257E-04, 7.86935E-04,
     X 6.10797E-04, 4.63912E-04, 4.05675E-04, 3.66230E-04, 4.86472E-04,
     X 5.31818E-04, 5.15865E-04, 4.87344E-04, 4.99857E-04, 5.35479E-04,
     X 5.27561E-04, 4.99000E-04, 4.77056E-04, 4.74242E-04, 4.66595E-04,
     X 4.66325E-04, 4.94704E-04, 5.12842E-04, 5.01795E-04, 4.80789E-04,
     X 5.73709E-04, 5.65214E-04, 5.11321E-04, 4.55242E-04, 4.29330E-04,
     X 5.09792E-04, 4.70489E-04, 4.82859E-04, 4.99195E-04, 4.07724E-04,
     X 4.99951E-04, 4.55755E-04, 4.42528E-04, 4.19433E-04, 3.31325E-04,
     X 3.70517E-04, 3.77708E-04, 2.97923E-04, 2.27470E-04, 2.47389E-04,
     X 2.38324E-04, 2.56706E-04, 2.45046E-04, 2.62539E-04, 3.37054E-04,
     X 3.33930E-04, 3.01390E-04, 3.08028E-04, 3.41464E-04, 3.70574E-04,
     X 3.47893E-04, 3.28433E-04, 3.46976E-04, 3.60351E-04, 3.50559E-04,
     X 3.56070E-04, 3.62782E-04, 3.37330E-04, 3.33763E-04, 3.57046E-04,
     X 3.08784E-04, 2.93898E-04, 2.80842E-04, 2.54114E-04, 2.38198E-04/
	DATA C11201 /
     X 3.48753E-04, 2.97334E-04, 2.82929E-04, 2.94150E-04, 3.07875E-04,
     X 3.21129E-04, 3.38335E-04, 3.49826E-04, 3.47647E-04, 3.35438E-04,
     X 3.58145E-04, 3.72391E-04, 3.59372E-04, 3.64755E-04, 4.16867E-04,
     X 3.43614E-04, 3.34932E-04, 3.12782E-04, 3.28220E-04, 4.32595E-04,
     X 3.49513E-04, 3.51861E-04, 3.81166E-04, 3.91194E-04, 3.38944E-04,
     X 2.63445E-04, 2.49520E-04, 2.46184E-04, 2.33203E-04, 2.16315E-04,
     X 1.89536E-04, 1.95730E-04, 1.99664E-04, 1.77139E-04, 1.27969E-04,
     X 5.17216E-05, 7.60445E-05, 1.24418E-04, 1.30989E-04, 2.31539E-04,
     X 2.21334E-04, 2.08757E-04, 2.18351E-04, 2.46202E-04, 2.29824E-04,
     X 2.28909E-04, 2.88826E-04, 3.58039E-04, 2.60800E-04, 2.33025E-04,
     X 2.52667E-04, 2.61394E-04, 2.31384E-04, 2.29388E-04, 2.54701E-04,
     X 2.21158E-04, 1.61506E-04, 1.36752E-04, 1.69481E-04, 8.64539E-05,
     X 1.64407E-04, 3.65674E-04, 3.18233E-04, 4.00755E-04, 3.33375E-04,
     X 2.62930E-04, 2.87052E-04, 2.51395E-04, 2.85274E-04, 2.66915E-04,
     X 2.10866E-04, 1.89517E-04, 1.67378E-04, 2.79951E-04, 2.97224E-04,
     X 1.89222E-04, 3.33825E-04, 3.56386E-04, 3.89727E-04, 4.30407E-04/
	DATA C11281 /
     X 4.45922E-04, 4.23446E-04, 4.41347E-04, 4.06723E-04, 3.00181E-04,
     X 1.85243E-04, 3.13176E-04, 4.08991E-04, 4.24776E-04, 3.56412E-04,
     X 3.84760E-04, 2.30602E-04, 1.77702E-04, 2.62329E-04, 2.49442E-04,
     X 3.76212E-04, 3.69176E-04, 2.97681E-04, 2.71662E-04, 2.05694E-04,
     X 2.11418E-04, 2.25439E-04, 2.27013E-04, 2.47845E-04, 3.14603E-04,
     X 2.68802E-04, 2.04334E-04, 2.77399E-04, 2.68273E-04, 2.04991E-04,
     X 2.24441E-04, 3.55074E-04, 2.90135E-04, 3.35680E-04, 3.59358E-04,
     X 3.44716E-04, 3.24496E-04, 3.48146E-04, 3.49042E-04, 3.54848E-04,
     X 3.86418E-04, 3.59198E-04, 3.47608E-04, 3.20522E-04, 2.78401E-04,
     X 2.64579E-04, 2.23694E-04, 2.34370E-04, 2.52559E-04, 1.88475E-04,
     X 2.01258E-04, 1.63979E-04, 1.45384E-04, 1.91215E-04, 1.76958E-04,
     X 1.69167E-04, 1.71767E-04, 1.86595E-04, 2.14969E-04, 2.48345E-04,
     X 2.46691E-04, 2.25234E-04, 2.26755E-04, 1.64112E-04, 1.87750E-04,
     X 2.22984E-04, 2.00443E-04, 2.38863E-04, 2.77590E-04, 2.91953E-04,
     X 2.80611E-04, 3.08215E-04, 1.79095E-04, 1.46920E-04, 2.29177E-04,
     X 2.54685E-04, 2.68866E-04, 2.13346E-04, 1.20122E-04, 5.55240E-05/
	DATA C11361 /
     X 5.99017E-05, 1.07768E-04, 1.67810E-04, 2.06886E-04, 2.36232E-04,
     X 2.24598E-04, 2.30792E-04, 2.71274E-04, 1.29062E-04, 1.92624E-04,
     X 2.38438E-04, 1.98994E-04, 1.81687E-04, 2.55733E-04, 2.84379E-04,
     X 2.54459E-04, 2.30884E-04, 2.68873E-04, 3.07231E-04, 3.15063E-04,
     X 2.46725E-04, 2.60370E-04, 2.66391E-04, 2.50708E-04, 2.04296E-04,
     X 1.66011E-04, 1.19164E-04, 1.06700E-04, 1.77576E-04, 1.91741E-04,
     X 1.66618E-04, 1.49824E-04, 1.80699E-04, 2.20905E-04, 1.38754E-04,
     X 6.27971E-05, 7.52567E-05, 1.89995E-04, 1.72489E-04, 1.40424E-04,
     X 1.52384E-04, 1.63942E-04, 1.19901E-04, 1.49234E-04, 2.68313E-04,
     X 2.08815E-04, 1.17218E-04, 1.42235E-04, 2.71237E-04, 1.38192E-04,
     X 2.15643E-04, 2.84476E-04, 2.78117E-04, 2.19234E-04, 1.59128E-04,
     X 1.78819E-04, 2.67785E-04, 2.66786E-04, 2.58545E-04, 2.68476E-04,
     X 2.88542E-04, 2.59726E-04, 3.00936E-04, 3.11237E-04, 2.61275E-04,
     X 1.37136E-04, 2.76566E-04, 3.82888E-04, 3.97564E-04, 4.43655E-04,
     X 3.15415E-04, 2.60869E-04, 3.19171E-04, 3.34205E-04, 2.02914E-04,
     X 1.16223E-04, 1.14737E-04, 6.10978E-05,-8.03695E-06,-1.07062E-05/
	DATA C11441 /
     X 6.50664E-05, 1.12586E-04, 1.56727E-04, 1.57927E-04, 1.05762E-04,
     X 1.03646E-04, 1.72520E-04, 2.23668E-04, 2.12775E-04, 2.33525E-04,
     X 2.75558E-04, 2.34256E-04, 5.10062E-05, 1.76007E-04, 1.70850E-04,
     X 1.43266E-04, 1.89626E-04, 2.97283E-04, 3.02773E-04, 2.74401E-04,
     X 3.00754E-04, 3.66813E-04, 3.54383E-04, 2.90580E-04, 2.32206E-04,
     X 1.58405E-04, 1.54663E-04, 1.84598E-04, 1.26408E-04, 2.14481E-04,
     X 2.00791E-04, 1.05796E-04, 2.39794E-04, 1.66105E-04, 7.88615E-05,
     X 4.30615E-05, 7.37518E-05, 1.24926E-04, 1.38295E-04, 8.54356E-05,
     X 6.12641E-05, 6.54466E-05, 6.17727E-05, 1.30688E-05, 6.00462E-05,
     X 1.52612E-04, 2.11656E-04, 9.67692E-05, 8.67858E-05, 1.34888E-04,
     X 1.90899E-04, 1.03234E-04, 1.03837E-04, 1.49767E-04, 2.19058E-04,
     X 2.26549E-04, 2.11506E-04, 1.85238E-04, 1.53774E-04, 1.32313E-04,
     X 6.10658E-05, 2.37782E-05, 1.24450E-04, 1.87610E-04, 1.44775E-04,
     X 5.60937E-05, 6.64032E-05, 1.28073E-04, 1.77512E-04, 1.84684E-04,
     X 5.73677E-05, 5.29679E-05, 9.95510E-05, 1.61423E-04, 3.19036E-04,
     X 3.17383E-04, 2.36505E-04, 1.80844E-04, 1.63722E-04, 1.21478E-04/
	DATA C11521 /
     X 6.85823E-05, 7.42058E-05, 1.14838E-04, 1.21131E-04, 8.01009E-05,
     X 1.52058E-04, 2.18368E-04, 2.53416E-04, 2.27116E-04, 1.25336E-04,
     X 6.26421E-05, 5.32471E-05, 1.34705E-04, 2.07005E-05,-5.18630E-05,
     X-3.25696E-05,-8.06171E-05,-1.09430E-04,-1.05637E-04,-4.96066E-05,
     X-7.76138E-05,-4.85930E-05, 3.65111E-06,-2.86933E-05,-4.61366E-05,
     X-4.88820E-05,-3.08816E-05, 8.43778E-05, 1.40484E-04, 1.31125E-04,
     X 3.55198E-05, 8.47412E-05, 1.23408E-04, 1.36799E-04, 1.21147E-04,
     X 1.25585E-04, 1.32337E-04, 1.34092E-04, 1.26652E-04, 1.12131E-04,
     X 1.00927E-04, 1.13828E-04, 1.06053E-04, 9.43643E-05, 8.33628E-05,
     X 8.65842E-05, 7.59315E-05, 8.28623E-05, 1.39681E-04, 1.80492E-04,
     X 1.65779E-04, 1.03843E-04, 3.10284E-05, 1.94408E-05, 4.57525E-05,
     X 1.02436E-04, 1.39750E-04, 1.43342E-04, 1.11999E-04, 2.94197E-05,
     X 2.76980E-05, 5.51685E-05, 9.39909E-05, 1.16108E-04, 7.72703E-05,
     X 4.37409E-05, 1.13925E-04, 8.18872E-05, 2.87657E-05,-2.41413E-05,
     X 1.24699E-05, 2.19589E-05,-5.88247E-06,-9.66151E-05,-2.06255E-05,
     X-1.83148E-06,-5.63625E-05,-8.65590E-05,-8.26020E-05,-5.06239E-05/
	DATA C11601 /
     X 1.28065E-05,-1.34669E-05, 1.59701E-05, 9.44755E-05, 1.63032E-05,
     X 2.51304E-05, 7.38226E-05, 1.28405E-04, 1.17413E-04, 9.92387E-05,
     X 9.51533E-05, 2.17008E-04, 2.25854E-04, 1.90448E-04, 1.77207E-04,
     X 1.80844E-04, 1.53501E-04, 9.80430E-05, 1.27404E-04, 1.16465E-04,
     X 9.98611E-05, 1.25556E-04, 1.73627E-04, 1.12347E-04,-7.73523E-05,
     X 5.66599E-05, 5.36347E-05, 1.20227E-06, 6.96325E-05, 4.79010E-05,
     X-1.09886E-05,-9.16457E-05,-7.09170E-05,-5.31410E-05,-2.68376E-05,
     X 6.32641E-05, 8.06052E-06,-4.99262E-05,-2.56644E-05,-8.76854E-05,
     X-8.21360E-05,-5.02403E-06, 4.66629E-05, 6.93127E-05, 5.53828E-05,
     X-2.32399E-05,-2.07514E-05,-7.33240E-05,-2.10483E-04,-1.53757E-04,
     X-7.13861E-05,-1.07356E-05,-1.26578E-04,-7.48854E-05, 3.25418E-06,
     X 2.97068E-05, 3.35685E-05, 3.15022E-05, 2.68904E-05, 3.87401E-05,
     X 5.12522E-05, 5.12172E-05, 1.05053E-05, 1.65321E-05, 3.47537E-05,
     X 5.62503E-05, 4.18666E-05, 3.13970E-05, 3.11750E-05, 7.21547E-05,
     X 2.55262E-05,-2.76061E-05, 5.43449E-06,-5.20575E-05,-1.08627E-04,
     X-1.40475E-04,-1.59926E-04,-1.32237E-04,-8.15458E-05,-1.31738E-04/
	DATA C11681 /
     X-1.64036E-04,-1.69351E-04,-1.24797E-04,-1.61950E-04,-2.01904E-04,
     X-2.22995E-04,-1.87647E-04,-1.70817E-04,-1.64583E-04,-1.12811E-04,
     X-8.38306E-05,-8.62707E-05,-1.54362E-04,-1.98090E-04,-2.12920E-04,
     X-1.89358E-04,-2.02988E-04,-1.72791E-04,-1.02863E-04,-1.09877E-04,
     X-1.04257E-04,-8.20734E-05,-2.18346E-05,-2.94593E-05,-4.18226E-05,
     X-1.86891E-05,-6.14620E-05,-3.21912E-05, 1.00844E-04, 6.92419E-05,
     X 3.16713E-05, 5.62042E-07, 5.18900E-05, 7.48835E-05, 8.03381E-05,
     X 7.24685E-05, 9.55588E-05, 9.22801E-05, 2.87159E-05, 2.26234E-05,
     X 2.62790E-05, 3.58332E-05, 6.23297E-05, 5.01998E-05, 1.81446E-05,
     X 3.33564E-05, 3.97765E-06,-2.60624E-05, 7.01802E-06,-4.16797E-05,
     X-8.70108E-05,-8.22182E-05,-6.64886E-05,-7.88704E-05,-1.28305E-04,
     X-1.29990E-04,-1.12646E-04,-8.68394E-05,-1.29584E-04,-1.44352E-04,
     X-1.42082E-04,-1.33790E-04,-1.27963E-04,-1.21233E-04,-1.09965E-04,
     X-1.02233E-04,-1.03804E-04,-1.19503E-04,-7.74707E-05,-4.66805E-05,
     X-3.52201E-05,-4.07406E-05,-4.66887E-05,-5.05962E-05,-3.30333E-05,
     X-3.47981E-05,-3.60962E-05, 1.44242E-05, 4.10478E-05, 3.68984E-05/
	DATA C11761 /
     X-2.81300E-05, 2.83171E-05, 7.48062E-05, 4.29333E-05, 8.50076E-06,
     X 4.98135E-06, 4.44854E-05, 2.51860E-05, 3.12189E-05, 6.39424E-05,
     X 7.20715E-05, 9.89688E-05, 1.33768E-04, 1.07781E-04, 9.76731E-05,
     X 9.21479E-05, 6.72624E-05, 5.41295E-05, 4.89022E-05, 5.28039E-05,
     X-4.48737E-06,-5.15409E-05,-3.57396E-05,-1.94752E-05,-2.09521E-05,
     X-5.13096E-05,-2.62781E-05,-2.75451E-05,-6.98423E-05,-1.25462E-04,
     X-1.68362E-04,-1.97456E-04,-1.90669E-04,-2.06890E-04,-2.36699E-04,
     X-1.97732E-04,-1.76504E-04,-1.67505E-04,-1.60694E-04,-1.85851E-04,
     X-2.01567E-04,-9.82507E-05,-1.33338E-04,-1.95199E-04,-1.40781E-04,
     X-8.90988E-05,-3.63239E-05, 2.16510E-05,-1.56807E-05,-4.21285E-05,
     X 5.50505E-06, 6.78937E-07, 3.12346E-06, 3.64202E-05, 3.50651E-05,
     X 6.20423E-05, 1.38667E-04, 7.74738E-05, 6.77036E-05, 1.38367E-04,
     X 1.17359E-04, 1.06637E-04, 1.12404E-04, 9.78586E-05, 1.03178E-04,
     X 1.28717E-04, 1.56642E-04, 1.62544E-04, 1.50109E-04, 1.43214E-04,
     X 1.33651E-04, 1.24352E-04, 1.41420E-04, 1.36340E-04, 1.18769E-04,
     X 1.31656E-04, 8.81533E-05, 1.55214E-05,-3.68736E-07,-1.76213E-05/
	DATA C11841 /
     X-2.85341E-05, 4.65155E-06, 5.41350E-06,-7.01247E-06, 6.57918E-06,
     X-2.45784E-05,-6.89104E-05,-6.90953E-05,-6.23937E-05,-6.72978E-05,
     X-1.39547E-04,-1.44228E-04,-1.42543E-04,-2.31080E-04,-2.12756E-04,
     X-1.62089E-04,-1.66063E-04,-1.61872E-04,-1.59764E-04,-1.80217E-04,
     X-1.38355E-04,-8.45661E-05,-7.58308E-05,-4.65144E-05,-2.76855E-05,
     X-7.48714E-05,-8.28561E-05,-6.45277E-05,-7.08509E-06,-1.05566E-05,
     X-1.96352E-05, 3.55561E-05, 2.24676E-05,-1.25648E-05,-1.87661E-05,
     X 6.99061E-06, 2.33676E-05,-5.25111E-05,-3.86758E-05, 1.03585E-06,
     X-1.65901E-05,-1.04855E-05, 5.03694E-06, 1.25937E-05,-8.31340E-06,
     X-4.37906E-05,-7.91444E-05,-4.62167E-05, 5.14238E-06,-4.52863E-05,
     X-5.86455E-05,-4.98093E-05,-3.03495E-05,-5.09377E-05,-8.88116E-05,
     X-6.21360E-05,-7.38148E-05,-1.07502E-04,-7.55276E-05,-6.39257E-05,
     X-6.86921E-05,-8.05504E-05,-9.24178E-05,-1.03991E-04,-1.00468E-04,
     X-6.71447E-05,-3.84897E-06,-5.99067E-06,-2.21894E-05,-5.21766E-05,
     X-3.93796E-05,-4.06712E-05,-6.21649E-05,-1.13073E-04,-1.20560E-04,
     X-5.92397E-05, 5.24432E-05, 9.41628E-05,-3.47458E-07, 5.33267E-05/
	DATA C11921 /
     X 8.92961E-05, 2.75694E-05,-7.48460E-06,-2.15504E-05, 1.05501E-06,
     X 6.30910E-06, 5.94620E-07,-2.45194E-05,-1.59657E-05, 7.93610E-07,
     X-1.05319E-05,-2.36584E-05,-3.95700E-05,-6.57225E-05,-5.23797E-05,
     X-1.82588E-05,-1.43240E-05,-3.29989E-05,-6.48909E-05,-2.41326E-05,
     X-1.89195E-05,-4.64607E-05,-1.00739E-05,-1.35033E-05,-6.49945E-05,
     X-5.19986E-05,-6.68505E-05,-1.31530E-04,-1.45464E-04,-1.46815E-04,
     X-1.39684E-04,-1.23205E-04,-1.26738E-04,-1.93822E-04,-2.37508E-04,
     X-2.52917E-04,-1.91110E-04,-1.36217E-04,-9.41093E-05,-1.20601E-04,
     X-1.17295E-04,-9.57420E-05,-1.57227E-04,-1.62795E-04,-1.12201E-04,
     X-1.20419E-04,-1.10597E-04,-7.61223E-05,-6.27167E-05,-5.54733E-05,
     X-5.50437E-05,-5.14148E-05,-3.59591E-05, 1.09906E-05, 5.94396E-06,
     X-1.38597E-05,-8.80857E-06,-3.13101E-05,-6.31715E-05,-4.04264E-05,
     X-1.66405E-05, 7.94396E-06,-3.41772E-06,-4.03175E-05,-1.06888E-04,
     X-9.50526E-05,-7.46111E-05,-5.09617E-05,-6.70981E-05,-7.93529E-05,
     X-5.58423E-05,-1.01523E-04,-1.62269E-04,-1.69958E-04,-1.37786E-04,
     X-8.79862E-05,-1.46838E-04,-1.66938E-04,-1.51380E-04,-1.62184E-04/
	DATA C12001 /
     X-1.61105E-04,-1.42088E-04,-1.57033E-04,-1.65294E-04,-1.45079E-04,
     X-9.76982E-05,-6.09891E-05,-1.01719E-04,-1.03049E-04,-8.85546E-05,
     X-1.47754E-04,-1.44542E-04,-8.34620E-05,-8.99440E-05,-7.11901E-05,
     X-1.57480E-05,-8.81797E-05,-1.56314E-04,-1.65952E-04,-1.80986E-04,
     X-2.04610E-04,-2.58669E-04,-2.16016E-04,-1.21582E-04,-1.44929E-04,
     X-1.72886E-04,-2.05950E-04,-1.93829E-04,-1.67518E-04,-1.22969E-04,
     X-1.13060E-04,-1.14854E-04,-1.26198E-04,-1.24288E-04,-1.19519E-04,
     X-1.50456E-04,-1.53286E-04,-1.32231E-04,-7.42672E-05,-2.23129E-05,
     X 1.79115E-05, 1.42073E-05,-1.21676E-05,-7.56567E-05,-1.03423E-04,
     X-1.10373E-04,-8.77244E-05,-6.43485E-05,-4.05156E-05,-6.24405E-05,
     X-5.70375E-05,-2.36695E-06,-3.75929E-05,-7.97119E-05,-6.70419E-05,
     X-6.99475E-05,-8.19748E-05,-1.06895E-04,-1.31422E-04,-1.55438E-04,
     X-1.61937E-04,-1.62626E-04,-1.54977E-04,-1.77814E-04,-2.00386E-04,
     X-1.87407E-04,-2.07243E-04,-2.44672E-04,-2.19014E-04,-2.13695E-04,
     X-2.32440E-04,-1.85194E-04,-1.51172E-04,-1.69834E-04,-1.73780E-04,
     X-1.75232E-04,-2.00698E-04,-1.82826E-04,-1.27786E-04,-1.33633E-04/
	DATA C12081 /
     X-1.21317E-04,-7.50390E-05,-1.06743E-04,-1.40805E-04,-1.06336E-04,
     X-9.46654E-05,-9.78182E-05,-1.19906E-04,-1.14160E-04,-7.28186E-05,
     X-1.07652E-04,-1.20978E-04,-3.79658E-05,-3.16113E-05,-6.02417E-05,
     X-7.51148E-05,-5.56145E-05,-6.77421E-06,-1.74321E-05,-4.67952E-05,
     X-1.05000E-04,-6.29932E-05,-4.74356E-06,-2.83397E-05,-4.65192E-05,
     X-6.04574E-05,-4.33970E-05,-3.18311E-05,-3.02321E-05,-4.49667E-05,
     X-6.85347E-05,-1.11375E-04,-1.16293E-04,-9.38757E-05,-1.38594E-04,
     X-1.60483E-04,-1.48344E-04,-1.33436E-04,-1.27387E-04,-1.59508E-04,
     X-1.74026E-04,-1.72170E-04,-1.49196E-04,-1.33233E-04,-1.22382E-04,
     X-1.78156E-04,-2.21349E-04,-2.41846E-04,-2.06549E-04,-1.68283E-04,
     X-1.89512E-04,-1.44523E-04,-4.67953E-05,-1.00334E-04,-1.23478E-04,
     X-8.14024E-05,-9.18016E-05,-1.17536E-04,-1.36160E-04,-1.38780E-04,
     X-1.27749E-04,-1.45598E-04,-1.55964E-04,-1.45120E-04,-1.25544E-04,
     X-1.05692E-04,-1.17639E-04,-1.24142E-04,-1.24749E-04,-1.63878E-04,
     X-1.97021E-04,-1.98617E-04,-2.69136E-04,-3.68357E-04,-2.33702E-04,
     X-1.61830E-04,-1.78578E-04,-2.01839E-04,-2.28731E-04,-2.63606E-04/
	DATA C12161 /
     X-2.44698E-04,-1.86451E-04,-2.20546E-04,-2.22752E-04,-1.55169E-04,
     X-1.25100E-04,-1.09794E-04,-9.59016E-05,-1.03857E-04,-1.35573E-04,
     X-1.73780E-04,-1.82457E-04,-9.39821E-05,-1.18245E-04,-2.11563E-04,
     X-1.37392E-04,-9.28173E-05,-9.71073E-05,-9.72535E-05,-9.39557E-05,
     X-7.50117E-05,-6.70754E-05,-7.01186E-05,-5.76151E-05,-5.18785E-05,
     X-7.14209E-05,-7.01682E-05,-5.61614E-05,-8.92769E-05,-1.06238E-04,
     X-9.70294E-05,-6.70229E-05,-4.69214E-05,-1.53105E-04,-2.02326E-04,
     X-1.90395E-04,-2.04367E-04,-2.16787E-04,-2.08725E-04,-1.78119E-04,
     X-1.31043E-04,-1.32204E-04,-1.51522E-04,-2.05143E-04,-1.77144E-04,
     X-1.16130E-04,-1.44440E-04,-1.66010E-04,-1.78206E-04,-1.61163E-04,
     X-1.46351E-04,-1.96722E-04,-2.27027E-04,-2.37243E-04,-2.25235E-04,
     X-1.99552E-04,-1.40238E-04,-1.26311E-04,-1.42746E-04,-1.19028E-04,
     X-1.18750E-04,-1.72076E-04,-1.72120E-04,-1.48285E-04,-1.85116E-04,
     X-1.98602E-04,-1.74016E-04,-1.37913E-04,-1.01221E-04,-9.69581E-05,
     X-1.08794E-04,-1.39433E-04,-1.38575E-04,-1.32088E-04,-1.37431E-04,
     X-1.30033E-04,-1.10829E-04,-1.35604E-04,-1.66515E-04,-1.98167E-04/
	DATA C12241 /
     X-1.97716E-04,-1.74019E-04,-1.64719E-04,-1.64779E-04,-1.85725E-04,
     X-2.28526E-04,-2.84329E-04,-1.82449E-04,-1.30747E-04,-1.93620E-04,
     X-2.28529E-04,-2.47361E-04,-1.90001E-04,-1.66278E-04,-2.02540E-04,
     X-2.31811E-04,-2.53772E-04,-2.08629E-04,-1.85021E-04,-1.93989E-04,
     X-2.16568E-04,-2.38288E-04,-1.94453E-04,-1.87154E-04,-2.30493E-04,
     X-2.34696E-04,-2.30351E-04,-2.60562E-04,-2.86427E-04,-3.06699E-04,
     X-2.79131E-04,-2.49392E-04,-3.03389E-04,-3.10346E-04,-2.61782E-04,
     X-2.30905E-04,-2.11669E-04,-2.37680E-04,-2.38194E-04,-2.10955E-04/
	END
c----------------------------------------------------------------------
	BLOCK DATA BO3HH2
	IMPLICIT REAL*8(A-H,O-Z)
C
C     RATIO (C2/C0)
C     DATA FROM BASS 1985
C
	COMMON /O3HH2/  V1C,V2C,DVC,
     X           C20001(80),C20081(80),C20161(80),C20241(80),C20321(80),
     X           C20401(80),C20481(80),C20561(80),C20641(80),C20721(80),
     X           C20801(80),C20881(80),C20961(80),C21041(80),C21121(80),
     X           C21201(80),C21281(80),C21361(80),C21441(80),C21521(80),
     X           C21601(80),C21681(80),C21761(80),C21841(80),C21921(80),
     X           C22001(80),C22081(80),C22161(80),C22241(40),
     +           NC
C
	DATA V1C /29405./, V2C /40800./ ,DVC /5./, NC /2280/
C
	DATA C20001 /
     X 2.16295E-05, 1.69111E-05, 5.39633E-05, 1.01866E-04, 8.28657E-05,
     X 9.16593E-05, 8.88666E-05, 1.37764E-04, 1.44322E-04, 1.20659E-04,
     X 1.10332E-04, 1.01317E-04, 9.09964E-05, 1.17148E-04, 1.18000E-04,
     X 7.21801E-05, 1.10550E-04, 1.32672E-04, 1.02474E-04, 1.10434E-04,
     X 1.38759E-04, 8.92135E-05, 9.18239E-05, 9.08256E-05, 7.02969E-05,
     X 1.12827E-04, 8.25561E-05, 1.39555E-04, 6.72239E-05, 7.82804E-05,
     X 8.56258E-05, 8.61068E-05, 7.16732E-05, 6.25720E-05, 5.23957E-05,
     X 3.78801E-05, 4.37281E-05, 4.99821E-05, 5.96976E-05, 7.19070E-05,
     X 3.89579E-05, 5.30171E-05, 3.92507E-05, 4.93901E-05, 4.53047E-05,
     X 4.89955E-05, 4.61649E-05, 3.75742E-05, 3.14124E-05, 2.37893E-05,
     X 3.34899E-06, 3.08080E-05, 5.35883E-05, 3.39838E-05, 7.02334E-05,
     X 7.24784E-05, 7.46533E-05, 6.22257E-05, 6.38945E-05, 6.73423E-05,
     X 4.51321E-05, 5.91854E-05, 5.51601E-05, 4.41923E-05, 3.59217E-05,
     X 4.08520E-05, 6.15981E-05, 6.66549E-05, 8.26031E-05, 1.13556E-04,
     X 8.72988E-05, 9.71052E-05, 9.31839E-05, 8.73745E-05, 8.61717E-05,
     X 6.05645E-05, 6.51131E-05, 6.93393E-05, 7.01096E-05, 6.43565E-05/
	DATA C20081 /
     X 7.36929E-05, 7.66881E-05, 7.60815E-05, 7.13570E-05, 8.40487E-05,
     X 8.51489E-05, 7.54168E-05, 6.72694E-05, 4.75508E-05, 3.59379E-05,
     X 4.24698E-05, 4.17850E-05, 4.56047E-05, 4.12779E-05, 4.55933E-05,
     X 4.27941E-05, 4.42230E-05, 3.68525E-05, 3.83392E-05, 3.83722E-05,
     X 4.64904E-05, 3.33878E-05, 3.53027E-05, 3.54694E-05, 2.36233E-05,
     X 2.99641E-05, 2.56097E-05, 2.14134E-05, 2.74403E-05, 2.83896E-05,
     X 3.17082E-05, 1.75526E-05, 2.80382E-05, 3.18009E-05, 4.08715E-05,
     X 4.77807E-05, 5.00609E-05, 5.12459E-05, 4.44062E-05, 4.74942E-05,
     X 4.99882E-05, 5.18837E-05, 5.03246E-05, 5.55168E-05, 5.35853E-05,
     X 4.81834E-05, 6.66231E-05, 5.26670E-05, 6.84700E-05, 6.53412E-05,
     X 5.71740E-05, 4.61076E-05, 3.90239E-05, 4.72924E-05, 6.32194E-05,
     X 5.20868E-05, 4.81039E-05, 3.71748E-05, 4.37492E-05, 3.63959E-05,
     X 3.79823E-05, 3.72225E-05, 3.02360E-05, 3.22961E-05, 3.43398E-05,
     X 3.57176E-05, 2.65446E-05, 3.29388E-05, 1.65455E-05, 2.66173E-05,
     X 1.74277E-05, 1.74324E-05, 1.27879E-05, 1.46247E-05, 1.92378E-05,
     X 2.20049E-05, 1.44790E-05, 2.49244E-05, 2.29209E-05, 1.76192E-05/
	DATA C20161 /
     X 1.84528E-05, 2.54350E-05, 3.33972E-05, 3.69190E-05, 2.92139E-05,
     X 2.47666E-05, 2.86764E-05, 1.48163E-05, 1.80461E-05, 2.84545E-05,
     X 2.41064E-05, 2.85721E-05, 3.31996E-05, 3.75973E-05, 3.73874E-05,
     X 4.69293E-05, 5.12665E-05, 5.35607E-05, 4.64577E-05, 3.59887E-05,
     X 3.39168E-05, 3.89746E-05, 3.12196E-05, 3.70907E-05, 3.95172E-05,
     X 4.61642E-05, 4.26029E-05, 4.17856E-05, 4.51437E-05, 4.04189E-05,
     X 4.19251E-05, 4.53977E-05, 3.69860E-05, 4.20904E-05, 3.69735E-05,
     X 3.57898E-05, 3.47729E-05, 3.14280E-05, 2.71197E-05, 3.34380E-05,
     X 2.69843E-05, 2.88036E-05, 2.51912E-05, 2.45699E-05, 2.23184E-05,
     X 2.50563E-05, 2.24493E-05, 1.77101E-05, 1.64763E-05, 1.34978E-05,
     X 1.57081E-05, 1.45966E-05, 1.02722E-05, 2.07177E-05, 1.47662E-05,
     X 1.50721E-05, 1.24431E-05, 1.51572E-05, 1.92210E-05, 2.06047E-05,
     X 2.02921E-05, 3.22062E-05, 2.37112E-05, 1.94803E-05, 2.40726E-05,
     X 2.11531E-05, 1.89158E-05, 2.46957E-05, 2.63175E-05, 2.57747E-05,
     X 2.22047E-05, 2.52755E-05, 2.80848E-05, 3.75157E-05, 4.09915E-05,
     X 4.04853E-05, 3.21661E-05, 3.15652E-05, 3.21576E-05, 3.67060E-05/
	DATA C20241 /
     X 3.13071E-05, 2.84939E-05, 2.71169E-05, 2.99559E-05, 2.94631E-05,
     X 3.26716E-05, 2.99028E-05, 2.60045E-05, 3.15375E-05, 3.12895E-05,
     X 2.77767E-05, 2.43976E-05, 2.10764E-05, 2.22725E-05, 2.04581E-05,
     X 1.63509E-05, 1.60028E-05, 1.60294E-05, 1.62366E-05, 1.89293E-05,
     X 1.79675E-05, 1.89259E-05, 1.68300E-05, 1.99460E-05, 2.42370E-05,
     X 2.64738E-05, 1.93137E-05, 1.39460E-05, 1.32222E-05, 1.38752E-05,
     X 1.62071E-05, 1.79652E-05, 1.63772E-05, 1.56251E-05, 1.81918E-05,
     X 1.46111E-05, 2.92174E-05, 2.94263E-05, 2.46180E-05, 2.93333E-05,
     X 3.13657E-05, 2.97686E-05, 2.78387E-05, 2.40924E-05, 2.93369E-05,
     X 2.93747E-05, 2.77665E-05, 3.00814E-05, 3.01068E-05, 3.62275E-05,
     X 3.56613E-05, 3.66913E-05, 3.56280E-05, 3.52856E-05, 3.63928E-05,
     X 2.96738E-05, 2.90314E-05, 2.62972E-05, 2.15250E-05, 1.97910E-05,
     X 2.02314E-05, 2.20209E-05, 2.05131E-05, 2.12017E-05, 1.96689E-05,
     X 1.61907E-05, 1.57662E-05, 1.58239E-05, 1.54650E-05, 1.46376E-05,
     X 1.32891E-05, 1.30511E-05, 1.17635E-05, 1.28585E-05, 1.12887E-05,
     X 1.32627E-05, 1.31833E-05, 1.68679E-05, 1.98092E-05, 2.70744E-05/
	DATA C20321 /
     X 2.22033E-05, 1.63430E-05, 1.61104E-05, 1.50865E-05, 1.54382E-05,
     X 1.55654E-05, 1.67924E-05, 1.89185E-05, 1.96791E-05, 2.14894E-05,
     X 2.76137E-05, 2.67339E-05, 2.79423E-05, 2.54664E-05, 3.10707E-05,
     X 2.72745E-05, 2.60940E-05, 2.47736E-05, 2.21105E-05, 2.20357E-05,
     X 2.26499E-05, 2.34137E-05, 2.29537E-05, 2.36157E-05, 2.48244E-05,
     X 2.26667E-05, 2.07781E-05, 2.11702E-05, 1.91214E-05, 1.62172E-05,
     X 1.61285E-05, 1.63952E-05, 1.68156E-05, 1.61236E-05, 1.56611E-05,
     X 1.47697E-05, 1.50856E-05, 1.44169E-05, 1.63816E-05, 1.74283E-05,
     X 1.49853E-05, 1.62444E-05, 1.70007E-05, 1.60371E-05, 1.22713E-05,
     X 1.45518E-05, 1.35051E-05, 1.40787E-05,-1.54925E-05,-2.15204E-05,
     X-4.04516E-06, 2.22439E-05, 3.21262E-05, 3.83792E-05, 4.44462E-05,
     X 4.44192E-05, 2.77328E-05, 4.10549E-06, 4.48758E-06,-1.27771E-05,
     X-2.17204E-05,-1.23979E-05,-1.04928E-05, 7.43085E-06, 1.55350E-05,
     X 3.15204E-05, 3.17601E-05, 2.93677E-05, 3.42485E-05, 3.87087E-05,
     X 3.61242E-05, 2.62406E-05, 3.31686E-05, 3.54314E-05, 2.50625E-05,
     X 2.60444E-05, 4.10729E-05, 3.47247E-05, 3.31716E-05, 3.34778E-05/
	DATA C20401 /
     X 4.03029E-05, 4.09241E-05, 3.96717E-05, 3.53410E-05, 2.81048E-05,
     X 1.98891E-05, 1.92314E-05, 2.82525E-05, 3.76641E-05, 4.34135E-05,
     X 4.24570E-05, 3.98429E-05, 3.29417E-05, 2.16679E-05, 8.88085E-06,
     X-5.05319E-06,-8.14815E-06,-5.01930E-06, 7.13565E-06, 2.00949E-05,
     X 2.65988E-05, 2.77656E-05, 2.09299E-05, 1.98968E-05, 2.04835E-05,
     X 1.75254E-05, 6.48674E-06, 3.14323E-06, 1.93242E-06, 3.86745E-06,
     X 1.39727E-05, 2.10731E-05, 2.66432E-05, 2.69551E-05, 2.57453E-05,
     X 2.72834E-05, 2.58860E-05, 2.51266E-05, 1.76048E-05, 2.03072E-05,
     X 2.61960E-05, 2.36230E-05, 1.81172E-05, 1.33972E-05, 1.60959E-05,
     X 1.61081E-05, 2.34099E-05, 2.64979E-05, 2.36894E-05, 2.13665E-05,
     X 2.16774E-05, 2.52566E-05, 1.99785E-05, 1.40414E-05, 1.39948E-05,
     X 1.32637E-05, 7.24742E-06, 1.11395E-06,-1.27323E-06, 4.56637E-07,
     X 6.93250E-06, 5.07198E-06, 7.90632E-06, 9.08149E-06, 1.03602E-05,
     X 2.17425E-05, 2.71741E-05, 2.16875E-05, 1.95088E-05, 1.56568E-05,
     X 8.41152E-06, 1.26749E-05, 1.17673E-05, 9.96037E-06, 1.21982E-05,
     X 1.31854E-05, 1.50216E-05, 1.72214E-05, 2.02773E-05, 2.09625E-05/
	DATA C20481 /
     X 1.66656E-05, 1.45666E-05, 1.66608E-05, 2.04989E-05, 2.21395E-05,
     X 2.35993E-05, 2.69390E-05, 2.13921E-05, 1.72643E-05, 1.70995E-05,
     X 1.78241E-05, 1.85308E-05, 1.80360E-05, 1.48619E-05, 1.90369E-05,
     X 1.51089E-05, 1.22705E-05, 1.62608E-05, 1.41637E-05, 1.23786E-05,
     X 7.02677E-06, 8.89811E-06, 1.07379E-05, 1.23677E-05, 1.48196E-05,
     X 2.05770E-05, 1.70994E-05, 1.00072E-05, 1.76119E-05, 1.41779E-05,
     X 1.34358E-05, 1.58674E-05, 1.65837E-05, 1.69569E-05, 1.40381E-05,
     X 1.46118E-05, 1.30556E-05, 1.97204E-05, 1.97488E-05, 1.64524E-05,
     X 1.73764E-05, 1.66355E-05, 1.64419E-05, 1.65486E-05, 1.21523E-05,
     X 1.51513E-05, 1.60354E-05, 1.38528E-05, 1.45538E-05, 1.71702E-05,
     X 1.56336E-05, 1.31279E-05, 1.47346E-05, 1.70719E-05, 1.75588E-05,
     X 1.55187E-05, 1.29598E-05, 1.38463E-05, 1.35382E-05, 1.16062E-05,
     X 1.37014E-05, 1.34487E-05, 1.15536E-05, 1.33597E-05, 9.24478E-06,
     X 7.28477E-06, 1.40321E-05, 1.31518E-05, 1.03118E-05, 8.59764E-06,
     X 1.57138E-05, 1.20792E-05, 1.49440E-05, 1.34375E-05, 1.54686E-05,
     X 1.65346E-05, 1.33823E-05, 1.37238E-05, 1.36128E-05, 1.46206E-05/
	DATA C20561 /
     X 1.40777E-05, 1.59980E-05, 1.30180E-05, 1.01390E-05, 1.12366E-05,
     X 9.86099E-06, 1.10702E-05, 1.26783E-05, 9.51072E-06, 8.07299E-06,
     X 1.22955E-05, 1.53506E-05, 1.29711E-05, 9.78759E-06, 1.28800E-05,
     X 1.39702E-05, 1.64832E-05, 1.06473E-05, 1.15419E-05, 1.63795E-05,
     X 1.69837E-05, 1.72726E-05, 1.77231E-05, 1.62337E-05, 1.20881E-05,
     X 1.13210E-05, 1.20531E-05, 1.31374E-05, 1.22259E-05, 1.27802E-05,
     X 1.38962E-05, 8.87355E-06, 9.42264E-06, 1.02075E-05, 7.91816E-06,
     X 9.66835E-06, 1.24921E-05, 8.43227E-06, 1.10637E-05, 1.03958E-05,
     X 9.40996E-06, 1.22922E-05, 1.21088E-05, 1.30116E-05, 1.18776E-05,
     X 1.42245E-05, 1.34745E-05, 1.11165E-05, 1.29914E-05, 1.29801E-05,
     X 1.10895E-05, 1.12331E-05, 9.03490E-06, 9.33726E-06, 9.63923E-06,
     X 1.11299E-05, 9.53481E-06, 1.21708E-05, 1.11951E-05, 7.22558E-06,
     X 6.66928E-06, 1.08926E-05, 1.07870E-05, 9.23485E-06, 8.50452E-06,
     X 9.41914E-06, 8.74027E-06, 8.93322E-06, 9.79061E-06, 8.26490E-06,
     X 8.37630E-06, 1.17064E-05, 1.10176E-05, 1.11587E-05, 9.45563E-06,
     X 1.18352E-05, 7.79327E-06, 9.22766E-06, 1.01868E-05, 8.23925E-06/
	DATA C20641 /
     X 9.23706E-06, 1.04428E-05, 8.80392E-06, 9.37098E-06, 7.43126E-06,
     X 7.01424E-06, 9.29360E-06, 8.97171E-06, 9.31718E-06, 9.87118E-06,
     X 8.11419E-06, 8.77416E-06, 9.96927E-06, 8.87533E-06, 9.33163E-06,
     X 7.41505E-06, 9.39988E-06, 1.17932E-05, 1.03287E-05, 9.17415E-06,
     X 8.43035E-06, 8.00040E-06, 8.33346E-06, 7.66787E-06, 7.18411E-06,
     X 1.06236E-05, 1.05559E-05, 8.49187E-06, 9.22472E-06, 8.16512E-06,
     X 8.35687E-06, 1.06325E-05, 9.80273E-06, 9.01599E-06, 9.20499E-06,
     X 9.98417E-06, 9.23191E-06, 6.98769E-06, 5.17748E-06, 4.57130E-06,
     X 8.18492E-06, 9.98095E-06, 7.52148E-06, 1.33038E-05, 8.17630E-06,
     X 1.02454E-05, 9.62706E-06, 9.44304E-06, 8.86704E-06, 8.88116E-06,
     X 8.79062E-06, 8.20042E-06, 8.55789E-06, 9.26249E-06, 1.00467E-05,
     X 7.96012E-06, 9.08773E-06, 1.01481E-05, 8.84360E-06, 7.94928E-06,
     X 6.68425E-06, 8.56576E-06, 1.05282E-05, 1.10647E-05, 9.91625E-06,
     X 7.95356E-06, 8.66443E-06, 9.13551E-06, 1.04870E-05, 9.79244E-06,
     X 1.26214E-05, 8.42148E-06, 8.13468E-06, 1.11338E-05, 1.06780E-05,
     X 8.54578E-06, 7.82119E-06, 8.33258E-06, 8.23644E-06, 5.95583E-06/
	DATA C20721 /
     X 5.85592E-06, 4.05898E-06, 6.39260E-06, 8.43280E-06, 8.76251E-06,
     X 6.70423E-06, 6.81368E-06, 7.43506E-06, 7.14376E-06, 6.51065E-06,
     X 5.65633E-06, 5.42394E-06, 7.10817E-06, 4.78831E-06, 6.29380E-06,
     X 4.87344E-06, 6.81764E-06, 6.51611E-06, 5.70526E-06, 6.50590E-06,
     X 6.61568E-06, 5.39248E-06, 6.32002E-06, 7.98976E-06, 7.73795E-06,
     X 4.85788E-06, 5.83443E-06, 6.11694E-06, 5.40408E-06, 5.00946E-06,
     X 5.62153E-06, 6.30263E-06, 6.05764E-06, 5.53274E-06, 5.80664E-06,
     X 5.18684E-06, 6.85555E-06, 6.22889E-06, 6.06959E-06, 6.49228E-06,
     X 5.64064E-06, 4.92690E-06, 5.77661E-06, 7.18450E-06, 7.38658E-06,
     X 6.77379E-06, 5.74668E-06, 6.68355E-06, 6.13655E-06, 6.43266E-06,
     X 7.08896E-06, 7.71187E-06, 7.37273E-06, 6.75882E-06, 6.39307E-06,
     X 4.59520E-06, 5.10323E-06, 5.80178E-06, 6.88172E-06, 6.68825E-06,
     X 7.50416E-06, 6.14975E-06, 6.51422E-06, 7.74942E-06, 8.11492E-06,
     X 1.19607E-05, 7.92722E-06, 4.47848E-06, 6.02524E-06, 9.74067E-06,
     X 1.02429E-05, 8.60819E-06, 8.57044E-06, 1.09196E-05, 1.02048E-05,
     X 3.86222E-06, 9.26104E-06, 7.33341E-06, 9.08181E-06, 1.05569E-05/
	DATA C20801 /
     X 1.06776E-05, 1.10247E-05, 1.04520E-05, 8.78328E-06, 7.60679E-06,
     X 7.27896E-06, 9.72776E-06, 5.16039E-06, 1.03134E-05, 1.09088E-05,
     X 8.12575E-06, 7.61685E-06, 8.16346E-06, 5.91269E-06, 3.61448E-06,
     X 8.74336E-06, 1.03990E-05, 6.25691E-06, 7.04541E-06, 7.94348E-06,
     X 8.39807E-06, 8.67342E-06, 8.32173E-06, 7.56015E-06, 8.31782E-06,
     X 6.36556E-06, 6.99328E-06, 6.24490E-06, 6.73080E-06, 6.95852E-06,
     X 7.55508E-06, 7.74168E-06, 7.90414E-06, 8.94934E-06, 7.99809E-06,
     X 6.12528E-06, 9.04115E-06, 7.14535E-06, 5.88625E-06, 6.43941E-06,
     X 7.11566E-06, 7.47425E-06, 8.23805E-06, 6.19919E-06, 7.31614E-06,
     X 8.24852E-06, 6.82172E-06, 5.45362E-06, 6.66115E-06, 8.44300E-06,
     X 8.07530E-06, 7.22735E-06, 5.85614E-06, 5.13900E-06, 6.03215E-06,
     X 6.59491E-06, 4.81592E-06, 4.48587E-06, 7.11525E-06, 8.36201E-06,
     X 7.11669E-06, 2.80033E-06, 6.50756E-06, 9.43974E-06, 5.22402E-06,
     X 3.82334E-06, 7.29963E-06, 8.62313E-06, 7.42018E-06, 4.56506E-06,
     X 5.29972E-06, 5.62787E-06, 4.63852E-06, 5.18329E-06, 7.01884E-06,
     X 7.24888E-06, 5.18157E-06, 5.40219E-06, 5.92412E-06, 4.97977E-06/
	DATA C20881 /
     X 5.29040E-06, 5.33812E-06, 4.76620E-06, 4.65759E-06, 5.10546E-06,
     X 6.49525E-06, 4.43416E-06, 5.30223E-06, 3.27044E-06, 2.55324E-06,
     X 4.85017E-06, 7.46556E-06, 8.04448E-06, 5.14009E-06, 6.09755E-06,
     X 5.38381E-06, 6.41959E-06, 6.59233E-06, 4.83160E-06, 3.81289E-06,
     X 5.37013E-06, 5.69212E-06, 5.54983E-06, 5.73495E-06, 4.00639E-06,
     X 2.33817E-06, 2.55751E-06, 3.29627E-06, 3.59845E-06, 6.20623E-06,
     X 4.47088E-06, 3.49267E-06, 3.09273E-06, 3.32506E-06, 4.83353E-06,
     X 6.39001E-06, 3.78074E-06, 4.07848E-06, 4.01811E-06, 3.19767E-06,
     X 3.34053E-06, 4.34246E-06, 3.68003E-06, 3.01090E-06, 3.98545E-06,
     X 2.72338E-06, 1.90024E-06, 2.77553E-06, 3.73381E-06, 2.58685E-06,
     X 1.70987E-06,-5.48480E-07, 1.64591E-06, 2.43481E-06, 2.52116E-06,
     X 2.19316E-06, 1.32392E-06, 1.75370E-06, 2.65409E-07, 2.22278E-06,
     X 2.53079E-06, 2.87260E-06, 1.87600E-06,-3.84453E-07, 1.80836E-06,
     X 9.28123E-07, 1.94986E-06, 2.40483E-06, 2.79865E-06, 2.86361E-06,
     X 2.63868E-06, 3.34704E-06, 3.32132E-06, 2.58463E-06, 2.45684E-06,
     X 3.35043E-06, 3.19848E-06, 1.73037E-06, 2.98206E-06, 2.77491E-06/
	DATA C20961 /
     X 6.51674E-07, 2.52219E-06, 2.97136E-06, 1.96700E-06, 2.29350E-06,
     X 3.01956E-06, 3.20811E-06, 1.30467E-06, 1.68172E-06, 2.56264E-06,
     X 2.46167E-06, 1.78221E-06, 2.31647E-06, 2.69480E-06, 2.63619E-06,
     X 1.81319E-06, 1.83448E-06, 2.23432E-06, 8.14045E-07, 8.75863E-07,
     X 1.61350E-06, 1.59796E-06, 2.08419E-06, 1.89665E-06, 6.93584E-07,
     X 1.09880E-06, 3.79031E-07,-3.36470E-07, 1.04326E-06, 1.06497E-06,
     X 2.15108E-07, 3.28774E-07,-5.17613E-07, 1.27762E-06, 8.22924E-07,
     X 4.92835E-07, 2.24698E-08,-1.99111E-07, 1.30262E-06,-3.81299E-07,
     X 9.55084E-07, 2.17641E-07,-6.03874E-08, 8.44121E-07, 1.72391E-06,
     X 1.66921E-06, 2.19855E-06, 1.17655E-06, 1.79637E-06, 3.31670E-06,
     X 3.40206E-06, 6.05670E-07, 2.08299E-06, 2.10121E-06, 1.68598E-06,
     X 2.21155E-06, 2.43221E-06, 5.81282E-08, 1.62613E-06,-5.49850E-07,
     X 2.14143E-07, 1.21751E-06, 2.30470E-06, 4.27911E-06, 2.96622E-06,
     X 8.67534E-07, 9.12041E-07, 2.48797E-06, 9.43519E-07,-3.60949E-06,
     X 2.01928E-06, 1.88873E-06, 8.06749E-07, 7.33519E-07, 1.17440E-06,
     X 1.69744E-06, 3.64492E-06, 3.11556E-06, 8.89471E-07, 1.93064E-06/
	DATA C21041 /
     X 3.02787E-06, 1.92575E-06, 1.73720E-06,-1.32700E-07, 1.41743E-06,
     X 2.24632E-06, 2.47945E-06, 2.05151E-06,-9.56031E-07, 2.57317E-07,
     X 3.00980E-06, 3.07981E-06, 2.78202E-06, 3.02555E-06, 5.48784E-09,
     X 2.37693E-06, 2.90011E-06, 2.93608E-06, 2.14837E-06, 6.55832E-07,
     X 3.41155E-07,-2.13884E-06, 2.52553E-06, 4.27109E-06, 3.33766E-06,
     X 3.07708E-06, 2.66405E-06, 3.22850E-06,-5.78879E-07,-6.06194E-07,
     X 1.72864E-06, 1.57072E-06,-3.39701E-07, 7.21540E-08, 1.67012E-06,
     X 2.48568E-06, 2.70214E-06, 3.62383E-06, 2.20408E-06, 1.19395E-06,
     X 1.53825E-06, 2.37511E-06, 2.66754E-06, 1.77020E-06, 5.40420E-07,
     X 2.01156E-06, 3.27498E-06, 3.04720E-06, 1.96213E-06, 3.71633E-06,
     X 2.07886E-06, 1.60069E-06, 5.33370E-07, 1.33966E-07, 2.16073E-06,
     X 8.81457E-07, 1.12880E-06, 2.40509E-06, 2.94252E-06, 2.22899E-06,
     X 1.80941E-06, 2.68577E-06, 2.44584E-06, 2.51720E-06, 2.64857E-06,
     X 2.24182E-06, 1.62007E-06, 2.60421E-06, 3.09782E-06, 3.11099E-06,
     X 3.81513E-06, 6.91606E-06, 3.28767E-06, 3.44175E-06, 4.16771E-06,
     X 3.75452E-06, 2.21050E-06, 2.99939E-06, 2.86993E-06, 2.47080E-06/
	DATA C21121 /
     X 2.33607E-06, 2.68568E-06, 3.39344E-06, 6.09518E-06, 5.10422E-06,
     X 4.04027E-06, 4.01363E-06, 4.53142E-06, 2.94424E-06, 4.76694E-06,
     X 6.44206E-06, 7.86435E-06, 8.55564E-06, 6.00857E-06, 5.48073E-06,
     X 1.56287E-06,-1.16619E-06,-1.85215E-06,-3.04762E-06,-3.45420E-07,
     X 2.48111E-07,-1.39302E-07,-6.27593E-07,-5.26792E-07, 4.81454E-08,
     X-3.08631E-08,-1.02976E-06,-1.54919E-06,-9.34044E-07,-1.02507E-06,
     X-1.39794E-06,-1.15709E-06,-1.04875E-06,-1.64379E-06,-2.97514E-06,
     X-3.22236E-07,-1.18978E-06,-2.85325E-06,-3.93143E-06,-4.15349E-06,
     X-2.33228E-06,-3.27125E-06,-2.44987E-06,-1.44460E-06,-3.59727E-06,
     X-7.18516E-07,-1.53237E-06,-1.53526E-06,-1.56450E-06,-2.91088E-06,
     X-8.52134E-07,-1.44575E-07,-1.50350E-06,-2.92806E-06,-2.47710E-06,
     X-9.71202E-07,-9.82754E-07,-1.09924E-06,-6.08199E-07, 3.62885E-07,
     X-6.67372E-07,-1.00033E-06,-1.12001E-06,-1.06624E-06,-9.23789E-07,
     X-9.83788E-07,-2.11656E-06,-2.45001E-06,-2.75874E-06,-3.36003E-06,
     X-3.38364E-06,-2.63747E-06,-3.11047E-06,-3.75258E-06,-3.83211E-06,
     X-3.52833E-06,-3.48464E-06,-3.77021E-06,-4.26887E-06,-4.23917E-06/
	DATA C21201 /
     X-1.42438E-06,-2.48477E-06,-2.84719E-06,-2.70247E-06,-2.50588E-06,
     X-2.22900E-06,-1.78393E-06,-1.76826E-06,-2.16396E-06,-2.67543E-06,
     X-2.23706E-06,-2.31793E-06,-2.87590E-06,-3.07803E-06,-2.50493E-06,
     X-4.54223E-06,-5.15511E-06,-5.39690E-06,-4.89633E-06,-3.33710E-06,
     X-4.56583E-06,-4.78877E-06,-3.93508E-06,-3.29027E-06,-4.95668E-06,
     X-6.01801E-06,-5.76016E-06,-5.34657E-06,-5.29080E-06,-5.57133E-06,
     X-5.73135E-06,-5.39374E-06,-5.09808E-06,-5.12874E-06,-5.20269E-06,
     X-7.30702E-06,-7.04220E-06,-5.96514E-06,-5.74802E-06,-4.53961E-06,
     X-4.42127E-06,-4.63922E-06,-4.80622E-06,-4.69659E-06,-5.96786E-06,
     X-6.29800E-06,-4.75452E-06,-2.85907E-06,-5.33662E-06,-5.31681E-06,
     X-5.04646E-06,-5.21729E-06,-5.93409E-06,-5.73462E-06,-5.44926E-06,
     X-6.43325E-06,-7.74451E-06,-7.83147E-06,-5.51568E-06,-7.37048E-06,
     X-4.25726E-06, 2.32917E-06,-5.61131E-07, 2.05234E-06, 3.74631E-07,
     X-7.66493E-07, 1.42689E-06,-7.79683E-07, 9.06809E-07, 5.13642E-07,
     X-1.52504E-06,-2.12058E-06,-2.50316E-06, 1.03637E-08, 5.60002E-07,
     X-1.48075E-06, 1.94155E-06, 1.91846E-06, 2.78507E-06, 3.90146E-06/
	DATA C21281 /
     X 3.61409E-06, 3.23677E-06, 4.00022E-06, 3.19157E-06, 4.03034E-07,
     X-2.03929E-06, 1.23366E-06, 3.28589E-06, 3.94168E-06, 3.94672E-06,
     X 3.84619E-06, 2.30400E-07,-2.07799E-06,-1.75115E-06,-5.71958E-07,
     X 2.33425E-06, 2.01664E-06, 6.05673E-07, 9.57363E-07,-8.89924E-07,
     X-4.71331E-07, 2.82826E-07, 5.10859E-07, 3.63512E-07, 9.86288E-07,
     X-4.86309E-07,-2.23163E-06,-1.23370E-06,-2.43131E-07,-2.11498E-06,
     X-1.56756E-06, 2.70905E-06, 1.87606E-08, 7.83721E-08, 1.58444E-06,
     X 2.88574E-06, 1.40306E-06, 2.40883E-06, 2.84063E-06, 3.13820E-06,
     X 3.71016E-06, 3.12975E-06, 3.21981E-06, 2.56191E-06, 1.04624E-06,
     X 1.87464E-07, 7.25329E-07, 1.03650E-06, 7.23663E-07,-4.18739E-07,
     X 9.95744E-07,-1.80878E-07,-1.04044E-06, 3.86965E-07,-9.36186E-07,
     X-4.02271E-07,-2.00231E-07,-5.94965E-07, 4.94038E-07, 3.34585E-07,
     X 4.82255E-07, 1.12599E-06, 2.11763E-06, 2.66807E-07, 2.29324E-07,
     X 7.07005E-07, 3.41907E-07,-1.17115E-07, 9.03089E-07, 1.76844E-06,
     X 1.87134E-06, 2.64057E-06, 4.00395E-07,-4.19679E-07, 6.30769E-07,
     X 1.02725E-06, 1.05876E-06,-4.08660E-07,-2.32668E-06,-2.73468E-06/
	DATA C21361 /
     X-2.40600E-06,-1.81203E-06,-7.96431E-07, 7.40789E-07, 2.73188E-07,
     X 1.68367E-07,-1.27227E-07,-1.05041E-06,-3.51726E-06,-1.64956E-06,
     X-5.63840E-07,-1.61242E-06,-1.33264E-06, 1.56604E-06, 2.35083E-06,
     X 9.26708E-07, 5.41983E-07, 3.54277E-07, 8.53743E-07, 1.54196E-06,
     X 1.19902E-06, 1.10552E-06, 1.63179E-06, 1.96366E-06, 7.82848E-07,
     X-3.34741E-08,-7.90842E-07,-6.45131E-07, 1.36158E-06, 1.62453E-06,
     X 6.68965E-07,-4.86203E-08, 6.83561E-07, 1.89652E-06,-2.80988E-07,
     X-2.30536E-06,-1.90777E-06, 1.31617E-06, 1.27309E-06, 5.90825E-07,
     X 5.65686E-07, 1.23631E-07,-1.70279E-06,-1.60768E-06, 9.69543E-07,
     X 1.01108E-07,-2.02473E-06,-1.75146E-06, 6.33201E-07,-3.59110E-06,
     X-9.71706E-07, 9.16822E-07, 1.40681E-07,-7.16745E-07,-2.11376E-06,
     X-1.00951E-06, 2.12465E-06, 1.06982E-06, 1.44032E-06, 1.49692E-06,
     X 1.07277E-06, 1.37006E-06, 1.66932E-06, 1.75820E-06, 1.41859E-06,
     X-5.84947E-08, 2.17349E-06, 4.27053E-06, 5.27286E-06, 5.87085E-06,
     X 2.42692E-06, 2.39305E-06, 6.19573E-06, 5.12518E-06, 1.27171E-06,
     X-6.81963E-07, 4.16199E-08,-1.36608E-06,-2.53272E-06,-2.37700E-06/
	DATA C21441 /
     X-7.96719E-07, 3.85367E-07,-1.08393E-07,-9.04587E-07,-1.54917E-06,
     X-3.11945E-06,-5.58484E-07, 1.61347E-06, 1.11736E-06, 2.11889E-06,
     X 2.43534E-06, 1.46709E-06,-1.05429E-06, 1.09978E-06, 7.22493E-07,
     X 8.53307E-08, 1.22733E-06, 2.99380E-06, 3.62416E-06, 3.81404E-06,
     X 4.46735E-06, 4.70753E-06, 4.54494E-06, 3.83002E-06, 2.28067E-06,
     X 2.03102E-06, 2.43844E-06, 2.93132E-06, 2.17555E-06, 3.92919E-06,
     X 3.53089E-06, 1.61388E-06, 5.09498E-06, 3.40067E-06, 1.58876E-06,
     X 1.17367E-06, 1.13344E-06, 1.17798E-06, 1.10976E-06, 7.90635E-07,
     X-4.15989E-07,-1.00581E-06,-9.60236E-07,-1.79111E-07,-5.70733E-07,
     X 1.49766E-06, 3.44374E-06, 6.45914E-07, 1.00532E-06, 2.01068E-06,
     X 2.59092E-06, 9.35770E-08, 6.00121E-07, 1.54409E-06, 2.03537E-06,
     X 8.10358E-07, 1.34126E-06, 1.88873E-06, 1.43283E-06,-2.05029E-07,
     X-1.09782E-06,-6.56149E-07, 2.01650E-06, 1.84770E-06, 4.39586E-08,
     X-2.03588E-06,-1.46366E-06,-3.45189E-07, 4.02577E-07, 3.10362E-07,
     X-2.16073E-06,-1.91861E-06,-2.90520E-07, 2.03692E-06, 3.47996E-06,
     X 4.21761E-06, 3.89000E-06, 1.86138E-06, 1.56143E-06, 4.88964E-07/
	DATA C21521 /
     X-9.28184E-07,-4.34315E-07, 8.74954E-07, 1.58417E-06, 1.36880E-06,
     X 2.65016E-06, 4.62623E-06, 5.81990E-06, 4.72139E-06, 1.95905E-06,
     X 1.54151E-06, 2.95768E-06, 4.71536E-06, 2.62359E-06, 9.11513E-07,
     X 4.75677E-07,-1.53801E-06,-2.32382E-06,-2.25220E-06,-1.46641E-06,
     X-2.23014E-06,-2.12604E-06,-1.66259E-06,-2.48856E-06,-2.38895E-06,
     X-2.18158E-06,-1.95841E-06, 4.43899E-07, 1.08517E-06, 1.66370E-07,
     X-2.42342E-06,-7.19331E-07, 3.19532E-07, 3.58690E-07,-2.01979E-07,
     X 5.07242E-07, 1.10562E-06, 1.00419E-06, 1.22379E-06, 7.05180E-07,
     X 1.42283E-07, 8.61092E-07, 8.95236E-07, 1.18043E-07,-1.23589E-06,
     X-6.16316E-07,-1.18947E-06,-1.45838E-06,-1.47522E-09, 1.33867E-06,
     X 9.18310E-07,-8.98949E-07,-2.27314E-06,-1.71510E-06,-7.16704E-07,
     X 8.60666E-09, 5.68015E-07, 1.31219E-06, 1.75478E-06, 5.11790E-07,
     X 3.35270E-07, 5.39243E-07, 9.08467E-07, 1.39382E-06, 1.08806E-06,
     X 1.18589E-06, 3.58461E-06, 2.78668E-06, 1.25964E-06,-2.72255E-07,
     X 1.72305E-06, 1.82937E-06, 7.46252E-07,-1.10555E-06, 2.24967E-07,
     X 6.45674E-07,-1.87591E-07,-8.84068E-07,-1.75433E-06,-2.17670E-06/
	DATA C21601 /
     X-1.37112E-06,-2.31722E-06,-2.23860E-06,-1.16796E-06,-2.23765E-06,
     X-1.86406E-06,-1.03517E-06,-5.90824E-07,-6.57710E-07,-7.00941E-07,
     X-4.46064E-07, 1.77205E-06, 2.45066E-06, 2.39371E-06, 2.30736E-06,
     X 2.35355E-06, 1.85070E-06, 9.62711E-07, 2.59644E-06, 2.05304E-06,
     X 9.70090E-07, 1.50942E-06, 3.79439E-06, 2.94597E-06,-1.91789E-06,
     X 6.44324E-08,-3.92094E-07,-1.55398E-06, 4.46701E-08,-4.78760E-07,
     X-1.70061E-06,-3.17252E-06,-2.93173E-06,-2.01455E-06,-7.76298E-07,
     X-2.74577E-07,-1.39907E-06,-2.16470E-06,-1.26010E-06,-2.76845E-06,
     X-2.38226E-06,-5.49068E-08, 9.65258E-07, 1.08650E-06, 5.64738E-07,
     X-5.78379E-07,-5.68918E-07,-1.90177E-06,-5.08874E-06,-3.03648E-06,
     X-1.30527E-06,-4.87669E-07,-2.83326E-06,-1.97823E-06,-5.94313E-07,
     X-1.50961E-07,-1.15908E-06,-1.43260E-06,-9.29331E-07,-1.39459E-06,
     X-1.27237E-06,-1.50189E-06,-3.79292E-06,-3.92038E-06,-3.58490E-06,
     X-3.26439E-06,-2.42138E-06,-2.70516E-06,-3.58080E-06,-1.71822E-06,
     X-2.41567E-06,-3.50193E-06,-2.62394E-06,-3.08424E-06,-3.89604E-06,
     X-4.84127E-06,-4.41385E-06,-3.22673E-06,-1.80987E-06,-2.93027E-06/
	DATA C21681 /
     X-3.17366E-06,-2.79721E-06,-1.78848E-06,-2.80254E-06,-3.55572E-06,
     X-3.34632E-06,-2.83979E-06,-2.48022E-06,-2.15090E-06,-1.08311E-06,
     X-6.15216E-07,-7.13008E-07,-1.70841E-06,-2.96098E-06,-3.57134E-06,
     X-3.04405E-06,-3.35280E-06,-2.97780E-06,-1.97966E-06,-2.33197E-06,
     X-2.76708E-06,-2.70409E-06,-4.51094E-07,-1.43068E-06,-2.83719E-06,
     X-2.98921E-06,-4.14949E-06,-3.63780E-06,-8.10138E-07,-1.61597E-06,
     X-2.25394E-06,-2.58110E-06,-1.57781E-06,-1.71520E-06,-2.30016E-06,
     X-2.61268E-06,-1.96696E-06,-1.86744E-06,-3.15645E-06,-3.59354E-06,
     X-3.61015E-06,-3.21793E-06,-2.57436E-06,-2.74347E-06,-3.33319E-06,
     X-2.93400E-06,-3.25986E-06,-3.46384E-06,-2.22114E-06,-2.92650E-06,
     X-3.73666E-06,-3.70485E-06,-2.75963E-06,-2.40652E-06,-2.93107E-06,
     X-1.77517E-06,-1.57096E-06,-2.17533E-06,-2.80190E-06,-2.27942E-06,
     X-1.37371E-06,-1.65974E-06,-1.26079E-06,-8.08050E-07,-8.41278E-07,
     X-1.53860E-06,-1.66687E-06,-6.56592E-07,-3.05110E-08, 1.08623E-07,
     X-2.87222E-07,-2.63555E-07,-7.89575E-07,-1.56059E-06,-6.42174E-07,
     X-9.43333E-07,-1.38671E-06, 6.50443E-07, 1.35301E-06, 9.27981E-07/
	DATA C21761 /
     X-1.21705E-06,-9.63848E-08, 8.73593E-07,-3.47278E-08,-1.79042E-06,
     X-2.15544E-06,-4.48668E-07,-1.17414E-06,-1.35437E-06,-8.90688E-07,
     X-4.54757E-07, 2.41484E-09, 3.88010E-07,-1.85349E-08, 1.58011E-07,
     X 3.70566E-07,-7.30268E-07,-8.42354E-07,-4.13738E-07, 3.96796E-07,
     X-5.55763E-07,-1.26877E-06,-2.89854E-07, 5.78676E-07, 9.51356E-07,
     X 5.56912E-07, 1.05014E-06, 9.75896E-07, 5.91573E-08,-6.15073E-07,
     X-1.48803E-06,-2.53397E-06,-1.77027E-06,-2.08546E-06,-3.10452E-06,
     X-1.65227E-06,-1.15981E-06,-1.25849E-06,-9.65711E-07,-1.90319E-06,
     X-2.71831E-06,-5.71559E-08,-1.20368E-06,-3.16820E-06,-2.22766E-06,
     X-1.19828E-06,-2.82573E-07, 2.53850E-07,-9.10547E-07,-1.65529E-06,
     X-6.00138E-07,-4.98898E-07,-3.45799E-07, 2.25160E-07, 1.14332E-07,
     X 3.16082E-07, 1.12681E-06,-6.04876E-07,-7.24616E-07, 1.48177E-06,
     X 1.05680E-06, 5.91076E-07, 2.07187E-07, 3.82385E-07, 5.91560E-07,
     X 8.26519E-07, 1.22139E-06, 1.63501E-06, 2.06423E-06, 2.50038E-06,
     X 2.38037E-06, 1.91688E-06, 2.46702E-06, 2.45066E-06, 2.16732E-06,
     X 3.13517E-06, 2.68221E-06, 1.39877E-06, 8.58945E-07, 6.83181E-07/
	DATA C21841 /
     X 8.46816E-07, 1.73491E-06, 1.98732E-06, 1.94059E-06, 2.19284E-06,
     X 1.73215E-06, 1.06865E-06, 1.14117E-06, 1.43213E-06, 1.42275E-06,
     X-4.15449E-07,-2.39911E-07, 3.46498E-08,-2.75022E-06,-2.43736E-06,
     X-1.06489E-06,-7.81941E-07,-8.04801E-07,-1.04984E-06,-1.65734E-06,
     X-1.03167E-06,-3.18255E-08, 5.70283E-07, 6.19050E-07, 2.92257E-07,
     X-6.01436E-07,-7.04005E-07,-3.70875E-07, 4.12830E-07, 1.31319E-07,
     X-1.61570E-07, 9.76170E-07, 7.99907E-07, 1.41860E-07,-1.98022E-07,
     X 3.13766E-07, 7.43982E-07,-6.11287E-07,-5.21146E-07, 1.11156E-07,
     X 3.91719E-07, 5.45566E-07, 6.39059E-07, 7.29515E-07, 4.59167E-07,
     X 6.13179E-08,-3.48146E-08, 5.32046E-07, 1.19736E-06, 3.83982E-07,
     X 1.73267E-07, 3.54304E-07, 9.34657E-07, 5.53819E-07,-2.86678E-07,
     X 2.01853E-08,-1.56159E-07,-6.08130E-07,-2.14929E-07, 1.66317E-08,
     X 9.32462E-08,-4.83623E-07,-9.16323E-07,-1.22772E-06,-1.61586E-06,
     X-1.27409E-06,-1.98119E-07,-3.69182E-08,-1.41061E-07,-5.12562E-07,
     X-4.55495E-07,-8.12132E-07,-1.71772E-06,-2.70741E-06,-2.98751E-06,
     X-2.19520E-06, 3.01900E-07, 1.17806E-06,-1.23067E-06, 4.17086E-07/
	DATA C21921 /
     X 1.68113E-06, 4.81677E-07,-1.55187E-07,-3.35287E-07, 2.94916E-07,
     X 4.57124E-07, 3.38692E-07,-2.49203E-07,-3.62585E-07,-2.39653E-07,
     X 3.72675E-08,-7.79964E-09,-2.83285E-07,-9.74713E-07,-6.91171E-07,
     X 1.21925E-07, 3.39940E-07, 3.68441E-08,-5.82188E-07, 2.12605E-07,
     X 4.65144E-07, 2.17190E-07, 7.50119E-07, 8.62008E-07, 4.63016E-07,
     X 1.25620E-06, 1.04567E-06,-8.17037E-07,-1.20023E-06,-1.06224E-06,
     X-3.77100E-07,-1.28057E-07,-2.76183E-07,-1.24304E-06,-2.56776E-06,
     X-3.36699E-06,-1.49408E-06,-1.01189E-07, 7.41870E-07,-6.45425E-07,
     X-7.47111E-07, 4.79055E-10,-1.32339E-06,-1.86135E-06,-1.61074E-06,
     X-1.82039E-06,-1.68040E-06,-1.08025E-06,-8.61965E-07,-7.00131E-07,
     X-5.63105E-07,-8.09843E-07,-8.09221E-07, 1.69474E-07,-1.33941E-07,
     X-7.49558E-07,-5.19013E-07,-8.53534E-07,-1.33703E-06,-3.11161E-07,
     X 8.99037E-07, 2.25330E-06, 1.44822E-06, 3.07437E-07,-1.22366E-06,
     X-7.64217E-07, 2.13156E-08, 1.07909E-06, 6.10755E-07, 1.81483E-07,
     X 8.12405E-07,-9.13283E-08,-1.35885E-06,-1.58366E-06,-7.88594E-07,
     X 4.48283E-07,-1.23754E-06,-1.65105E-06,-8.93014E-07,-1.48622E-06/
	DATA C22001 /
     X-1.67948E-06,-1.24310E-06,-1.54411E-06,-1.65677E-06,-1.04998E-06,
     X-1.46985E-07, 4.61778E-07,-4.87832E-07,-4.89452E-07,-1.24840E-07,
     X-1.70101E-06,-1.66976E-06,-1.48528E-07,-1.12621E-07,-2.30607E-08,
     X 1.82301E-07,-8.58152E-07,-1.89794E-06,-2.46464E-06,-2.32745E-06,
     X-2.02112E-06,-2.07656E-06,-1.43824E-06,-5.16583E-07,-1.80702E-06,
     X-2.93490E-06,-3.89216E-06,-3.36211E-06,-2.41393E-06,-9.53406E-07,
     X-1.16269E-06,-1.66431E-06,-1.77150E-06,-1.82496E-06,-1.93095E-06,
     X-2.75759E-06,-2.83618E-06,-2.27908E-06,-6.33348E-07, 5.61257E-07,
     X 1.00142E-06, 7.73337E-07, 3.17721E-07,-3.69804E-07,-8.82058E-07,
     X-1.17364E-06,-4.53480E-07,-2.47824E-07,-4.79624E-07,-5.17032E-07,
     X-3.46498E-07, 1.42669E-07,-1.59168E-07,-5.06580E-07,-3.18573E-07,
     X-2.74092E-07,-2.68860E-07, 1.32811E-07,-2.35567E-09,-6.71971E-07,
     X-9.75302E-07,-8.70978E-07,-3.59071E-08,-3.01726E-07,-8.27641E-07,
     X-1.14899E-06,-1.50160E-06,-1.83660E-06,-1.26290E-06,-1.07659E-06,
     X-1.34878E-06,-5.24626E-07,-7.85094E-08,-8.79473E-07,-1.19291E-06,
     X-1.33298E-06,-1.59750E-06,-1.31836E-06,-5.73079E-07,-1.10349E-06/
	DATA C22081 /
     X-1.11807E-06,-1.99530E-07,-8.10496E-07,-1.42679E-06,-5.34617E-07,
     X-2.05001E-07,-2.51690E-07,-1.01740E-06,-1.02841E-06,-7.48750E-08,
     X-1.01770E-06,-1.50413E-06, 1.80898E-07, 3.63788E-07,-1.97900E-07,
     X-1.16721E-06,-1.05497E-06,-2.07218E-08,-1.90590E-07,-8.25501E-07,
     X-2.21142E-06,-1.19905E-06, 2.16271E-07,-2.52574E-07,-4.35837E-07,
     X-3.95272E-07, 5.97065E-08, 2.76639E-07, 9.22569E-08, 1.20142E-07,
     X-2.95030E-09,-1.08216E-06,-1.32386E-06,-9.62248E-07,-1.99430E-06,
     X-2.13890E-06,-9.56082E-07,-6.94022E-07,-7.75721E-07,-1.31048E-06,
     X-1.50080E-06,-1.35873E-06,-7.48378E-07,-4.83436E-07,-4.69624E-07,
     X-1.51156E-06,-2.48221E-06,-3.30134E-06,-2.79114E-06,-2.08976E-06,
     X-2.24768E-06,-1.06947E-06, 1.17462E-06,-2.51423E-07,-7.85729E-07,
     X 5.37467E-07,-9.39876E-08,-1.11303E-06,-7.46860E-07,-9.36220E-07,
     X-1.59880E-06,-1.61420E-06,-1.54368E-06,-1.41036E-06,-7.20350E-07,
     X 1.35544E-07, 3.14481E-07, 6.29265E-07, 1.09161E-06,-1.36044E-07,
     X-1.22932E-06,-1.29847E-06,-3.26429E-06,-6.01062E-06,-2.09945E-06,
     X 1.26878E-07,-2.88050E-08,-6.82802E-07,-1.39340E-06,-1.82986E-06/
	DATA C22161 /
     X-1.67208E-06,-1.07994E-06,-1.89195E-06,-2.10782E-06,-1.04519E-06,
     X-3.27672E-07, 1.95516E-07, 1.63838E-07,-2.29575E-07,-1.01609E-06,
     X-2.19286E-06,-2.71850E-06,-9.77485E-07,-1.48830E-06,-3.37826E-06,
     X-1.59130E-06,-5.74498E-07,-8.27962E-07,-9.92211E-07,-1.14422E-06,
     X-1.41420E-06,-1.11629E-06,-2.51575E-07, 1.60805E-07, 1.82934E-07,
     X-7.28868E-07,-2.57062E-07, 1.06520E-06, 4.16488E-07, 2.97049E-08,
     X 6.62797E-08, 8.29435E-07, 1.29657E-06,-2.27961E-06,-3.40386E-06,
     X-1.88594E-06,-2.29732E-06,-2.72594E-06,-2.09847E-06,-1.31771E-06,
     X-4.23693E-07,-4.96348E-07,-9.40209E-07,-2.08707E-06,-1.21368E-06,
     X 4.79409E-07,-1.12548E-08,-4.57316E-07,-8.40885E-07,-5.03210E-07,
     X-1.61036E-07,-1.05835E-06,-1.66417E-06,-1.97827E-06,-1.63737E-06,
     X-1.11711E-06,-3.16081E-07,-6.81746E-07,-1.82599E-06,-1.12895E-06,
     X-9.19712E-07,-1.91707E-06,-2.14767E-06,-2.03629E-06,-2.86441E-06,
     X-3.07735E-06,-2.28656E-06,-1.40256E-06,-5.50649E-07,-3.11627E-07,
     X-7.90261E-07,-2.10728E-06,-1.89739E-06,-1.53762E-06,-2.39947E-06,
     X-2.28765E-06,-1.27564E-06,-2.15154E-06,-3.17932E-06,-3.84234E-06/
	DATA C22241 /
     X-3.65102E-06,-2.84055E-06,-2.48744E-06,-2.27683E-06,-2.33087E-06,
     X-3.44460E-06,-5.19613E-06,-2.85882E-06,-1.39921E-06,-2.00579E-06,
     X-2.80593E-06,-3.65940E-06,-2.39526E-06,-1.70389E-06,-2.03532E-06,
     X-2.71522E-06,-3.42227E-06,-2.23606E-06,-1.77845E-06,-2.42071E-06,
     X-2.61515E-06,-2.56413E-06,-1.49601E-06,-1.23245E-06,-2.08440E-06,
     X-2.11121E-06,-1.93424E-06,-2.27439E-06,-2.58183E-06,-2.84705E-06,
     X-2.32183E-06,-1.80966E-06,-3.04089E-06,-3.14334E-06,-1.91331E-06,
     X-1.51037E-06,-1.43610E-06,-2.11316E-06,-2.45184E-06,-2.42262E-06/
	END


