/*
 * @(#)Inflater.java	1.39 03/02/07
 *
 * Copyright 2003 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */

package java.util.zip;

/**
 * This class provides support for general purpose decompression using
 * popular ZLIB compression library. The ZLIB compression library was
 * initially developed as part of the PNG graphics standard and is not
 * protected by patents. It is fully described in the specifications at 
 * the <a href="package-summary.html#package_description">java.util.zip
 * package description</a>.
 *
 * <p>The following code fragment demonstrates a trivial compression
 * and decompression of a string using <tt>Deflater</tt> and
 * <tt>Inflater</tt>.
 *
 * <blockquote><pre>
 * // Encode a String into bytes
 * String inputString = "blahblahblah\u20AC\u20AC";
 * byte[] input = inputString.getBytes("UTF-8");
 *
 * // Compress the bytes
 * byte[] output = new byte[100];
 * Deflater compresser = new Deflater();
 * compresser.setInput(input);
 * compresser.finish();
 * int compressedDataLength = compresser.deflate(output);
 *
 * // Decompress the bytes
 * Inflater decompresser = new Inflater();
 * decompresser.setInput(output, 0, compressedDataLength);
 * byte[] result = new byte[100];
 * int resultLength = decompresser.inflate(result);
 * decompresser.end();
 *
 * // Decode the bytes into a String
 * String outputString = new String(result, 0, resultLength, "UTF-8");
 * </pre></blockquote>
 *
 * @see		Deflater
 * @version 	1.39, 02/07/03
 * @author 	David Connelly
 *
 */
public
class Inflater {
    private long strm;
    private byte[] buf = new byte[0];
    private int off, len;
    private boolean finished;
    private boolean needDict;

    /*
     * Loads the ZLIB library.
     */
    static {
	java.security.AccessController.doPrivileged(
		  new sun.security.action.LoadLibraryAction("zip"));
	initIDs();
    }

    /**
     * Creates a new decompressor. If the parameter 'nowrap' is true then
     * the ZLIB header and checksum fields will not be used. This provides
     * compatibility with the compression format used by both GZIP and PKZIP.
     * <p>
     * Note: When using the 'nowrap' option it is also necessary to provide
     * an extra "dummy" byte as input. This is required by the ZLIB native
     * library in order to support certain optimizations.
     *
     * @param nowrap if true then support GZIP compatible compression
     */
    public Inflater(boolean nowrap) {
	strm = init(nowrap);
    }

    /**
     * Creates a new decompressor.
     */
    public Inflater() {
	this(false);
    }

    /**
     * Sets input data for decompression. Should be called whenever
     * needsInput() returns true indicating that more input data is
     * required.
     * @param b the input data bytes
     * @param off the start offset of the input data
     * @param len the length of the input data
     * @see Inflater#needsInput
     */
    public synchronized void setInput(byte[] b, int off, int len) {
	if (b == null) {
	    throw new NullPointerException();
	}
	if (off < 0 || len < 0 || off > b.length - len) {
	    throw new ArrayIndexOutOfBoundsException();
	}
	this.buf = b;
	this.off = off;
	this.len = len;
    }

    /**
     * Sets input data for decompression. Should be called whenever
     * needsInput() returns true indicating that more input data is
     * required.
     * @param b the input data bytes
     * @see Inflater#needsInput
     */
    public void setInput(byte[] b) {
	setInput(b, 0, b.length);
    }

    /**
     * Sets the preset dictionary to the given array of bytes. Should be
     * called when inflate() returns 0 and needsDictionary() returns true
     * indicating that a preset dictionary is required. The method getAdler()
     * can be used to get the Adler-32 value of the dictionary needed.
     * @param b the dictionary data bytes
     * @param off the start offset of the data
     * @param len the length of the data
     * @see Inflater#needsDictionary
     * @see Inflater#getAdler
     */
    public synchronized void setDictionary(byte[] b, int off, int len) {
	if (strm == 0 || b == null) {
	    throw new NullPointerException();
	}
	if (off < 0 || len < 0 || off > b.length - len) {
	    throw new ArrayIndexOutOfBoundsException();
	}
	setDictionary(strm, b, off, len);
	needDict = false;
    }

    /**
     * Sets the preset dictionary to the given array of bytes. Should be
     * called when inflate() returns 0 and needsDictionary() returns true
     * indicating that a preset dictionary is required. The method getAdler()
     * can be used to get the Adler-32 value of the dictionary needed.
     * @param b the dictionary data bytes
     * @see Inflater#needsDictionary
     * @see Inflater#getAdler
     */
    public void setDictionary(byte[] b) {
	setDictionary(b, 0, b.length);
    }

    /**
     * Returns the total number of bytes remaining in the input buffer.
     * This can be used to find out what bytes still remain in the input
     * buffer after decompression has finished.
     * @return the total number of bytes remaining in the input buffer
     */
    public synchronized int getRemaining() {
	return len;
    }

    /**
     * Returns true if no data remains in the input buffer. This can
     * be used to determine if #setInput should be called in order
     * to provide more input.
     * @return true if no data remains in the input buffer
     */
    public synchronized boolean needsInput() {
	return len <= 0;
    }

    /**
     * Returns true if a preset dictionary is needed for decompression.
     * @return true if a preset dictionary is needed for decompression
     * @see Inflater#setDictionary
     */
    public synchronized boolean needsDictionary() {
	return needDict;
    }

    /**
     * Return true if the end of the compressed data stream has been
     * reached.
     * @return true if the end of the compressed data stream has been
     * reached
     */
    public synchronized boolean finished() {
	return finished;
    }

    /**
     * Uncompresses bytes into specified buffer. Returns actual number
     * of bytes uncompressed. A return value of 0 indicates that
     * needsInput() or needsDictionary() should be called in order to
     * determine if more input data or a preset dictionary is required.
     * In the later case, getAdler() can be used to get the Adler-32
     * value of the dictionary required.
     * @param b the buffer for the uncompressed data
     * @param off the start offset of the data
     * @param len the maximum number of uncompressed bytes
     * @return the actual number of uncompressed bytes
     * @exception DataFormatException if the compressed data format is invalid
     * @see Inflater#needsInput
     * @see Inflater#needsDictionary
     */
    public synchronized int inflate(byte[] b, int off, int len)
	throws DataFormatException
    {
	if (b == null) {
	    throw new NullPointerException();
	}
	if (off < 0 || len < 0 || off > b.length - len) {
	    throw new ArrayIndexOutOfBoundsException();
	}
	return inflateBytes(b, off, len);
    }

    /**
     * Uncompresses bytes into specified buffer. Returns actual number
     * of bytes uncompressed. A return value of 0 indicates that
     * needsInput() or needsDictionary() should be called in order to
     * determine if more input data or a preset dictionary is required.
     * In the later case, getAdler() can be used to get the Adler-32
     * value of the dictionary required.
     * @param b the buffer for the uncompressed data
     * @return the actual number of uncompressed bytes
     * @exception DataFormatException if the compressed data format is invalid
     * @see Inflater#needsInput
     * @see Inflater#needsDictionary
     */
    public int inflate(byte[] b) throws DataFormatException {
	return inflate(b, 0, b.length);
    }

    /**
     * Returns the ADLER-32 value of the uncompressed data.
     * @return the ADLER-32 value of the uncompressed data
     */
    public synchronized int getAdler() {
	if (strm == 0) {
	    throw new NullPointerException();
	}
	return getAdler(strm);
    }

    /**
     * Returns the total number of bytes input so far.
     * @return the total number of bytes input so far
     */
    public synchronized int getTotalIn() {
	if (strm == 0) {
	    throw new NullPointerException();
	}
	return getTotalIn(strm);
    }

    /**
     * Returns the total number of bytes output so far.
     * @return the total number of bytes output so far
     */
    public synchronized int getTotalOut() {
	if (strm == 0) {
	    throw new NullPointerException();
	}
	return getTotalOut(strm);
    }

    /**
     * Resets inflater so that a new set of input data can be processed.
     */
    public synchronized void reset() {
	if (strm == 0) {
	    throw new NullPointerException();
	}
	reset(strm);
	finished = false;
	needDict = false;
	off = len = 0;
    }

    /**
     * Closes the decompressor and discards any unprocessed input.
     * This method should be called when the decompressor is no longer
     * being used, but will also be called automatically by the finalize()
     * method. Once this method is called, the behavior of the Inflater
     * object is undefined.
     */
    public synchronized void end() {
	if (strm != 0) {
	    end(strm);
	    strm = 0;
	}
    }

    /**
     * Closes the decompressor when garbage is collected.
     */
    protected void finalize() {
	end();
    }

    private native static void initIDs();
    private native static long init(boolean nowrap);
    private native static void setDictionary(long strm, byte[] b, int off,
					     int len);
    private native int inflateBytes(byte[] b, int off, int len)
	    throws DataFormatException;
    private native static int getAdler(long strm);
    private native static int getTotalIn(long strm);
    private native static int getTotalOut(long strm);
    private native static void reset(long strm);
    private native static void end(long strm);
}
