#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>

#include "gatr3d.h"
#include "tr3d.h"

double evaluateTruss(int &nb, int &nn, int &ns, double *R, double *P, 
		int *NM, int *NP, double *A, double *Fx, double *L, double diter, 
		int &feasible, bool echo);
double getVolume(int &nb, double *A, double *L);
void setLength(int &nb, double *R, int *NM, int *NP, double *L);

double sigma_allow = 36.0;

int main() {	
	int i,j,k,n,ii,utime,ngen,npop,p1,p2,imin,echo,nfeasible;
	long ltime;
	double d,Ai,sum,min,max;
	double *divs, *func;
	int nb,nn,ns,na;
	double *R, *P, *A, *sizes, *Fx, *L;
	int *NM, *NP, *population, *pnew;

	// initialize parameters
	npop = 15;
	ngen = 500;
	echo = 0;

	// seed rand with current time
	ltime = time(NULL);
	utime = (unsigned int) ltime/2;
	srand(utime);
	// showRandom();

	// open file, read the basic model, and compute
	// member length (once)
	FILE *file;
	if((file = fopen("in.txt", "r")) == NULL) {
		printf("*** Unable to open file");
		exit(0);
	}
	// problem dimensions
	fscanf(file, "%5d%5d%5d%5d", &nb, &nn, &ns, &na);
	if(echo) printf("%5d%5d%5d%5d\n", nb, nn, ns, na);
	// allocate space
	R = new double[3*nn];
	P = new double[3*nn];
	A = new double[nb];
	NM = new int[nb];
	NP = new int[nb];
	n = pow(2, na);
	sizes = new double[n];
	n = readFile(file, nb, nn, na, R, P, NM, NP, A, sizes, echo>0);
	// printf("gatr3d: parameters in main\n");
	// printf("%5d%5d%5d%5d\n", nb, nn, ns, na);
	// printf("gatr3d: node 0 %f\t%f\t%f\n", R[0], R[1], R[2]);
	if(n < 0) {
		printf("*** Problem reading file");
		exit(0);
	}
	if(file != NULL) fclose(file);
	Fx = new double[nb];
	L = new double[nb];
	setLength(nb, R, NM, NP, L);

	// ga dimensions
	population 	= new int[npop*nb];
	pnew 		= new int[npop*nb];
	divs 		= new double[npop];
	func 		= new double[npop];

	// create initial population using random selections
	n = pow(2, na);
	for(i=0; i<npop; i++) {
		if(echo > 0) printf("\npopulation %d\n", i);
		for(j=0; j<nb; j++) {
			k = (int)(rand1()*n);
			population[i*nb + j] = k;
			// Ai = sizes[k];
			if(echo > 0) printf("\t%d\t%d\t%10.3f\n", j, k, sizes[k]);
		}
	}

	// begin iterations
	for(ii=0; ii<ngen; ii++) {
		if(echo > 0) printf("\nbegin generation %d\n", ii);
		// evaluate fitness
		imin = -1;
		max = 0.0;
		nfeasible = 0;
		for(i=0; i<npop; i++) {
			for(j=0; j<nb; j++) {
				// printf("\t%d\n", population[i*nb+j] );
				A[j] = sizes[ population[i*nb + j] ];
			}
			// evaluate this design
			func[i] = evaluateTruss(nb, nn, ns, R, P, NM, NP, A, 
					Fx, L, (double)ii/ngen, j, echo > 3);
			nfeasible += j;
			if(echo > 1) printf("obj %d\t%10.3e\n", i, func[i]);
			// sum += func[i];
			// divs[i] = sum;
			if(func[i] > max) {
				max = func[i];
			}
			if(imin == -1 || func[i] < min) {
				min = func[i];
				imin = i;
			}
		}

		// printf("normalize\n");
		// Normalize fitness divisions to [0,1] for roulette-wheel
		// selection and change maximization to minimization.
		sum = 0.0;
		for(i=0; i<npop; i++) {
			sum += max - func[i];
			divs[i] = sum;
		}
		if(sum == 0.0) {
			printf("\n***\n*** Generation %d\n", ii);
			printf("*** No variation in the population.\n");
			d = getVolume(nb, A, L);
			printf("*** best solution obj %10.3f, vol %10.3f\n", min, d);
			printf("*** Stop.\n***\n");
			exit(0);
		}
		// printf("sum %10.5f\n", sum);
		for(i=0; i<npop; i++) {
			divs[i] /= sum;
		}

		// show best solution at this iteration
		if(ii == ngen-1 || echo > 2) {
			for(j=0; j<nb; j++) {
				A[j] = sizes[ population[imin*nb + j] ];
			}
			evaluateTruss(nb, nn, ns, R, P, NM, NP, A, Fx, L, 0, j, true);
			d = getVolume(nb, A, L);
			printf("\niter %d found %d feasible solutions (%2.0f%%)\n",
				ii, nfeasible, (double)nfeasible/npop*100.0);
			printf("best solution at index %d, obj %10.3f, vol %10.3f\n",
					imin, min, d);
		}

		// save best solution
		n = imin*nb;
		for(i=0; i<nb; i++) {
			pnew[i] = population[n+i];
		}

		// create new generation
		for(i=1; i<npop; i++) {
			// select two parents
			p1 = selectParent(divs, npop);
			p2 = selectParent(divs, npop);
			// printf("parents %d and %d\n", p1, p2);
			reproduce(population, pnew, i*nb, p1*nb, p2*nb, nb, na);
		}

		// printf("copy new generation\n");
		for(i=0; i<npop*nb; i++) {
			population[i] = pnew[i];
			// printf("\t%d\t%d\n", i, population[i]);
		}
		// printf("done\n");

	} // next generation

	// last solution
	/*
	printf("last population\n");
	for(i=0; i<npop*nb; i++) {
		printf("\t%d\t%d\n", i, population[i]);
	}
	*/

	printf("Stop. Program Terminated.\n");
	return 0;
}

/********************************************************
 * Evaluate a 3D truss model subject to allowable-stress 
 * constraints.
 */
double evaluateTruss(int &nb, int &nn, int &ns, double *R, double *P, 
		int *NM, int *NP, double *A, double *Fx, double *L, double diter,
		int &feasible, bool echo) {
	int i;
	double a,sig,v;

	// printf("evaluateTruss: begin\n");
	feasible = 1;
	analyze(nb, nn, ns, R, P, NM, NP, A, L, Fx);
	v = getVolume(nb, A, L);

	// add separation to feasible solutions and compute
	// constraint violation factor
	diter *= 50;
	v *= pow(1.05, (diter+1));
	a = pow(2.0, diter+1);

	// check for constraint violation
	for(i=0; i<nb; i++) {
		sig = fabs(Fx[i])/A[i];
		if(echo) printf("\t%d\t%10.3f\t%10.3f\t%10.3f\n",
			i, Fx[i], A[i], sig);
		if(sig > sigma_allow) {
			v += a*(sig - sigma_allow);
			// v *= a*sig/sigma_allow;
			feasible = 0;
		}
	}
	return v;
}

/*********************************************************
 * Compute the volume of a 3D truss.
 */
double getVolume(int &nb, double *A, double *L) {
	int i;
	double v = 0.0;
	for(i=0; i<nb; i++) {
		v += A[i]*L[i];
	}
	return v;
}

/********************************************************
 * Generate a real random number [0,1]
 */
double rand1() {
	return rand()/(double)RAND_MAX;
}

/*****************************************
 * Create new entry in 'pnew' at index 'ind' using parent indices
 * 'p1' and 'p2' from 'population'.
 * 	population	array of indices for population
 * 	pnew		new array of indices for population
 * 	ind		index in pnew to begin placing vars
 * 	p1		index in population of first parent
 * 	p2		index in population of second parent
 * 	nvar		number of variables
 * 	bpv		number of bits per variable
 */
void reproduce(int *population, int *pnew, int ind, int p1, int p2,
	       	int nvar, int bpv) {
	int i,j,k,ii,n,p,redo,ibit,val,imax;
	int nbits;
	int *splt;
	double eps = 0.005;

	nbits = nvar*bpv;

	// determine number and location of split point(s)
	// n = 5;
	n = 0.25*nvar;
	int nsplt = rand1()*n;
	splt = new int[nsplt];
	for(i=0; i<nsplt; i++) {
		splt[i] = (int)(rand1()*nbits);
	}

	// put split points in order
	do {
		redo = 0;
		for(i=1; i<nsplt; i++) {
			if(splt[i-1] > splt[i]) {
				j = splt[i-1];
				splt[i-1] = splt[i];
				splt[i] = j;
				redo = 1;
			}
		}
	} while(redo);

	// create new member
	k = 0;
	p = p1;
	ibit = bpv;
	for(i=0; i<nvar; i++) {
		val = population[p+i];
		while(ibit >= splt[k] && k < nsplt-1) {
			// change parents and split this var
			// printf("split %d at index %d\n", k, splt[k]);
			p = p == p1 ? p2 : p1;
			ii = ibit - splt[k];
			val = splitVars(val, population[p+i], ii);
			k++;
		}
		for(j=0; j<bpv; j++) {
			if(rand1() < eps) {
				// printf("mutate bit %d of var %d, val %d\n", 
						// j, i, val);
				n = pow(2, j);
				if((val & n) == n) {
				       // printf("subtract %d from %d\n",
					       // n, val);
			       	       val -= n;
				} else val += n;
			}
		}
		ibit += bpv;
		if(val < 0) {
			printf("***\n*** val %d out of range.\n*** Stop\n***",
					val);
			exit(0);
		}
		pnew[ind+i] = val;
	}
	delete(splt);
}

/*****************************************
 * Return an index in divs according to a randomly selected number
 * and the intervals in divs. Parameter divs should be [0,1] and
 * in increasing order. Parameter n is the length of divs.
 */
int selectParent(double *divs, int n) {
	int i;
	double d;
	d = rand1();
	i = 0;
	do {
		if(divs[i] > d) return i;
		i++;
	} while(i < n);
	// printf("*** selectParent: unable to find value in divs greater than %10.5f\n", d);
	return n-1;
}

/******************************************
 * Compute and set the length of all given elements.
 */
void setLength(int &nb, double *R, int *NM, int *NP, double *L) {
	int i,j;
	double d,dd;
	for(i=0; i<nb; i++) {
		dd = 0.0;
		for(j=0; j<3; j++) {
			d = R[3*NP[i]+j] - R[3*NM[i]+j];
			dd += d*d;
		}
		L[i] = sqrt(dd);
		// printf("setLength\t%d\t%10.4f\n", i, L[i]);
	}
}

/******************************************
 * Generate some random number and direct to terminal
 * to test generator
 */
void showRandom() {
	int i,n;
	double d;
	printf("random numbers from 0 to %d\n", RAND_MAX);
	for(i=0; i<10; i++) {
		n = rand();
		d = n/(double)RAND_MAX;
		printf("\t%d\t%8.5f\n", n, d);
	}
}

/******************************************
 * Combine vars v1 and v2 splitting them at point ii and
 * return the result. Use the beginning of v1 and the end
 * of v2.
 */
int splitVars(int v1, int v2, int ii) {
	int i,n,val;

	// clear bits 0 to ii
	val = v2;
	// val = val >> ii;
	// val = val << ii;
	val >>= ii;
	val <<= ii;

	// use bits 0 to ii from v1
	n = 0;
	for(i=0; i<ii; i++) {
		val += (v1 & n);
		// n = n << 1;
		n <<= 1;
	}
	// printf("splitVars %d and %d at %d make %d\n", v1, v2, ii, val);

	return val;
}
