/******************************************************
 * This file contains functions to work with a 3D truss
 */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "tr3d.h"
#include "cmath.h"


static double E_MOD = 29000.0;

/* This function assembles the stiffness matrix, Ke, which 
 * should already be allocated.
 * 	nfree	number of free nodes
 * 	R	coordinate array
 */

int assemble(int nb, int nfree, double *R, int *NM, int *NP, double *A,
		double *L, double *Ke, double *vec) {
	 
	int i,j,k,ii,dof,n1,n2,b,row,col;
	double ki;
	double *uv = new double[3];
	double *nkn = new double[9];

	// System.out.println("begin assemble");
	dof = 3*nfree;

	for(i=0; i<dof*dof; i++) Ke[i] = 0.0; 
	for (b=0; b<nb; b++) {
		n1 = 3*NM[b];
		n2 = 3*NP[b];
		unitv(n1, n2, R, uv);
		j = 3*b;
		for(i=0; i<3; i++) {
			vec[j+i] = uv[i];
		}
		// printf("tr3d::assemble bar %d, area %5.3f\n", b, A[b]);
		ki = E_MOD*A[b]/L[b];
		for(i=0; i<3; i++) {
			for(j=0; j<3; j++) {
				nkn[3*i+j] = ki*uv[i]*uv[j];
			}
		}
		if(n1 < dof) {
			// NM is a free node
			for(i=0; i<3; i++)
			   for(j=0; j<3; j++) {
				row = n1;
				col = n1;
				Ke[dof*(row+i)+col+j] += nkn[3*i+j];
			   }
			if(n2 < dof) {
			   // both NM and NP are free nodes
			   for(i=0; i<3; i++)
			      for(j=0; j<3; j++) {
				row = n1;
				col = n2;
				Ke[dof*(row+i)+col+j] += -nkn[3*i+j];
				Ke[dof*(row+i)+col+j] += Ke[i*dof+j];
			     }
			}
		} 

		if(n2 < dof) {
			// NP is a free node (but not NM)
			for(i=0; i<3; i++)
			   for(j=0; j<3; j++) {
				row = n2;
				col = n2;
				Ke[dof*(row+i)+col+j] += nkn[3*i+j];
			   }
		}
	}

	// show matrix
	/*
	printf("tr3d::assemble Ke\n");
	for(i=0; i<dof; i++) {
		for(j=0; j<dof; j++) {
			printf("\t%d\t%d\t%10.3f\n", i, j, Ke[dof*i+j]);
		}
	}
	*/

	// free memory
	delete(uv);
	delete(nkn);

	return 0;
}

int analyze(int &nb, int &nn, int &ns, double *R, double *P, int *NM, int *NP, double *A,
		double *L, double *Fx) {
	int i,j,k,ii,nfree,dof,ib;
	double dl,fac;

	// matrix x(dof, 1);
	// matrix b(dof, 1);
	double *x, *b, *Ke, *vec;
	// b = new double[dof];
	nfree = nn - ns;
	dof = 3*nfree;
	x = new double[dof];
	Ke = new double[dof*dof];
	vec = new double[3*nb];
	// printf("tr3d::analyze begin\n");

	// assemble system matrix
	assemble(nb, nfree, R, NM, NP, A, L, Ke, vec);
	// printf("tr3d::analyze done assemble\n");

	// solve the system
	solve(Ke, P, x, dof);
	// printf("tr3d::analyze done solve\n");

	// printf("tr3d::analyze solution:\n\tdof\tdisp\n");
	// for(i=0; i<dof; i++) printf("\t%d\t%10.4f\n", i, x[i]);

	//  compute member forces and displacements from joint displacements
	for(i=0; i<nb; i++) {
		dl = 0.0;
		ib = 3*i;
		ii = NM[i];
		fac = -1.0;
		for(j=0; j<2; j++) {
			if(ii < nfree) {
				ii *= 3;
				// printf("\t%d\t%10.3f\t%10.3f\t%10.3f\n",
						// i, vec[ii], vec[ii+1], vec[ii+2]);
				for(k=0; k<3; k++) {
					dl += fac * x[ii+k]*vec[ib+k];
				}
			}
			fac = 1.0;
			ii = NP[i]; 
		}
		Fx[i] = A[i]*E_MOD/L[i] * dl;
		// printf("\t%d\t%10.3f\t%10.3f\n", i, dl, Fx[i]);
	}

	// free memory
	delete(x);
	delete(Ke);
	delete(vec);

	return 0;
}

/*************************************************
 * Compute a unit vector from indices n1 and n2 in 
 * coordinate array R and return in v
 * 	n1	index in R of node 1 (multiple of 3)
 * 	n2	index in R of node 2 (multiple of 3)
 * 	R	coordinate array
 * 	v	unit vector computed here
 */
void unitv(int n1, int n2, double *R, double *v) {
	int i;
	double d;

	for(i=0; i<3; i++) {
		v[i] = R[n2+i] - R[n1+i];
		d += v[i]*v[i];
	}
	if(d <= 0.0) return;
	d = sqrt(d);
	for(i=0; i<3; i++) {
		v[i] /= d;
	}
}

/********************************************************
 * Read a 3D truss model and available sizes for optimization.
 */
int readFile(FILE *f, int &nb, int &nn, int &na, double *R, double *P, int *NM, int *NP, double *A, double *sizes, bool echo) {
	int i,j,k,i3,n;
	if(f == NULL) {
		printf("tr3d:readFile f is null");
		return -1;
	}

	// problem dimensions
	/*
	fscanf(f, "%5d%5d%5d%5d", &nb, &nn, &ns, &na);
	printf("%5d%5d%5d%5d\n", nb, nn, ns, na);

	// allocate space
	R = new double[3*nn];
	P = new double[3*nn];
	A = new double[nb];
	NM = new int[nb];
	NP = new int[nb];
	*/

	// nodes
	for(i=0; i<nn; i++) {
		i3 = 3*i;
		fscanf(f, "%d%lf%lf%lf%lf%lf%lf", &n,
				&R[i3], &R[i3+1], &R[i3+2], &P[i3], &P[i3+1], &P[i3+2]);
		if(echo) printf("%5d%10.3f%10.3f%10.3f%10.3f%10.3f%10.3f\n", n,
				R[i3], R[i3+1], R[i3+2], P[i3], P[i3+1], P[i3+2]);
		if(n != i+1) {
			printf("*** Input file must have consecutive node numbers beginning at 1\n");
			printf("*** problem encountered at node %d, index %d\n", n, i);
			exit(0);
		}
	}

	// members
	for(i=0; i<nb; i++) {
		fscanf(f, "%d%d%d%lf", &n, &NM[i], &NP[i], &A[i]);
		if(echo) printf("%5d%5d%5d%10.3f\n", n, NM[i], NP[i], A[i]);
		NM[i]--;
		NP[i]--;
		if(n != i+1) {
			printf("*** Input file must have consecutive member numbers beginning at 1\n");
			printf("*** problem encountered at member %d, index %d\n", n, i);
			exit(0);
		}
	}

	// available sizes
	n = pow(2, na);
	i = 0;
	do {
		fscanf(f, "%lf", &sizes[i]);
		if(feof(f) ) {
			i--;
		} else {
			if(echo) printf("%10.3f\n", sizes[i]);
		}
		i++;
	} while(i < n && f != NULL && !feof(f) );

	// pad sizes with beginning of list if not enough sizes given
	if(i < n) {
		k = 0;
		for(j=i; j<n; j++) {
			sizes[j] = sizes[k++];
			if(echo) printf("%10.3f\n", sizes[i]);
			// printf("pad entry %d from %d %10.3f\n", j, i, sizes[j]);
		}
	}
	// printf("tr3d:readFile node 0 %f\t%f\t%f\n", R[0], R[1], R[2]);
	if(echo) printf("done reading file\n\n");

	return 0;
}

