// file: Tr3d.java
import java.text.*;		// format numbers

/***********************************************
 * Class to work with 3D Truss objects. Currently there are no 
 * analysis routines included.
 */
public class Tr3d {

	int NB,NN,NS;
	TrussNode 	nodes[];
	TrussMember	bars[];

	/** bit-mask flag to indicate fixity */
	final static int NODE_FREE = 0;
	final static int NODE_PINNED = 7;

	/** basic constructor */
	public Tr3d() {
		NB = 0;
		NN = 0;
		NS = 0;
		nodes = null;
		bars = null;
	}

	/** basic constructor which allocates space */
	public Tr3d(int nb, int nn, int ns) {
		NB = nb;
		NN = nn;
		NS = ns;
		nodes = new TrussNode[nn];
		bars = new TrussMember[nb];
	}

	/** determine if the given node index is fixed */
	public boolean isFixed(int ind) {
		if(ind < 0 || ind >= NN) return false;
		return nodes[ind].fixity != NODE_FREE;
	}

	/****************************************************
	 * Return the index of a member with the given number. This
	 * will return the lowest number if there are duplicates.
	 */
	public int memberIndex(int n) {
		int i;
		for(i=0; i<NB; i++) {
			if(bars[i] != null && bars[i].number == n) return i;
		}
		return -1;
	}

	/****************************************************
	 * Return the index of a node with the given number. This
	 * will return the lowest number if there are duplicates.
	 */
	public int nodeIndex(int n) {
		int i;
		for(i=0; i<NN; i++) {
			if(nodes[i] != null && nodes[i].number == n) return i;
		}
		return -1;
	}

} // end of 'Tr3d' class definition


/** Class to handle truss nodes */
class TrussNode {
	/** node number */			int number;
	/** bit-masked flag for fixity */	int fixity;
 	/** node coordinates */			double x, y, z;
	/** node displacements */		double dx, dy, dz;
	/** applied loads */			double Px, Py, Pz;

	/** empty constructor */
	public TrussNode() { }

	/** create a new object from a String definition */
	public TrussNode(String s) {
		int i;
		if(s == null) return;
		this.number	= View3D.getInt(s, 0);
		i = 1;
		this.x		= View3D.getDouble(s, 10*i++);
		this.y		= View3D.getDouble(s, 10*i++);
		this.z		= View3D.getDouble(s, 10*i++);
	}
} // end of 'TrussNode' class definition


/** class to handle truss members */
class TrussMember {
	/** positive (+) node */	int NP;
	/** negative (-) node */ 	int NM;
	/** member number */ 		int number;
	/** bar area */ 		double area;
	/** Young's modulus */ 		double E;
	/** bar length */ 		double Length;
	/** axial bar force */ 		double Fx;
	/** bar stress */ 		double stress;
	/** change in length */ 	double dl;

	/** empty constructor */
	public TrussMember() { }

	/** construct a new object from a given string defintion */
	public TrussMember(String s) {
		if(s == null) return;
		this.number 	= View3D.getInt(s, 0);
		this.NM 	= View3D.getInt(s, 5);
		this.NP		= View3D.getInt(s, 10);
	}
}  // end of 'TrussMember' class definition
