// file: Tr3dCanvas.java
import java.awt.*;
import java.awt.event.*;
import java.text.*;						// DecimalFormat

/** class to draw a Tr3d object on a Canvas */
class Tr3dCanvas extends Canvas implements MouseListener, MouseMotionListener {

	// public variables
	/** drawing settings */ 	public boolean 	showPointNumber, showLineNumber,
							showAxes, axesAtOrigin;
	/** the object to draw */ 	public Tr3d 	tr3d;

	// private variables
	/** rotated coordinates */ 	private double 	rXYZ[][];
	/** rotation matrix */ 		private double 	R[][][];
	/** axes of system */ 		private double 	axes[][];
	/** rotation angle */ 		private double 	thetaX, thetaY, thetaZ;
	/** drawing parameter */ 	private double 	XSHIFT, YSHIFT, SCALE;
	/** usable plotting area */ 	private double 	AMAXX, AMAYY;
	/** drawing parameter */ 	private int 	canvasWidth, canvasHeight;
	/** mouse position */ 		private int 	lastMouseX, lastMouseY;

	/** type of mouse-click */ 	private boolean	middleButtonClick;
	/** color to draw with */ 	private Color 	lineColor,
							supportColor,
							pointNumberColor,
							lineNumberColor;
	/** text in top-left */		private String 	tlString;

	/** basic constructor */
	public Tr3dCanvas(int height, int width) {
		int i, j;
		R 		= new double[3][3][3];
		axes 		= new double[3][3];
		canvasWidth 	= width;
		canvasHeight 	= height;

		addMouseListener(this);
		addMouseMotionListener(this);

		// initialize booleans
		showLineNumber  	= false;
		showPointNumber 	= false;
		showAxes		= true;

		// initialize colors
		lineColor 		= Color.green;
		supportColor 		= Color.red;
		lineNumberColor 	= Color.yellow;
		pointNumberColor  	= Color.red;

		// usable plotting area
		AMAXX = canvasWidth - 80;
		AMAYY = canvasHeight - 80;

		// initialize rotation matrix
		for(i=0; i<3; i++) 
		   for(j=0; j<3; j++) 
		      R[i][i][j] = 1.0;
		setSize(canvasWidth, canvasHeight);
		setBackground(Color.black);
	}

	/** call appropriate drawing functions */
	public void paint(Graphics g) {
		int x,y,K,h,w;
		double x1,y1,r;

		if(tr3d == null || tr3d.NN == 0) {
			drawSplash(g);
			return;
		}

		drawObject( (Graphics2D)g );

		// draw the coordinate axes
		if(showAxes) {
			// determine shift point
			if(axesAtOrigin) {
				x = (int)(-XSHIFT);
				y = (int)(-YSHIFT);
			} else {
				x = 50;
				y = 60;
			}
			r = 35;

			// xyz are RGB
			((Graphics2D)g).setStroke(new BasicStroke((float)3.0));
			g.setColor(Color.red);
			x1 = x + r*axes[0][0];
			y1 = y + r*axes[0][1];
			g.drawLine( x, y, (int)x1, (int)y1);

			g.setColor(Color.green);
			x1 = x + r*axes[1][0];
			y1 = y + r*axes[1][1];
			g.drawLine( x, y, (int)x1, (int)y1);

			g.setColor(Color.blue);
			x1 = x + r*axes[2][0];
			y1 = y + r*axes[2][1];
			g.drawLine( x, y, (int)x1, (int)y1);

			((Graphics2D)g).setStroke(new BasicStroke((float)0.0));
		}
	}

	/******************************************
	 * Draw an opening figure when there is no model loaded.
	 */
	public void drawSplash(Graphics g) {
		int x0,y0;
		setBackground(Color.black);
		x0 = 100;
		y0 = 150;
		g.setColor(Color.cyan);
		g.setFont(new Font("Arial", Font.ITALIC, 30));
		g.drawString(View3D.programTitle, x0, y0);
	}

	/******************************************
	 * Force the object to be drawn in the size specified
	 * as in for printing to a page.
	 */
	public void drawObject(Graphics2D g, int xsize, int ysize) {
		// keep track of existing values
		int oldWidth = canvasWidth;
		int oldHeight = canvasHeight;
		resetSize(xsize, ysize);
		drawObject(g);
		resetSize(oldWidth, oldHeight);
	}

	/******************************************
	 * Called by 'paint' to draw the initial mesh
	 */
	public void drawObject(Graphics2D g) {
		int i,j,typ,K,M,r;
		double x1,y1,x2,y2;
		float r1;

		if(tr3d == null || tr3d.NN == 0) return;

		// Plot the mesh
		g.setColor(lineColor );
		for(i=0; i<tr3d.NB; i++) {
			K = tr3d.bars[i].NM;
			M = tr3d.bars[i].NP;
			K = tr3d.nodeIndex(K);
			M = tr3d.nodeIndex(M);
			if(K >= 0 && M >= 0) {
				x1 = (rXYZ[K][0] / SCALE) - XSHIFT;
				y1 = (rXYZ[K][1] / SCALE) - YSHIFT;
				x2 = (rXYZ[M][0] / SCALE) - XSHIFT;
				y2 = (rXYZ[M][1] / SCALE) - YSHIFT;
				g.drawLine( (int)x1, (int)y1, (int)x2, (int)y2);
			}
		}

		// Show points and/or numbers
		r1 = 2;
		g.setColor(Color.cyan);
		if(r1 > 0 || showPointNumber) {
			for(i=0; i<tr3d.NN; i++) {
				x1 = (rXYZ[i][0] / SCALE) - XSHIFT;
				y1 = (rXYZ[i][1] / SCALE) - YSHIFT;
				if(showPointNumber) {
					g.drawString( " "+
						tr3d.nodes[i].number,(int)x1, (int)y1);
				}
				// show nodes as dots
				// if(r1 > 0) {
					// g.setColor(Color.blue);
					// g.fillOval( (int)(x1-r1), (int)(y1-r1), 
						// (int)(2*r1), (int)(2*r1) );
				// }
			}
		}

		// Show line numbers
		if(showLineNumber) {
		  g.setColor(lineNumberColor);
		  for(i=0; i<tr3d.NB; i++) {
			K = tr3d.bars[i].NM;
			M = tr3d.bars[i].NP;
			K = tr3d.nodeIndex(K);
			M = tr3d.nodeIndex(M);
			if(K >= 0 && M >= 0) {
				x1 = (rXYZ[K][0] / SCALE) - XSHIFT;
				y1 = (rXYZ[K][1] / SCALE) - YSHIFT;
				x2 = (rXYZ[M][0] / SCALE) - XSHIFT;
				y2 = (rXYZ[M][1] / SCALE) - YSHIFT;
				x1 = (x1 + x2)/2;
				y1 = (y1 + y2)/2;
				g.drawString( " "+
					tr3d.bars[i].number, (int)x1, (int)y1);
			}
		  }
		}

		// Show the filename tag at the top left
		if(tlString != null) {
			g.setColor(Color.white);
			g.drawString(tlString,10, 20);
		}
	}

	// required for MouseListener
	public void mouseClicked( MouseEvent e)  { }
	public void mouseEntered( MouseEvent e)  { }
	public void mouseExited( MouseEvent e)   { }
	public void mouseMoved( MouseEvent e)    { }
	public void mouseReleased( MouseEvent e) { }

	/** Respond to mouse drag events */
	public void mouseDragged( MouseEvent e) {
		int dx, dy;
		double angle;

		if(tr3d == null || (tr3d.NN == 0 && tr3d.NB == 0) ) return;

		dx = e.getX() - lastMouseX;
		dy = e.getY() - lastMouseY;
		lastMouseX = e.getX();
		lastMouseY = e.getY();

		if(middleButtonClick) {
			angle = dx / 100.0;
			addRotation(angle);
		} else {
			addRotation(dx, dy);
		}

		repaint();
	}

	/******************************************
	 * Respond to mouse clicks. Call methods based on 'mouseFunction'.
	 */
	public void mousePressed( MouseEvent e) {
		int n;
		long dt;
		lastMouseX = e.getX();
		lastMouseY = e.getY();
		middleButtonClick = false;

		if(tr3d == null || (tr3d.NN == 0 && tr3d.NB == 0) ) return;
		if(e.isAltDown() ) {
			middleButtonClick = true;
		}
	}

	/***************************************************
	 * Create three rotation matrices, R[][][0] for thetaX, 1 for thetaY
	 * and 2 for thetaZ.
	 */
	private void mkRotationMatrix() {

		R[0][0][0] = 1.0;
		R[1][1][0] = Math.cos(thetaX);
		R[1][2][0] = Math.sin(thetaX);
		R[2][1][0] = -Math.sin(thetaX);
		R[2][2][0] = Math.cos(thetaX);

		R[0][0][1] = Math.cos(thetaY);
		R[0][2][1] = -Math.sin(thetaY);
		R[1][1][1] = 1.0;
		R[2][0][1] = Math.sin(thetaY);
		R[2][2][1] = Math.cos(thetaY);

		R[0][0][2] = Math.cos(thetaZ);
		R[0][1][2] = Math.sin(thetaZ);
		R[1][0][2] = -Math.sin(thetaZ);
		R[1][1][2] = Math.cos(thetaZ);
		R[2][2][2] = 1.0;

	}

	/***************************************************
	 * Set the rotated coordinates 'rXYZ' from the (unchanging) set of
	 * coordinates in the tr3d object.
	 * When looking at the screen, consider it to be in a local xy plane 
	 * and the viewer is looking along the z-axis.
	 * The (global) rotation matrix <b>R</b> should already be set.
	 */
	private void rotate() {
		int i, j, k;
		double temp[] = new double[3];
		if(tr3d == null) return;

		// allocate space if needed
		if(rXYZ == null || rXYZ.length < tr3d.NN) 
			rXYZ = new double[tr3d.NN][3];

		// initialize to un-rotated coordinates
		for(i=0; i<tr3d.NN; i++) {
			rXYZ[i][0] = tr3d.nodes[i].x;
			rXYZ[i][1] = tr3d.nodes[i].y;
			rXYZ[i][2] = tr3d.nodes[i].z;
		}

		// initialize coordinate axes
		for(i=0; i<3; i++) {
			for(j=0; j<3; j++) {
				axes[i][j] = 0.0;
			}
			axes[i][i] = 1.0;
		}

		// rotate the object
		for(i=0; i<tr3d.NN; i++) {
		  for(k=0; k<3; k++) {		// do rotation about x, y, and z axis
		    for(j=0; j<3; j++) temp[j] = R[j][0][k]*rXYZ[i][0] 
			+ R[j][1][k]*rXYZ[i][1] + R[j][2][k]*rXYZ[i][2];
		    for(j=0; j<3; j++) rXYZ[i][j] = temp[j];
		  }
		}

		// rotate the coordinate axes
		for(i=0; i<3; i++) {
		  for(k=0; k<3; k++) {
		    for(j=0; j<3; j++) temp[j] = R[j][0][k]*axes[i][0] 
			+ R[j][1][k]*axes[i][1] + R[j][2][k]*axes[i][2];
		    for(j=0; j<3; j++) axes[i][j] = temp[j];
		  }
		}

		setRange();
	}

	/***************************************************
	 * Change 'thetaX' and 'thetaY' according to mouse motion (dx,dy).
	 * This will update the rotation matrix 'R', and call method 'rotate'.
	 */
	private void addRotation(int dx, int dy) {
		// System.out.println(" drag dx = " + dx + ", dy = " + dy );
		// note: 0.17 rad ~ 10 degrees
		thetaX += -dy / 100.0;
		thetaY += dx / 100.0;
		mkRotationMatrix();
		rotate();
	}

	/***************************************************
	 * Add 'angle' to variable 'thetaZ', update the rotation 
	 * matrix 'R', and call method 'rotate'.
	 */
	private void addRotation(double angle) {
		// Rotation about z-axis
		thetaZ += angle;
		mkRotationMatrix();
		rotate();
	}

	/** Return the current rotation angles in degrees */
	public double[] getRotation() {
		double r[] = new double[3];
		r[0] = thetaX * 180.0 / Math.PI;
		r[1] = thetaY * 180.0 / Math.PI;
		r[2] = thetaZ * 180.0 / Math.PI;
		return r;
	}

	/***************************************************
	 * Explicily call the 'rotate' method.
	 */
	public void refresh() {
		rotate();
	}

	/***************************************************
	 * Reset the size of this object. Should be called explicitly
	 * when the container is resized.
	 */
	public void resetSize(int w, int h) {
		canvasWidth 	= w;
		canvasHeight 	= h;
		AMAXX = canvasWidth - 80;			// usable plotting area
		AMAYY = canvasHeight - 80;
		setRange();
		repaint();
	}

	/***************************************************
	 * Set the shift and scale factors for the object.
	 */
	private void setRange() {
		int i,j;
		double Xrange,Yrange;
		double max[] = new double[2];
		double min[] = new double[2];

		if(tr3d == null || tr3d.NN < 1) return;

		// get extremes
		for(i=0; i<2; i++) {
			max[i] = Double.NEGATIVE_INFINITY;
			min[i] = Double.POSITIVE_INFINITY;
		}
		for(i=0; i<tr3d.NN; i++) {
			for(j=0; j<2; j++) {
				if(rXYZ[i][j] > max[j]) max[j] = rXYZ[i][j];
				if(rXYZ[i][j] < min[j]) min[j] = rXYZ[i][j];
			}
		}

		// range of coordinates
		Xrange = max[0] - min[0];
		Yrange = max[1] - min[1];

		// scale so everything fits but x and y are proportional
		SCALE = Xrange/AMAXX > Yrange/AMAYY ?  Xrange/AMAXX : Yrange/AMAYY;

		// shift (in pixels) to align centroid of 2D object
		if(SCALE == 0.0) SCALE = 1.0;
		XSHIFT = (max[0]+min[0])/2.0/SCALE - canvasWidth/2;
		YSHIFT = (max[1]+min[1])/2.0/SCALE - canvasHeight/2;
	}

	/***************************************************
	 * Set the view to the specified rotation
	 * angles in degrees
	 */
	public void setRotation(double x, double y, double z) {
		thetaX = x * Math.PI / 180;
		thetaY = y * Math.PI / 180;
		thetaZ = z * Math.PI / 180;
		mkRotationMatrix();
		rotate();
		return;
	}

	/***************************************************
	 * Set the string to 's' that will be shown at the top-left of
	 * of the frame.
	 */
	public void setTopLeftString(String s) {
		tlString = s;
	}
}

//
// end Tr3dCanvas class
//
//////////////////////////////////////////////////////////
