// file: View3D.java
// Program to view a 3D object
//
import java.awt.*;
import java.io.*;
import java.text.*; 		// DecimalFormat
import java.awt.event.*;
import javax.swing.*;
import java.awt.print.*;

/** Main class for program. Contains GUI objects */
public class View3D extends Frame implements ActionListener, ComponentListener, 
	ItemListener, Printable, WindowListener { 

	private String 		currentFileName;
	private String		pwd;

	private CheckboxMenuItem viewNodeNums, viewMemberNums,
		viewAxes, viewAxesAtOrigin;
	private PageFormat 	pgFormat = new PageFormat();
	private Paper 		pgPaper = new Paper();

	// project-specific objects
	public Tr3d 			tr3d;
	public Tr3dCanvas 		drawingArea;

	// finals
	public final static String programTitle = "View3D Program 1.0";

	/** main program entry point */
	public static void main(String args[]) {
		View3D f = null;
		String s;

		f = new View3D();
		if(args != null && args.length > 0) {
			System.out.println("main: args " + args[0]);
			f.readNewFile(args[0]);
		}
	}

	// Constructor
	public View3D() {
		super(programTitle);
		int width  = 700;
		int height = 700;
		drawingArea = new Tr3dCanvas(height-50, width);

                // create menus
		Menu fileMenu   = new Menu("File");
		Menu viewMenu 	= new Menu("View");
		Menu helpMenu   = new Menu("About");

                // add items to the menus
		fileMenu.add("New");
		fileMenu.add("Open");
		fileMenu.add( new MenuItem("-"));
		fileMenu.add( "Print");
		fileMenu.add( new MenuItem("-"));
		fileMenu.add( "Exit");

		viewNodeNums 	= new CheckboxMenuItem("Node Numbers", false);
		viewMemberNums 	= new CheckboxMenuItem("Member Numbers", false);
		viewAxes 	= new CheckboxMenuItem( "View Axes (RGB=xyz)", 
				drawingArea.showAxes);
		viewAxesAtOrigin = new CheckboxMenuItem( "Place Axes At Origin",
				false);

		viewMenu.add(viewNodeNums);
		viewMenu.add(viewMemberNums);
		viewMenu.add( new MenuItem("-"));
		viewMenu.add(viewAxes);
		viewMenu.add(viewAxesAtOrigin);

		helpMenu.add("About");

                // add menus to menu bar
		MenuBar bar = new MenuBar();
                bar.add(fileMenu);
		bar.add(viewMenu);
		bar.add(helpMenu);
		setMenuBar(bar);

		// listen for user actions on these components
		fileMenu.addActionListener(this);
		viewMenu.addActionListener(this);
		helpMenu.addActionListener(this);
		viewNodeNums.addItemListener(this);
		viewMemberNums.addItemListener(this);
		viewAxes.addItemListener(this);
		viewAxesAtOrigin.addItemListener(this);
		addComponentListener(this);
		addWindowListener(this);

		// Setup the frame and show it
		add(drawingArea);
		setSize(width, height);
		newFile();
		setVisible(true);
	}

	// required for ItemListener
	public void itemStateChanged(ItemEvent e) {
		String s;
		Object o = e.getSource();

		// System.out.println("clicked on " + e.getItem());
		if(o == viewNodeNums) {
			drawingArea.showPointNumber = !drawingArea.showPointNumber;
			drawingArea.repaint();
		} else if(o == viewMemberNums) {
			drawingArea.showLineNumber = !drawingArea.showLineNumber;
			drawingArea.repaint();
		} else if(o == viewAxes) {
			drawingArea.showAxes = !drawingArea.showAxes;
			drawingArea.repaint();
		} else if(o == viewAxesAtOrigin) {
			drawingArea.axesAtOrigin = !drawingArea.axesAtOrigin;
			drawingArea.repaint();
		}
	}

	// required for ComponentListener
	public void componentHidden(ComponentEvent e) { }
	public void componentMoved(ComponentEvent e) { }
	public void componentResized(ComponentEvent e) {
		drawingArea.resetSize(this.getWidth(), this.getHeight()-50 );
	}
	public void componentShown(ComponentEvent e) { }

	// required for WindowListener
	public void windowClosed(WindowEvent event) { }
	public void windowDeiconified(WindowEvent event) { }
	public void windowIconified(WindowEvent event) { }
	public void windowActivated(WindowEvent event) { }
	public void windowDeactivated(WindowEvent event) { }
	public void windowOpened(WindowEvent event) { }
	public void windowClosing(WindowEvent e) { 
		setVisible(false);
		dispose();
		System.exit(0); 
	}

	/** respond to actions */
	public void actionPerformed(ActionEvent e) {
		// System.out.println("View3D::actionPerformed");
		// Menu Items
		if(e.getSource() instanceof MenuItem) {
		  String cmd = e.getActionCommand();
		  // File menu items
		  if (cmd == "New") {
			  newFile();
		  } else if (cmd == "Open") {
			MyFiles m = new MyFiles(this, FileDialog.LOAD, pwd);
			if(m.fileName != "") {
				pwd = m.pwd;
				System.out.println("Reading " + m.fullName);
				readNewFile(m.fullName);
			}
		  } else if (cmd == "Print") {
		  	printImage(0);

		  } else if (cmd == "Exit") {
			setVisible(false);
			dispose();
			System.exit(0);

		  // View menu items
		  } else if (cmd == "Node Numbers") {
			drawingArea.showPointNumber = !drawingArea.showPointNumber;
		  } else if (cmd == "Member Numbers") {
			drawingArea.showLineNumber = !drawingArea.showLineNumber;

		  // Help menu
		  } else if (cmd == "About") {
			postAbout();
		  }
		}
		drawingArea.repaint();
	}

	/***********************************************************
	 * Post the About dialog.
	 */
	public void postAbout() {
		JOptionPane.showMessageDialog(this, 
			programTitle + "\nby K. M. MacBain and W. R. Spillers"
			+ "\n\nProgram to view a 3D Object.",
			programTitle, JOptionPane.PLAIN_MESSAGE);
	}

	/** print the image */
	public int printImage(int flag) {
		// This method will set the 'Paper' to simulate
		// either full-page (8.5 x 11) or "image" which is
		// defined here. For 'flag' = 0, full-page is used.
		
		// pgPaper = new Paper();
		if(flag == 1) {
			// System.out.println("printImage:: flag = 1");
			// setSize(double width, double height)
			double w = 5.0;
			double h = 5.0;
			double margin = 0.5;
			pgPaper.setSize( w*72.0, h*72.0);
			pgPaper.setImageableArea( margin*72.0, margin*72.0,
			(w-2*margin)*72.0, (h-2*margin)*72.0);
		}

		PrinterJob pj = PrinterJob.getPrinterJob();
		// pgFormat.setOrientation(PageFormat.LANDSCAPE);
		pgFormat.setPaper(pgPaper);
		pgFormat = pj.validatePage(pgFormat);
		// pj.setPrintable(this);
		pj.setPrintable(this, pgFormat );

		if (pj.printDialog()) {
			try {
				pj.print();
			} catch (Exception ex) {
				ex.printStackTrace();
			}
		}
		return 0;
	}
	
	/** print the image */
	public int print(Graphics g, PageFormat pf, int pi)
			throws PrinterException {
		if (pi >= 1) {
		    return Printable.NO_SUCH_PAGE;
		}

		pf = pgFormat;
		pf.setPaper(pgPaper);

		// Shift and scale this object to fit on a page.
		// System.out.println(" pf img " + pf.getImageableHeight()+" "
		// 	+pf.getImageableWidth() );
		// System.out.println(" pf w x h " + pf.getWidth()+" "+ pf.getHeight() );
		// System.out.println(" paper w x h " + pgPaper.getWidth()+" "
		// 	+ pgPaper.getHeight() );
		g.translate( (int)pf.getImageableX(), (int)pf.getImageableY() );

		if(pf.getOrientation() == PageFormat.LANDSCAPE) {
			double r[] = new double[3];
			r = drawingArea.getRotation();
			r[2] -= 45.0;
			drawingArea.setRotation(r[0], r[1], r[2]);
			drawingArea.drawObject((Graphics2D) g, (int)pf.getImageableWidth(),
				(int)pf.getImageableHeight() );
			r[2] += 45.0;
			drawingArea.setRotation(r[0], r[1], r[2]);
		} else {
			drawingArea.drawObject((Graphics2D) g, (int)pf.getImageableWidth(),
				(int)pf.getImageableHeight() );
		}
		
		return Printable.PAGE_EXISTS;
	}
  
	/*************************************************************
	 * Return the value in a 10-character portion of the given 
	 * string parsed as a double starting at <b>start</b>.
	 */
	static public double getDouble(String line, int start) {
		return getDouble(line, start, 10);
	}

	/*************************************************************
	 * Return the value in a portion of the given string parsed as a
	 * double starting at <b>start</b>.
	 */
	static public double getDouble(String line, int start, int length) {
		int n;
		int end;
		String s;
		double d;

		if(start >= line.length() ) {
			return 0;
		}

		end = start + length;
		if(end > line.length() ) end = line.length() ;

		s = line.substring(start, end);
		s = s.trim();
		try {
			d = Double.parseDouble(s);
			return d;
		} catch(Exception e) {
			return 0.0;
		}
	}

	/*************************************************************
	 * Return the value in a portion of the given string parsed as an int
	 * starting at <b>start</b> including 5 characters (I5).
	 */
	static public int getInt(String line, int start) {
		int n;
		int end;
		String s;

		if(start >= line.length() ) {
			return 0;
		}

		end = start + 5;
		if(end > line.length() ) end = line.length() ;

		s = line.substring(start, end);
		s = s.trim();
		try {
			n = Integer.parseInt(s);
			return n;
		} catch(Exception e) {
			return 0;
		}
	}

	/** Set workspace for a new file and read one. */
	public boolean readNewFile(String fileName) {
		newFile();
		if(readFile(fileName)) {
			currentFileName = fileName;
			updateDrawing();
			drawingArea.setRotation(180.0, 0.0, 0.0);
			setTitle(programTitle + ": " + currentFileName);
			return true;
		}
		return false;
	}

	/** Set workspace for a new file. */
	public void newFile() {
		System.out.println("View3D::newFile");
		tr3d = null;
		currentFileName = null;
		updateDrawing();
		drawingArea.repaint();
	}

	/** read a file */
	public boolean readFile(String fileName) {
		int i,nb,nn,ns;
		BufferedReader b = null;
		String s;

		// System.out.println("readFile(" + fileName + ")");
		try {
			b = new BufferedReader( new FileReader(fileName) );
		} catch(IOException e) {
			JOptionPane.showMessageDialog(this, "File Open Failed\n"
			  + e, "File Not Opened", JOptionPane.INFORMATION_MESSAGE);
			b = null;
			return false;
		}

		if(b == null) return false;
		// Read the file
		try {
			// first line is system size
			s = b.readLine();
			nb = getInt(s, 0);
			nn = getInt(s, 5);
			ns = getInt(s, 10);
			s = null;
			if(nb <= 0) {
				s = "There must be at least 1 member.";
			} else if(nn <= 1) {
				s = "There must be at least 2 nodes.";
			} else if(ns <= 2) {
				// s = "There must be at least 3 supports.";
			}
			if(s != null) {
				JOptionPane.showMessageDialog(this, s,
					"Invalid Input File", JOptionPane.ERROR_MESSAGE);
				return false;
			}
			tr3d = new Tr3d(nb, nn, ns);

			// read nodes
			for(i=0; i<nn; i++) {
				s = b.readLine();
				tr3d.nodes[i] = new TrussNode(s);
			}

			// read members
			for(i=0; i<nb; i++) {
				s = b.readLine();
				tr3d.bars[i] = new TrussMember(s);
			}

			b.close();
			drawingArea.tr3d = tr3d;
			updateDrawing();
			// System.out.println("View3D::readFile done");
			return true;
		} catch(Exception e) {
			System.err.println("View3D::readFile Exception\n"+e.toString() );
			JOptionPane.showMessageDialog(this,
				"Unable to fully read file. Check your input.",
				"Problem Reading", JOptionPane.WARNING_MESSAGE);
		}
		return false;
	}

	public void updateDrawing() {
		drawingArea.tr3d = tr3d;
		drawingArea.setTopLeftString("File: " + currentFileName);
		drawingArea.refresh();
		drawingArea.repaint();
	}
}

//
// 	end class View3D 
//
/////////////////////////////////////////////////////////////////////////////////
