#!/usr/bin/gawk -f
#
# This script parses an INVERT .plt-file and writes .dat-files, which are
# suitable for loading (label-files) or plotting (other files) by GNUPLOT.
# This version of the script only parses bodies of type SPHERE, CYLINDER,
# PLATE and PROFILE. See "INVERT User Guide" for a complete description of
# the .plt-file.
#
# Variables assignments expected on the command line:
# outdir:     Directory (which must exists) where output files are written to
#             (default: ".").
# dim:        Dimensionalty of resulting plot: 2 or 3 (i.e. 2D or 3D)
#             (default: 2)
# xmin, xmax,
# ymin,ymax:  Area of interest for label positions (default: 0). 
# seed:       Seed (integer) for random vertex selection defining possible
#             label positions (default: 0).
#
# Names of output files:
#  labels_in.dat,labels_out.dat: density(Mass) labels of input-/output model.
#  spheres_in.dat,spheres_out.dat
#  profiles_in.dat,profiles_out.dat
#  plates_in.dat,plates_out.dat
#
# Label files are always created or overwritten. Body files will only be
# created or overwritten for bodies available in the .plt-file.
#
# Dependencies on external programs:
#   gawk
#
# (C) 2008 Peter L. Smilde
#


#
# Return a 3D or 2D (selection of x or y by a selection function) label
# position string from given x/y/z coordinates.
#
function at(x,y,z) {
	if (dim==3) {
		return sprintf("%g,%g,%g", x,y,z);
	}

	else {
		return sprintf("selectxy(xyselector,%g,%g),%g", x,y,z);
	}
}


#
# Select a random vertex from coordinate array xa and ya (with na elements
# each) within the area of interest(defined by the global variables xmin,
# xmax, ymin and ymax). If the area has zero extension in one direction
# (indicated by global variables xrange/yrange <= 0) all vertices in that
# direction will be accepted. Output the selected vertex coordinates
# in global variables xclip and yclip and output the index of the selected
# vertex in global variable iclip.
#
function clipxy(na,xa,ya,   __i,__ia,__na)
{
	for (__i=1; __i<=na; ++__i) {
		__ia[__i]=__i;
	}

	__na = na;
	while (__na > 0) {
		__i = int(__na*rand())+1;
		iclip = __ia[__i];
		xclip = xa[iclip];
		yclip = ya[iclip];
		if ((xrange<=0 || (xclip > xmin && xclip < xmax)) &&
				(yrange<=0 || (yclip > ymin && yclip < ymax))) {
			break;
		}
		else {
			__ia[__i] = __ia[__na];
			delete __ia[__na];
			--__na;
		}
	}
#	exit

	if (xrange<=0) xclip=(xmin+xmax)/2;
	if (yrange<=0) yclip=(ymin+ymax)/2;
}

#
# Absolute value
#
function abs(x) {
	return (x<0) ? -x : x;
}

#
# Maximum value
#
function max(x,y) {
	return (x>y) ? x : y;
}

#
# Maximum distance between vertices in arrays xa/ya and corners of the area
# of interest.
#
function maxr(na,xa,ya,   __i)
{
	rx = max(abs(xmin-xa[1]),abs(xmax-xa[1]));
	ry = max(abs(ymin-ya[1]),abs(ymax-ya[1]));
	for (__i=2; __i<=na; ++__i) {
		rx = max(max(abs(xmin-xa[__i]),abs(xmax-xa[__i])),rx);
		ry = max(max(abs(ymin-ya[__i]),abs(ymax-ya[__i])),ry);
	}
	return sqrt(rx*rx + ry*ry);
}

function incrementcount(type) {
	if (isout) {
		if (type == 0) {
			++label_out_count;
		}
		else if (type == 4) {
			++sphere_out_count;
		}
		else if (type == 7) {
			++plate_out_count;
		}
		else if (type == 8) {
			++profile_out_count;
		}
	}
	else {
		if (type == 0) {
			++label_in_count;
		}
		else if (type == 4) {
			++sphere_in_count;
		}
		else if (type == 7) {
			++plate_in_count;
		}
		else if (type == 8) {
			++profile_in_count;
		}
	}
}

BEGIN {
	pi=3.1415926;

	if (length(outdir)==0) outdir="."

	if (dim!=3) dim=2;

	xrange = xmax - xmin;
	yrange = ymax - ymin;

	print "" > "test.dat"

	if (seed < 0) {
		srand();
	}
	else {
		srand(seed);
	}

#
# GNUPLOT line types (color) for labels of input/output model (fixed for now).
#
	ltin = 3;
	ltout = 1;

  isout=1;
	state = 0;

	bodynumber = 0;

	label_in_count = 0;
	label_out_count = 0;

	sphere_in_count = 0;
	sphere_out_count = 0;
	plate_in_count = 0;
	plate_out_count = 0;
	profile_in_count = 0;
	profile_out_count = 0;


#
# Define a GNUPLOT function to select the x- or y- coordinate for the position
# of 2D labels.
#
	if (dim==2) {
		print "selectxy(s,x,y) = (s==1) ? x : y\n" > outdir"/labels_in.dat";
		print "selectxy(s,x,y) = (s==1) ? x : y\n" > outdir"/labels_out.dat";
	}
	else {
		print "" > outdir"/labels_in.dat";
		print "" > outdir"/labels_out.dat";
	}
}

#
# Skip comment lines.
#
/^[!#]/ && NF>0 {
	 next;
}

#
# State 1 and 2 if new body is found.
#
/^OUTPUT$/ {
# Eventually restore seed of corresponding INPUT body to select the same
# vertex in OUTPUT body:
	if (isout) seed = rand()*2147483647;
	srand(seed);

	isout = 1;
	lskipl = " ";
	lskipr = "";
	ljust = "left";
	lt = ltout;
	labelfile=outdir"/labels_out.dat";

	state = 1;
	next;
}

/^INPUT$/ {
# Save new random seed for corresponding OUTPUT body:
	seed=rand()*2147483647;
	srand(seed);

	isout = 0;
	lskipl = "";
	lskipr = " ";
	ljust = "right";
	lt = ltin;
	labelfile = outdir"/labels_in.dat";

	state = 1;
	next;
}

state==1 {
	bodynumber = $1;
	print "BN: ",$1," seed: ",seed >> "test.dat"
	state = 2;
	next
}

#
# State 40 to 43 for subsequent SPHERE line types.
#
/^SPHERE$/ {
	if (isout) {
		bodyfile = outdir"/spheres_out.dat";
		if (sphere_out_count == 0) print "" > bodyfile;
	}
	else {
		bodyfile = outdir"/spheres_in.dat";
		if (sphere_in_count == 0) print "" > bodyfile;
	}

	state = 40;
	next;
}

state==40 {
	nob = $1;
	if (nob>0) {
		iob = 0;
		state = 41;
	}
	else {
		state = 0;
	}

	next;
}

state==41 {
	++iob;

	erd = $2;
	if (!erd) {
		r = $3;
	}
	else {
		d = $3;
	}

	state = 42;

	next;
}

state==42 {
	ns = $1;
	if (ns>0) {
		is = 0;
		state = 43;
	}
	else if (iob<nob) {
		state = 41;
	}
	else {
		state = 0;
	}

	next;
}

state==43 {
	++is;

	if (!erd) {
		d = $1;
	}
	else {
		r = $1;
	}

#
# Mass calculation from density and radius.
#
	m=d*4/3*pi*r^3;

	print $2,$3,$4,r,d >> bodyfile
	incrementcount(4);

	printf("set label \"%s%s%d=%g (M=%g)%s\" at %s front %s tc lt %d\n",
				 lskipl,erd?"r":"d",bodynumber,d,m,lskipr,at($2,$3,$4),ljust,lt) >> labelfile;
	incrementcount(0);

	if (is==ns) {
		if (iob<nob) {
			state = 41;
		}
		else {
			state = 0;
		}
	}

	next;
}

#
# State 70 to 73 for subsequent PLATE line types.
#
/^PLATE/ {
	if (isout) {
		bodyfile = outdir"/plates_out.dat";
		if (plate_out_count == 0) print "" > bodyfile;
	}
	else {
		bodyfile = outdir"/plates_in.dat";
		if (plate_in_count == 0) print "" > bodyfile;
	}

	state = 70;
	next;
}

state==70 {
	nob = $1;
	if (nob>0) {
		iob = 0;
		state = 71;
	}
	else {
		state = 0;
	}

	next;
}

state==71 {
	++iob;

	d    = $3;
	zmin = $4;
	zmax = $5;

	zl = (zmax+zmin)/2;

	state = 72;

	next;
}

state==72 {
	nv = $1;
	if (nv>0) {
		iv = 0;
		state = 73;
	}
	else if (iob<nob) {
		state = 71;
	}
	else {
		state = 0;
	}

	next;
}

state==73 {
	++iv;

	xa[iv] = $1;
	ya[iv] = $2;

	if (iv==nv) {
#
# Lower horizontal surface
#
		for (i=1; i<=nv; ++i) {
			print xa[i],ya[i],zmin,d >> bodyfile;
		}
		print "" >> bodyfile;

#
# Upper horizontal surface
#
		for (i=1; i<=nv; ++i) {
			print xa[i],ya[i],zmax,d >> bodyfile;
		}
		print "" >> bodyfile;

#
# Vertical surfaces
#
		for (i=1; i<=nv; ++i) {
			print xa[i],ya[i],zmin,d >> bodyfile;
			print xa[i],ya[i],zmax,d >> bodyfile;
			print "" >> bodyfile;
		}
		print "" >> bodyfile;

		incrementcount(7);

#
# Label
#
		clipxy(nv,xa,ya);
		printf("set label \"%sd%d=%g%s\" at %s front %s tc lt %d\n",
					 lskipl,bodynumber,d,lskipr,at(xclip,yclip,zl),ljust,lt) >> labelfile;
		incrementcount(0);

		if (iob<nob) {
			state = 71;
		}
		else {
			state = 0;
		}
	}

	next;
}

#
# State 80 to 83 for subsequent PROFILE line types.
#
/^PROFILE/ {
	if (isout) {
		bodyfile = outdir"/profiles_out.dat";
		if (profile_out_count == 0) print "" > bodyfile;
	}
	else {
		bodyfile = outdir"/profiles_in.dat";
		if (profile_in_count == 0) print "" > bodyfile;
	}

	state = 80;
	next;
}

state==80 {
	nob = $1;
	if (nob>0) {
		iob = 0;
		state = 81;
	}
	else {
		state = 0;
	}

	next;
}

state==81 {
	++iob;

	x0 = $2;
	y0 = $3;
	d  = $4;
	a  = $5;
	sa = sin(a/180*pi);
	ca = cos(a/180*pi);

	state = 82;

	next;
}

state==82 {
	nv = $1;
	if (nv>0) {
		iv = 0;
		state = 83;
	}
	else if (iob<nob) {
		state = 81;
	}
	else {
		state = 0;
	}

	next;
}

state==83 {
	++iv;

	xa[iv] = x0 + $1*sa;
	ya[iv] = y0 + $1*ca;
	za[iv] = $2;

	print xa[iv],ya[iv],$2,d >> bodyfile;

	if (iv==nv) {
		print "" >> bodyfile;

# #
# # Horizontal extension of vertices
# #
# 		r = maxr(nv,xa,ya);
# 		for (i=1; i<=nv; ++i) {
# 			print xa[i]+r*ca,ya[i]-r*sa,za[i],d >> bodyfile;
# 			print xa[i],ya[i],za[i],d >> bodyfile;
# 			print xa[i]-r*ca,ya[i]+r*sa,za[i],d >> bodyfile;
# 			print "" >> bodyfile;
# 		}
# 		print "" >> bodyfile;

		incrementcount(8);

		clipxy(nv,xa,ya);
		printf("set label \"%sd%d=%g%s\" at %s front %s tc lt %d\n",
					 lskipl,bodynumber,d,lskipr,at(xclip,yclip,za[iclip]),ljust,lt) >> labelfile;
		incrementcount(0);

		if (iob<nob) {
			state = 81;
		}
		else {
			state = 0;
		}
	}

	next;
}

#
# Return number of parsed labels and bodies for input and output model.
#
END {
	print \
		label_in_count, label_out_count, \
		sphere_in_count, sphere_out_count, \
		plate_in_count, plate_out_count, \
		profile_in_count, profile_out_count 
}
